package edu.cmu.cs.lti.letras.features;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

/**
 * Represents some combination of unique features via a bitmap, to allow for
 * very fast subset searching.
 */
public class FeatureGroup<T> {

	protected final long[] compositeMask;
	private final T value;

	/**
	 * @param features
	 * @param value
	 *            A value of any sort that might be associated with this "node"
	 */
	public FeatureGroup(Collection<Feature> features, T value) {

		// determine how many longs the longest feature mask takes
		int arrayLength = 0;
		for (Feature f : features)
			arrayLength = Math.max(f.maskArrayIndex, arrayLength);

		// compensate for zero-based array
		arrayLength++;

		// OR all of the feature masks together
		long[] compositeMask = new long[arrayLength];
		for (Feature f : features)
			compositeMask[f.maskArrayIndex] |= f.maskBits;

		this.compositeMask = compositeMask;
		this.value = value;
	}

	/**
	 * Create a feature group consisting of only one feature
	 * 
	 * @param feature
	 * @param value
	 */
	public FeatureGroup(Feature feature, T value) {
		long[] compositeMask = new long[feature.maskArrayIndex + 1];
		compositeMask[feature.maskArrayIndex] = feature.maskBits;

		this.compositeMask = compositeMask;
		this.value = value;
	}

	private FeatureGroup(long[] compositeMask, T value) {
		this.compositeMask = compositeMask;
		this.value = value;
	}

	public int hashCode() {
		return Arrays.hashCode(compositeMask);
	}

	public boolean equals(Object obj) {
		if (this instanceof FeatureGroup) {
			FeatureGroup other = (FeatureGroup) obj;
			return Arrays.equals(this.compositeMask, other.compositeMask);
		} else {
			return false;
		}
	}

	/**
	 * Get the set-wise complement of this FeatureGroup with respect to the
	 * global set of features.
	 * 
	 * @return
	 */
	public FeatureGroup<T> complement() {
		int bitCount = FeatureFactory.getCurrentBitCount();
		int requiredArrayLength = (bitCount / Long.SIZE) + 1;

		assert requiredArrayLength <= compositeMask.length;

		long[] complementArray = new long[requiredArrayLength];
		System.arraycopy(compositeMask, 0, complementArray, 0, compositeMask.length);

		for (int i = 0; i < complementArray.length; i++) {
			complementArray[i] = ~complementArray[i];
		}

		// handle the last element in the array
		// we must keep the unused bit range 0 to avoid creating bogus features
		// so we create a mask to kill off the unused bits with an AND
		int numUsedBitsInLastElement = bitCount % Long.SIZE;
		int numUnusedBitsInLastElement = Long.SIZE - numUsedBitsInLastElement;
		long maskForUsedBits = 0xffffffffffffffffL >>> numUnusedBitsInLastElement;
		complementArray[complementArray.length - 1] &= maskForUsedBits;

		return new FeatureGroup<T>(complementArray, this.value);
	}

	public ArrayList<Feature> getFeatures() {
		ArrayList<Feature> features = new ArrayList<Feature>();

		for (int i = 0; i < compositeMask.length; i++) {
			for (int j = 0; j < Long.SIZE; j++) {

				// check each bit (feature) of this group
				int maskArrayIndex = i;
				long maskBits = (1L << j);

				// System.err.println("Searching " + maskBits + " @
				// maskArrayIndex");

				if ((compositeMask[i] & maskBits) != 0) {
					Feature feature = FeatureFactory.getFeatureByMask(maskArrayIndex, maskBits);

					assert feature != null : "No such feature found: " + maskBits + " @ "
							+ maskArrayIndex;

					features.add(feature);
				}

			}
		}

		return features;
	}

	/**
	 * Convert this feature group from a bitmap into a human-readable list of
	 * feature names
	 */
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{ ");

		ArrayList<Feature> features = getFeatures();
		for (Feature feature : features) {
			builder.append(feature.getName() + ", ");
		}

		// remove last comma and add
		builder.deleteCharAt(builder.length() - 2);
		builder.append("}");
		return builder.toString();
	}

	/**
	 * Find out if this feature group is a subset of another, ignoring the
	 * parameterized type of the FeatureGroup.
	 * 
	 * @param <X>
	 * @param superset
	 * @return
	 */
	public <X> boolean isSubsetOf(FeatureGroup<X> superset) {

		assert superset != null;

		if (this.compositeMask.length > superset.compositeMask.length) {

			// check common elements
			for (int i = 0; i < superset.compositeMask.length; i++) {
				if ((this.compositeMask[i] & superset.compositeMask[i]) != this.compositeMask[i])
					return false;
			}

			// check "orphan" elements
			for (int i = superset.compositeMask.length; i < this.compositeMask.length; i++) {
				if (this.compositeMask[i] != 0)
					return false;
			}
		} else {
			// else if(this.compositeMask.length <= other.compositeMask.length)

			// check common elements
			for (int i = 0; i < this.compositeMask.length; i++) {
				if ((this.compositeMask[i] & superset.compositeMask[i]) != this.compositeMask[i])
					return false;
			}

			// check "orphan" elements
			for (int i = this.compositeMask.length; i < superset.compositeMask.length; i++) {
				if (superset.compositeMask[i] != 0)
					return false;
			}
		}

		return true;
	}

	public T getValue() {
		return value;
	}

}
