/************************************************************************
 *                                                                      *
 *  Program package 'som_pak':                                          *
 *                                                                      *
 *  Xvisual.c                                                           *
 *  -draw the values of one weight plane to graphics (for X11r5)	*
 *   and the trajectory (if there is data)                              *
 *                                                                      *
 *  Version 1.0                                                         *
 *  Date: 9 Oct 1992                                                    *
 *                                                                      *
 *  NOTE: This program package is copyrighted in the sense that it      *
 *  may be used for scientific purposes. The package as a whole, or     *
 *  parts thereof, cannot be included or used in any commercial         *
 *  application without written permission granted by its producents.   *
 *  No programs contained in this package may be copied for commercial  *
 *  distribution.                                                       *
 *  									*
 *  version 1.1a							*
 *									*
 *  X11R5 revision A.S. Miller March 1993				*
 *  asm@phastr.soton.ac.uk                                              *
 *									*
 *  All comments  concerning this program package may be sent to the    *
 *  e-mail address 'lvq@cochlea.hut.fi'.                                *
 *                                                                      *
 ************************************************************************/

#include <stdio.h>
#include <float.h>
#include <math.h>
/*
	Use X11 header files.
*/
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
/*
	Include LVQ header.
*/
#include "lvq_pak.h"

extern void draw_plane(struct entries *data, struct entries *codes, int plane,
                       char *code_filename, int argc, char **argv);

extern void plot_plane(struct entries *data, struct entries *codes, int plane,
                       Window win,Display *display, GC Gc, 
                       XFontStruct *FontInfo);

/*
	Main body of code unchanged from Silicon Graphics GL version.

	All X Windows calls are made from the two routines draw_plane and
	plot_plane.
*/
main(int argc, char **argv)
{
  int plane;
  char *in_code_file;
  char *in_data_file = NULL;
  struct entries *codes;
  struct entries *data = NULL;

  in_code_file = extract_parameter(argc, argv, IN_CODE_FILE, ALWAYS);
  in_data_file = extract_parameter(argc, argv, IN_DATA_FILE, OPTION);
  plane = (int) oatoi(extract_parameter(argc, argv, PLANE, OPTION), 1);
  verbose((int) oatoi(extract_parameter(argc, argv, VERBOSE, OPTION), 1));
	
  label_not_needed(1);
  if (verbose(-1) > 1)
    fprintf(stdout, "Codebook entries are read from file %s\n", in_code_file);
  codes = read_entries(in_code_file);

  if (codes->topol < topol_type("hexa")) {
    printf("File %s is not a map file\n", in_code_file);
    exit(0);
  }

  if (in_data_file != NULL) {
    if (verbose(-1) > 1)
      fprintf(stdout, "Data entries are read from file %s\n", in_data_file);
    data = read_entries(in_data_file);

    if (data->dimension > codes->dimension) {
      errormsg("Dimensions in data and codebook files are different");
      exit(0);
    }
  }

  if (plane > codes->dimension) {
    errormsg("Required plane is bigger than codebook vector dimension");
    exit(0);
  }
/*
	Draw Display.
*/
  if (verbose(-1) > 1)
    fprintf(stdout, "Displaying Data...\n");
  draw_plane(data, codes, plane, in_code_file, argc, argv);
  if (verbose(-1) > 1)    
    fprintf(stdout, "Finished...\n\n");         
  return(0);
}

/*
	Some useful definitions used by draw_plane and plot_plane.
*/
/*
	Boolean values.
*/
#define TRUE 1
#define FALSE 0	
/*
	Define some values for useful colours.
	To add more colours change ColourName[] too!
*/
#define MAP_COLOURS 100			/* Colours required by greyscale. */
#define WHITE  255			/* Define specific colours.       */
#define YELLOW 254
#define GREEN  253
#define BLUE   252
#define STEEL  251
#define RED    250
#define BLACK  249
#define SET_COLOURS (WHITE - BLACK)	/* Set colourtable size.          */
#define MAP_START (BLACK - 1)
/*
	Define what each button does.
*/
#define UP_BUTTON 1		/* Increase plane number.	*/
#define DOWN_BUTTON 2		/* Decrease plane number.	*/
#define EXIT_BUTTON 3		/* Kill window.			*/
/*
	Define this for trajectory plot rather than node highlights.

#define traj_plot
*/
/*
	Calculate colour from ranges and value.
*/
int colour_value(float value, float minval, float maxval)
{
  int colour  = MAP_START - 
                (int)((MAP_COLOURS - 1) * (value - minval) / 
                (maxval - minval));
  return(colour);
}

/*
	Plots SOM image plane.

	This is similar to the gvisual routine "draw_plane",
	but uses X11 calls.

	Change this routine to alter the display output.
*/
void plot_plane(struct entries *data, struct entries *codes, int plane,
                Window win,Display *display, GC Gc, XFontStruct *FontInfo)
{
  int cv;
  int i;
  int x, y, xp, yp, xsize, ysize, ystep;
  int index;
  int offset = 0;
  char string[80];
  float diff, diffsf, difference;
  float minval = FLT_MAX,
        maxval = FLT_MIN;
  struct entries *codetmp = codes;
  struct entries *datatmp;

  if (codes->topol = topol_type("hexa"))
    offset = 25;
  xsize = 50 * codes->xdim + offset + 20;
  ysize = 50 * codes->ydim + 20;

  if (verbose(-1) > 1)
    fprintf(stderr, "Plotting frame %d\n", plane);
/*
	Find image range.
*/
  while (codetmp != NULL) {
    if (maxval < codetmp->points[plane])
      maxval = codetmp->points[plane];
    if (minval > codetmp->points[plane])
      minval = codetmp->points[plane];
    codetmp = codetmp->next;
  }
/*
	Clear window and paint white, then draw palette.
*/
  XSetWindowBackground(display, win, STEEL);
  XClearWindow(display, win);
  y = 10;
  ystep = (int)((float)(ysize / 2) / (float)MAP_COLOURS);
  for (i = 0;i<MAP_COLOURS;i++) {
    XSetForeground(display, Gc, (MAP_START - i));
    XFillRectangle(display, win, Gc, 5, y, 15, ystep);
    y += ystep;
  }
  XSetForeground(display, Gc, BLACK);
  XDrawRectangle(display, win, Gc, 5,10,15,(ystep*MAP_COLOURS));
/*
	Print title info.
*/
  XSetForeground(display,Gc,BLACK);
  sprintf(string,"Plane %d range (%3.2f, %3.2f).",(plane+1),minval,maxval);
  XDrawString(display,win,Gc,40,13,string,strlen(string));
/*
	Now plot out activation array.
*/
  index = 0;
  codetmp = codes;
  while (codetmp != NULL) {
    xp = 50 * (index % codes->xdim) + 25;
    yp = 50 * (index / codes->xdim) + 25;
    if (((index / codes->xdim) % 2)) xp += offset;
    cv = colour_value(codetmp->points[plane],minval,maxval);
/* 
	Show the gray level circles 
*/
    XSetForeground(display,Gc,cv);
    XFillArc(display,win,Gc,xp,yp,22,22,0,23040);
/* 
	If there is label then show it 
*/
    if (codetmp->index != find_conv_to_ind(nolab())) {
      strcpy(string,find_conv_to_lab(codetmp->index));
      XSetForeground(display,Gc,GREEN);
      XDrawString(display,win,Gc,xp,yp,string,strlen(string));
    }
    codetmp = codetmp->next;
    index++;
  }

    /* If data values are present, draw trajectory */
  if (data != NULL) {
    long bel[2], cul[2];
    int bpos;
    int ind;
    int first = 1;
    if (verbose(-1) > 1) 
      fprintf(stderr,"Drawing trajectory...");
    datatmp = data;
 
   /* Scan all input entries */
    XSetLineAttributes(display,Gc,2,LineSolid,CapRound,JoinBevel);
    while (datatmp != NULL) {
    
      codetmp = codes;
      diffsf = FLT_MAX;
      ind = 0;

	/* Compare all codebook entries against the input entry */
      while (codetmp != NULL) {
        difference = 0.0;
    
	/* Compute the distance between the input and codebook entries */
        for (i = 0; i < datatmp->dimension; i++) {
          diff = datatmp->points[i] - codetmp->points[i];
          difference += diff * diff;
          if (difference > diffsf) break;
        }
    
	/* If distance is smaller than the previous distances */
        if (difference < diffsf) {
          diffsf = difference;
          bpos = ind;
        }
    
	  /* Take the next codebook entry */
        codetmp = codetmp->next;
        ind++;
      }
    	/* Draw the trajectory */
      if (first) {
        first = 0;
        XSetForeground(display, Gc, BLUE);
        cul[0] = 50 * (bpos % codes->xdim) + 25;
        cul[1] = 50 * (bpos / codes->xdim) + 25;
        if (((bpos / codes->xdim) % 2)) cul[0] += offset;
#ifndef traj_plot
        XSetForeground(display,Gc,RED);
        XDrawArc(display,win,Gc,cul[0],cul[1],22,22,0,23040);
#endif
      }
      else {
        bel[0] = cul[0]; bel[1] = cul[1];
        cul[0] = 50 * (bpos % codes->xdim) + 25;
        cul[1] = 50 * (bpos / codes->xdim) + 25;
        if (((bpos / codes->xdim) % 2)) cul[0] += offset;
#ifdef traj_plot
/*
	To plot data set trajectories over map use the following.
*/
        XSetForeground(display,Gc,BLUE);
        XDrawLine(display,win,Gc,bel[0],bel[1],cul[0],cul[1]);
#else
/*
	Highlight regions in data set.
*/
        XSetForeground(display,Gc,RED);
        XDrawArc(display,win,Gc,cul[0],cul[1],22,22,0,23040);
#endif
      }
	/* Take the next input entry */
      datatmp = datatmp->next;
    }
  }
  return;
}


/*
	Opens window and does all event handling.

	This routine does all the messy X Windows bits, it should not
	need to be altered to change the actual display content.  It does
	control the event handling, such as button actions, and the 
	colour allocation.
*/
void draw_plane(struct entries *data, struct entries *codes, int plane,
                char *code_filename, int argc, char **argv)
{
/*
	Other Variables.
*/
  int offset, i, xsize, ysize, col;

  short running;			/*  X Window event loop flag.	*/
  int Ncolours;				/*  Colour counter.		*/	
/*
	X variables.
	variables commented with a # may be specified at command line.
*/
  Display *display;			/*  Display server link.     	*/ 
  int ScreenNum;			/*  Display screen no.       	*/
  Window  win;				/*  Window structure.        	*/
  GC Gc;				/*  Graphics Context struct. 	*/

  char *AppName = "Xvisual";		/*  Application Name.	     	*/	
  char *DisplayName = NULL;		/* #Display name (default).  	*/
  char BarText[50]; 			/*  Window bar text.	     	*/
  char *IconText = BarText;		/*  Icon name text.          	*/

  XFontStruct *font_info;		/*  Font information.        	*/
  XColor color;			   	/*  Colour information       	*/
  int DefDepth;		  	   	/*  Default depth.		*/
  Visual *DefVisual;	  	   	/*  Default Visual 		*/
  Colormap DefColMap;	  	   	/*  Default colourmap.		*/
  Colormap NewColMap;			/*  Xvisual colourmap.		*/
  XVisualInfo VisInfo;	
  XColor ExactColDef[MAP_COLOURS];	/*  Exact colour information.	*/
  int PlaneMasks[1];
/*
	List of names corresponding to Visual types.
	Used for returning errors etc.
*/
  static char *VisualClassName[] = {"StaticGrey","GreyScale","StaticColour",
                                   "PseudoColour","TrueColour","DirectColour"};
/*
	X window call variables (structures).
*/
  XSizeHints size_hints;		/*  Window size Hints.	     	*/
  XWMHints XWMhints;			/*  Window manager hints.    	*/
  XSetWindowAttributes XWattrib;	/*  Window attributes.       	*/
  XGCValues GcValues;			/*  Context values.          	*/

  char DefGeom[80];			/* #Default geometry string. 	*/
  unsigned int border_width = 4;	/*  Border is 4 pixels wide. 	*/
  unsigned long foreground_pixel;	/*  Initial pixel values.    	*/
  unsigned long background_pixel;
  unsigned long border_pixel;
/*
	X window return value variables.
*/
  char *tmpstr;				/* tmp pointer for XR values.	*/
  char ColString[15];			/* for storing colour names. 	*/
  int bitmask;				/* return mask for XGeometry.	*/
  XEvent report;			/* Event report value.       	*/
/*
	Some useful colour names.
*/
  char *ColourName[] = {"white","yellow",
                      "green","blue","steel blue","red","black"}; 
/*
	Define a useful structure for storing X resources.
*/ 
typedef struct {   
  char *name;				/* Resource name. 		*/ 
  char **value;				/* Ptr to store resource in. 	*/
} Resources;

/*
	Define some useful resource defaults.
*/
  char *XRbgcol    = "Black",		/* Background colour.		*/
       *XRfgcol    = "White",		/* Foreground colour.		*/
       *XRfont     = "Fixed",		/* Font name. 			*/
       *XRgeom     = NULL,		/* Geometry (default)		*/
       *XRtext     = BarText;		/* Window bar text.		*/
/*
	Define which X resources are required by name.
	set pointers to variables in which to store values.
*/
#define NXr 5				/* Number of resources.		*/
  Resources Xr[NXr];
  Xr[0].name = "background"; Xr[0].value = &XRbgcol;
  Xr[1].name = "foreground"; Xr[1].value = &XRfgcol;
  Xr[2].name = "font";       Xr[2].value = &XRfont;
  Xr[3].name = "geometry";   Xr[3].value = &XRgeom;
  Xr[4].name = "text";       Xr[4].value = &XRtext;

/*
	Now define window size (add 20 for border space),
	and set title bar string.
*/
  if (codes->topol = topol_type("hexa"))
    offset = 25;
  xsize = 50 * codes->xdim + offset + 20;
  ysize = 50 * codes->ydim + 20;
  sprintf(BarText, "%s: %s", AppName,code_filename);
/* 
	Modify the input plane index by -1. 
*/
  plane--;
/*
	Open graphics Window and get screen number.
*/
  if ((display = XOpenDisplay(DisplayName)) == NULL) {
     fprintf(stderr,"%s: Error - Can't open display\n", AppName);
     exit(0);
  }
  ScreenNum  = DefaultScreen(display);
/*
	Get X resource values from "~/.Xdefaults".
*/
  for (i = 0; i < NXr; i++) {
    if ((tmpstr = XGetDefault(display, BarText, Xr[i].name)) != NULL)
       *Xr[i].value = tmpstr;
  }
/*
	Get font required by resource value.
*/
  if ((font_info = XLoadQueryFont(display, XRfont)) == NULL) {
     fprintf(stderr, "%s: cannot load font %s\n", AppName, XRfont);
     exit(0);
  }
/*
	Get default colour map.
*/
  DefColMap = DefaultColormap(display, ScreenNum);  
/*
	Set window hints: position, size, minimum size
*/
  size_hints.flags      = (PPosition | PSize | PMinSize);
  size_hints.min_width  = 200;
  size_hints.min_height = 200;
  size_hints.height     = ysize;
  size_hints.width      = xsize;
  size_hints.x          = 10;
  size_hints.y          = 10;
/*
	Write default geometry string, this could be supplied by the user.
*/
  sprintf(DefGeom, "%dx%d+%d+%d", size_hints.width, size_hints.height, 
                                  size_hints.x, size_hints.y);
  XRgeom = DefGeom;
/*
	Now try to set up this geometry.
	XGeometry interprets the DefGeom string and sets flags if
	various parameters were defined there.  It also writes the new
	values to the size_hints structure.
*/	
  bitmask = XGeometry(display, ScreenNum, XRgeom, DefGeom,  border_width,
                      font_info->max_bounds.width,
                      font_info->max_bounds.ascent +
                      font_info->max_bounds.descent,
                      1, 1, &(size_hints.x), &(size_hints.y),
                      &(size_hints.width), &(size_hints.height));
/*
	Now set hint flags.
*/
  if (bitmask & (XValue | YValue)) size_hints.flags |= USPosition;
  if (bitmask & (WidthValue | HeightValue)) size_hints.flags |= USSize;
/*
	Create Simple (?) window.
	This sets up lots of other default values.
*/
  win = XCreateSimpleWindow(display, DefaultRootWindow(display),
                            size_hints.x, size_hints.y,
                            size_hints.width, size_hints.height, 
			    border_width,
                            border_pixel, background_pixel);
/*
	Setup some more standard window properties.
*/	
  XSetStandardProperties(display, win, AppName, IconText,   
                         None, argv, argc, &size_hints);
/*
	Set X Window Manager Hints.
*/
  XWMhints.flags         = (InputHint | StateHint);
  XWMhints.input         = False;
  XWMhints.initial_state = NormalState;
  XSetWMHints(display, win, &XWMhints);
/*
	Setup a graphics context for general drawing.	
*/
  GcValues.font       = font_info->fid;
  GcValues.foreground = foreground_pixel;
  GcValues.background = background_pixel;
  Gc = XCreateGC(display, win, (GCFont | GCForeground | GCBackground), 
                 &GcValues);
/*
	Change a few of the Window attributes:
		set colourmap to default
		bit gravity	
*/
  XWattrib.colormap    = DefaultColormap(display, ScreenNum);
  XWattrib.bit_gravity = CenterGravity;
  XChangeWindowAttributes(display, win,(CWColormap | CWBitGravity), &XWattrib);
/*
	Select which events will be sent to the window.
		expose window
		button press
*/
  XSelectInput(display, win, (ExposureMask | ButtonPressMask));
/*
	Get default values.
        	depth
		visual
		colour map
*/
  DefDepth = DefaultDepth(display, ScreenNum);
  DefVisual = DefaultVisual(display, ScreenNum);
  DefColMap = DefaultColormap(display, ScreenNum);
/*
	If black and white display exit.
*/
  if (verbose(-1) > 2) 
    fprintf(stderr, "Found visual depth %d\n",DefDepth);
  if (DefDepth == 1){
    border_pixel = BlackPixel(display, ScreenNum);
    background_pixel = WhitePixel(display, ScreenNum);
    foreground_pixel = BlackPixel(display, ScreenNum);
    fprintf(stderr,"No colour display found!\n");
    exit(0);
  }
/*
	Now try and find colour visual or greyscale visual.
*/
  i = 5;
  while(!XMatchVisualInfo(display,ScreenNum,DefDepth,i--,&VisInfo))
  ;
  i++;
  if (verbose(-1) > 2)
    fprintf(stderr, "Found visual class %s\n",VisualClassName[i]);
/*
	Check to see is we have the default visual.
*/
  if ((VisInfo.visual != DefVisual) && (verbose(-1) > 2))
    printf("Visual class %s at default depth.\n",VisualClassName[i]);
  else
    if (verbose(-1) > 2)
      printf("Visual class %s at not at default depth!\n",VisualClassName[i]);
/*
	Create new colour map.
	We will allocate colurs from the top down to preserve 
	as many of the existing colours as possible.
*/
  NewColMap = XCreateColormap(display,win,DefVisual,AllocAll);
  XWattrib.colormap = NewColMap;
  XChangeWindowAttributes(display,win,CWColormap,&XWattrib);
/*
	Allocate specific colours.
*/
  Ncolours = 0;
  for(i=0;i<SET_COLOURS;i++){
    if (!XParseColor(display,NewColMap,ColourName[i],
                   &ExactColDef[0])){
      fprintf(stderr,"%s is an unknown colour!\n",ColourName[i]);
      exit(0);
    }
    if (!XAllocColor(display,NewColMap,&ExactColDef[0])){
      ExactColDef[0].pixel = 255 - Ncolours;
      ExactColDef[0].flags = DoRed | DoGreen | DoBlue;
      XStoreColor(display,NewColMap,&ExactColDef[0]);
    }
    Ncolours++;
  }
/*
	Allocate read/write greyscale colours.
*/
  for (i=0;i<MAP_COLOURS;i++) {
    ExactColDef[i].pixel = MAP_START - i;
    ExactColDef[i].flags = DoRed | DoGreen | DoBlue;   
    ExactColDef[i].red = ExactColDef[i].blue = ExactColDef[i].green = 
                         (int)(63270 * ((float)i / (float)MAP_COLOURS));
    Ncolours++;
  }
  XStoreColors(display,NewColMap,ExactColDef,MAP_COLOURS);
/*
	Copy all remaining colours from default colour pallette.
*/
  for(i=0;i<(MAP_START - MAP_COLOURS - 1);i++){
        ExactColDef[0].pixel = i;
    ExactColDef[0].flags = DoRed | DoGreen | DoBlue;
    XQueryColor(display,DefColMap,&ExactColDef[0]); 

    XStoreColor(display,NewColMap,&ExactColDef[0]);
  }
/*
	Now... map window onto display (at last!)
*/
  XMapWindow(display, win);
  if (verbose(-1) > 2)
    fprintf(stderr, "Window opened on %s\n", DisplayName);
/*
	Main Window event handle loop.
*/
  running = TRUE;
  while (running) {
    XNextEvent(display, &report);
    switch(report.type){
/*
		Expose: redraw window.
*/
      case Expose:
        plot_plane(data,codes,plane,win,display,Gc, font_info);
        break;
/*
		Button press:
*/
      case ButtonPress:
        switch(report.xbutton.button){
/*
			Exit: quit application.
*/
          case EXIT_BUTTON:
            running = FALSE;
            break;
/*
			Up: increase plane number and redraw.
*/
          case UP_BUTTON:
            plane++;
            if (plane > (codes->dimension-1)) 
              plane = codes->dimension - 1;
            else
              plot_plane(data,codes,plane,win,display,Gc, font_info);
            break;
/*
			Down: decrease plane number and redraw.
*/
          case DOWN_BUTTON:   
            plane--;
            if (plane < 0) 
              plane = 0;
            else
              plot_plane(data,codes,plane,win,display,Gc, font_info);
            break;
          default:
            fprintf(stderr,"Bad Button type!\n");
            exit(0);     
        }
        break;
      default:
        fprintf(stderr,"Bad Event type!\n");
        exit(0);
    }
  }
/*
	Unload font, free colours and close window.
*/
  XUnloadFont(display, font_info->fid);
  XFreeGC(display, Gc);
  XCloseDisplay(display);
  return;
}
