/*  $Id: rnswrite.c,v 1.18 1991/10/10 16:47:05 richard Exp richard $  */

/*  Part of RNS -- Recurrent Network Simulator
 *
 *     by R. Kooijman
 *        T.U. Delft
 *        Faculteit Elektrotechniek
 *        Vakgroep Computerarchitectuur
 *        Sectie Neurale Netwerken
 */


/*  $Log: rnswrite.c,v $
 * Revision 1.18  1991/10/10  16:47:05  richard
 * Switched to UNIX RCS
 *
 * Revision 1.17  91/05/09  14:43:26  ROOT_DOS
 * Added output of generated seed value
 * Changed don't care and undefined values
 *
 * Revision 1.16  91/03/24  16:02:00  ROOT_DOS
 * Added RANDOM learning property
 * Added DONTCARE command
 *
 * Revision 1.15  91/02/26  21:51:01  ROOT_DOS
 * Added comments
 * Added support for SEED command
 *
 * Revision 1.14  91/02/17  14:29:32  ROOT_DOS
 * Minor changes
 * Let UNIX version handle DISPLAY
 *
 * Revision 1.13  91/02/05  21:53:33  ROOT_DOS
 * Added MAXSTEP command
 *
 * Revision 1.12  91/02/05  01:30:06  ROOT_DOS
 * Made version that can be compiled to single or double arithmetic simulator
 *
 * Revision 1.11  91/02/01  23:32:28  ROOT_DOS
 * Added ->constraint==NULL checks
 *
 * Revision 1.10  91/01/30  21:45:05  ROOT_DOS
 * Forget ANSI C and make traditional C / Turbo C dual mode
 *
 * Revision 1.9  91/01/27  21:04:52  ROOT_DOS
 * Changes to make code strict ANSI C compatible
 * Removed DISPLAY command in ANSI C mode (no graphics)
 *
 * Revision 1.8  91/01/22  16:54:47  ROOT_DOS
 * Made ITER command conditional compilable
 *
 * Revision 1.7  91/01/21  16:17:34  ROOT_DOS
 * Added "don't care" values to real-valued patterns
 *
 * Revision 1.6  91/01/20  00:45:28  ROOT_DOS
 * Added check for NULL result part of pattern in case it wasn't allocated
 *
 * Revision 1.5  91/01/19  22:28:06  ROOT_DOS
 * Added support for real-valued patterns, relative learning rates
 *
 * Revision 1.3  91/01/03  14:47:22  ROOT_DOS
 * Added support for simulated annealing algorithme
 *
 * Revision 1.2  90/12/08  03:51:48  ROOT_DOS
 * Some minor cosmetic changes
 *
 * Revision 1.1  90/12/08  03:19:10  ROOT_DOS
 * Initial revision
 *   */



/*  functies voor het wegschrijven van de resultaten  */


#define EXTERN
#include "rnsdefs.h"
#include "rnsfuncs.h"


#ifdef TURBOC
#include <stdio.h>
#include <string.h>
#else
#include <stdio.h>
#include <string.h>
#endif



#ifdef USEPROTOS
int write_output(void)
#else
int write_output()
#endif
{
int rc = OK;

   if (step==0)                 /*  als er geen stappen zijn uitgevoerd  */
      return(OK);               /*  zijn er nog geen resultaten, dus...  */

   if ((output_file=fopen(output_name, OUTPUTMODE))==NULL)
      return(OPNOUT);           /*  fout bij openen output bestand  */

   setbuf(output_file, file_buffer);
                                /*  buffer de uitvoer  */
   rc=write_header();           /*  schrijf de header  */

   if (rc==OK)
      rc=write_neurons();       /*  schrijf network weg  */

   if (rc==OK)
      rc=write_epsilon();       /*  schrijf leerparameter weg  */

   if (rc==OK)
      rc=write_momentum();      /*  schrijf momentum term  */

   if (rc==OK)
      rc=write_dmomentum();     /*  schrijf dmomentum term  */

   if (rc==OK)
      rc=write_decay();         /*  schrijf update term weg  */

   if (rc==OK)
      rc=write_addrate();       /*  schrijf addrate term weg  */

   if (rc==OK)
      rc=write_mulrate();       /*  schrijf mulrate term weg  */

   if (rc==OK)
      rc=write_adjust();        /*  schrijf adjust term weg  */

   if (rc==OK)
      rc=write_forget();        /*  schrijf forget term  */

   if (rc==OK)
      rc=write_maxweight();     /*  schrijf maxweight term  */

   if (rc==OK)
      rc=write_pselect();       /*  schrijf patroon select parameters weg  */

   if (rc==OK)
      rc=write_step();          /*  schrijf maximaal aantal stappen weg  */

   if (rc==OK)
      rc=write_dontcare();      /*  schrijf dontcare waarde weg  */

   if (rc==OK)
      rc=write_seed();          /*  schrijf seed waarde weg  */

#ifndef NOITER
   if (rc==OK)
      rc=write_iter();          /*  schrijf maximaal aantal iterations weg  */
#endif

   if (rc==OK)
      rc=write_learn();         /*  schrijf leeralgoritme weg  */

   if (rc==OK)
      rc=write_error();         /*  schrijf doelwaarde globale fout weg  */

   if (rc==OK)
      rc=write_tolerance();     /*  schrijf tolerantie in bitwaarde weg  */

   if (rc==OK)
      rc=write_display();       /*  schrijf vensterdata weg  */

   if (rc==OK)
      rc=write_patterns();      /*  schrijf patronen weg  */

   if (fclose(output_file)==EOF)
      return(CLSOUT);           /*  fout bij sluiten van output bestand  */

   return(rc);
}



#ifdef USEPROTOS
int write_header(void)
#else
int write_header()
#endif
{
   /*  header informatie met bestanden en tijdmetingen  */
   fprintf(output_file, "// %s:  %s\n\n", RNSNAME, VERSION);
   fprintf(output_file, "// %s:  Input file:   %s\n", RNSNAME, input_name);
   fprintf(output_file, "// %s:  Output file:  %s\n", RNSNAME, output_name);
   fprintf(output_file, "// %s:  Log file:     %s\n\n", RNSNAME, log_name);
   fprintf(output_file, "// %s:  Start time:   %s", RNSNAME, begin_str);
   fprintf(output_file, "// %s:  End time:     %s\n\n\n", RNSNAME, end_str);
                                /*  er staat al een \n in de tijd strings  */

   return(OK);
}



#ifdef USEPROTOS
int write_neurons(void)
#else
int write_neurons()
#endif
{
int number;
Neuronptr neuronptr;

   /*  schrijf alle input neuronen weg  */
   fprintf(output_file, "INPUT:\n");
   for (number=0; number<nrinputs; number++)
      write_neuron(INPUTNEURON(number));
   fprintf(output_file, "END;\n\n\n");

   /*  schrijf alle hidden neuronen weg  */
   fprintf(output_file, "HIDDEN:\n");
   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      if (neuronptr->type==HIDDEN)
         write_neuron(neuronptr);
   fprintf(output_file, "END;\n\n\n");

   /*  schrijf alle output neuronen weg  */
   fprintf(output_file, "OUTPUT:\n");
   for (number=0; number<nroutputs; number++)
      write_neuron(OUTPUTNEURON(number));
   fprintf(output_file, "END;\n\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_neuron(Neuronptr neuronptr)
#else
int write_neuron(neuronptr)
       Neuronptr neuronptr;
#endif
{
int weight;
Weightptr weightptr;
char *functionstr;

   /*  bepaal functie van neuron in tekstvorm  */
   switch(neuronptr->function)
   {
      case STEP:
         functionstr="STEP";
         break;
      case LOGISTIC:
         functionstr="LOGISTIC";
         break;
      case LOGC4:
         functionstr="LOGC4";
         break;
      case LOGC8:
         functionstr="LOGC8";
         break;
      case LINEAR:
         functionstr="LINEAR";
         break;
   }

   /*  schrijf 'nummer: functie : constraint bias' weg  */
   fprintf(output_file, "%3d: %8s : %2s% 8.4f",
           neuronptr->number, functionstr,
           neuronptr->constraint!=NULL ? neuronptr->constraint : "  ",
           neuronptr->constraint!=NULL &&
           (strchr(neuronptr->constraint, '<')!=NULL ||
            strchr(neuronptr->constraint, '>')!=NULL) ?
           neuronptr->orgbias : neuronptr->bias);

   /*  als er links zijn, schrijf deze dan weg  */
   if (neuronptr->nrweights!=0)
   {
      fprintf(output_file, " <= ");
      for (weight=0; weight<neuronptr->nrweights; weight++)
      {                         /*  ga alle links af  */
         weightptr=neuronptr->weights+weight;
         if (weightptr->from2==NULL)   /*  geen Sigma-Pi link  */
            fprintf(output_file, "     %3d", weightptr->from->number);
         else
            fprintf(output_file, "%3d *%3d", weightptr->from->number,
                                             weightptr->from2->number);
         /*  schrijf eigenschappen van link weg  */
         fprintf(output_file, " %c : %2s% 8.4f",
                 weightptr->delay==TRUE ? '~' : ' ',
                 weightptr->constraint!=NULL ? weightptr->constraint : "  ",
                 weightptr->constraint!=NULL &&
                 (strchr(weightptr->constraint, '<')!=NULL ||
                  strchr(weightptr->constraint, '>')!=NULL) ?
                 weightptr->orgvalue : weightptr->value);

         /*  als er nog links komen, plaats dan een komma en begin  */
         /*   eventueel een nieuwe regel                            */
         if (weight<neuronptr->nrweights-1)
         {
            fprintf(output_file, ", ");
            if (((weight+1) % 2)==0)
               fprintf(output_file, "\n                              ");
         }
      }
   }

   fprintf(output_file, ";\n");

   return(OK);
}



#ifdef USEPROTOS
int write_epsilon(void)
#else
int write_epsilon()
#endif
{
   /*  schrijf RATE:   learning_rate_value;  */
   fprintf(output_file, "RATE:     %7.4f;\n", epsilon);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_momentum(void)
#else
int write_momentum()
#endif
{
   /*  schrijf MOMENTUM:   momentum_value;  */
   fprintf(output_file, "MOMENTUM: %7.4f;\n", momentum);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_error(void)
#else
int write_error()
#endif
{
   /*  schrijf ERROR:   target_global_error_value;  */
   fprintf(output_file, "ERROR:    %7.4f;\n", error);
   fprintf(output_file, "// %s:  %s reached an error of %.4f in %ld steps = %.0f s\n\n\n",
                        RNSNAME, RNSNAME, total_error, step, difftime(end_time, begin_time));

   return(OK);
}



#ifdef USEPROTOS
int write_display(void)
#else
int write_display()
#endif
{
   /*  schrijf venster coordinaten en titels weg  */
#ifdef TURBOC
   fprintf(output_file, "DISPLAY:\n"
                        "%4d, %4d, %4d, %4d, \"%s\",\n"
                        "%4d, %4d, %4d, %4d, \"%s\",\n"
                        "%4d, %4d, %4d, %4d, \"%s\",\n"
                        "%4d, %4d, %4d, %4d, \"%s\";\n",
           left_in, top_in, right_in, bottom_in, inputstr,
           left_out, top_out, right_out, bottom_out, outputstr,
           left_res, top_res, right_res, bottom_res, resultstr,
           left_info, top_info, right_info, bottom_info, infostr);
#else
   fprintf(output_file, "DISPLAY:\n\
%4d, %4d, %4d, %4d, \"%s\",\n\
%4d, %4d, %4d, %4d, \"%s\",\n\
%4d, %4d, %4d, %4d, \"%s\",\n\
%4d, %4d, %4d, %4d, \"%s\";\n",
           left_in, top_in, right_in, bottom_in, inputstr,
           left_out, top_out, right_out, bottom_out, outputstr,
           left_res, top_res, right_res, bottom_res, resultstr,
           left_info, top_info, right_info, bottom_info, infostr);
#endif
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_patterns(void)
#else
int write_patterns()
#endif
{
int pattern, row, colom;
FLOAT dinput, doutput, result;
int pair;

   /*  schrijf PATTERNS:   width_in[,height_in]   width_out[,height_out]  */
   fprintf(output_file, "\n");
   fprintf(output_file, "PATTERNS:");

   fprintf(output_file, "   %3d", width_in);

   if (height_in>1)
      fprintf(output_file, ", %3d", height_in);

   fprintf(output_file, "   %3d", width_out);

   if (height_out>1)
      fprintf(output_file, ", %3d", height_out);

   fprintf(output_file, "\n\n\n");

   for (pattern=0; pattern<nrpatterns+nrtests; pattern++)
   {                            /*  ga alle patronen af  */
      if (pattern==nrpatterns)  /*  zijn we bij de testpatronen aangekomen?  */
         fprintf(output_file, "TEST:\n\n\n");

      /*  schrijf nummer van patroon weg  */
      fprintf(output_file, "// %s:  Pattern    %3d\n\n", RNSNAME, pattern);

      /*  schrijf relatieve factor weg indien andere waarde dan 1  */
      if (GETPATTERN(pattern)->relative!=1)
         fprintf(output_file, "RELATIVE: %6.4f\n\n", GETPATTERN(pattern)->relative);

      /*  schrijf type van patroon weg  */
      if (GETPATTERN(pattern)->type==BIT)
         fprintf(output_file, "BIT:\n");
      else
         fprintf(output_file, "REAL:\n");

      /*  schrijf SET directive weg indien meer dan 1 patroonpaar  */
      if (PAIRS(pattern)>1)
         fprintf(output_file, "SET:\n");

      for (pair=0; pair<PAIRS(pattern); pair++)
      {                         /*  ga alle paren af in set  */
#ifdef VERBOSE
         fprintf(output_file, "// %s:  Input:\n", RNSNAME);
#endif
         for (row=0; row<height_in; row++)
         {                      /*  ga alle rijen af  */
            for (colom=0; colom<width_in; colom++)
            {                   /*  ga alle kolommen af  */
               dinput=pattern_val(pattern, pair, row*width_in+colom, INPUT);
               if (GETPATTERN(pattern)->type==BIT)
               {                /*  als bit pattern dan uitvoer in bits  */
                  fprintf(output_file, "%c",
                          dinput==UNDEFINED ? 'x'
                                     : (dinput==MAXOUTPUT ? 'X'
                                        : (dinput==DONTCARE ? '*' : '-')));
                  if (((colom+1) % 80)==0 && row<height_in && colom<width_in-1)
                     fprintf(output_file, "\n");
               }
               else
               {                /*  anders uitvoer in reeele waarden  */
                  if (dinput==DONTCARE)
                     fprintf(output_file, "    *  ");
                  else
                  if (dinput==UNDEFINED)
                     fprintf(output_file, "    x  ");
                  else          /*  normaliseer tot logisch niveau  */
                     fprintf(output_file, "%6.4f ", (dinput-MINOUTPUT)/MAXMINOUT);
                  if (((colom+1) % 10)==0 && row<height_in && colom<width_in-1)
                     fprintf(output_file, "\n");
               }
            }
            fprintf(output_file, "\n");
         }

#ifdef VERBOSE
         fprintf(output_file, "// %s:  Output:\n", RNSNAME);
#endif
         for (row=0; row<height_out; row++)
         {                      /*  ga alle rijen af  */
            for (colom=0; colom<width_out; colom++)
            {                   /*  ga alle kolommen af  */
               doutput=pattern_val(pattern, pair, row*width_out+colom, OUTPUT);
               if (GETPATTERN(pattern)->type==BIT)
               {                /*  als bit pattern dan uitvoer in bits  */
                  fprintf(output_file, "%c",
                          doutput==UNDEFINED ? 'x'
                                      : (doutput==MAXOUTPUT ? 'X'
                                         : (doutput==DONTCARE ? '*' : '-')));
                  if (((colom+1) % 80)==0 && row<height_out && colom<width_out-1)
                     fprintf(output_file, "\n");
               }
               else
               {                /*  anders uitvoer in reeele waarden  */
                  if (doutput==DONTCARE)
                     fprintf(output_file, "    *  ");
                  else
                  if (doutput==UNDEFINED)
                     fprintf(output_file, "    x  ");
                  else          /*  normaliseer tot logisch niveau  */
                     fprintf(output_file, "%6.4f ", (doutput-MINOUTPUT)/MAXMINOUT);
                  if (((colom+1) % 10)==0 && row<height_out && colom<width_out-1)
                     fprintf(output_file, "\n");
               }
            }
            fprintf(output_file, "\n");
         }

         /*  als ->result==NULL dan geldt dit voor alle patroonparen  */
         /*  dus zal dit voor elk paar aangeroepen worden waardoor    */
         /*  dit ook werkt voor sets met patroonparen                 */
         if (GETPATTERN(pattern)->result==NULL)
         {                             /*  result is niet gealloceerd  */
            reset_activations(pattern, pair, OUT|RESULT);
            while (calc_output(pattern, pair)==FALSE)
               ;

            /*  calc_pattern_error() kan overgeslagen worden  */
            /*  omdat dit al in run_tests() is gebeurd        */

            calc_delay();
         }

         fprintf(output_file, "// %s:  Result:\n", RNSNAME);
         fprintf(output_file, "// %s:  ", RNSNAME);
         for (row=0; row<height_out; row++)
         {                /*  ga alle rijen af  */
            for (colom=0; colom<width_out; colom++)
            {             /*  ga alle kolommen af  */
               /*  als ->result==NULL dan waarde uit netwerk anders uit ->result  */
               if (GETPATTERN(pattern)->result==NULL)
                  result=OUTPUTNEURON(row*width_out+colom)->output;
               else
                  result=pattern_val(pattern, pair, row*width_out+colom, RESULT);

               /*  normaliseer tot logisch niveau  */
               fprintf(output_file, "%6.4f ", (result-MINOUTPUT)/MAXMINOUT);
               if (((colom+1) % 10)==0 && row<height_out && colom<width_out-1)
                  fprintf(output_file, "\n// %s:  ", RNSNAME);
            }
            fprintf(output_file, "\n\n");
         }
      }
      fprintf(output_file, "// %s:  Error: %7.4f\n",
                           RNSNAME, GETPATTERN(pattern)->error);

      if (PAIRS(pattern)>1)
         fprintf(output_file, "END;\n");

      fprintf(output_file, "\n\n");
   }

   fprintf(output_file, "END;\n\n\n");

   return(OK);
}



#ifndef NOITER
#ifdef USEPROTOS
int write_iter(void)
#else
int write_iter()
#endif
{
   /*  schrijf ITER:   max_number_of_iterations;  */
   fprintf(output_file, "ITER:     % ld;\n", max_iter);
   fprintf(output_file, "\n\n");

   return(OK);
}
#endif



#ifdef USEPROTOS
int write_learn(void)
#else
int write_learn()
#endif
{
int learntmp;
char *learnstr;

   learntmp=learn;

   /*  schrijf LEARN:   learning_mode;  */
   fprintf(output_file, "LEARN:     ");

   /*  zolang er nog leermethode bits zijn  */
   while (learntmp>0)
   {
      if (learntmp & HEBB)
      {
         learntmp&=~HEBB;
         learnstr="HEBB";
      }
      else
       if (learntmp & FBPR)
       {
          learntmp&=~FBPR;
          learnstr="FBPR";
       }
       else
        if (learntmp & FORCED)
        {
           learntmp&=~FORCED;
           learnstr="FORCED";
        }
        else
         if (learntmp & SELF)
         {
            learntmp&=~SELF;
            learnstr="SELF";
         }
         else
          if (learntmp & DHEBB)
          {
             learntmp&=~DHEBB;
             learnstr="DHEBB";
          }
          else
           if (learntmp & GROSSBERG)
           {
              learntmp&=~GROSSBERG;
              learnstr="GROSSBERG";
           }
           else
            if (learntmp & CLIP424)
            {
               learntmp&=~CLIP424;
               learnstr="CLIP424";
            }
            else
             if (learntmp & CLIP262)
             {
                learntmp&=~CLIP262;
                learnstr="CLIP262";
             }
             else
              if (learntmp & ANNEAL)
              {
                 learntmp&=~ANNEAL;
                 learnstr="ANNEAL";
              }
              else
               if (learntmp & RANDOM)
               {
                  learntmp&=~RANDOM;
                  learnstr="RANDOM";
               }
               else
                if (learntmp & USERERR)
                {
                   learntmp&=~USERERR;
                   learnstr="USERERR";
                }
                else
                {
                   learntmp=0;  /*  maak 'learntmp' hard 0  */
                   learnstr="";
                }

      /*  schrijf gevormde string weg  */
      fprintf(output_file, "%s", learnstr);

      /*  schrijf & teken weg als er nog meer bits zijn  */
      if (learntmp>0)
         fprintf(output_file, "%s", " & ");
   }

   fprintf(output_file, ";\n\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_decay(void)
#else
int write_decay()
#endif
{
   /*  schrijf UPDATE:   update_step_rate;  */
   fprintf(output_file, "UPDATE:   % ld;\n", decay);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_step(void)
#else
int write_step()
#endif
{
   /*  schrijf MAXSTEP:   max_number_of_steps;  */
   fprintf(output_file, "MAXSTEP:  % ld;\n", lim_step);
   fprintf(output_file, "\n\n");
   return(OK);
}



#ifdef USEPROTOS
int write_seed(void)
#else
int write_seed()
#endif
{
   fprintf(output_file, "SEED:     %u;\n", seed);
   if (seed==0)
      fprintf(output_file, "// %s:  %s used a generated seed value of %u\n", RNSNAME, RNSNAME, gseed);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_dontcare(void)
#else
int write_dontcare()
#endif
{
   fprintf(output_file, "DONTCARE: %7.4f;\n", dontcare);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_pselect(void)
#else
int write_pselect()
#endif
{
   if (pselect_adapt == FALSE)
      fprintf(output_file, "PSELECT:  %7.4f % ld;\n", pselect_error, pselect_steps);
   else
      fprintf(output_file, "PSELECT:   %s % ld;\n// %s:  Final pattern select error was %.4f\n", "ADAPT", pselect_steps, RNSNAME, pselect_error);

   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_tolerance(void)
#else
int write_tolerance()
#endif
{
   /*  schrijf TOLERANCE:   tolerance_value;  */
   fprintf(output_file, "TOLERANCE:%7.4f;\n", tolerance);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_maxweight(void)
#else
int write_maxweight()
#endif
{
   /*  schrijf MAXWEIGHT:   maxweight_value;  */
   fprintf(output_file, "MAXWEIGHT: %6.3f;\n", maxweight);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_forget(void)
#else
int write_forget()
#endif
{
   /*  schrijf FORGET:      forget_value;  */
   fprintf(output_file, "FORGET:   %7.4f;\n", forget);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_dmomentum(void)
#else
int write_dmomentum()
#endif
{
   /*  schrijf DMOMENTUM:   dmomentum_value;  */
   fprintf(output_file, "DMOMENTUM:%7.4f;\n", dmomentum);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_addrate(void)
#else
int write_addrate()
#endif
{
   /*  schrijf ADDRATE:     addrate_value;  */
   fprintf(output_file, "ADDRATE:  %7.4f;\n", addrate);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_mulrate(void)
#else
int write_mulrate()
#endif
{
   /*  schrijf MULRATE:     mulrate_value;  */
   fprintf(output_file, "MULRATE:  %7.4f;\n", mulrate);
   fprintf(output_file, "\n\n");

   return(OK);
}



#ifdef USEPROTOS
int write_adjust(void)
#else
int write_adjust()
#endif
{
   /*  schrijf ADJUST:      adjust_value;  */
   fprintf(output_file, "ADJUST:    %s  %s;\n", (adjust_mode==ADJUST_RELATIVE ? "RELATIVE" : (adjust_mode==ADJUST_LOCAL ? "LOCAL" : "GLOBAL")),
                                                (adjust_net==ADJUST_RECURRENT ? "RECURRENT" : "ALL"));
   fprintf(output_file, "\n\n");

   return(OK);
}
