/*  $Id: rnsinput.c,v 1.23 1991/10/15 07:57:12 richard Exp richard $  */

/*  Part of RNS -- Recurrent Network Simulator
 *
 *     by R. Kooijman
 *        T.U. Delft
 *        Faculteit Elektrotechniek
 *        Vakgroep Computerarchitectuur
 *        Sectie Neurale Netwerken
 */


/*  $Log: rnsinput.c,v $
 * Revision 1.23  1991/10/15  07:57:12  richard
 * Fixed error return in convert_to_bitpat
 *
 * Revision 1.22  1991/10/10  16:45:46  richard
 * Switched to UNIX RCS
 * Added ATOH, BTOH, FTOH and NTOH for random bit pattern support
 *
 * Revision 1.21  91/05/09  14:41:29  ROOT_DOS
 * Added USERERR learn mode property
 * Changed don't care and undefined values
 * Minor changes
 *
 * Revision 1.20  91/03/24  16:04:01  ROOT_DOS
 * Added dead-end checking
 * Added DONTCARE command
 *
 * Revision 1.19  91/02/26  21:50:09  ROOT_DOS
 * Added support for SEED command
 *
 * Revision 1.18  91/02/17  14:31:05  ROOT_DOS
 * Added NUMBER directive
 * Let UNIX version handle DISPLAY
 * Minor changes
 *
 * Revision 1.17  91/02/05  21:53:59  ROOT_DOS
 * Added MAXSTEP command
 *
 * Revision 1.16  91/02/05  09:17:34  ROOT_DOS
 * Made new REALLOC and FREE functions to support older style realloc and free
 *
 * Revision 1.15  91/02/05  01:30:40  ROOT_DOS
 * Made version that can be compiled to single or double arithmetic simulator
 *
 * Revision 1.14  91/02/01  23:35:24  ROOT_DOS
 * Added support for older style realloc's
 *
 * Revision 1.13  91/01/30  21:45:21  ROOT_DOS
 * Forget ANSI C and make traditional C / Turbo C dual mode
 *
 * Revision 1.12  91/01/27  21:05:29  ROOT_DOS
 * Changes to make code strict ANSI C compatible
 * Removed DISPLAY command in ANSI C mode (no graphics)
 *
 * Revision 1.11  91/01/22  16:54:09  ROOT_DOS
 * Made ITER command conditional compilable
 *
 * Revision 1.10  91/01/21  16:16:47  ROOT_DOS
 * Changed and fixed handling of "don't care" values
 *
 * Revision 1.9  91/01/20  00:44:02  ROOT_DOS
 * Extra free of temporary pattern storage
 *
 * Revision 1.8  91/01/19  22:26:28  ROOT_DOS
 * Added support for real-valued patterns, relative learning rates
 *
 * Revision 1.6  91/01/03  14:45:50  ROOT_DOS
 * Added support for simulated annealing
 *
 * Revision 1.5  90/12/19  21:27:43  ROOT_DOS
 * Fixed bug with ALPHA command
 * Optimized get_next_word somewhat
 *
 * Revision 1.4  90/12/18  22:22:09  ROOT_DOS
 * Added ALPHA command
 *
 * Revision 1.3  90/12/12  20:50:52  ROOT_DOS
 * Added check for missing learning mode
 *
 * Revision 1.2  90/12/08  03:51:10  ROOT_DOS
 * Some minor cosmetic changes
 *
 * Revision 1.1  90/12/08  03:16:34  ROOT_DOS
 * Initial revision
 *   */



/*  functies voor het inlezen uit het invoerbestand  */


#define EXTERN
#include "rnsdefs.h"
#include "rnsfuncs.h"

#include "rnsalloc.h"


#ifdef TURBOC
#include <alloc.h>
#include <ctype.h>
#include <limits.h>
#include <math.h>               /*  voor pow()  */
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#else
#include <ctype.h>
#include <limits.h>
#include <math.h>               /*  voor pow()  */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#endif



#ifdef USEPROTOS
int read_input(void)
#else
int read_input()
#endif
{
int rc = OK;

   if ((input_file=fopen(input_name, INPUTMODE))==NULL)
      return(OPENIN);

   if (plot==TRUE && (learn_file=fopen(learn_name, LEARNMODE))==NULL)
      return(OPNLRN);

   if (plot==TRUE && (test_file=fopen(test_name, TESTMODE))==NULL)
      return(OPNTST);

   setbuf(input_file, file_buffer);

   while (rc==OK)
   {
      switch (rc=read_command())
      {
         case READ_INPUT:
            rc=read_neurons(INPUT);
            break;
         case READ_HIDDEN:
            rc=read_neurons(HIDDEN);
            break;
         case READ_OUTPUT:
            rc=read_neurons(OUTPUT);
            break;
         case READ_EPSILON:
            rc=read_epsilon();
            break;
         case READ_MOMENTUM:
            rc=read_momentum();
            break;
         case READ_ERROR:
            rc=read_error();
            break;
         case READ_PATTERNS:
            rc=read_patterns();
            break;
         case READ_DISPLAY:
            rc=read_display();
            break;
#ifndef NOITER
         case READ_ITER:
            rc=read_iter();
            break;
#endif
         case READ_LEARN:
            rc=read_learn();
            break;
         case READ_DECAY:
            rc=read_decay();
            break;
         case READ_STEP:
            rc=read_step();
            break;
         case READ_SEED:
            rc=read_seed();
            break;
         case READ_DONTCARE:
            rc=read_dontcare();
            break;
         case READ_PSELECT:
            rc=read_pselect();
            break;
         case READ_TOLERANCE:
            rc=read_tolerance();
            break;
         case READ_MAXWEIGHT:
            rc=read_maxweight();
            break;
         case READ_FORGET:
            rc=read_forget();
            break;
         case READ_DMOMENTUM:
            rc=read_dmomentum();
            break;
         case READ_ADDRATE:
            rc=read_addrate();
            break;
         case READ_MULRATE:
            rc=read_mulrate();
            break;
         case READ_ADJUST:
            rc=read_adjust();
            break;
      }
   }

   rc= rc==FILEND ? OK : rc;

   if (fclose(input_file)==EOF)
      return(CLOSIN);

   if (plot==TRUE && fclose(learn_file)==EOF)
      return(CLSLRN);

   if (plot==TRUE && fclose(test_file)==EOF)
      return(CLSTST);

   return(rc);
}



#ifdef USEPROTOS
int read_line(FILE *fp)
#else
int read_line(fp)
       FILE *fp;
#endif
{
char *input_lineptr;

   if (fgets(input_line, MAXLINE, fp)==NULL)
      if (feof(fp))
         return(FILEND);

   input_row++;
   input_colom=0;

   if ((input_lineptr=strstr(input_line, "//"))!=NULL)
      *input_lineptr='\0';

   return(OK);
}



#ifdef USEPROTOS
char *get_next_word(void)
#else
char *get_next_word()
#endif
{
char token;
int rc, index;
boolean stop, string;

   rc=OK;

   input_word[0]='\0';


   token=input_line[input_colom];

   while (rc!=FILEND && (isspace(token) || token=='\0'))
   {
      if (token=='\0')
         rc=read_line(input_file);
      else
         input_colom++;

      token=input_line[input_colom];
   }


   string=FALSE;

   if (token=='\"')
   {
      string=TRUE;
      token=input_line[++input_colom];
   }


   stop=FALSE;

   index=0;

   do
   {
      if (string && token=='\"')
      {
         string=FALSE;
         token=input_line[++input_colom];
      }
      else
      {
         if (isalnum(token) || ispunct(token) ||
             (string && (token==' ' || token=='\t')))
            input_word[index++]=token;

         input_word[index]='\0';

         if (token=='\0')
            rc=read_line(input_file);
         else
            input_colom++;

         token=input_line[input_colom];
      }


      if (!string &&
          ((token=='.' && input_line[input_colom+1]=='.') ||
          (token=='<' && input_line[input_colom+1]=='=') ||
          strcmp(input_word, "..")==0 || strcmp(input_word, "<=")==0 ||
          !(isalnum(input_word[0]) || strchr("+-*xX.<>=!|", input_word[0])!=NULL) ||
          !(isalnum(token) || strchr("+-*xX.<>=!|", token)!=NULL) ||
          (isdigit(input_word[0]) && token=='*') ||
          (isdigit(token) && input_word[0]=='*')))
         stop=TRUE;

      if (stop && index==1)
         if ((token=='.' && input_word[0]=='.') ||
             (token=='=' && input_word[0]=='<'))
            stop=FALSE;

   } while (!(rc==FILEND || stop) || string);

   if (rc==FILEND)
      return(NULL);

   return(input_word);
}



#ifdef USEPROTOS
int read_command(void)
#else
int read_command()
#endif
{
char *command;

   command=get_next_word();

   if (command==NULL)
      return(FILEND);
   else
    if (strcmp(command, "INPUT")==0)
       return(READ_INPUT);
    else
     if (strcmp(command, "HIDDEN")==0)
        return(READ_HIDDEN);
     else
      if (strcmp(command, "OUTPUT")==0)
         return(READ_OUTPUT);
      else
       if (strcmp(command, "RATE")==0)
          return(READ_EPSILON);
       else
        if (strcmp(command, "MOMENTUM")==0)
           return(READ_MOMENTUM);
        else
         if (strcmp(command, "ERROR")==0)
            return(READ_ERROR);
         else
          if (strcmp(command, "PATTERNS")==0)
             return(READ_PATTERNS);
          else
           if (strcmp(command, "DISPLAY")==0)
              return(READ_DISPLAY);
           else
#ifndef NOITER
            if (strcmp(command, "ITER")==0)
               return(READ_ITER);
            else
#endif
             if (strcmp(command, "LEARN")==0)
                return(READ_LEARN);
             else
              if (strcmp(command, "UPDATE")==0)
                 return(READ_DECAY);
              else
               if (strcmp(command, "MAXSTEP")==0)
                  return(READ_STEP);
               else
                if (strcmp(command, "SEED")==0)
                   return(READ_SEED);
                else
                 if (strcmp(command, "DONTCARE")==0)
                    return(READ_DONTCARE);
                 else
                  if (strcmp(command, "PSELECT")==0)
                     return(READ_PSELECT);
                  else
                   if (strcmp(command, "TOLERANCE")==0)
                      return(READ_TOLERANCE);
                   else
                    if (strcmp(command, "MAXWEIGHT")==0)
                       return(READ_MAXWEIGHT);
                    else
                     if (strcmp(command, "FORGET")==0)
                        return(READ_FORGET);
                     else
                      if (strcmp(command, "DMOMENTUM")==0)
                         return(READ_DMOMENTUM);
                      else
                       if (strcmp(command, "ADDRATE")==0)
                          return(READ_ADDRATE);
                       else
                        if (strcmp(command, "MULRATE")==0)
                           return(READ_MULRATE);
                        else
                         if (strcmp(command, "ADJUST")==0)
                            return(READ_ADJUST);
                         else
                            return(UNKNWN);
}



#ifdef USEPROTOS
int read_neurons(char type)
#else
int read_neurons(type)
       char type;
#endif
{
int rc = OK;
char *neuron_tok;
int number, tonumber, fromnumber, from2number, tofromnumber,
    index, fromindex, tmp;
Neuronptr neuronptr, fromptr, from2ptr;
FLOAT bias, weight;
char function;
char constraint[MAXSTR];
boolean delay, changed;

   neuron_tok=get_next_word();

   if (neuron_tok==NULL)
      return(ENINFO);

   if (*neuron_tok!=':')
      return(CNEXPD);

   neuron_tok=get_next_word();

   if (neuron_tok==NULL)
      return(ENINFO);

   while (strcmp(neuron_tok, "END")!=0)
   {
      if ((number=atoi(neuron_tok))<=0)
         return(INVNUM);

      tonumber=number;

      neuron_tok=get_next_word();

      if (neuron_tok==NULL)
         return(ENINFO);

      if (strcmp(neuron_tok, "..")==0)
      {
         neuron_tok=get_next_word();

         if (neuron_tok==NULL)
            return(ENINFO);

         if ((tonumber=atoi(neuron_tok))<=0)
            return(INVNUM);

         if (number>tonumber)
         {
            tmp=number;
            number=tonumber;
            tonumber=tmp;
         }

         neuron_tok=get_next_word();

         if (neuron_tok==NULL)
            return(ENINFO);
      }


      if (*neuron_tok==':')
      {
         neuron_tok=get_next_word();

         if (neuron_tok==NULL)
            return(ENINFO);

         if (strcmp(neuron_tok, "STEP")==0)
            function=STEP;
         else
          if (strcmp(neuron_tok, "LOGISTIC")==0)
             function=LOGISTIC;
          else
           if (strcmp(neuron_tok, "LOGC4")==0)
              function=LOGC4;
           else
            if (strcmp(neuron_tok, "LOGC8")==0)
               function=LOGC8;
            else
             if (strcmp(neuron_tok, "LINEAR")==0)
                function=LINEAR;
             else
                return(UNFUNC);

         neuron_tok=get_next_word();

         if (neuron_tok==NULL)
            return(ENINFO);
      }
      else
         return(NOFUNC);


      bias=0;

      strcpy(constraint, "");

      if (*neuron_tok==':')
      {
         neuron_tok=get_next_word();

         if (neuron_tok==NULL)
            return(ENINFO);

         do
         {
            changed=FALSE;

            if (strspn(neuron_tok, "<>=")==strlen(neuron_tok))
            {
               strcat(constraint, neuron_tok);

               changed=TRUE;

               neuron_tok=get_next_word();

               if (neuron_tok==NULL)
                  return(ENINFO);
            }
            else
            if (isdigit(*neuron_tok) || strchr("+-.", *neuron_tok)!=NULL)
            {
               bias=atof(neuron_tok);

               bias=max(bias, -maxweight);
               bias=min(bias, maxweight);

               /*  verander 'changed' niet: na een getal gewoon verder  */

               neuron_tok=get_next_word();

               if (neuron_tok==NULL)
                  return(ENINFO);
            }
            else
               return(BADCNS);

         } while (changed);
      }


      for (index=number; index<=tonumber; index++)
      {
         if ((neuronptr=add_neuron(index))==NULL)
            return(MEMFUL);

         if (neuronptr->type==TYPE)
         {
            switch (type)
            {
               case INPUT:
                  if ((rc=add_input(neuronptr))!=OK)
                     return(rc);
                  break;
               case OUTPUT:
                  if ((rc=add_output(neuronptr))!=OK)
                     return(rc);
                  break;
            }
         }
         else
            return(DBLDEF);

         neuronptr->type=type;

         neuronptr->bias=bias;
         neuronptr->orgbias=bias;

         if (*constraint!='\0')
         {
            if ((neuronptr->constraint=(char *)malloc(strlen(constraint)+1))==NULL)
               return(MEMFUL);
            strcpy(neuronptr->constraint, constraint);
         }

         neuronptr->function=function;
      }


      if (strcmp(neuron_tok, "<=")==0)
      {
         do
         {
            neuron_tok=get_next_word();

            if (neuron_tok==NULL)
               return(ENINFO);

            if ((fromnumber=atoi(neuron_tok))<=0)
               return(INVNUM);

            tofromnumber=fromnumber;

            neuron_tok=get_next_word();

            if (neuron_tok==NULL)
               return(ENINFO);

            from2number=0;

            if (strcmp(neuron_tok, "..")==0)
            {
               neuron_tok=get_next_word();

               if (neuron_tok==NULL)
                  return(ENINFO);

               if ((tofromnumber=atoi(neuron_tok))<=0)
                  return(INVNUM);

               if (fromnumber>tofromnumber)
               {
                  tmp=fromnumber;
                  fromnumber=tofromnumber;
                  tofromnumber=tmp;
               }

               neuron_tok=get_next_word();

               if (neuron_tok==NULL)
                  return(ENINFO);
            }
            else
               if (strcmp(neuron_tok, "*")==0)
               {
                  neuron_tok=get_next_word();

                  if (neuron_tok==NULL)
                     return(ENINFO);

                  if ((from2number=atoi(neuron_tok))<=0)
                     return(INVNUM);

                  neuron_tok=get_next_word();

                  if (neuron_tok==NULL)
                     return(ENINFO);
               }


            delay=FALSE;

            if (*neuron_tok=='~')
            {
               delay=TRUE;

               neuron_tok=get_next_word();

               if (neuron_tok==NULL)
                  return(ENINFO);
            }


            weight=0;

            strcpy(constraint, "");

            if (*neuron_tok==':')
            {
               neuron_tok=get_next_word();

               if (neuron_tok==NULL)
                  return(EWINFO);

               do
               {
                  changed=FALSE;

                  if (strspn(neuron_tok, "<>=!|")==strlen(neuron_tok))
                  {
                     strcat(constraint, neuron_tok);

                     changed=TRUE;

                     neuron_tok=get_next_word();

                     if (neuron_tok==NULL)
                        return(EWINFO);
                  }
                  else
                  if (isdigit(*neuron_tok) || strchr("+-.", *neuron_tok)!=NULL)
                  {
                     weight=atof(neuron_tok);

                     weight=max(weight, -maxweight);
                     weight=min(weight, maxweight);

                     /*  verander 'changed' niet: na een getal gewoon verder  */

                     neuron_tok=get_next_word();

                     if (neuron_tok==NULL)
                        return(EWINFO);
                  }
                  else
                     return(BADCNS);

               } while (changed);
            }

            if (from2number==0)
            {
               for (fromindex=fromnumber; fromindex<=tofromnumber; fromindex++)
               {
                  if ((fromptr=add_neuron(fromindex))==NULL)
                     return(MEMFUL);

                  for (index=number; index<=tonumber; index++)
                  {
                     neuronptr=get_neuron(index);
                     if (connect_neuron(neuronptr, fromptr, NULL,
                                        weight, constraint, delay)!=OK)
                        return(CNTCON);
                  }
               }
            }
            else
            {
               if ((fromptr=add_neuron(fromnumber))==NULL)
                  return(MEMFUL);

               if ((from2ptr=add_neuron(from2number))==NULL)
                  return(MEMFUL);

               for (index=number; index<=tonumber; index++)
               {
                  neuronptr=get_neuron(index);
                  if (connect_neuron(neuronptr, fromptr, from2ptr,
                                     weight, constraint, delay)!=OK)
                     return(CNTCON);
               }
            }

         } while (*neuron_tok==',');

      }

      if (*neuron_tok!=';')
         return(SCEXPD);

      neuron_tok=get_next_word();

      if (neuron_tok==NULL)
         return(ENINFO);
   }

   neuron_tok=get_next_word();

   if (neuron_tok==NULL)
      return(ENINFO);

   if (*neuron_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_epsilon(void)
#else
int read_epsilon()
#endif
{
char *epsilon_tok;

   epsilon_tok=get_next_word();

   if (epsilon_tok==NULL)
      return(EAINFO);

   if (*epsilon_tok!=':')
      return(CNEXPD);

   epsilon_tok=get_next_word();

   if (epsilon_tok==NULL)
      return(EAINFO);

   if ((epsilon=atof(epsilon_tok))<=0)
      return(BADEPS);

   epsilon_tok=get_next_word();

   if (epsilon_tok==NULL)
      return(EAINFO);

   if (*epsilon_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_momentum(void)
#else
int read_momentum()
#endif
{
char *momentum_tok;

   momentum_tok=get_next_word();

   if (momentum_tok==NULL)
      return(EMINFO);

   if (*momentum_tok!=':')
      return(CNEXPD);

   momentum_tok=get_next_word();

   if (momentum_tok==NULL)
      return(EMINFO);

   if ((momentum=atof(momentum_tok))<0 || momentum>1)
      return(BADMOM);

   momentum_tok=get_next_word();

   if (momentum_tok==NULL)
      return(EMINFO);

   if (*momentum_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_error(void)
#else
int read_error()
#endif
{
char *error_tok;

   error_tok=get_next_word();

   if (error_tok==NULL)
      return(EEINFO);

   if (*error_tok!=':')
      return(CNEXPD);

   error_tok=get_next_word();

   if (error_tok==NULL)
      return(EEINFO);

   if ((error=atof(error_tok))<=0)
      return(BADERR);

   error_tok=get_next_word();

   if (error_tok==NULL)
      return(EEINFO);

   if (*error_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_patterns(void)
#else
int read_patterns()
#endif
{
int rc;
char *pattern_tok;
int  patlen, sublen;
int  pair;
int  nrbits, distance;
int  index;
boolean test, set, changed;
FLOAT relative;
char type, mode;
FILE *pat_file;

   if (seed==0)
      Srand(gseed=(unsigned) time(NULL));
   else                                /*  geef random-generator  */
      Srand(seed);                     /*  een startwaarde        */

   pattern_tok=get_next_word();

   if (pattern_tok==NULL)
      return(EPINFO);

   if (*pattern_tok!=':')
      return(CNEXPD);

   pattern_tok=get_next_word();

   if (pattern_tok==NULL)
      return(EPINFO);

   if ((width_in=atoi(pattern_tok))<1)
      return(INVWDT);

   pattern_tok=get_next_word();

   if (pattern_tok==NULL)
      return(EPINFO);

   height_in=1;

   if (*pattern_tok==',')
   {
      pattern_tok=get_next_word();

      if (pattern_tok==NULL)
         return(EPINFO);

      if ((height_in=atoi(pattern_tok))<1)
         return(INVHGT);

      pattern_tok=get_next_word();

      if (pattern_tok==NULL)
         return(EPINFO);
   }

   if ((width_out=atoi(pattern_tok))<1)
      return(INVWDT);

   pattern_tok=get_next_word();

   if (pattern_tok==NULL)
      return(EPINFO);

   height_out=1;

   if (*pattern_tok==',')
   {
      pattern_tok=get_next_word();

      if (pattern_tok==NULL)
         return(EPINFO);

      if ((height_out=atoi(pattern_tok))<1)
         return(INVHGT);

      pattern_tok=get_next_word();

      if (pattern_tok==NULL)
         return(EPINFO);
   }

   if (nrinputs!=width_in*height_in)
      return(NEQINP);

   if (nroutputs!=width_out*height_out)
      return(NEQOUT);

   input_pattern=(FLOAT *)malloc(nrinputs*sizeof(*(*patternlistptr)->in));
   output_pattern=(FLOAT *)malloc(nroutputs*sizeof(*(*patternlistptr)->out));

   if (input_pattern==NULL || output_pattern==NULL)
      return(MEMFUL);


   test=FALSE;

   pat_file=learn_file;

   while (strcmp(pattern_tok, "END")!=0)
   {
      if (strcmp(pattern_tok, "TEST")==0)
      {
         pattern_tok=get_next_word();

         if (pattern_tok==NULL)
            return(EPINFO);

         if (*pattern_tok!=':')
            return(CNEXPD);

         if (!test)
            test=TRUE;
         else
            return(TESTST);

         pat_file=test_file;    /*  schakel over op test patroon bestand  */

         pattern_tok=get_next_word();

         if (pattern_tok==NULL)
            return(EPINFO);

         if (strcmp(pattern_tok, "END")==0)
            break;              /*  test nogmaals op END  */
      }


      relative=1.0;

      type=BIT;

      do
      {
         changed=FALSE;

         if (strcmp(pattern_tok, "RELATIVE")==0)
         {
            pattern_tok=get_next_word();

            if (pattern_tok==NULL)
               return(EPINFO);

            if (*pattern_tok!=':')
               return(CNEXPD);

            pattern_tok=get_next_word();

            if (pattern_tok==NULL)
               return(EPINFO);

            if ((relative=atof(pattern_tok))<0 || relative>1)
               return(BADREL);

            changed=TRUE;

            pattern_tok=get_next_word();

            if (pattern_tok==NULL)
               return(EPINFO);
         }

         if (strcmp(pattern_tok, "BIT")==0)
         {
            pattern_tok=get_next_word();

            if (pattern_tok==NULL)
               return(EPINFO);

            if (*pattern_tok!=':')
               return(CNEXPD);

            type=BIT;

            changed=TRUE;

            pattern_tok=get_next_word();

            if (pattern_tok==NULL)
               return(EPINFO);
         }
         else
          if (strcmp(pattern_tok, "REAL")==0)
          {
             pattern_tok=get_next_word();

             if (pattern_tok==NULL)
                return(EPINFO);

             if (*pattern_tok!=':')
                return(CNEXPD);

             type=REAL;

             changed=TRUE;

             pattern_tok=get_next_word();

             if (pattern_tok==NULL)
                return(EPINFO);
          }

      } while (changed);


      if (plot==TRUE)
         fprintf(pat_file, "RELATIVE: %6.4f\n", relative);

      if (plot==TRUE)
         if (type==BIT)
            fprintf(pat_file, "BIT:\n");
         else
            fprintf(pat_file, "REAL:\n");


      set=FALSE;

      pair=0;

      if (strcmp(pattern_tok, "SET")==0)
      {
         pattern_tok=get_next_word();

         if (pattern_tok==NULL)
            return(EPINFO);

         if (*pattern_tok!=':')
            return(CNEXPD);

         if (!set)
            set=TRUE;
         else
            return(SETSET);

         if (plot==TRUE)
            fprintf(pat_file, "SET:\n");

         pattern_tok=get_next_word();

         if (pattern_tok==NULL)
            return(EPINFO);
      }


      do
      {
         patlen=0;

         while (patlen!=width_in*height_in)
         {
            if (patlen>width_in*height_in)
               return(PATLNG);

            if (type==BIT)
            {
               mode=NONE;

               if (strcmp(pattern_tok, "ALPHA")==0)
                  mode=ALPHA;
               else
               if (strcmp(pattern_tok, "NUMBER")==0)
                  mode=NUMBER;
               else
               if (strcmp(pattern_tok, "ATOH")==0)
                  mode=ATOH;
               else
               if (strcmp(pattern_tok, "BTOH")==0)
                  mode=BTOH;
               else
               if (strcmp(pattern_tok, "FTOH")==0)
                  mode=FTOH;
               else
               if (strcmp(pattern_tok, "NTOH")==0)
                  mode=NTOH;

               if (mode!=NONE)
               {
                  pattern_tok=get_next_word();

                  if (pattern_tok==NULL)
                     return(EPINFO);

                  if (*pattern_tok!=':')
                     return(CNEXPD);

                  pattern_tok=get_next_word();

                  if (pattern_tok==NULL)
                     return(EPINFO);

                  if ((nrbits=atoi(pattern_tok))<0 || nrbits>MAXPATLEN)
                     return(TMBITS);

                  pattern_tok=get_next_word();

                  if (pattern_tok==NULL)
                     return(EPINFO);

                  distance=0;

                  if (mode==ATOH || mode==BTOH || mode==FTOH || mode==NTOH)
                  {
                     if ((distance=atoi(pattern_tok))<0 || distance>MAXPATLEN)
                        return(DSTLRG);

                     pattern_tok=get_next_word();

                     if (pattern_tok==NULL)
                        return(EPINFO);
                  }

                  if (plot==TRUE)
                     switch(mode)
                     {
                     case ALPHA:
                        fprintf(pat_file, "ALPHA: %d  ", nrbits);
                        break;
                     case NUMBER:
                        fprintf(pat_file, "NUMBER: %d  ", nrbits);
                        break;
                     case ATOH:
                        fprintf(pat_file, "ATOH: %d %d  ", nrbits, distance);
                        break;
                     case BTOH:
                        fprintf(pat_file, "BTOH: %d %d  ", nrbits, distance);
                        break;
                     case FTOH:
                        fprintf(pat_file, "FTOH: %d %d  ", nrbits, distance);
                        break;
                     case NTOH:
                        fprintf(pat_file, "NTOH: %d %d  ", nrbits, distance);
                        break;
                     }

                  if (plot==TRUE)
                     if (mode==ALPHA || mode==ATOH)
                        fprintf(pat_file, "\"%s\"   ", pattern_tok);
                     else
                        fprintf(pat_file, "%s   ", pattern_tok);

                  if ((rc=convert_to_bitpat(mode, pattern_tok, nrbits, distance, &pattern_tok))!=OK)
                     return(rc);
               }
               else
                  if (plot==TRUE)
                     fprintf(pat_file, "%s", pattern_tok);

               sublen=strlen(pattern_tok);

               if (strspn(pattern_tok, "-*xX")!=sublen)
                  return(BADPAT);

               index=0;
               while (index<sublen && patlen<=nrinputs)
               {
                  if (pattern_tok[index]=='-')
                     input_pattern[patlen]=MINOUTPUT;
                  else
                   if (pattern_tok[index]=='x')
                      input_pattern[patlen]=UNDEFINED;
                   else
                    if (pattern_tok[index]=='X')
                       input_pattern[patlen]=MAXOUTPUT;
                    else
                     if (pattern_tok[index]=='*')
                        input_pattern[patlen]=DONTCARE;

                  patlen++;
                  index++;
               }

               pattern_tok=get_next_word();

               if (pattern_tok==NULL)
                  return(EPINFO);
            }
            else
            {
               if (plot==TRUE)
                  fprintf(pat_file, "%s   ", pattern_tok);

               sublen=strlen(pattern_tok);

               if (strspn(pattern_tok, ".0123456789x*")!=sublen)
                  return(BADPAT);

               if (*pattern_tok=='x')
                  input_pattern[patlen]=UNDEFINED;
               else
               if (*pattern_tok=='*')
                  input_pattern[patlen]=DONTCARE;
               else
                  if ((input_pattern[patlen]=
                       atof(pattern_tok)*MAXMINOUT+MINOUTPUT)==UNDEFINED)
                     input_pattern[patlen]+=1E-8;
                                       /*  tel een kleine waarde op bij  */
                                       /*  pattern waarde om UNDEFINED   */
                                       /*  waarde te vermijden           */

               patlen++;

               pattern_tok=get_next_word();

               if (pattern_tok==NULL)
                  return(EPINFO);
            }
         }

         if (plot==TRUE)
         {
            fprintf(pat_file, "\n");
            fprintf_pat(pat_file, type, nrinputs, input_pattern);
         }


         patlen=0;

         while (patlen!=width_out*height_out)
         {
            if (patlen>width_out*height_out)
               return(PATLNG);

            if (type==BIT)
            {
               mode=NONE;

               if (strcmp(pattern_tok, "ALPHA")==0)
                  mode=ALPHA;
               else
               if (strcmp(pattern_tok, "NUMBER")==0)
                  mode=NUMBER;
               else
               if (strcmp(pattern_tok, "ATOH")==0)
                  mode=ATOH;
               else
               if (strcmp(pattern_tok, "BTOH")==0)
                  mode=BTOH;
               else
               if (strcmp(pattern_tok, "FTOH")==0)
                  mode=FTOH;
               else
               if (strcmp(pattern_tok, "NTOH")==0)
                  mode=NTOH;

               if (mode!=NONE)
               {
                  pattern_tok=get_next_word();

                  if (pattern_tok==NULL)
                     return(EPINFO);

                  if (*pattern_tok!=':')
                     return(CNEXPD);

                  pattern_tok=get_next_word();

                  if (pattern_tok==NULL)
                     return(EPINFO);

                  if ((nrbits=atoi(pattern_tok))<0 || nrbits>MAXPATLEN)
                     return(TMBITS);

                  pattern_tok=get_next_word();

                  if (pattern_tok==NULL)
                     return(EPINFO);

                  distance=0;

                  if (mode==ATOH || mode==BTOH || mode==FTOH || mode==NTOH)
                  {
                     if ((distance=atoi(pattern_tok))<0 || distance>MAXPATLEN)
                        return(DSTLRG);

                     pattern_tok=get_next_word();

                     if (pattern_tok==NULL)
                        return(EPINFO);
                  }

                  if (plot==TRUE)
                     switch(mode)
                     {
                     case ALPHA:
                        fprintf(pat_file, "ALPHA: %d  ", nrbits);
                        break;
                     case NUMBER:
                        fprintf(pat_file, "NUMBER: %d  ", nrbits);
                        break;
                     case ATOH:
                        fprintf(pat_file, "ATOH: %d %d  ", nrbits, distance);
                        break;
                     case BTOH:
                        fprintf(pat_file, "BTOH: %d %d  ", nrbits, distance);
                        break;
                     case FTOH:
                        fprintf(pat_file, "FTOH: %d %d  ", nrbits, distance);
                        break;
                     case NTOH:
                        fprintf(pat_file, "NTOH: %d %d  ", nrbits, distance);
                        break;
                     }

                  if (plot==TRUE)
                     if (mode==ALPHA || mode==ATOH)
                        fprintf(pat_file, "\"%s\"   ", pattern_tok);
                     else
                        fprintf(pat_file, "%s   ", pattern_tok);

                  if ((rc=convert_to_bitpat(mode, pattern_tok, nrbits, distance, &pattern_tok))!=OK)
                     return(rc);
               }
               else
                  if (plot==TRUE)
                     fprintf(pat_file, "%s", pattern_tok);

               sublen=strlen(pattern_tok);

               if (strspn(pattern_tok, "-*xX")!=sublen)
                  return(BADPAT);

               index=0;
               while (index<sublen && patlen<=nroutputs)
               {
                  if (pattern_tok[index]=='-')
                     output_pattern[patlen]=MINOUTPUT;
                  else
                   if (pattern_tok[index]=='x')
                      output_pattern[patlen]=UNDEFINED;
                   else
                    if (pattern_tok[index]=='X')
                       output_pattern[patlen]=MAXOUTPUT;
                    else
                     if (pattern_tok[index]=='*')
                        output_pattern[patlen]=DONTCARE;

                  patlen++;
                  index++;
               }

               pattern_tok=get_next_word();

               if (pattern_tok==NULL)
                  return(EPINFO);
            }
            else
            {
               if (plot==TRUE)
                  fprintf(pat_file, "%s   ", pattern_tok);

               sublen=strlen(pattern_tok);

               if (strspn(pattern_tok, ".0123456789x*")!=sublen)
                  return(BADPAT);

               if (*pattern_tok=='x')
                  output_pattern[patlen]=UNDEFINED;
               else
               if (*pattern_tok=='*')
                  output_pattern[patlen]=DONTCARE;
               else
                  if ((output_pattern[patlen]=
                       atof(pattern_tok)*MAXMINOUT+MINOUTPUT)==UNDEFINED)
                     output_pattern[patlen]+=1E-8;
                                       /*  tel een kleine waarde op bij  */
                                       /*  pattern waarde om UNDEFINED   */
                                       /*  waarde te vermijden           */

               patlen++;

               pattern_tok=get_next_word();

               if (pattern_tok==NULL)
                  return(EPINFO);
            }
         }

         if (plot==TRUE)
         {
            fprintf(pat_file, "\n");
            fprintf_pat(pat_file, type, nroutputs, output_pattern);
         }


         if (add_pattern(type, pair++, input_pattern, output_pattern, relative)!=OK)
            return(CNTPAT);

      } while (set==TRUE && strcmp(pattern_tok, "END")!=0);

      if (set==TRUE)
      {
         pattern_tok=get_next_word();

         if (pattern_tok==NULL)
            return(EPINFO);

         if (*pattern_tok!=';')
            return(SCEXPD);

         if (plot==TRUE)
            fprintf(pat_file, "END;\n");

         pattern_tok=get_next_word();

         if (pattern_tok==NULL)
            return(EPINFO);
      }

      if (test)
         nrtests++;
      else
         nrpatterns++;
   }

   pattern_tok=get_next_word();

   if (pattern_tok==NULL)
      return(EPINFO);

   if (*pattern_tok!=';')
      return(SCEXPD);

   FREE(input_pattern);
   FREE(output_pattern);

   input_pattern=NULL;
   output_pattern=NULL;                /*  herhaald in cleanup()  */

   return(OK);
}



#ifdef USEPROTOS
void fprintf_pat(FILE *output_file, char type, int length, FLOAT *pattern)
#else
void fprintf_pat(output_file, type, length, pattern)
        FILE  *output_file;
        char   type;
        int    length;
        FLOAT *pattern;
#endif
{
int index;
FLOAT elemval;

   for (index=0; index<length; index++)
   {
      elemval=pattern[index];
      if (type==BIT)
      {                /*  als bit pattern dan uitvoer in bits  */
         fprintf(output_file, "%c",
                 elemval==UNDEFINED ? 'x'
                             : (elemval==MAXOUTPUT ? 'X'
                                : (elemval==DONTCARE ? '*' : '-')));
      }
      else
      {                /*  anders uitvoer in reeele waarden  */
         if (elemval==DONTCARE)
            fprintf(output_file, "    *  ");
         else
         if (elemval==UNDEFINED)
            fprintf(output_file, "    x  ");
         else          /*  normaliseer tot logisch niveau  */
            fprintf(output_file, "%6.4f ", (elemval-MINOUTPUT)/MAXMINOUT);
      }
   }
   fprintf(output_file, "\n");
}



#ifdef USEPROTOS
int read_display(void)
#else
int read_display()
#endif
{
char *display_tok;

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=':')
      return(CNEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((left_in=atoi(display_tok))<0)
      return(BADLIN);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((top_in=atoi(display_tok))<0)
      return(BADTIN);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((right_in=atoi(display_tok))<0 || right_in>=VIRTUALX)
      return(BADRIN);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((bottom_in=atoi(display_tok))<0 || bottom_in>=VIRTUALY)
      return(BADBIN);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
   {
      if (strlen(display_tok)>MAXSTR)
         return(STRLNG);

      strcpy(inputstr, display_tok);

      if ((display_tok=get_next_word())==NULL)
         return(EDINFO);

      if (*display_tok!=',')
         return(CMEXPD);
   }

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((left_out=atoi(display_tok))<0)
      return(BDLOUT);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((top_out=atoi(display_tok))<0)
      return(BDTOUT);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((right_out=atoi(display_tok))<0 || right_out>=VIRTUALX)
      return(BDROUT);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((bottom_out=atoi(display_tok))<0 || bottom_out>=VIRTUALY)
      return(BDBOUT);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
   {
      if (strlen(display_tok)>MAXSTR)
         return(STRLNG);

      strcpy(outputstr, display_tok);

      if ((display_tok=get_next_word())==NULL)
         return(EDINFO);

      if (*display_tok!=',')
         return(CMEXPD);
   }

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((left_res=atoi(display_tok))<0)
      return(BDLRES);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((top_res=atoi(display_tok))<0)
      return(BDTRES);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((right_res=atoi(display_tok))<0 || right_res>=VIRTUALX)
      return(BDRRES);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((bottom_res=atoi(display_tok))<0 || bottom_res>=VIRTUALY)
      return(BDBRES);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
   {
      if (strlen(display_tok)>MAXSTR)
         return(STRLNG);

      strcpy(resultstr, display_tok);

      if ((display_tok=get_next_word())==NULL)
         return(EDINFO);

      if (*display_tok!=',')
         return(CMEXPD);
   }

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((left_info=atoi(display_tok))<0)
      return(BDLINF);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((top_info=atoi(display_tok))<0)
      return(BDTINF);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((right_info=atoi(display_tok))<0 || right_info>=VIRTUALX)
      return(BDRINF);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if ((bottom_info=atoi(display_tok))<0 || bottom_info>=VIRTUALY)
      return(BDBINF);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=',')
      return(CMEXPD);

   if ((display_tok=get_next_word())==NULL)
      return(EDINFO);

   if (*display_tok!=';')
   {
      if (strlen(infostr)+strlen(display_tok)>MAXSTR)
         return(STRLNG);

      strcpy(infostr, display_tok);

      if ((display_tok=get_next_word())==NULL)
         return(EDINFO);

      if (*display_tok!=';')
         return(SCEXPD);
   }

   return(OK);
}



#ifndef NOITER
#ifdef USEPROTOS
int read_iter(void)
#else
int read_iter()
#endif
{
char *iter_tok;

   iter_tok=get_next_word();

   if (iter_tok==NULL)
      return(EIINFO);

   if (*iter_tok!=':')
      return(CNEXPD);

   iter_tok=get_next_word();

   if (iter_tok==NULL)
      return(EIINFO);

   if ((max_iter=atoi(iter_tok))<=0)
      return(BADITR);

   iter_tok=get_next_word();

   if (iter_tok==NULL)
      return(EIINFO);

   if (*iter_tok!=';')
      return(SCEXPD);

   return(OK);
}
#endif



#ifdef USEPROTOS
int read_learn(void)
#else
int read_learn()
#endif
{
char *learn_tok;

   learn_tok=get_next_word();

   if (learn_tok==NULL)
      return(ELINFO);

   if (*learn_tok!=':')
      return(CNEXPD);


   learn=LEARN;

   do
   {
      learn_tok=get_next_word();

      if (learn_tok==NULL)
         return(ELINFO);

      if (strcmp(learn_tok, "FORCED")==0)
         learn|=FORCED;
      else
       if (strcmp(learn_tok, "HEBB")==0)
          learn|=HEBB;
       else
        if (strcmp(learn_tok, "FBPR")==0)
           learn|=FBPR;
        else
         if (strcmp(learn_tok, "SELF")==0)
            learn|=SELF;
         else
          if (strcmp(learn_tok, "DHEBB")==0)
             learn|=DHEBB;
          else
           if (strcmp(learn_tok, "GROSSBERG")==0)
              learn|=GROSSBERG;
           else
            if (strcmp(learn_tok, "CLIP424")==0)
               learn|=CLIP424;
            else
             if (strcmp(learn_tok, "CLIP262")==0)
                learn|=CLIP262;
             else
              if (strcmp(learn_tok, "ANNEAL")==0)
                 learn|=ANNEAL;
              else
               if (strcmp(learn_tok, "RANDOM")==0)
                  learn|=RANDOM;
               else
                if (strcmp(learn_tok, "USERERR")==0)
                   learn|=USERERR;
                else
                   return(BADLRN);

      learn_tok=get_next_word();

      if (learn_tok==NULL)
         return(ELINFO);

   } while (*learn_tok=='&');

   if (*learn_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_decay(void)
#else
int read_decay()
#endif
{
char *decay_tok;

   decay_tok=get_next_word();

   if (decay_tok==NULL)
      return(ECINFO);

   if (*decay_tok!=':')
      return(CNEXPD);

   decay_tok=get_next_word();

   if (decay_tok==NULL)
      return(ECINFO);

   decay=atoi(decay_tok);

   if (decay<0)
      decay_val=1;
   else
      decay_val=10;

   decay_tok=get_next_word();

   if (decay_tok==NULL)
      return(ECINFO);

   if (*decay_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_step(void)
#else
int read_step()
#endif
{
char *step_tok;

   step_tok=get_next_word();

   if (step_tok==NULL)
      return(ESINFO);

   if (*step_tok!=':')
      return(CNEXPD);

   step_tok=get_next_word();

   if (step_tok==NULL)
      return(ESINFO);

   if ((lim_step=atol(step_tok))<0)
      return(BADSTP);

   step_tok=get_next_word();

   if (step_tok==NULL)
      return(ESINFO);

   if (*step_tok!=';')
      return(SCEXPD);
   return(OK);
}



#ifdef USEPROTOS
int read_seed(void)
#else
int read_seed()
#endif
{
char *seed_tok;

   seed_tok=get_next_word();

   if (seed_tok==NULL)
      return(ERINFO);

   if (*seed_tok!=':')
      return(CNEXPD);

   seed_tok=get_next_word();

   if (seed_tok==NULL)
      return(ERINFO);

#ifdef TURBOC
   if ((seed=atoi(seed_tok))==1 || seed>65535U)
      return(BADSED);
#else
   if ((seed=atoi(seed_tok))==1 || seed>65535)
      return(BADSED);
#endif

   seed_tok=get_next_word();

   if (seed_tok==NULL)
      return(ERINFO);

   if (*seed_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_dontcare(void)
#else
int read_dontcare()
#endif
{
char *dontcare_tok;

   dontcare_tok=get_next_word();

   if (dontcare_tok==NULL)
      return(EOINFO);

   if (*dontcare_tok!=':')
      return(CNEXPD);

   dontcare_tok=get_next_word();

   if (dontcare_tok==NULL)
      return(EOINFO);

   if ((dontcare=atof(dontcare_tok))<0 || dontcare>1)
      return(BADCAR);

   dontcare_tok=get_next_word();

   if (dontcare_tok==NULL)
      return(EOINFO);

   if (*dontcare_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_pselect(void)
#else
int read_pselect()
#endif
{
char *pselect_tok;

   pselect_tok=get_next_word();

   if (pselect_tok==NULL)
      return(ETINFO);

   if (*pselect_tok!=':')
      return(CNEXPD);

   pselect_tok=get_next_word();

   if (pselect_tok==NULL)
      return(ETINFO);

   if (!strcmp(pselect_tok, "ADAPT"))
      pselect_adapt = TRUE;
   else
   if ((pselect_error=atof(pselect_tok))<0 || pselect_error>MAXPATLEN)
      return(BADPSE);

   pselect_tok=get_next_word();

   if (pselect_tok==NULL)
      return(ETINFO);

   if ((pselect_steps=atol(pselect_tok))<0 || pselect_steps>MAXINT)
      return(BADPSS);

   pselect_tok=get_next_word();

   if (pselect_tok==NULL)
      return(ETINFO);

   if (*pselect_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_tolerance(void)
#else
int read_tolerance()
#endif
{
char *tolerance_tok;

   tolerance_tok=get_next_word();

   if (tolerance_tok==NULL)
      return(ETLINF);

   if (*tolerance_tok!=':')
      return(CNEXPD);

   tolerance_tok=get_next_word();

   if (tolerance_tok==NULL)
      return(ETLINF);

   if ((tolerance=atof(tolerance_tok))<0 || tolerance>0.5)
      return(BADTOL);

   tolerance_tok=get_next_word();

   if (tolerance_tok==NULL)
      return(ETLINF);

   if (*tolerance_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_maxweight(void)
#else
int read_maxweight()
#endif
{
char *maxweight_tok;

   maxweight_tok=get_next_word();

   if (maxweight_tok==NULL)
      return(EMWINF);

   if (*maxweight_tok!=':')
      return(CNEXPD);

   maxweight_tok=get_next_word();

   if (maxweight_tok==NULL)
      return(EMWINF);

   if (fabs(maxweight=atof(maxweight_tok))>MAXWEIGHT || maxweight<=0)
      return(BADMAX);

   maxweight_tok=get_next_word();

   if (maxweight_tok==NULL)
      return(EMWINF);

   if (*maxweight_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_forget(void)
#else
int read_forget()
#endif
{
char *forget_tok;

   forget_tok=get_next_word();

   if (forget_tok==NULL)
      return(EFGINF);

   if (*forget_tok!=':')
      return(CNEXPD);

   forget_tok=get_next_word();

   if (forget_tok==NULL)
      return(EFGINF);

   if ((forget=atof(forget_tok))<0 || forget>1)
      return(BADFGT);

   forget_tok=get_next_word();

   if (forget_tok==NULL)
      return(EFGINF);

   if (*forget_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_dmomentum(void)
#else
int read_dmomentum()
#endif
{
char *dmomentum_tok;

   dmomentum_tok=get_next_word();

   if (dmomentum_tok==NULL)
      return(EDMINF);

   if (*dmomentum_tok!=':')
      return(CNEXPD);

   dmomentum_tok=get_next_word();

   if (dmomentum_tok==NULL)
      return(EDMINF);

   if ((dmomentum=atof(dmomentum_tok))<0 || dmomentum>1)
      return(BADDMM);

   dmomentum_tok=get_next_word();

   if (dmomentum_tok==NULL)
      return(EDMINF);

   if (*dmomentum_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_addrate(void)
#else
int read_addrate()
#endif
{
char *addrate_tok;

   addrate_tok=get_next_word();

   if (addrate_tok==NULL)
      return(EARINF);

   if (*addrate_tok!=':')
      return(CNEXPD);

   addrate_tok=get_next_word();

   if (addrate_tok==NULL)
      return(EARINF);

   if ((addrate=atof(addrate_tok))<0 || addrate>1)
      return(BADADR);

   addrate_tok=get_next_word();

   if (addrate_tok==NULL)
      return(EARINF);

   if (*addrate_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_mulrate(void)
#else
int read_mulrate()
#endif
{
char *mulrate_tok;

   mulrate_tok=get_next_word();

   if (mulrate_tok==NULL)
      return(EMRINF);

   if (*mulrate_tok!=':')
      return(CNEXPD);

   mulrate_tok=get_next_word();

   if (mulrate_tok==NULL)
      return(EMRINF);

   if ((mulrate=atof(mulrate_tok))<0 || mulrate>1)
      return(BADMLR);

   mulrate_tok=get_next_word();

   if (mulrate_tok==NULL)
      return(EMRINF);

   if (*mulrate_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
int read_adjust(void)
#else
int read_adjust()
#endif
{
char *adjust_tok;

   adjust_tok=get_next_word();

   if (adjust_tok==NULL)
      return(EADINF);

   if (*adjust_tok!=':')
      return(CNEXPD);

   adjust_tok=get_next_word();

   if (adjust_tok==NULL)
      return(EADINF);

   if (strcmp(adjust_tok, "RELATIVE")==0)
      adjust_mode=ADJUST_RELATIVE;
   else
   if (strcmp(adjust_tok, "LOCAL")==0)
      adjust_mode=ADJUST_LOCAL;
   else
   if (strcmp(adjust_tok, "GLOBAL")==0)
      adjust_mode=ADJUST_GLOBAL;
   else
      return(BADADG);

   adjust_tok=get_next_word();

   if (adjust_tok==NULL)
      return(EADINF);

   if (strcmp(adjust_tok, "RECURRENT")==0)
      adjust_net=ADJUST_RECURRENT;
   else
   if (strcmp(adjust_tok, "ALL")==0)
      adjust_net=ADJUST_ALL;
   else
      return(BADADA);

   adjust_tok=get_next_word();

   if (adjust_tok==NULL)
      return(EADINF);

   if (*adjust_tok!=';')
      return(SCEXPD);

   return(OK);
}



#ifdef USEPROTOS
Neuronptr add_neuron(int number)
#else
Neuronptr add_neuron(number)
             int number;
#endif
{
Neuronptr neuronptr, prvptr, newptr;

   prvptr=NULL;

   for (neuronptr=neuronlist;
        neuronptr!=NULL && neuronptr->number<=number;
        prvptr=neuronptr, neuronptr=neuronptr->next)
      if (neuronptr->number==number)
         return(neuronptr);

   if ((newptr=(Neuronptr)malloc(sizeof(Neuron)))==NULL)
      return(NULL);


   /*  initialiseer indien nieuw neuron  */

   newptr->number=number;
   newptr->ionum=0;
   newptr->type=TYPE;
   newptr->epsilon=0;
   newptr->addrate=0;
   newptr->mulrate=0;
   newptr->activation=0;
   newptr->bias=0;
   newptr->dbias=0;
   newptr->prevdbias=0;
   newptr->orgbias=0;
   newptr->constraint=NULL;
   newptr->function=FUNCTION;
   newptr->output=0;
   newptr->prevoutput=0;
   newptr->derivation=0;
   newptr->deltasum=0;
   newptr->delta=0;
   newptr->prevdelta=0;
   newptr->changed.output=FALSE;
   newptr->changed.prev=FALSE;
   newptr->changed.delta=FALSE;
   newptr->changed.extra=FALSE;
   newptr->nrweights=0;
   newptr->weights=NULL;


   /*  kort maar krachtig:  */

   newptr->next=neuronptr;

   if (neuronlist==NULL || prvptr==NULL)
      neuronlist=newptr;
   else
      prvptr->next=newptr;

   nrneurons++;

   return(newptr);
}



#ifdef USEPROTOS
Neuronptr get_neuron(int number)
#else
Neuronptr get_neuron(number)
             int number;
#endif
{
Neuronptr neuronptr;

   for (neuronptr=neuronlist;
        neuronptr!=NULL && neuronptr->number<number;
        neuronptr=neuronptr->next) ;

   if (neuronptr->number!=number)
      return(NULL);

   return(neuronptr);
}



#ifdef USEPROTOS
int connect_neuron(Neuronptr neuronptr, Neuronptr fromptr, Neuronptr from2ptr,
                   FLOAT weight, char *constraint, boolean delay)
#else
int connect_neuron(neuronptr, fromptr, from2ptr,
                   weight, constraint, delay)
       Neuronptr neuronptr;
       Neuronptr fromptr;
       Neuronptr from2ptr;
       FLOAT weight;
       char *constraint;
       boolean delay;
#endif
{
Weightptr weightptr;

   if ((neuronptr->nrweights % MINALLC)==0)
      if ((neuronptr->weights=
         (Weightptr)REALLOC(neuronptr->weights,
                            (neuronptr->nrweights+MINALLC)*sizeof(Weight)))==NULL)
         return(CNTCON);

   weightptr=neuronptr->weights+neuronptr->nrweights;

   weightptr->epsilon=0;
   weightptr->addrate=0;
   weightptr->mulrate=0;
   weightptr->value=weight;
   weightptr->orgvalue=weight;
   weightptr->partact=0;
   weightptr->partdel=0;
   weightptr->partdel2=0;
   weightptr->delta=0;
   weightptr->prevdelta=0;

   weightptr->constraint=NULL;

   if (*constraint!='\0')
   {
      if ((weightptr->constraint=(char *)malloc(strlen(constraint)+1))==NULL)
         return(MEMFUL);
      strcpy(weightptr->constraint, constraint);
   }

   weightptr->delay=delay;
   weightptr->from=fromptr;
   weightptr->from2=from2ptr;

   neuronptr->nrweights++;

   return(OK);
}



#ifdef USEPROTOS
int add_input(Neuronptr neuronptr)
#else
int add_input(neuronptr)
       Neuronptr neuronptr;
#endif
{
int input;

   for (input=0; input<nrinputs; input++)
      if (INPUTNEURON(input)==neuronptr)
         return(IALRDY);

   if ((nrinputs % MINALLC)==0)
      if ((inputlistptr=
            (Neuronptr *)REALLOC(inputlistptr,
                                 (nrinputs+MINALLC)*sizeof(Neuronptr)))==NULL)
         return(CNTADD);

   neuronptr->ionum=nrinputs;

   INPUTNEURON(nrinputs++)=neuronptr;

   return(OK);
}



#ifdef USEPROTOS
int add_output(Neuronptr neuronptr)
#else
int add_output(neuronptr)
       Neuronptr neuronptr;
#endif
{
int output;

   for (output=0; output<nroutputs; output++)
      if (OUTPUTNEURON(output)==neuronptr)
         return(OALRDY);

   if ((nroutputs % MINALLC)==0)
      if ((outputlistptr=
            (Neuronptr *)REALLOC(outputlistptr,
                                 (nroutputs+MINALLC)*sizeof(Neuronptr)))==NULL)
         return(CNTADD);

   neuronptr->ionum=nroutputs;

   OUTPUTNEURON(nroutputs++)=neuronptr;

   return(OK);
}



#ifdef USEPROTOS
int convert_to_bitpat(char mode, char *source_pattern, int nrbits, int distance, char **dest_pattern)
#else
int convert_to_bitpat(mode, source_pattern, nrbits, distance, dest_pattern)
         char mode;
         char *source_pattern;
         int  nrbits, distance;
         char **dest_pattern;
#endif
{
int rc;
static char bit_pattern[MAXPATLEN+1];
char *bit_ptr = bit_pattern;
int  bitvalue;
unsigned long value, mask;
FLOAT fvalue;

   switch (mode)
   {
   case ALPHA:
      while ((value=*source_pattern++)!='\0')
      {
         if ((!iscntrl((char) value) && !isspace((char) value)) || value==' ')
         {
            mask=(unsigned long) pow(2.0, (double) nrbits-1);

            while (mask!=0)
            {
               if (value & mask)
                  *bit_ptr++='X';
               else
                  *bit_ptr++='-';

               mask>>=1;
            }
         }
      }
      break;
   case NUMBER:
      if (strspn(source_pattern, "0123456789")!=strlen(source_pattern))
         return(BADPAT);

      value=atol(source_pattern);

      if (value>=(unsigned long) pow(2.0, (double) nrbits))
         return(NUMLNG);

      mask=(unsigned long) pow(2.0, (double) nrbits-1);

      while (mask!=0)
      {
         if (value & mask)
            *bit_ptr++='X';
         else
            *bit_ptr++='-';

         mask>>=1;
      }
      break;
   case ATOH:
      while ((value=*source_pattern++)!='\0')
      {
         if ((!iscntrl((char) value) && !isspace((char) value)) || value==' ')
         {
            if ((rc=value_to_h(value, nrbits, distance, bit_ptr))!=OK)
               return(rc);

            bit_ptr+=nrbits;
         }
      }
      break;
   case BTOH:
      if (strspn(source_pattern, "-X")!=strlen(source_pattern))
         return(BADPAT);

      value=0;

      while ((bitvalue=*source_pattern++)!='\0')
         value=2*value+(bitvalue=='X' ? 1 : 0);

      if (value>=(unsigned long) pow(2.0, (double) nrbits))
         return(BITLNG);

      if ((rc=value_to_h(value, nrbits, distance, bit_ptr))!=OK)
         return(rc);

      bit_ptr+=nrbits;

      break;
   case FTOH:
      if (strspn(source_pattern, "+-0123456789.eE")!=strlen(source_pattern))
         return(BADPAT);

      if ((fvalue=atof(source_pattern))<0 || fvalue>1)
         return(FLTRNG);

      value=(unsigned long) (fvalue*(pow(2.0, (double) nrbits-distance+1)-1)+0.5);

      if ((rc=value_to_h(value, nrbits, distance, bit_ptr))!=OK)
         return(rc);

      bit_ptr+=nrbits;

      break;
   case NTOH:
      if (strspn(source_pattern, "0123456789")!=strlen(source_pattern))
         return(BADPAT);

      value=atol(source_pattern);

      if (value>=(long) (unsigned long) pow(2.0, (double) nrbits))
         return(NUMLNG);

      if ((rc=value_to_h(value, nrbits, distance, bit_ptr))!=OK)
         return(rc);

      bit_ptr+=nrbits;

      break;
   }

   *bit_ptr='\0';

   *dest_pattern=bit_pattern;

   return(OK);
}



#ifdef USEPROTOS
int add_pattern(char type, int pair,
                FLOAT *input_pattern, FLOAT *output_pattern,
                FLOAT relative)
#else
int add_pattern(type, pair,
                input_pattern, output_pattern,
                relative)
       char type;
       int pair;
       FLOAT *input_pattern;
       FLOAT *output_pattern;
       FLOAT relative;
#endif
{
Patternptr patternptr;

   if (pair==0)
   {
      if (((nrpatterns+nrtests) % MINALLC)==0)
         if ((patternlistptr=
               (Patternptr *)REALLOC(patternlistptr,
                                     (nrpatterns+nrtests+MINALLC)*sizeof(Patternptr)))==NULL)
            return(MEMFUL);

      if ((patternptr=(Patternptr)malloc(sizeof(Pattern)))==NULL)
         return(MEMFUL);

      patternptr->type=type;
      patternptr->relative=relative;
      patternptr->count=0;
      patternptr->nrpairs=0;
      patternptr->in=NULL;
      patternptr->out=NULL;
      patternptr->result=NULL;
      patternptr->error=(FLOAT)nroutputs/8;

      SETPATTERN(nrpatterns+nrtests)=patternptr;
   }
   else
      patternptr=GETPATTERN(nrpatterns+nrtests);


   if ((patternptr->nrpairs % MINALLC)==0)
   {
      if ((patternptr->in=
            (FLOAT *)REALLOC(patternptr->in,
                             (patternptr->nrpairs+MINALLC)*nrinputs*
                              sizeof(*patternptr->in)))==NULL)
         return(MEMFUL);

      if ((patternptr->out=
            (FLOAT *)REALLOC(patternptr->out,
                             (patternptr->nrpairs+MINALLC)*nroutputs*
                              sizeof(*patternptr->out)))==NULL)
         return(MEMFUL);
   }

   memcpy(patternptr->in+patternptr->nrpairs*nrinputs, input_pattern,
          nrinputs*sizeof(*patternptr->in));
   memcpy(patternptr->out+patternptr->nrpairs*nroutputs, output_pattern,
          nroutputs*sizeof(*patternptr->out));

   patternptr->nrpairs++;

   return(OK);
}



#ifdef USEPROTOS
int check_input(void)
#else
int check_input()
#endif
{
Neuronptr neuronptr;
Weightptr weightptr;
int weight;
boolean changed;

   if (nrinputs==0)
      return(NOINDF);

   if (nroutputs==0)
      return(NOUTDF);

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      if (neuronptr->function==FUNCTION)
         return(UNDEFN);               /*  test functie definitie  */

      if (neuronptr->type==TYPE)
         return(UNTYPE);
      else
         if (neuronptr->type==INPUT)
            neuronptr->changed.output=TRUE;
         else
            neuronptr->changed.output=FALSE;

      neuronptr->changed.extra=FALSE;
   }

   do
   {
      changed=FALSE;

      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      {
         for (weight=0; weight<neuronptr->nrweights; weight++)
         {
            weightptr=neuronptr->weights+weight;

            if (neuronptr->changed.output==FALSE &&
                (weightptr->from->changed.output==TRUE ||
                 (weightptr->from2!=NULL && weightptr->from2->changed.output==TRUE)))
            {
               neuronptr->changed.output=TRUE;
               changed=TRUE;
            }

            weightptr->from->changed.extra=TRUE;

            if (weightptr->from2!=NULL)
               weightptr->from2->changed.extra=TRUE;
         }
      }
   } while (changed);


   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      if (neuronptr->changed.output==FALSE ||
          (neuronptr->type!=OUTPUT && neuronptr->changed.extra==FALSE))
         return(UNCONN);               /*  test op 'output' en 'extra' bit  */
      else
      {
         neuronptr->changed.output=FALSE;
         neuronptr->changed.extra=FALSE;
      }
   }

   if (epsilon==0)
      return(NORATE);

   if (!(learn & (HEBB | DHEBB | GROSSBERG | FBPR | ANNEAL)))
      return(NOLRNM);

   if (!(left_in || top_in || right_in || bottom_in ||
         left_out || top_out || right_out || bottom_out ||
         left_res || top_res || right_res || bottom_res ||
         left_info || top_info || right_info || bottom_info))
      quiet=batch=TRUE;

   if (nrpatterns+nrtests==0)
      return(NOPATD);

   return(OK);
}
