'Compile with QuickBasic 4.5
'
'This is a demo backpropagation neural net used to learn the XOR function.
'It differs from backpropgation in two significant ways:
'(1) It uses Newton's Method to find a zero of the "error function",
'    instead of Gradient Descent to find a minimum.  This drastically
'    speeds up the rate of convergence under certain conditions.
'(2) It is recurrent: meaning it has cyclic loops in it.  This extra
'    feature enormously adds to the power of the neural net.
'
' OUTPUT:       E
'             ^   ^
'            /     \
'           C <---> D
'           ^^     ^^
'           | \   / |
'           |  \ /  |
'           |   X   |
'           |  / \  |
'           | /   \ |
' INPUT:    A       B
'
' ------------------------------------------------------------
'
' Mark Hopkins (markh@csd4.csd.uwm.edu)
' November 12, 1990
' Milwaukee, Wisconsin

const TOL = 0.0001
const INPUTS = 2, CELLS = 2, OUTPUTS = 1
dim shared zI(0 to INPUTS - 1) as single, x(0 to INPUTS - 1) as integer
dim shared zH(0 to CELLS - 1)
dim shared zO(0 to OUTPUTS - 1) as single, y(0 to OUTPUTS - 1) as integer
dim shared wIH(0 to INPUTS - 1, 0 to CELLS - 1) as single
dim shared wHH(0 to CELLS - 1, 0 to CELLS - 1) as single
dim shared tH(0 to CELLS - 1) as single, dtH(0 to CELLS - 1) as single
dim shared wHO(0 to CELLS - 1, 0 to OUTPUTS - 1) as single
dim shared toO(0 to OUTPUTS - 1) as single, dtO(0 to OUTPUTS - 1) as single
dim shared Er as single, Eta as double
dim shared SEED as double
dim shared Iter as integer
const MaxStep = 250

cls
print "(C)... Quit"
print "(Q)... Pause"
print "(S)... Resume"
Iter = 0: Initialize: Display
do
   for I = 0 to 1: for J = 0 to 1
      select case inkey$
         case "c", "C": end
         case "q", "Q"
            locate 4, 1: print "PAUSED";
            do: Ch$ = inkey$: loop until Ch$ = "s" or Ch$ = "S"
            locate 4, 1: print "      ";
      end select
      x(0) = I: x(1) = J
      y(0) = (x(0) + x(1)) mod 2
      locate 5, 1
      print using "LEARNING #, # -> #"; x(0); x(1); y(0)
      Tick x(), y(): Display
   next J: next I
   Iter = Iter + 1
loop
end

function SIGMA#(x as single)
   SIGMA# = 1/(1 + exp(-x))
end function

function TAU#(y as single)
   TAU# = y*(1 - y)
end function

sub SRand
   SEED = timer
end sub

function Rand#
   SEED = (SEED*3053 + 1027 + timer)
   SEED = SEED - 65536#*int(SEED/65536#)
   Rand# = SEED - int(SEED)
end function

sub Initialize
   dim I as integer, H as integer, H1 as integer, O as integer
   SRand
   for I = 0 to INPUTS - 1: zI(I) = 0.0: next I
   for H = 0 to CELLS - 1
      for I = 0 to INPUTS - 1: wIH(I, H) = Rand#: next I
      for H1 = 0 to CELLS - 1: wHH(H1, H) = Rand#: next H1
      for O = 0 to OUTPUTS - 1: wHO(H, O) = Rand#: next O
      tH(H) = Rand#: zH(H) = 0.0: dtH(H) = 0.0
   next H
   for O = 0 to OUTPUTS - 1: toO(O) = Rand#: zO(O) = 0.0: next O
end sub

sub Activate(x() as integer)
   dim Net as single, z as single, Er1 as double, Stp as integer
   dim I as integer, H as integer, H1 as integer, O as integer

   for I = 0 to INPUTS - 1: zI(I) = x(I): next I
   Stp = 0
   do
      Er1 = 0.0
      for H = 0 to CELLS - 1
         Net = tH(H)
         for I = 0 to INPUTS - 1: Net = Net + zI(I)*wIH(I, H): next I
         for H1 = 0 to CELLS - 1: Net = Net + zH(H1)*wHH(H1, H): next H1
         z = SIGMA#(Net)
         Er1 = Er1 + (zH(H) - z)*(zH(H) - z): zH(H) = z
      next H
      if sqr(Er1) < 0.000001 then exit do
      Stp = Stp + 1
      if Stp = MaxStep then locate 20, 1: print "Indecision in activations": end
   loop
   locate 20, 1: print using "## step(s) in activation."; Stp
   for O = 0 to OUTPUTS - 1
      Net = toO(O)
      for H = 0 to CELLS - 1: Net = Net + zH(H)*wHO(H, O): next H
      zO(O) = SIGMA#(Net)
   next O
end sub

sub Gradient(y() as integer)
   dim dt as single, Er1 as double, Stp as integer
   dim O as integer

   for O = 0 to OUTPUTS - 1
      dtO(O) = TAU#(zO(O))*(zO(O) - y(O))
   next O
   Stp = 0
   do
      Er1 = 0.0
      for H = 0 to CELLS - 1
         Net = 0.0
         for O = 0 to OUTPUTS - 1: Net = Net + wHO(H, O)*dtO(O): next O
         for H1 = 0 to CELLS - 1: Net = Net + wHH(H, H1)*dtH(H1): next H1
         dt = TAU#(zH(H))*Net
         Er1 = Er1 + (dtH(H) - dt)*(dtH(H) - dt): dtH(H) = dt
      next H
      if sqr(Er1) < 0.000001 then exit do
      Stp = Stp + 1
      if Stp = MaxStep then locate 21, 1: print "Indecision in gradient": end
   loop
   locate 21, 1: print using "## step(s) in gradient."; Stp
   Er = 0
   for O = 0 to OUTPUTS - 1
      Er = Er + (zO(O) - y(O))*(zO(O) - y(O))/2
   next O
end sub

function FindEta#
   dim Active as single, Rate as single
   dim I as integer, H as integer, O as integer
   if Er < TOL then FindEta# = 0.0: exit function
   Rate = 0.0
   for O = 0 to OUTPUTS - 1: Rate = Rate + dtO(O)*dtO(O): next O
   for H = 0 to CELLS - 1: Rate = Rate + dtH(H)*dtH(H): next H
   Active = 1
   for O = 0 to OUTPUTS - 1: Active = Active + zO(O)*zO(O): next O
   for H = 0 to CELLS - 1: Active = Active + zH(H)*zH(H): next H
   for I = 0 to INPUTS - 1: Active = Active + zI(I)*zI(I): next I
   FindEta# = (Er/Rate)/Active
end sub

sub Update(Eta as double)
   dim I as integer, H as integer, H1 as integer, O as integer
   if Eta <= 0.01 then exit sub
   for O = 0 to OUTPUTS - 1
      toO(O) = toO(O) - Eta*dtO(O)
      for H = 0 to CELLS - 1
         wHO(H, O) = wHO(H, O) - Eta*zH(H)*dtO(O)
      next H
   next O
   for H = 0 to CELLS - 1
      tH(H) = tH(H) - Eta*dtH(H)
      for H1 = 0 to CELLS - 1
         wHH(H1, H) = wHH(H1, H) - Eta*zH(H1)*dtH(H)
      next H1
      for I = 0 to INPUTS - 1
         wIH(I, H) = wIH(I, H) - Eta*zI(I)*dtH(H)
      next I
   next H
end sub

sub Tick(x() as integer, y() as integer)
   Activate x(): Gradient y(): Eta = FindEta#: Update Eta
end sub

sub Display
   locate 8, 1
   print using "ACTIVATIONS: A:#.###, B:#.###, C:#.###, "; zI(0); zI(1); zH(0);
   print using "D:#.###, E:#.###"; zH(1); zO(0)
   print using "THRESHOLDS: C:###.###, D:###.###, "; tH(0); tH(1);
   print using "E:###.###"; toO(0)
   print "WEIGHTS:"
   print using "   A->C: ##.###, A->D: ##.###, "; wIH(0, 0); wIH(0, 1);
   print using "B->C: ##.###, B->D: ##.###, "; wIH(1, 0); wIH(1, 1)
   print using "   C->C: ##.###, C->D: ##.###, "; wHH(0, 0); wHH(0, 1);
   print using "D->C: ##.###, D->D: ##.###"; wHH(1, 0); wHH(1, 1)
   print using "   C->E: ##.###, D->E: ##.###, "; wHO(0, 0); wHO(1, 0)
   print using "ERROR: #.#####, ETA: ###.####       "; Er; Eta
   print
   print using "STEPS: #####"; Iter
end sub
