/*
 *
 *    Runtime system:     condela.c
 *
 */

#define __CONDELA_C 1

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "condela.h"

#ifndef LRATE
#define LRATE 0.2
#endif


#ifndef TFAK
#define TFAK 1.0
#endif

struct {
    short lb,
          ub;
} rng[MAXEXP][MAXRANGELEN];

short   idx     [MAXEXP], /* aktueller index des Vectors */
        rng_idx [MAXEXP], /* aktueller range des Vectors */
        rflag   [MAXEXP]; /* ist ein range dieses Vectors in Verwendung ? */

VECTOR res = { -1, 0 };

int runtime_err (int nErr)
{
    fprintf (stderr, "CONDELA Runtime Error #%d ", nErr);

    switch(nErr){
      case 1:  fprintf (stderr, "memory allocation failed.\n"); break;
      case 2:  fprintf (stderr, "too many ranges.\n"         ); break;
      default: fprintf (stderr, "-unknown-\n"                ); break;
    }

    return exit (nErr);
}

short resize_vec (VECTOR* v, short len)
{
        float* aux = (float*)CALLOC(len, float);
        int    ll,
               i;

        if(!aux)
           return runtime_err(1);

        for(i=0, ll = min (len, v->vlen);i<ll;i++)
            aux[i]=v->vloc[i];

        if(v->vloc)
            free(v->vloc);

        v->vloc=aux;
        v->vlen=len;

        return 1;
}

void clear_vec(VECTOR v, short start)
{
        int i;
        for(i=start;i<v.vlen;i++)
            v.vloc[i]=0;
}

short create_vec(VECTOR *v, short len)
{
    v->vlen = len;
    v->vloc = (float *)CALLOC(len,float);

    if(v->vloc)
        return 1;

    return  runtime_err(1);
}

void read_vec(VECTOR* v)
{
    float f;
    int   i=0;

    do
    {
        fscanf(stdin,"%f",&f);
        v->vloc[i++]=f;
    }
    while (f != 0);

    v->vlen=i-1;
}

void write_vec(VECTOR v)
{
    int i;
    for(i=0;i<v.vlen;i++)
        printf("%-2.2f ", v.vloc[i]);
}

short vec_eq(VECTOR a, VECTOR b)
{
    int i;

    if(a.vlen!=b.vlen)
        return 0;

    for(i=0;i<a.vlen;i++)
        if(a.vloc[i]!=b.vloc[i])
            return 0;

    return 1;
}


VECTOR vec_add (VECTOR a, VECTOR b)
{
    short i,
          ml = min(a.vlen,b.vlen);

    if(res.vlen<0)
        create_vec(&res,ml);
    else
        if(ml!=res.vlen)
            resize_vec(&res,ml);

    for(i=0; i<ml; i++)
        res.vloc[i]=a.vloc[i]+b.vloc[i];

    return res;
}

VECTOR vec_sub (VECTOR a, VECTOR b)
{
    short i,
          ml=min(a.vlen,b.vlen);

    if(res.vlen<0)
        create_vec(&res,ml);
    else
        if(ml!=res.vlen)
            resize_vec(&res,ml);

    for(i=0; i<ml; i++)
        res.vloc[i]=a.vloc[i]-b.vloc[i];

    return res;
}

VECTOR vec_neg (VECTOR a)
{
    int    i;
    VECTOR res;

    create_vec(&res,a.vlen);

    for(i=0; i<a.vlen; i++)
        res.vloc[i]= -a.vloc[i];

    res.vlen=a.vlen;

    return res;
}

float vec_mul(VECTOR a, VECTOR b)
{
    float res = 0;
    int   ml  = min(a.vlen,b.vlen),
          i;

    for(i=0;i<ml;i++)
        res+= a.vloc[i] * b.vloc[i];

    return res;
}

VECTOR vec_skal(float k, VECTOR b)
{
    int i;

    if(res.vlen<0)
        create_vec(&res,b.vlen);
    else
        if(b.vlen!=res.vlen)
            resize_vec(&res,b.vlen);
    for(i=0;i<b.vlen;i++)
        res.vloc[i]=k*b.vloc[i];

    return res;
}

short vec_asgn(VECTOR* t, VECTOR s, short clear)
{
    short i, len=s.vlen;

    if(t->vlen<len)
    {
        float *new_vloc;
        if(t->vloc)
            free(t->vloc);

        new_vloc=(float *) calloc (len,sizeof(float));

        if(!new_vloc)
            return  runtime_err(1);
        t->vloc=new_vloc; t->vlen=len;
    }
    else
    {
        if(t->vlen>s.vlen)
        {
            if(clear)
                clear_vec(*t,s.vlen);
            else
                resize_vec(t,len);
        }
        else
            t->vlen=len;
    }

    for(i=0;i<len;i++)
        t->vloc[i]=s.vloc[i];

    return 1;
}

void reset_rng()
{
    int i,j;
    for(i=0;i<MAXEXP;i++)
    {
        rng_idx[i] = 0;
        rflag  [i] = 0;

        for(j=0;j<MAXRANGELEN;j++)
             rng[i][j].lb= -1;

    }
}

void init_idx(short veci)
{
   idx[veci] = rflag[veci] ? rng[veci][0].lb : 0;
}

short next_idx(short veci) /* 1: Ueberlauf, 0: weiter, -1: kein neuer Index */
{
    if(rflag[veci])
    {
         if(++idx[veci] > rng[veci][rng_idx[veci]].ub)
         {
             if(++rng_idx[veci] > MAXRANGELEN)
                  return runtime_err(2);

             if((idx[veci]=rng[veci][rng_idx[veci]].lb)==-1)
                  return -1;
          }
    }
    else
        idx[veci]++;

    return 0;
}

short cmplx_vec_eq(VECTOR a, short ai, VECTOR b, short bi)
{
    short nai,
          nbi;

    for(;;)
    {
        if(a.vloc[idx[ai]]!=b.vloc[idx[bi]])
            return 0;

        nai=next_idx(ai);
        nbi=next_idx(bi);

        if(nai!=nbi)
            return 0;  /* |Ia| <> |Ib|  kardinalitaet der indexmengen ungleich ? */

        if(nai==-1)
            return 1;   /* |Ia| =  |Ib| */

        if(idx[ai]>a.vlen || idx[bi]>b.vlen)
            return 0;
    }
}

void cmplx_vec_asgn(VECTOR *t, short ti, VECTOR s, short si, short clear)
{
    short i      = 0,
          maxtlen= t->vlen,nti,nts;

    for(;;)
    {
        t->vloc[idx[ti]]=s.vloc[idx[si]];

        nti = next_idx(ti);
        nts = next_idx(si);

        i++;

        if(nti || nts) break;
        if(idx[si]>maxtlen)
        {
            resize_vec (t,rng [si][rng_idx[si]].ub);
            maxtlen=t->vlen;
        }
    }

    if(clear && !nti)
        clear_vec(*t,idx[si]);

    if(!rflag[ti])
        t->vlen=i;
}

void cmplx_vec_init(VECTOR* t, short ti, float v)
{
    short i=0,maxtlen=t->vlen,nti;

    for(;;)
    {
         t->vloc[idx[ti]] = v;
         nti              = next_idx(ti);

         i++;

         if(nti)
             break;

         if(idx[ti]>maxtlen)
         {
             resize_vec(t,rng[ti][rng_idx[ti]].ub);
             maxtlen=t->vlen;
         }
    }

    if(!rflag[ti]) t->vlen=i;
}

VECTOR cmplx_vec_neg(VECTOR a, short ai)
{
    int i=0;
    VECTOR res;

    create_vec(&res,a.vlen);

    while(idx[ai]<a.vlen)
    {
        res.vloc[i++]= -a.vloc[idx[ai]];

        if(next_idx(ai))
            break;
    }

    res.vlen=i;

    return res;
}

VECTOR cmplx_vec_add(VECTOR a, short ai, VECTOR b, short bi)
{
    int i,
        mal=max (a.vlen,b.vlen);

    if(res.vlen<0)
        create_vec(&res,mal);
    else if(mal!=res.vlen)
        resize_vec(&res,mal);

    init_idx(ai);
    init_idx(bi);

    for(i=0; i<mal; i++)
    {
         res.vloc[i]=a.vloc[idx[ai]]+b.vloc[idx[bi]];
         if(
             next_idx(ai)     ||
             next_idx(bi)     ||
             idx[ai] > a.vlen ||
             idx[bi] > b.vlen
         )
             break;
    }

    res.vlen= ++i;

    if(mal>i)
        resize_vec(&res,i);

    return res;
}

VECTOR cmplx_vec_sub(VECTOR a, short ai, VECTOR b, short bi)
{
    int i,
        mal = max(a.vlen,b.vlen);

    if(res.vlen<0)
        create_vec(&res,mal);
    else if(mal!=res.vlen)
        resize_vec(&res,mal);

    init_idx(ai);
    init_idx(bi);

    for(i=0; i<mal; i++)
    {
        res.vloc[i]=a.vloc[idx[ai]]-b.vloc[idx[bi]];
        if(
            next_idx(ai)   ||
            next_idx(bi)   ||
            idx[ai]>a.vlen ||
            idx[bi]>b.vlen
        )
            break;
    }

    res.vlen= ++i;

    if(mal>i)
        resize_vec(&res,i);

    return res;
}

float cmplx_vec_mul(VECTOR a, short ai, VECTOR b, short bi)
{
    int   ml  = min (a.vlen,b.vlen);
    float res = 0;

    for(;;)
    {
        res+= a.vloc[idx[ai]] * b.vloc[idx[bi]];

        if(next_idx(ai) || next_idx(bi) || idx[ai]>ml || idx[bi]>ml)
            return res;
    }
}

VECTOR cmplx_vec_scal(float c, VECTOR a, short ai)
{
    int i=0;
    VECTOR res;

    create_vec(&res,a.vlen);
    while(idx[ai]<a.vlen)
    {
        res.vloc[i++]=c*a.vloc[idx[ai]];
        if(next_idx(ai))
            break;
    }

    res.vlen=i;

    return res;
}

void ins_usc(USEL **root, UNIT *new)
{
    static USEL *last=0;
           USEL *aux;

    for (aux= *root; aux; aux=aux->u_next)
        if(aux->p==new)
            return;

    aux    = calloc (1, sizeof (USEL));
    aux->p = new;

    if(*root)
    {
        last->u_next = aux;
        last         = aux;

        return;
    }

    *root = aux;
    last  = aux;
}

void less_usc (USEL **root, UNIT *new)
{
    USEL *aux,*ap;

    if(!*root)
        return;

    aux = *root;
    ap  = aux->u_next;

    if(aux->p==new)
    {
        free(*root);
        *root=ap;
        return;
    }

    while(ap)
    {
        if(ap->p==new)
        {
            aux->u_next=ap->u_next;
            free(ap);
            return;
        }
        else
        {
            aux = ap;
            ap  = ap->u_next;
        }
    }
}

float random() /* returns random float in [0..1] */
{
    return (float)rand()/(float)RAND_MAX-.5;
}

int random_to(int max)
{
    return (int) (max*((float)rand()/((float)RAND_MAX)));
}

void free_all_usel (USEL *s)
{
    if(s)
    {
        free_all_usel (s->u_next);
        free          (s        );
    }
}


void connect_u (
         UNIT*    su,
         USEL*    t,
         STRING   wtype,
         INIT_FKT init_fkt,
         float    init_val,
         short    init_flg,
         short    dup_flg
     )
{
    WEIGHT *new_w,
           *q;

    if(!t)
        return;

    new_w=calloc (1, sizeof (WEIGHT));

    q=su->c_to;

    if(q)
    {
        while (q->to_next)
            q=q->to_next;

        q->to_next=new_w;
    }
    else
        su->c_to=new_w;

    new_w->c_from = su;
    new_w->c_to   = t->p;
    new_w->gew    = init_flg ? init_fkt () : init_val;

    q=t->p->c_from;

    if(q)
    {
        while(q->from_next)
            q=q->from_next;

        q->from_next = new_w;
    }
    else
        t->p->c_from = new_w;

    connect_u(su, t->u_next, wtype, init_fkt, init_val, init_flg, dup_flg);
}

 /* the function connect connects two unit-selections: If you want to */
 /* create new types of connections, you have to modify the existing */
 /* procedure connect: check for the string wtype to distinguish */
 /* between the default-type (in this case the value of wtype is */
 /* "default") and other types. You might also exploit the  usage of */
 /* the NUDUPLICATES keyword: if the keyword NUDUPLICATES was NOT */
 /* present, the dup_flg is true (i.e. multiple connections between */
 /* units are permitted) and false otherwise. */
 /* If the initialization of the CONNECT statement was done with a */
 /* function-call the init_flg is true, if a constant of variable was */
 /* used this flag is false: see the function connect_u */

void connect(
         USEL*    s,
         USEL*    t,
         STRING   wtype,
         INIT_FKT init_fkt,
         float    init_val,
         short    init_flg,
         short    dup_flg
     )
{
    for (; s; s=s->u_next)
        connect_u (s->p, t, wtype, init_fkt, init_val, init_flg, dup_flg);
}

short in_list(short val, short* list, short siz)
{
     return (siz>0) && (*list==val || in_list(val, list+1, siz-1));
}

short rnd_p(short k, short n)
{
    static short anz_deliv=0, di=0, *delivered;
    short ret;

    if(!anz_deliv)
        delivered = calloc (k, sizeof (short));
    else if(anz_deliv==k)
    {
        anz_deliv = 0;
        di        = 0;

        free(delivered);

        return -1;
     }
     do
        ret=random_to(n);
     while (in_list(ret,delivered,di));

     delivered [di] = ret;

     anz_deliv++;
     di++;

     return ret;
}

void re_connect (UNIT **oldloc, int oldlen, int newlen)
{
    WEIGHT *tos,*froms;
    UNIT **newloc;

    newloc = calloc (newlen, sizeof (UNIT*));

    for (;oldlen; oldlen--)
    {
        froms                   =
        newloc [oldlen]->c_from = oldloc [oldlen]->c_from;

        for (;froms; froms = froms->from_next)
            froms->c_to = newloc [oldlen];

        tos                  =
        newloc[oldlen]->c_to = oldloc[oldlen]->c_to;

        for (;tos; tos=tos->to_next)
            tos->c_from=newloc[oldlen];

    }

    oldloc = newloc;
}

int trunc(float x)
{
    return (int)x;
}

float transfer(float x)
{
    return (float)((double)1.0/((double)1.0+exp(-TFAK*x)));
}

float transfer_1(float x)
{
    double tmp=exp(-TFAK*x);

    return (float)((double)TFAK*tmp/(((double)1.0+tmp)*((double)1.0+tmp)));
}

void ff(USEL* r)
{
    WEIGHT *from;
    double tmp;

    for (;r; r=r->u_next)
    {
        from = r->p->c_from;
        tmp  = 0;

        /* compute the net input of unit r->p */

        while(from)
        {
            tmp += from->gew * from->c_from->out;
            from = from->from_next;
        }

        /* assign net input to unit */

        r->p->net = (float)tmp;

        /* compute output value (the activation of the
           unit is identical to the output) */

        r->p->out = transfer((float) tmp);
    }
}

void bp_out (VECTOR o, USEL* r)
{
    WEIGHT* p1;
    int i;

    for (i=0; r; r=r->u_next, i++)
    {
        /*  compute errorsignal for the output-unit : */

        r->p->del= (o.vloc[i] - r->p->out) * transfer_1(r->p->net);

        for (p1=r->p->c_from; p1; p1=p1->from_next)
        {
            /*  Sum up the components of the errorsignal
                of the underlying hidden units */

            p1->c_from->del += p1->gew * r->p->del;
            p1->gew += LRATE * r->p->del * p1->c_from->out;

        }

    }
}


void bp_hid (USEL* r)
{
    WEIGHT *p1;
    for (;r; r=r->u_next)
    {
        /*  This value ~~~~~~! was already computed in bp_out,
             v (the errorsignal for the hidden unit). */

        r->p->del = r->p->del * transfer_1 (r->p->net);

        for (p1=r->p->c_from; p1; p1=p1->from_next)
        {
            /*  p1->c_from->del += p1->gew * r->p->del;
             *  not needed in a 3 layer network */

            p1->gew += LRATE * r->p->del * p1->c_from->out;
        }
    }
}
