/*
 * modcmac.c - MEX-file implementation of CMAC output generation function
 *
 * Matlab form is op = modcmac(wts,ip,target,beta,iprange,c,width,memsize)
 * where op is network output after training,
 * ip is an input vector, target is the desired network output,
 * beta is the learning rate, iprange is the range of values for
 * each element of ip, c is the number of weights summed to form
 * an output, width is the generalisation width and
 * memsize is the range of addresses used.
 *
 * written 26 August 1993 by Donald Reay
 *
 * using MetaWare High C/C++ version 3.1
 *
 * Power Electronics Group
 * Department of Computing and Electrical Engineering
 * Heriot-Watt University
 * Edinburgh UK
 *
 * e-mail dsr@cee.hw.ac.uk
 *
 */

#include <math.h>
#include "cmex.h"

extern  void modcmac(double *, double *, double *, double *, double *, int, double *, int, double *, long);

#define NO_CMAC_INARGS 8 
#define NO_CMAC_OUTARGS 1

#define WTS_IN prhs[0]
#define IP_IN prhs[1]
#define TARGET_IN prhs[2]
#define BETA_IN prhs[3]
#define IPRANGE_IN prhs[4]
#define C_IN prhs[5]
#define WIDTH_IN prhs[6]
#define MEMSIZE_IN prhs[7]
#define OP_OUT plhs[0]

#define HASH 12345

/*
 * first part of MEX-file source is MATLAB interface
 */

void mexFunction(int nlhs, Matrix *plhs[], int nrhs, Matrix *prhs[])

{
  /* check for proper number of arguments */

  if (nrhs != NO_CMAC_INARGS)
    mexErrMsgTxt("MODCMAC requires eight input arguments.");
  else if (nlhs != NO_CMAC_OUTARGS)
    mexErrMsgTxt("MODCMAC requires one output argument.");

  /* check the dimensions of ip.  ip should be a vector */

  if ((int)(mxGetN(IP_IN)) != 1)
    mexErrMsgTxt("MODCMAC requires that IP be a vector.");

  /* Create a matrix for the return argument */

  OP_OUT = mxCreateFull(1, 1, REAL);

  /* do the actual computations in a subroutine */

  modcmac(mxGetPr(OP_OUT), mxGetPr(WTS_IN), mxGetPr(IP_IN),
	  mxGetPr(TARGET_IN), mxGetPr(BETA_IN), (int)(mxGetM(IP_IN)),
	  mxGetPr(IPRANGE_IN), (int)(*mxGetPr(C_IN)),
	  mxGetPr(WIDTH_IN), (long)(*mxGetPr(MEMSIZE_IN)));

} /* mexFunction */

/*
 * second part of MEX-file source is C function that does the work 
 */

void modcmac(double *op, double *wts, double *ip, double *target,
	     double *beta, int ipdim, double *iprange, int c,
	     double *width, long memsize)

{

  int i, j;
  int quantisation;
  double shift, address, offset, ofs, sum, delta;
  long *addrs;

  addrs = mxCalloc(c, sizeof(long));

  /* compute network parameters */

  quantisation = (int)((*iprange)/(*width));
  offset = (*width)/c;
  ofs = 0.0;
  sum = 0.0;

  /* compute addresses */

  for (i=0 ; i<c ; i++)
  {
    address = 0; 
    shift = 1.0;
    for (j=0 ; j<ipdim ; j++)
    {
      address += (((int)((*(ip+j) + ofs)/(*width))) % quantisation)*shift;
      shift *= quantisation;
    }
    address += shift*i;

  /* check for hashing */

    if (memsize != 0)
      *(addrs+i) = (long)(log(address+1)*HASH) % memsize;
    else
      *(addrs+i) = (long)(address);  

    ofs += offset;
    sum += *(wts + *(addrs+i));

  }

  delta = (*beta)*(*target - sum)/c;
  for (i=0 ; i<c ; i++)
  {
    *(wts + *(addrs+i)) += delta;
  }
  *op = sum + delta*c;

  mxFree(addrs);
  
} /* modcmac */
