/*
	File:			Vec2.h

	Function:		Defines a length-2 vector.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Vec2__
#define __Vec2__

#include "VL.h"


// --- Vec2 Class -------------------------------------------------------------


TMPLVec class Vec2
{
public:

	// Constructors
	
	inline				Vec2();
	inline				Vec2(TReal x, TReal y);			// (x, y)
	inline				Vec2(const TVec2 &v);			// Copy constructor
	inline				Vec2(ZeroOrOne k);
		
	// Accessor functions
	
	inline TReal 	   &operator () (Int i);		   	// v(1) - Indexing
	inline const TReal &operator () (Int i) const;		  

	inline TReal 	   &operator [] (Int i);		   	// v[1] - Has no Index
	inline const TReal &operator [] (Int i) const;		//        check

	inline Int			Elts() const { return(3); };
	inline TReal		*Ref() const;					// Return ptr to data

	// Assignment operators
	
	inline TVec2	&operator =  (const TVec2 &a);	    				
	inline TVec2	&operator =  (ZeroOrOne k);	    				
	
	inline TVec2	&operator += (const TVec2 &a);
	inline TVec2	&operator -= (const TVec2 &a);
	inline TVec2	&operator *= (const TVec2 &a);
	inline TVec2	&operator *= (TReal s);
	inline TVec2	&operator /= (const TVec2 &a);
	inline TVec2	&operator /= (TReal s);

	// Comparison operators

	Bool			operator == (const TVec2 &a) const;		// v == a?
	Bool			operator != (const TVec2 &a) const;		// v != a?

	// Arithmetic operators
		
	inline TVec2	operator + (const TVec2 &a) const;		// v + a
	inline TVec2	operator - (const TVec2 &a) const;		// v - a
	inline TVec2	operator - () const;					// -v
	inline TVec2	operator * (const TVec2 &a) const;		// v * a (vx * ax, ...)
	inline TVec2	operator * (TReal s) const;				// v * s
	inline TVec2	operator / (const TVec2 &a) const;		// v / a (vx / ax, ...)
	inline TVec2	operator / (TReal s) const;				// v / s
	inline TReal	operator dot (const TVec2 &a) const;	// v . a

	// Initialisers
	
	inline void		MakeZero();								// Zero vector
	inline void		MakeUnit(Int i, TReal k = vl_one);		// I[i]
	inline void		MakeBlock(TReal k = vl_one);			// All-k vector
	
	// Private...
	
protected:

	TReal elt[2]; 
};


// --- Vec operators -------------------------------------------------

TMPLVec inline 
	TVec2 			operator * (TReal s, const TVec2 &v);	// Left mult. by s
TMPLVec TReal		len(const TVec2 &v);					// || v ||
TMPLVec TReal		sqrlen(const TVec2 &v);					// v . v
TMPLVec TVec2		norm(const TVec2 &v);					// v / || v ||
TMPLVec inline 
	TVec2			cross(const TVec2 &v);					// cross prod.
	
TMPLVec ostream	&operator << (ostream &s, const TVec2 &v);
TMPLVec istream	&operator >> (istream &s, TVec2 &v);

	
// --- Inlines ----------------------------------------------------------------


TMPLVec inline TReal &TVec2::operator [] (Int i)
{
    return(elt[i]);
}

TMPLVec inline const TReal &TVec2::operator [] (Int i) const
{
    return(elt[i]);
}

TMPLVec inline TReal &TVec2::operator () (Int i)
{
	CheckRange(i, 0, 2, "(Vec2::(i)) index out of range");
	
    return(elt[i]);
}

TMPLVec inline const TReal &TVec2::operator () (Int i) const
{
	CheckRange(i, 0, 2, "(Vec2::(i)) index out of range");
	
    return(elt[i]);
}

TMPLVec inline TVec2::Vec2()
{
}

TMPLVec inline TVec2::Vec2(TReal x, TReal y)
{
	elt[0] = x;
	elt[1] = y;
}

TMPLVec inline TVec2::Vec2(const TVec2 &v) 
{
	elt[0] = v[0];
	elt[1] = v[1];
}

TMPLVec inline TVec2::Vec2(ZeroOrOne k) 
{
	elt[0] = k;
	elt[1] = k;
}

TMPLVec inline TReal *TVec2::Ref() const
{
	return((TReal *) elt);
}

TMPLVec inline TVec2 &TVec2::operator = (const TVec2 &v)
{
	elt[0] = v[0];
	elt[1] = v[1];
	
	return(SELF);
}

TMPLVec	inline TVec2 &TVec2::operator = (ZeroOrOne k)
{
	elt[0] = k; elt[1] = k;
	
	return(SELF);
}

TMPLVec inline TVec2 &TVec2::operator += (const TVec2 &v)
{
	elt[0] += v[0];
	elt[1] += v[1];
	
	return(SELF);
}

TMPLVec inline TVec2 &TVec2::operator -= (const TVec2 &v)
{
	elt[0] -= v[0];
	elt[1] -= v[1];
	
	return(SELF);
}

TMPLVec inline TVec2 &TVec2::operator *= (const TVec2 &v)
{
	elt[0] *= v[0];
	elt[1] *= v[1];
	
	return(SELF);
}

TMPLVec inline TVec2 &TVec2::operator *= (TReal s)
{
	elt[0] *= s;
	elt[1] *= s;
	
	return(SELF);
}

TMPLVec inline TVec2 &TVec2::operator /= (const TVec2 &v)
{
	elt[0] /= v[0];
	elt[1] /= v[1];
	
	return(SELF);
}

TMPLVec inline TVec2 &TVec2::operator /= (TReal s)
{
	elt[0] /= s;
	elt[1] /= s;
	
	return(SELF);
}

TMPLVec inline TVec2 TVec2::operator + (const TVec2 &a) const
{
	TVec2 result;
	
	result[0] = elt[0] + a[0];
	result[1] = elt[1] + a[1];
	
	return(result);
}

TMPLVec inline TVec2 TVec2::operator - (const TVec2 &a) const
{
	TVec2 result;
	
	result[0] = elt[0] - a[0];
	result[1] = elt[1] - a[1];
	
	return(result);
}

TMPLVec inline TVec2 TVec2::operator - () const
{
	TVec2 result;
	
	result[0] = -elt[0];
	result[1] = -elt[1];
	
	return(result);
}

TMPLVec inline TVec2 TVec2::operator * (const TVec2 &a) const			
{
	TVec2 result;
	
	result[0] = elt[0] * a[0];
	result[1] = elt[1] * a[1];
	
	return(result);
}

TMPLVec inline TVec2 TVec2::operator * (TReal s) const
{
	TVec2 result;
	
	result[0] = elt[0] * s;
	result[1] = elt[1] * s;
	
	return(result);
}

TMPLVec inline TVec2 operator * (TReal s, const TVec2 &v)
{
	return(v * s);
}

TMPLVec inline TVec2 TVec2::operator / (const TVec2 &a) const
{
	TVec2 result;
	
	result[0] = elt[0] / a[0];
	result[1] = elt[1] / a[1];
	
	return(result);
}

TMPLVec inline TVec2 TVec2::operator / (TReal s) const
{
	TVec2 result;
	
	result[0] = elt[0] / s;
	result[1] = elt[1] / s;
	
	return(result);
}

TMPLVec inline TReal TVec2::operator dot (const TVec2 &a) const
{
	return(elt[0] * a[0] + elt[1] * a[1]);
}

TMPLVec inline TVec2 cross(const TVec2 &a)
{
	TVec2 result;
	
	result[0] =  a[1];
	result[1] = -a[0];
	
	return(result);
}

TMPLVec inline void TVec2::MakeUnit(Int i, TReal k)
{
	if (i == 0)
	{ elt[0] = k; elt[1] = vl_zero; }
	else if (i == 1)
	{ elt[0] = vl_zero; elt[1] = k; }
	else 
		Assert(false, "(Vec2::Unit) illegal unit vector");
}

TMPLVec inline void TVec2::MakeZero()
{
	elt[0] = vl_zero; elt[1] = vl_zero;
}

TMPLVec inline void TVec2::MakeBlock(TReal k)
{
	elt[0] = k; elt[1] = k;
}


#ifdef VL_TEMPLATE
#include "Vec2.cc"
#endif

#endif
