/*
	File:			Mat2.h

	Function:		Defines a 2 x 2 matrix.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Mat2__
#define __Mat2__

#include "VL.h"
#include "Vec2.h"


// --- Mat2 Class -------------------------------------------------------------


TMPLMat class Mat2 
{
public:
	
	// Constructors
	
	inline 		Mat2();
	inline 		Mat2(TMReal a, TMReal b, TMReal c, TMReal d); 	// Create from rows
	inline 		Mat2(const TMat2 &m);							// Copy constructor
	inline		Mat2(ZeroOrOne k);
	
	// Accessor functions
	
	inline Int			Rows() const { return(2); };
	inline Int			Cols() const { return(2); };
	
	inline TMVec2    	&operator () (Int i);		    // m(1) - Indexing
	inline const TMVec2 &operator () (Int i) const;		  
	
	inline TMVec2    	&operator [] (Int i);		    // m[1] - Has no index
	inline const TMVec2 &operator [] (Int i) const;		//        check

	inline TMReal       &operator () (Int i, Int j);	// Indexing by elt.
	inline TMReal		operator () (Int i, Int j) const;		  

	inline TMReal 		*Ref() const;					// Return pointer to data

	// Assignment operators

	inline TMat2 		&operator =  (const TMat2 &m); 
	inline TMat2 		&operator =  (ZeroOrOne k); 
	inline TMat2 		&operator += (const TMat2 &m);	   
	inline TMat2 		&operator -= (const TMat2 &m);	    
	inline TMat2 		&operator *= (const TMat2 &m);	    
	inline TMat2 		&operator *= (TMReal s);	    		
	inline TMat2 		&operator /= (TMReal s);	    		
	
	// Comparison operators
	
	Bool				operator == (const TMat2 &m) const;	// M == N?
	Bool				operator != (const TMat2 &m) const;	// M != N?
	
	// Arithmetic operators
	
	inline TMat2		operator + (const TMat2 &m) const;	// M + N
	inline TMat2		operator - (const TMat2 &m) const;	// M - N
	inline TMat2		operator - () const;				// -M
	inline TMat2		operator * (const TMat2 &m) const;	// M * N
	inline TVec2		operator * (const TVec2 &v) const;	// M * v
	inline TMat2		operator * (TMReal s) const;			// M * s
	inline TMat2		operator / (TMReal s) const;			// M / s

	inline TMat2		operator ~ () const;				// trans(M)
	inline TMat2		operator ! () const;				// adj(M)
	
	// Matrix-vector functions
	
	
	// Initialisers
	
	inline void			MakeZero();								// Zero matrix
	inline void			MakeUnit(TMReal k = vl_one);			// I
	inline void			MakeBlock(TMReal k = vl_one);			// all elts = k
	
	// Vector Transformations
	
	void 				MakeRot(Real theta);	  		  
	void 				MakeScale(const TVec2 &s);

	// Private...

protected:	
	
	TMVec2 row[2];		// Rows of the matrix
};


// --- Matrix operators ---------------------------------------------------


TMPLMat inline TVec2 &operator *= (TVec2 &v, const TMat2 &m);	  // v *= m
TMPLMat inline TVec2  operator *  (const TVec2 &v, const TMat2 &m);// v * m
TMPLMat inline TMat2  operator *  (TMReal s, const TMat2 &m);	  // s * m

TMPLMat inline TMat2	trans(const TMat2 &m);				// Transpose
TMPLMat inline TMReal	trace(const TMat2 &m);				// Trace
TMPLMat inline TMat2	adj(const TMat2 &m);				// Adjoint
TMPLMat TMReal			det(const TMat2 &m);				// Determinant
TMPLMat TMat2			inv(const TMat2 &m);				// Inverse

TMPLMat ostream		&operator << (ostream &s, const TMat2 &m);
TMPLMat istream		&operator >> (istream &s, TMat2 &m);


// --- Inlines ----------------------------------------------------------------


TMPLMat inline TMVec2 &TMat2::operator () (Int i)
{
	CheckRange(i, 0, 2, "(Mat2::(i)) index out of range");
	
    return(row[i]);
}

TMPLMat inline const TMVec2 &TMat2::operator () (Int i) const
{
	CheckRange(i, 0, 2, "(Mat2::(i)) index out of range");
	
    return(row[i]);
}

TMPLMat inline TMVec2 &TMat2::operator [] (Int i)
{
    return(row[i]);
}

TMPLMat inline const TMVec2 &TMat2::operator [] (Int i) const
{
    return(row[i]);
}

TMPLMat inline TMReal &TMat2::operator () (Int i, Int j)
{
	CheckRange(i, 0, 2, "(Mat2::(i, j)) index out of range");
	CheckRange(j, 0, 2, "(Mat2::(i, j)) index out of range");

	return(row[i][j]);
}

TMPLMat inline TMReal TMat2::operator () (Int i, Int j) const
{
	CheckRange(i, 0, 2, "(Mat2::(i, j)) index out of range");
	CheckRange(j, 0, 2, "(Mat2::(i, j)) index out of range");

	return(row[i][j]);
}

TMPLMat inline TMReal *TMat2::Ref() const
{
	return((TMReal *) row);
}

TMPLMat inline TMat2::Mat2()
{
}

TMPLMat inline TMat2::Mat2(TMReal a, TMReal b, TMReal c, TMReal d)
{
	row[0][0] = a;	row[0][1] = b;
	row[1][0] = c;	row[1][1] = d;
}

TMPLMat inline TMat2::Mat2(const TMat2 &m)
{
	row[0] = m[0];
	row[1] = m[1];
}


TMPLMat inline void TMat2::MakeZero()
{
	row[0][0] = vl_zero; row[0][1] = vl_zero;
	row[1][0] = vl_zero; row[1][1] = vl_zero;
}

TMPLMat inline void TMat2::MakeUnit(TMReal k)
{
	row[0][0] = k; 		row[0][1] = vl_zero;
	row[1][0] = vl_zero;	row[1][1] = k;
}

TMPLMat inline void TMat2::MakeBlock(TMReal k)
{
	row[0][0] = k; row[0][1] = k;
	row[1][0] = k; row[1][1] = k;
}

TMPLMat inline TMat2::Mat2(ZeroOrOne k)
{
	MakeUnit(k);
}

TMPLMat inline TMat2 &TMat2::operator = (ZeroOrOne k)
{
	MakeUnit(k);

	return(SELF);
}
	  
TMPLMat inline TMat2 &TMat2::operator = (const TMat2 &m)
{
	row[0] = m[0];
	row[1] = m[1];
	
	return(SELF);
}
	  
TMPLMat inline TMat2 &TMat2::operator += (const TMat2 &m)
{
	row[0] += m[0];
	row[1] += m[1];
	
	return(SELF);
}

TMPLMat inline TMat2 &TMat2::operator -= (const TMat2 &m)
{
	row[0] -= m[0];
	row[1] -= m[1];

	return(SELF);
}

TMPLMat inline TMat2 &TMat2::operator *= (const TMat2 &m)
{
	SELF = SELF * m; 

	return(SELF);
}

TMPLMat inline TMat2 &TMat2::operator *= (TMReal s)
{
	row[0] *= s;
	row[1] *= s;

	return(SELF);
}

TMPLMat inline TMat2 &TMat2::operator /= (TMReal s)
{
	row[0] /= s;
	row[1] /= s;

	return(SELF);
}


TMPLMat inline TMat2 TMat2::operator + (const TMat2 &m) const
{
	TMat2 result;
	
	result[0] = row[0] + m[0];
	result[1] = row[1] + m[1];

	return(result);	
}

TMPLMat inline TMat2 TMat2::operator - (const TMat2 &m) const
{
	TMat2 result;
	
	result[0] = row[0] - m[0];
	result[1] = row[1] - m[1];

	return(result);	
}

TMPLMat inline TMat2 TMat2::operator - () const
{
	TMat2 result;
	
	result[0] = -row[0];
	result[1] = -row[1];

	return(result);	
}

TMPLMat inline TMat2 TMat2::operator * (const TMat2 &m) const
{
#define N(x,y) row[x][y]
#define M(x,y) m[x][y]
#define R(x,y) result[x][y]

	TMat2 result;
	
	R(0,0) = N(0,0) * M(0,0) + N(0,1) * M(1,0);
	R(0,1) = N(0,0) * M(0,1) + N(0,1) * M(1,1);
	R(1,0) = N(1,0) * M(0,0) + N(1,1) * M(1,0);
	R(1,1) = N(1,0) * M(0,1) + N(1,1) * M(1,1);

	return(result);	
	
#undef N
#undef M
#undef R
}

TMPLMat inline TVec2 TMat2::operator * (const TVec2 &v) const
{
	TVec2 result;

	result[0] = row[0][0] * v[0] + row[0][1] * v[1];
	result[1] = row[1][0] * v[0] + row[1][1] * v[1];

	return(result);
}

TMPLMat inline TMat2 TMat2::operator * (TMReal s) const
{
	TMat2 result;
	
	result[0] = row[0] * s;
	result[1] = row[1] * s;

	return(result);	
}

TMPLMat inline TMat2 TMat2::operator / (TMReal s) const
{
	TMat2 result;
	
	result[0] = row[0] / s;
	result[1] = row[1] / s;

	return(result);	
}

TMPLMat inline TMat2 TMat2::operator ~ () const						
{
	TMat2 result;

	result[0][0] = row[0][0]; result[0][1] = row[1][0]; 
	result[1][0] = row[0][1]; result[1][1] = row[1][1]; 
		
	return(result);
}

TMPLMat inline TMat2 TMat2::operator ! () const			
{
	TMat2 result;

	result[0] =  cross(row[1]);
	result[0] = -cross(row[0]);
			
	return(result);
}


TMPLMat inline TVec2 operator * (const TVec2 &v, const TMat2 &m)			
{
	TVec2 result;

	result[0] = v[0] * m[0][0] + v[1] * m[1][0];
	result[1] = v[0] * m[0][1] + v[1] * m[1][1];

	return(result);
}

TMPLMat inline TMat2  operator *  (TMReal s, const TMat2 &m)
{
	return(m * s);
}

TMPLMat inline TVec2 &operator *= (TVec2 &v, const TMat2 &m)		
{
	TReal t;
	
	t    = v[0] * m[0][0] + v[1] * m[1][0];
	v[1] = v[0] * m[0][1] + v[1] * m[1][1];
	v[0] = t;

	return(v);
}


TMPLMat inline TMat2 trans(const TMat2 &m)
{
	return(~m);
}

TMPLMat inline TMReal trace(const TMat2 &m)
{
	return(m[0][0] + m[1][1]);
}
			
TMPLMat inline TMat2 adj(const TMat2 &m)			
{
	return(!m);
}


#ifdef VL_TEMPLATE
#include "Mat2.cc"
#endif

#endif
