/*
	File:			Mat.h

	Function:		Defines a generic resizeable matrix.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Mat__
#define __Mat__

#include "VL.h"
#include "Vec.h"
#include "Action.h"
#include "SGMat.h"
#include "Mat2.h"
#include "Mat3.h"
#include "Mat4.h"

#include <iostream.h>


// --- Mat Class --------------------------------------------------------------


TMPLMat class Mat
{
public:
	
	// Constructors
	
	Mat();										// Null matrix: no space allocated
	Mat(Int rows, Int cols);					// Ordinary uninitialised matrix
	Mat(Int rows, Int cols, double elt0 ...);	// Mat(2, 2, 1.0, 2.0, 3.0, 4.0)
	Mat(Int nrows, Int ncols, TMReal *ndata);	// Create reference matrix
	Mat(Int rows, Int cols, ZeroOrOne k);		// zero or identity matrix
	Mat(const TMat &m);							// Copy constructor
	Mat(const TSGMat &m);						// Conversion constructors...
	Mat(TMat2 &m);			
	Mat(TMat3 &m);			
	Mat(TMat4 &m);			

   ~Mat();			
  	
	// Accessor methods
	
	inline Int	Rows() const { return rows; };
	inline Int	Cols() const { return cols; };

	inline TMVec		operator () (Int i);			// Indexing by row
	inline const TMVec	operator ()	(Int i) const;		  

	inline TMReal 		&operator () (Int i, Int j);	// Indexing by elt.
	inline TMReal		operator () (Int i, Int j) const;		  

	inline TMVec    	operator [] (Int i);			// Indexing by row; no checking
	inline const TMVec  operator [] (Int i) const;		  

	inline TMReal		&elt(Int i, Int j);	    		// Indexing by elt; no checking
	inline TMReal		elt(Int i, Int j) const;
	
	inline TMReal 		*Ref() const;					// Return pointer to data

	// Assignment operators
	
	TMat 	   		&operator =  (const TMat &m);		// Assignment of a matrix
	TMat 	   		&operator =  (const TSGMat &m);		// Assignment of a sub-matrix
	TMat			&operator =	 (ZeroOrOne zeroOrOne);	// Setting to a constant...
	TMat 	   		&operator =  (const TMat2 &m);	
	TMat 	   		&operator =  (const TMat3 &m);	
	TMat 	   		&operator =  (const TMat4 &m);	

	TMat			&operator >>  (Action<TMReal> &a);
	void 			SetSize(Int nrows, Int ncols);

	//	Matrix initialisers
	
	void			MakeZero();
	void			MakeUnit(TMReal k);
	void			MakeUnit();
	void			MakeBlock(TMReal k);
	void			MakeBlock();

	// Private...
	
protected:
	
	TMReal	*data; 
	UInt	rows  : 31;
	UInt	isRef :  1;
	UInt	cols;
};


// --- Mat In-Place Operators -------------------------------------------------

TMPLMat TMat		&operator += (TMat &m, const TMat &n);
TMPLMat TMat		&operator -= (TMat &m, const TMat &n);
TMPLMat TMat		&operator *= (TMat &m, const TMat &n);
TMPLMat TMat		&operator *= (TMat &m, TMReal s);
TMPLMat TMat		&operator /= (TMat &m, TMReal s);

// --- Mat Comparison Operators -----------------------------------------------

TMPLMat Bool		operator == (const TMat &m, const TMat &n);
TMPLMat Bool		operator != (const TMat &m, const TMat &n);

// --- Mat Arithmetic Operators -----------------------------------------------

TMPLMat TMat		operator + (const TMat &m, const TMat &n);
TMPLMat TMat		operator - (const TMat &m, const TMat &n);
TMPLMat TMat		operator - (const TMat &m);
TMPLMat TMat		operator * (const TMat &m, const TMat &n);
TMPLMat TVec		operator * (const TMat &m, const TVec &v);
TMPLMat TMat		operator * (const TMat &m, TMReal s);
TMPLMat inline TMat	operator * (TMReal s, const TMat &m);
TMPLMat TMat		operator / (const TMat &m, TMReal s);

TMPLMat TVec		operator * (const TVec &v, const TMat &m);
TMPLMat TVec		&operator *= (TVec &v, const TMat &m);

TMPLMat TMat 		trans(const TMat &m);
TMPLMat TMReal		trace(const TMat &m);

// --- Mat Input & Output -----------------------------------------------------

TMPLMat ostream		&operator << (ostream &s, const TMat &m);
TMPLMat istream		&operator >> (istream &s, TMat &m);

TMPLMat void		operator << (FILE *, const TMat &m);
TMPLMat Int			operator >> (FILE *, TMat &m);

// --- Sub-matrix operators --------------------------------------------------
	
TMPLMat TSGMat 		sub(const TMat &m, Int top, Int left, Int rows, Int cols);
TMPLMat TSGMat 		sub(const TMat &m, Int rows, Int cols);
TMPLMat TMSGVec 	col(const TMat &m, Int i);		
TMPLMat TMSGVec 	diag(const TMat &m, Int diagNum);
//	-i = diag. starting on row i, +i = diag. starting on col i


// --- Mat Inlines ------------------------------------------------------------


TMPLMat inline TMVec TMat::operator () (Int i)
{
	CheckRange(i, 0, Rows(), "(Mat::(i)) i index out of range");
	
    return(TMVec(cols, data + i * cols));
}

TMPLMat inline const TMVec TMat::operator () (Int i) const
{
	CheckRange(i, 0, Rows(), "(Mat::(i)) i index out of range");
	
    return(TMVec(cols, data + i * cols));
}

TMPLMat inline TMVec TMat::operator [] (Int i)
{
	CheckRange(i, 0, Rows(), "(Mat::[i]) i index out of range");
	
    return(TMVec(cols, data + i * cols));
}

TMPLMat inline const TMVec TMat::operator [] (Int i) const
{
	CheckRange(i, 0, Rows(), "(Mat::[i]) i index out of range");
	
    return(TMVec(cols, data + i * cols));
}

TMPLMat inline TMReal &TMat::operator () (Int i, Int j)
{
	CheckRange(i, 0, Rows(), "(Mat::(i,j)) i index out of range");
	CheckRange(j, 0, Cols(), "(Mat::(i,j)) j index out of range");
	
    return(data[i * cols + j]);
}

TMPLMat inline TMReal TMat::operator () (Int i, Int j) const
{
	CheckRange(i, 0, Rows(), "(Mat::(i,j)) i index out of range");
	CheckRange(j, 0, Cols(), "(Mat::(i,j)) j index out of range");
	
    return(data[i * cols + j]);
}

TMPLMat inline TMReal &TMat::elt(Int i, Int j)
{
	CheckRange(i, 0, Rows(), "(Mat::e(i,j)) i index out of range");
	CheckRange(j, 0, Cols(), "(Mat::e(i,j)) j index out of range");
	
    return(data[i * cols + j]);
}

TMPLMat inline TMReal TMat::elt(Int i, Int j) const
{
	CheckRange(i, 0, Rows(), "(Mat::e(i,j)) i index out of range");
	CheckRange(j, 0, Cols(), "(Mat::e(i,j)) j index out of range");
	
    return(data[i * cols + j]);
}

TMPLMat inline TMReal *TMat::Ref() const
{
	return(data);
}

TMPLMat inline TMat operator * (TMReal s, const TMat &m)
{
	return(m * s);
}

#ifdef VL_TEMPLATE
#include "Mat.cc"
#endif

#endif

