/*
	File:			Array.h

	Function:		Defines an array type that manages its own storage space, and can be
					used as a stack or a rudimentary list.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Array__
#define __Array__

#include <iostream.h>
#include "Basics.h"
#include "Action.h"


#define TMPLArray	template<class T>
#define TArray		Array<T>

const kFirstAllocation = 16;	// Default number of items to initially allocate to
								// the array

TMPLArray class Array
{
public:
					Array();
					Array(Int size, Int alloc = kFirstAllocation);		
					Array(const TArray &array);
				   ~Array();
	
//	Array operators
	
	inline T		&operator [] (Int i);		// indexing operator
	inline const T	&operator [] (Int i) const; // indexing operator
	inline Int		NumItems() const;			// Number of items in the array
	TArray			&operator >> (TAction &a);	// Apply action to array
	
	TArray			&operator = (const TArray &array);	// Assignment!
	
//	Useful for stack implementations

	inline T		&Top();						// Return last item (top-of-stack)
	inline void		Pop();						// Delete last item (pop)
	inline void		Push(const T &t);			// Add item to end of array (push)
	
// List Operations

	inline void		Append(const T &t);			// Append single item to end of array.
	inline T		&Last();					// Return last item (top-of-stack)
	void			Clear();					// Delete all items

	void			SetSize(Int newSize);		// Set array size directly.
	void			Add(Int n);					// Add n items to the array
	void			Shrink(Int n);				// shrink the array by n items
	void 			Insert(Int i, Int n);		// Insert n items at i
	void			Delete(Int i, Int n);		// Delete n items at i
	void			ShrinkWrap();				// Ensure allocated space = space being used.
	
// Low level access

	inline T		*Ref();						// Return pointer to array
	inline T		*Detach();					// As above, but the array no longer owns the data.

//	Private...

protected:
	
	void 		Grow();
	
	T			*item;		// pointer to array
	Int32 		items;		// items in the array
	Int32		allocated;	// number of items we have space allocated for. 
};	

TMPLArray ostream &operator << (ostream &s, TArray &array);
TMPLArray istream &operator >> (istream &s, TArray &array);


// --- Inlines ----------------------------------------------------------------


TMPLArray inline TArray::Array() : items(0), item(0), allocated(0)
{
}

TMPLArray inline Int TArray::NumItems() const
{
	return(items);
}

TMPLArray inline T &TArray::operator [] (Int i)
{
	CheckRange(i, 0, items, "(Array::[]) index out of range");

	return(item[i]);
}

TMPLArray inline const T &TArray::operator [] (Int i) const
{
	CheckRange(i, 0, items, "(Array::[]) index out of range");

	return(item[i]);
}

TMPLArray inline T &TArray::Top()
{
	return(item[items - 1]);
}

TMPLArray inline T &TArray::Last()
{
	return(item[items - 1]);
}

TMPLArray inline void TArray::Push(const T &t)
{
	if (items >= allocated)
		Grow();
	
	item[items++] = t;
}

TMPLArray inline void TArray::Append(const T &t)
{
	if (items >= allocated)
		Grow();
	
	item[items++] = t;
}

TMPLArray inline void TArray::Pop()
{	
	items--;
}

TMPLArray inline void TArray::Clear()
{	
	items = 0;
	allocated = 0;
	delete item;
	item = 0;
}

TMPLArray inline T *TArray::Ref()
{
	return(item);
}

TMPLArray inline T *TArray::Detach()
{
	T* result = item;

	items = 0;
	allocated = 0;
	item = 0;

	return(result);
}


#include "Array.cc"

#endif
