// Oren-Nayar reflectance model

// reference:
// Generalization of Lambert's Reflectance Model
// SIGGRAPH '94
//
// http://www.cs.columbia.edu/~oren/
// ftp://ftp.cs.columbia.edu/pub/CAVE/papers/oren/

// Rob DeLine, Nov. 1996

#include <math.h>
#include "oren-nayar.H"

Oren_Nayar::Oren_Nayar() {
  iname = new char[11];
  iname = "Oren-Nayar";
  nparam = 5;
  reset();
}

void Oren_Nayar::print(ostream &s) const {
  s << "[Oren-Nayar: ";
  for (int i=0; i<nparam; i++) {
    s << param[i].name << "=" << param[i].val;
    if (i<nparam-1) s << ",";
  }
  s << "]" << endl;
}


const double pi = 3.14159;


void Oren_Nayar::reset() {
  param[Sigma].reset("sigma", 1., 0., pi);
  param[Rho].reset("rho", 1., 0., 1.);
  param[Mat_R].reset("mat_r", 1., 0., 1.);
  param[Mat_G].reset("mat_g", 1., 0., 1.);
  param[Mat_B].reset("mat_b", 1., 0., 1.);
}


inline double min(double a, double b) { return (a < b)? a : b; }
inline double max(double a, double b) { return (a < b)? a : b; }
inline double squared(double a) { return a*a; }
inline double cubed(double a) { return a*a*a; }


inline double angle(const Vec3& v1, const Vec3& v2)
{ return acos( (dot(v1, v2)) / (len(v1)*len(v2)) ); }


Vec3 Oren_Nayar::brdf(const Vec3& n, const Vec3& l, const Vec3& v) const {

  double theta_l = angle(n,l);
  double theta_v = angle(n,v);
  double alpha = max(theta_l, theta_v);
  double beta = min(theta_l, theta_v);

  // project incoming and outgoing vectors to the tangent plane and
  // find cosine of angle between them (cosine of delta_azimuth)
  Vec3 vn = v - n*dot(n, v);
  Vec3 ln = l - n*dot(n, l);
  double norm = len(vn)*len(ln);
  double cos_delta_phi = (norm == 0)? 1. : dot(vn, ln) / norm;

  double sigma = param[Sigma].val;
  double sigma2 = squared(sigma);
  double rho = param[Rho].val;

  double C1 = 1. - 0.5*(sigma2 / (sigma2 + 0.33));
  double C2 = 0.45 * (sigma2 / (sigma2 + 0.09)) *
    (sin(alpha) - ((cos_delta_phi >= 0)? 0. : cubed(2.*beta/pi)));
  double C3 = 0.125 * (sigma2 / (sigma2 + 0.09)) *
    squared(4.*alpha*beta/squared(pi));

  double L1 = rho/pi * ( C1 +
			 cos_delta_phi * C2 * tan(beta) +
			 (1 - fabs(cos_delta_phi)) * C3 * tan((alpha+beta)/2.) );

  double L2 = 0.17 * squared(rho)/pi *
    (sigma2 / (sigma2 + 0.13)) *
    (1. - cos_delta_phi * squared(2.*beta/pi));

  Vec3 result(param[Mat_R].val,
	      param[Mat_G].val,
	      param[Mat_B].val);
//  cout << "sigma=" << sigma <<
//    " C1=" << C1 << " C2=" << C2 << " C3=" << C3 <<
//    " res=" << (L1+L2) << endl;
  return result * (L1 + L2);
}
