/*
 *  brdfview.C: The UI code for visualizing
 *  Bidirectional Reflectance Distribution Functions (BRDF's).
 *  This program uses XForms and OpenGL.
 *
 *  Thomas Kang and Paul Heckbert
 *
 *  October 26, 1996
 */


#include "brdfview.H"       // UI header
#include "phong.H"          // Phong model
#include "cook_torrance.H"  // Cook-Torrance model
#include "oren-nayar.H"     // Oren-Nayar model
#include "heBRDF.H"         // He-Torrance-Sillion-Greenberg model


/* **************** *
 * global variables *
 * **************** */


static Glxf_pane pane;      // X-OpenGL integration
FD_BRDFv *form;             // global form

Vec3 N(0., 1., 0.);         // surface normal
Vec3 L(1.41, 1.41, 1.41);   // light vector
double zoomval;             // view zoom factor

Illum *RM[NUM_MODELS];      // array of ptrs to models
int cur_model;              // currently active model
double rad_multiplier;      // radiance multiplier

Point_norm brdf[90/STEP + 1][360/STEP];  // cache of brdf points

#define HSTEP (STEP*2)


/* ************************************ *
 * form handlers and rendering routines *
 * ************************************ */


void light_init() {
  // set material properties
  GLfloat white6[] = {.6, .6, .6, 1.};
  GLfloat white4[] = {.4, .4, .4, 1.};
  GLfloat black[] = {0., 0., 0., 1.};
  GLfloat mat_shininess[] = {50.};		// Phong exponent
  glMaterialfv(GL_FRONT, GL_AMBIENT, black);	// no ambient
  glMaterialfv(GL_FRONT, GL_DIFFUSE, white6);
  glMaterialfv(GL_FRONT, GL_SPECULAR, white4);
  glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);

  // set up several lights
  // one white light for the front, red and blue lights for the left & top

  GLfloat light0_position[] = {1., 1., 5., 0.};	  // directional light (w=0)
  GLfloat white[] = {1., 1., 1., 1.};
  glLightfv(GL_LIGHT0, GL_POSITION, light0_position);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, white);
  glLightfv(GL_LIGHT0, GL_SPECULAR, white);
  glEnable(GL_LIGHT0);

  GLfloat light1_position[] = {-3., 1., -1., 0.};
  GLfloat red[] = {1., .3, .3, 1.};
  glLightfv(GL_LIGHT1, GL_POSITION, light1_position);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, red);
  glLightfv(GL_LIGHT1, GL_SPECULAR, red);
  glEnable(GL_LIGHT1);

  GLfloat light2_position[] = {1., 8., -2., 0.};
  GLfloat blue[] = {.3, .4, 1., 1.};
  glLightfv(GL_LIGHT2, GL_POSITION, light2_position);
  glLightfv(GL_LIGHT2, GL_DIFFUSE, blue);
  glLightfv(GL_LIGHT2, GL_SPECULAR, blue);
  glEnable(GL_LIGHT2);

  glEnable(GL_NORMALIZE);	// normalize normal vectors
      // need this because of glScale(zoomval)

  // do the following when you want to turn on lighting
  // glEnable(GL_LIGHTING);
}


void brdf_init() {
  // pre-compute brdf points in case BRDF function is slow (e.g., HTSG)
  int t, p, step;
  double theta, phi;
  
  if (cur_model != 3)
	step = STEP;
  else
	step = HSTEP;

  for (t = 0; t <= 90/step; t++) {
    theta = deg_to_rad((t<90/step ? t : t-.01) * step);
	// fudge the last ring of points so they're at theta=90-epsilon
	// many reflectance models will return brdf=0 at theta=90
    for (p = 0; p < 360/step; p++) {
      phi = deg_to_rad(p*step);

      // convert from spherical to cartesian
      Vec3 V(sin(theta)*sin(phi), cos(theta), sin(theta)*cos(phi));

      // get reflectance value,
      // save BRDF point for red channel in array
      // we throw away the green & blue values
      brdf[t][p].point = rad_multiplier * RM[cur_model]->brdf(N, L, V)[0] * V;
    }
  }

  // compute normal vectors by differencing neighbors & cross product
  Vec3 u, v;	// tangent vectors in theta and phi directions
  int p0 = 360/step-2, p1 = p0+1, p2;
  for (p2 = 0; p2 < 360/step; p0 = p1, p1 = p2, p2++) {
    for (t = 1; t <= 90/step; t++) {
      if (t<90/step)				// internal grid vertex
	u = brdf[t+1][p1].point - brdf[t-1][p1].point;
      else					// at edge of grid (equator)
	u = 3.*brdf[t][p1].point - 4.*brdf[t-1][p1].point
	  + brdf[t-2][p1].point;		// extrapolate using 3 samples
      v = brdf[t][p2].point - brdf[t][p0].point;
      brdf[t][p1].normal = cross(u, v);
      // since we have glEnable(NORMALIZE), we don't normalize normal here
    }
  }
  
  // the above does all but first parallel (the north pole), now do that
  Vec3 pole(0,0,0);
  for (p = 0; p < 360/step; p++)		// average 1st parallel normals
    pole += brdf[1][p].normal;
  // pole = norm(pole);				// no need to normalize
  for (p = 0; p < 360/step; p++)
    brdf[0][p].normal = pole;
}

void render_brdf(int rerender) {

  // draw floor
  glColor3f(0.3, 0.2, 0.1);
  glBegin(GL_POLYGON);
  glScalef(3, 3, 3);
  glVertex3f(-1,0,-1);
  glVertex3f(1,0,-1);
  glVertex3f(1,0,1);
  glVertex3f(-1,0,1);
  glEnd();

  // draw light vector
  glLineWidth(2.0);
  glColor3f(1.,1.,1.);
  glBegin(GL_LINES);
  glVertex3f(0.,0.,0.);
  glVertex3f(2.*L[0], 2.*L[1], 2.*L[2]);
  glEnd();

  // draw reflected vector
  glColor3f(.3,1.,.3);
  glBegin(GL_LINES);
  glVertex3f(0.,0.,0.);
  glVertex3f(-2.*L[0], 2.*L[1], -2.*L[2]);
  glEnd();

  // draw BRDF in shades of gray
  glEnable(GL_LIGHTING);
  glColor3f(1.,1.,1.);

  // re-compute BRDF values if we need to
  if (rerender)
    brdf_init();

  int t, p0, p1, step;

  if (cur_model != 3)
	step = STEP;
  else
	step = HSTEP;

  for (p0 = 360/step-1, p1 = 0; p1 < 360/step; p0 = p1, p1++) {
    // p0 is phi index on one meridian, p1 is phi index on next meridian
    glBegin(GL_QUAD_STRIP);
    for (t = 0; t <= 90/step; t++) {
      // t is theta index on current parallel

      glNormal3dv(&brdf[t][p1].normal[0]);
      glVertex3dv(&brdf[t][p1].point[0]);

      glNormal3dv(&brdf[t][p0].normal[0]);
      glVertex3dv(&brdf[t][p0].point[0]);
    }
    glEnd();
  }
  glDisable(GL_LIGHTING);
}

void redisplay(int rerender) {
  // clear image- & z-buffer
  glClearColor(0, 0, 0, 0);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);

  // render model
  glPushMatrix();
  glRotatef(fl_get_slider_value(form->RotX), 1., 0., 0.);
  glRotatef(fl_get_slider_value(form->RotY), 0., 1., 0.);
  glRotatef(fl_get_slider_value(form->RotZ), 0., 0., 1.);
  glScalef(zoomval, zoomval, zoomval);
  render_brdf(rerender);
  glPopMatrix();

  // swap buffers (for double buffering)
  glXSwapBuffers(pane.display, pane.window);
}

void manipulate(FL_OBJECT *obj, long foo) {
  obj; foo;
  for (int i = 0; i < RM[cur_model]->nparam; i++)
    RM[cur_model]->param[i].val = fl_get_slider_value(form->slider[i]);
  redisplay(1);
}

void radiance(FL_OBJECT *obj, long foo) {
  foo;
  rad_multiplier = fl_get_button(obj) ?
    cos(deg_to_rad(fl_get_slider_value(form->RotTheta))) : 1.0;
  redisplay(1);
}

void orient(FL_OBJECT *obj, long foo) {
  obj; foo;

  // re-calculate the light vector
  double theta = deg_to_rad(fl_get_slider_value(form->RotTheta));
  double phi = deg_to_rad(fl_get_slider_value(form->RotPhi));
  L[0] = sin(theta) * sin(phi);
  L[1] = cos(theta);
  L[2] = sin(theta) * cos(phi);

  // re-compute only if light vector was altered
  if ((obj == form->RotX) || (obj == form->RotY) || (obj == form->RotZ))
    redisplay(0);
  else
    // check if we need to update radiance multiplier, then re-render
    radiance(form->Scale, 0);
}

void print(FL_OBJECT *obj, long foo) {
  obj; foo;
  RM[cur_model]->print(cout);
}

void read_params() {
  for (int i = 0; i < RM[cur_model]->nparam; i++)
    fl_set_slider_value(form->slider[i],
			RM[cur_model]->param[i].val);
}

void reset(FL_OBJECT *obj, long foo) {
  obj; foo;
  RM[cur_model]->reset();
  read_params();
  redisplay(1);
}

void init_sliders() {
  // set slider min/max/tags
  for (int i = 0; i < RM[cur_model]->nparam; i++) {
    fl_show_object(form->slider[i]);
    fl_set_slider_bounds(form->slider[i],
			 RM[cur_model]->param[i].min,
			 RM[cur_model]->param[i].max);
    fl_set_object_label(form->slider[i],
			RM[cur_model]->param[i].name);
  }

  // hide the unused sliders
  for (; i < NUM_SLIDERS; i++)
    fl_hide_object(form->slider[i]);

  // reflect current settings with the sliders
  read_params();
}

void choose_model(FL_OBJECT *obj, long dum) {
  obj; dum;
  cur_model = fl_get_choice(form->ReflModel) - 1;
  init_sliders();
  redisplay(1);
}


/* *************************** *
 * general execution functions *
 * *************************** */


void handle_event(XEvent *event) {
  switch (event->type) {
  case Expose:         // window just uncovered
    redisplay(1);
    break;

  case ButtonRelease:  // button release
    zoomval += (event->xbutton.button == 1) ? 0.25 : -0.25;
    if (event->xbutton.button == 2)
      zoomval = 2.0;
    redisplay(0);
    break;

  default:
    break;
  }
}

int main(int argc, char **argv) {
  Phong RM_Phong;        // Phong model
  CT RM_CT;              // Cook-Torrance model
  Oren_Nayar RM_ON;      // Oren-Nayar model
  HTSG RM_HTSG;          // He-Torrance-Sillion-Greenberg model

  RM[0] = (Illum *)&RM_Phong;
  RM[1] = (Illum *)&RM_CT;
  RM[2] = (Illum *)&RM_ON;
  RM[3] = (Illum *)&RM_HTSG;

  // initialize form
  fl_initialize(&argc, argv, "BRDF Viewer", 0, 0);
  form = create_form_BRDFv();
  fl_set_slider_value(form->RotTheta, 30.);
  fl_set_slider_value(form->RotPhi, 90.);
  fl_set_slider_value(form->RotX, 20.);

  // register models into the chooser
  for (int i = 0; i < NUM_MODELS; i++)
    fl_addto_choice(form->ReflModel, RM[i]->iname);

  // set start-up defaults
  cur_model = 0;
  zoomval = 2.0;
  init_sliders();

  // reveal the form
  fl_show_form(form->BRDFv,    // form
	       FL_PLACE_SIZE,  // pos & size flags
	       FL_FULLBORDER,  // border flags
	       argv[0]         // window name
	       );

  // bind OpenGL pane to display
  glxf_bind_pane(&pane,
		 form->pane,
		 1,           // double buffer?
		 handle_event
		 );

  // set global state for OpenGL
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(45.,   // vertical field of view
		 1.,    // aspect ratio
		 .1,    // znear
		 1000.  // zfar
		 );
  glMatrixMode(GL_MODELVIEW);
  glTranslatef(0., 0., -10.);

  // initialize lights
  light_init();

  // finally, display using the default orientation
  orient(NULL, 0);

  // have Xforms handle events until "exit" button is hit
  while (fl_do_forms() != form->exit);

  return(0);
}
