; %
; %    K S t r e a m s
; %
; % As in Abelson & Sussman, Section 3.4.3, page 259 ff.
; %
; Kstreams are an efficient implementation of long, or even infinite,
; streams, using {\tt delay} and {\tt force}.  These Scheme primitives
; provide delayed evaluation, which is necessary for decent
; efficiency.  Without them, long streams are obscenely inefficient.
;
; I named these optimized streams kstreams just so I could write {\tt
; kons} for the kstream konstructor.  The names of other kstream
; objects all begin with ``k''.  The convention that the name of every
; kstream object begin with ``k'' prevents name conflicts with Scheme
; primitives such as {\tt cons} and {\tt map}.  This naming convention
; also allows kstreams to coexist with streams implemented as ordinary
; lists -- {\em Lstreams}. Streams so implemented are appropriate for
; short streams, such as those representing vectors and matrices (see
; page 252 of A&S and ``Lstream.scm'').  For short lists, the
; function-call overhead inherent in {\tt delay} and {\tt force} is a
; little too costly.
;
; The following operators are defined for KStreams:
;
; essential operators:
;
; (kons <head> <tail>)
; (kar <ks>)
; (kdr <ks>)
; (kth <k> <ks>)
; (kempty? <ks>)
; knil % cute, huh?
; (kappend <ks1> <ks2>)
; (kaccum <combiner> <initial-value> <ks>)
; (kflatten <ks>)
; (kmap <proc> <ks>)
; (kflatmap <proc> <ks>)
; (kfilter <predicate?> <ks>)
; (kfor-each <proc> <ks>)
; (kprint <ks>)
; (ks->list <ks>)
; (klist->ks <list>)
; (kstream <elt1> <elt2> ... <eltn>)
; (kaccum-n <op> <initial-value> <ks-of-ks's>)
; (kollect <result> ((<v1> <set1>) ... (<vn> <setn>)) <restriction>)
;
; application and example operators:
;
; (enumerate-interval <n1> <n2>)
; (kleft-accum <combiner> <initial-value> <ks>)
; (ks-of-pairs <max-number>)
; (k*vv <vector1> <vector2>)
; (k*mv <matrix> <vector>)
; (k%m  <matrix>)
; (k*mn <matrix1> <matrix2>)
; (define (ks-of-pairs n)
; (prime-sum-pairs n)
; (kremove item ks)
; (kpermutations s)
; (integers-from n)
; integers
; no-sevens
; (fibgen a b)
; fibs
; (sieve <ks>)
; primes-sieve
; primes
; (prime? <n>)
; (kadd <k1> <k2>)
; (kscale <scalar> <ks>)


(define knil ())

(macro kons (lambda (l)
              `(cons ,(cadr l) (delay ,(caddr l)))))
;
; The definition of {\tt enumerate-interval} is needed here to help
; test the implementation of kons.
;

(define (enumerate-interval lo hi)
  (if (> lo hi)
      knil
      (kons lo (enumerate-interval (1+ lo) hi))))

(define (kar ks) (car ks))

(define (kdr ks) (force (cdr ks)))

(define (kth k ks)
  (if (= k 0)
      (kar ks)
      (kth (-1+ k) (kdr ks))))

(define kempty? null?)

; The following variables are abstract operators on streams,
; independent of the representation above.

(define (kappend ks1 ks2)
  (cond
   (  (kempty? ks1)  ks2  )
   (  t  (kons (kar ks1) (kappend (kdr ks1) ks2)))))

(define (kaccum combiner initial-value ks)
  (cond
   (  (kempty? ks)  initial-value  )
   (  t  (combiner
          (kar ks)
          (kaccum combiner initial-value (kdr ks)))  )))

(define (kflatten ks)
  (kaccum kappend knil ks))

;
; The following test application evaluates a polynomial, represented
; by the kstream {\em coeff-stream} of its coefficients, at the
; supplied point {\em x}.
;

(define (horner-eval x coeff-stream)
  (define (add-term coeff higher-terms)
    (+ coeff (* x higher-terms)))
  (kaccum add-term 0 coeff-stream))

(define (kfilter predicate? ks)
  (cond
   (  (kempty? ks)  knil  )
   (  (predicate? (kar ks))
        (kons
         (kar ks)
         (kfilter predicate? (kdr ks)))  )
   (  t  (kfilter predicate? (kdr ks))  )))

(define (kmap proc ks)
  (cond
   (  (kempty? ks)  knil  )
   (  t  (kons
          (proc (kar ks))
          (kmap proc (kdr ks)))  )))

(define (kflatmap f s) (kflatten (kmap f s)))

(define (kfor-each proc ks)
  (cond
   (  (kempty? ks)  'done  )
   (  t  (begin
          (proc (kar ks))
          (kfor-each proc (kdr ks)))  )))

(define (probably-kstream? ks)
  (cond
   (  (kempty? ks)           t  )
   (  (atom?   ks)           #f )
   (  (procedure? (cdr ks))  t  )
   (  t                      #f )))

(define kprint
  (letrec
   ((foo
     (lambda (ks)
       (cond ((kempty? ks) (begin (display "]") (newline) t))
             (t
              (begin (let (  (h (kar ks))  )
                       (if (probably-kstream? h) (kprint h)
                                                 (display h)))
                     (if (not (kempty? (kdr ks))) (display " "))
                     (foo (kdr ks))))))))
    (lambda (ks) (display "[") (foo ks))))

;;; page 251, Ex. 3.38

(define (kleft-accum combiner initial-value ks)
  (cond
   (  (kempty? ks)  initial-value  )
   (  t  (kleft-accum combiner
                      (combiner initial-value (kar ks))
                      (kdr ks))  )))
;;; end of exercise

(define (ks->list ks) (kaccum cons () ks))

(define (klist->ks list)
  (cond
   (  (null? list)  knil  )
   (  t  (kons (car list) (klist->ks (cdr list)))  )))

(define kstream
  (lambda args
    (cond
     (  (null? args)  knil  )
     (  t  (kons (car args) (klist->ks (cdr args)))  ))))

;;; page 252, Ex. 3.39

(define (kaccum-n op init ks-of-kss)
  (cond
   (  (kempty? (kar ks-of-kss))  knil  )
   (  t  (kons
          (kaccum   op init (kmap kar ks-of-kss))
          (kaccum-n op init (kmap kdr ks-of-kss)))  )))

;
; The syntax extension library is needed at this point (see Dybvig).
;
; (load "top 20:[scheme]:contributed:extendsyntax:extendsyntax.sch")
;
; This library is necessary, but not sufficient for the definition of
; {\tt kollect}.  {\tt Kollect} is syntactic sugar for nested {\tt
; kflatten} and {\tt kmap} operators.  The innermost operator is {\tt
; kmap}, and the outer operators are {\tt kflatmap}s.  {\tt Kflatmap},
; of course, is simply a composition of {\tt kflatten} and {\tt kmap}.
; Dybvig's {\tt extend-syntax} is not quite up to the task of creating
; {\tt kollect}, since its ellipsis notation allows one to repeat an
; expansion for all present arguments, but not to give special
; treatment to the innermost expression of a sequence of expressions.
; Either a new kstream operator, say {\tt kmap-or-kflatmap}, that
; tests the contents of its stream argument, and only flattens if the
; stream contains streams, or a new syntactic sugar is needed.
;% @@ This is yet another thing to finish.
;

;% page 252, Ex 3.40

;
; The following application operators furnish an arbitrary-dimensional
; matrix library.  Amazingly, they are all one-liners! The present
; kstream version of this library is appropriate for very large
; matrices, with dimension over 20 or so. For smaller matrices, the
; library implemented in terms of ordinary lstreams -- streams as
; lists -- is appropriate.
;

(define (k*vv v w) (kaccum + 0 (kaccum-n * 1 (kons v (kons w knil)))))

(define (k*mv m v) (kmap (lambda (row) (k*vv row v)) m))

; The following definition of {\tt k%m}, the transpose operator, doesn't
; work because {\tt kons} is not a procedure, but a syntactic form.
; However, it does express the intention.
;
; (define (k%m m) (kaccum-n kons () m))  ;; TRANSPOSE
;
; The correct definition of {\tt k%m} follows as a macro: @@ TO DO
;

(define (k*mm m n) (let ((%n (k%m n))) (kmap (lambda (v) (k*mv %n v)) m)))

; Here are the examples from page 253, leading to the definition of
; {\tt kollect}.  These examples are useful for testing, and that is why
; they are defined at this point.

(define (ks-of-pairs n)
  (kflatten (kmap (lambda (i)
                    (kmap (lambda (j) (list i j))
                          (enumerate-interval 1 (-1+ i))))
                  (enumerate-interval 1 n))))

(load "scheme code:*.scm:primes.scm")

(define (prime-sum-pairs n)
  (kmap (lambda (pair)
          (list (car pair)
                (cadr pair)
                (+ (car pair) (cadr pair))))
        (kfilter (lambda (pair)
                   (prime? (+ (car pair) (cadr pair))))
                 (ks-of-pairs n))))

(define (kremove item ks)
  (kfilter (lambda (x) (not (equal? x item))) ks))

(define (kpermutations s)
  (cond
   (  (kempty? s)  (kons () knil)  )
   (  t  (kflatmap
          (lambda (x)
            (kmap
             (lambda (p)
               (kons x p))
             (kpermutations (kremove x s))))
          s)  )))

;% Infinitely long streams.

(define (integers-from n)
  (kons n (integers-from (1+ n))))

(define integers (integers-from 1))

(define no-sevens
  (kfilter (lambda (x) (not (divides? x 7)))
           integers))

(kth 100 no-sevens)

(define (fibgen a b)
  (kons a (fibgen b (+ a b))))

(define fibs (fibgen 0 1))

(define (sieve ks)
  (kons
   (kar ks)
   (sieve
    (kfilter
     (lambda (x) (not (divides? (kar ks) x)))
     (kdr ks)))))

(define kprimes-sieve (sieve (integers-from 2)))

(define kprimes
  (kons 2 (kfilter kprime? (integers-from 3))))

(define (kprime? n)
  (define (iter ps)
    (cond
     (  (> (square (kar ps)) n)  t  )
     (  (divides? (kar ps) n)  ()  )
     (  else  (iter (kdr ps))  )))
  (iter kprimes))

(define (kfactors n)
  (define ps kprimes)
  (define (next factor)
    (set! ps (kdr ps))
    (kar ps))
  (define squirt (sqrt n))
  (define (iter factor n)
    (cond
     (  (< n 2)              ()  )
     (  (> factor squirt)    (list n)  )
     (  (divides? factor n)  (cons factor (iter factor (/ n factor)))  )
     (  else                 (iter (next factor) n)  )))
  (iter 2 n))

(define (kprime-factors n)
  (powers-list (kfactors n)))

;% This baby is a hog for memory, so be sure to (define primes 0) (gc)
;% when you're done.

(define (kadd k1 k2)
  (cond
   (  (kempty? k1)  k2  )
   (  (kempty? k2)  k1  )
   (  t             (kons
                     (+ (kar k1) (kar k2))
                     (kadd (kdr k1) (kdr k2))))))

(define (kscale scalar ks)
  (kmap (lambda (x) (* x scalar)) ks))
