(herald throw (env tsys))

;;; Copyright (c) 1985 Yale University
;;;     Authors: N Adams, R Kelsey, D Kranz, J Philbin, J Rees.
;;; This material was developed by the T Project at the Yale University Computer 
;;; Science Department.  Permission to copy this software, to redistribute it, 
;;; and to use it for any purpose is granted, subject to the following restric-
;;; tions and understandings.
;;; 1. Any copy made of this software must include this copyright notice in full.
;;; 2. Users of this software agree to make their best efforts (a) to return
;;;    to the T Project at Yale any improvements or extensions that they make,
;;;    so that these may be included in future releases; and (b) to inform
;;;    the T Project of noteworthy uses of this software.
;;; 3. All materials developed as a consequence of the use of this software
;;;    shall duly acknowledge such use, in accordance with the usual standards
;;;    of acknowledging credit in academic research.
;;; 4. Yale has made no warrantee or representation that the operation of
;;;    this software will be error-free, and Yale is under no obligation to
;;;    provide any services, by way of maintenance, update, or otherwise.
;;; 5. In conjunction with products arising from the use of this material,
;;;    there shall be no use of the name of the Yale University nor of any
;;;    adaptation thereof in any advertising, promotional, or sales literature
;;;    without prior written consent from Yale in each case.
;;;

;;;; dynamic state manipulation

;;; deficiencies:
;;;  - there's no way to temporarily "bind" the dynamic state (a la pdl
;;;    pointer args to eval in maclisp).
;;;  - there's no proviso for coroutines/multitasking (stack groups).
;;; these both stem from the assumption rooted fairly deep that changes in
;;; the "dynamic state" are irreversible.  this will change.

;;; binding:

(define-predicate stack?)

(define-handler stack
  (object nil
    ((stack? self) t)
    ((print-type-string self) "Stack")))
  

(define (bind-handler wind stuff unwind)
  ;; someday worry about doing things atomically.
  (wind)
  (push-magic-frame unwind stuff wind))


(define (unwind-protect-handler stuff unwind)
  (push-magic-frame unwind stuff false))

(define (bind-internal state stuff)
  (set (system-global slink/dynamic-state) state)
  (receive results
           (stuff)
    (perform-unwind state)
    (set (system-global slink/dynamic-state) (state-previous state))
    (apply return results)))

(define (perform-unwind state)          ; want better name!
  (let ((unwind (state-unwinder state)))
    (set (state-unwinder state)
         (if (eq? unwind throw-out-of-unwind)      ; kludge
             false
             throw-out-of-unwind))
    (unwind)
    (set (state-unwinder state) false)))

(define (throw-out-of-unwind)
  (error "attempting to throw out of an unwind or unbind action -~%~
          **~10tdoing (ret) or (reset) will abort the unwind action and~%~
          **~10tproceed with the throw anyhow."))

;;; throwing: one-stack model.

;;; the following is invariant, for now at least:
;;;  (eq? *dynamic-state* (get-dynamic-state (current-frame)))

(lset *the-current-throw-value* nil)          ; el hacko grossness
(lset *the-current-throw-frame* nil)
                                                    
(define-operation (escape-procedure-frame proc))
(define-predicate escape-procedure?)
                                
(define (t-code-catch proc sp link state frame)
    (proc (object (lambda vals (known-frame-throw state sp vals link))
            ((print-type-string self) "Escape-procedure")
            ((escape-procedure? self) t)
            ((escape-procedure-frame self) frame))))

         
(define (t-code-call/cc proc sp)
  (let* ((current-state (system-global slink/dynamic-state))
         (base-state (get-base-state current-state))
         (stack (copy-stack sp)))
    (proc (object (lambda vals 
              (continuation-throw sp stack vals current-state base-state))
            ((print-type-string self) "Upward-continuation")))))
                                                                
(define (get-base-state state)
  (iterate loop ((state state))
    (let ((prev (state-previous state)))
      (cond ((null? prev) state)
            (else (loop prev))))))
           
;;; COPY-STACK The make-pointer is to pretend that the
;;; stack has a header at the top

(define (copy-stack sp)
  (let* ((size (fx+ (fx- (system-global slink/stack) sp) 1))
         (stack (make-vector-extend header/stack 
                                    (enforce acceptable-vector-size? size)
                                    size)))
;    (disable-interrupts)
    (%copy-extend stack 
                  (make-pointer (gc-extend->pair (gc-extend->pair sp)) -2)
                  size)
;    (enable-interrupts)
    stack))

(define (continuation-throw sp stack vals k-state base-state)
  (cond ((stack? stack)
         (let ((a (swap *the-current-throw-value* vals))
               (b (swap *the-current-throw-frame* stack)))
           (unwind-to-top)
           (set *the-current-throw-frame* b)
           (set *the-current-throw-value* a)
           (set (system-global slink/dynamic-state) k-state)
           (invoke-continuation sp stack vals base-state k-state)))
        (else
         (error "throwing ~s to bad continuation ~s" vals stack))))
                                  
                             
(define (rewind-state-and-continue from to vals)
  (do ((state from (state-next state)))
      ((eq? state to) 
       ((state-winder state))
       (apply return vals))
    ((state-winder state))))

(define (frame-throw fr vals)
  (cond ((reasonable-frame? fr)
	 (let ((state (get-dynamic-state fr)))
	   (known-frame-throw state (frame-sp fr) vals (frame-header fr))))
        (else
         (frame-throw (error "invalid frame - (~s ~s ~s)"
                             'frame-throw fr vals)
                      vals))))

(define (known-frame-throw to-state sp vals link)
  (let ((a (swap *the-current-throw-value* vals))
	(b (swap *the-current-throw-frame* sp)))
    (unwind-to-state to-state)
    (set *the-current-throw-frame* b)
    (set *the-current-throw-value* a)
    (set (system-global slink/dynamic-state) to-state)
    (invoke-stack-continuation sp vals link)))
    

(define (unwind-to-state to-state)
  (iterate loop ((state (system-global slink/dynamic-state)))
    (cond ((eq? state to-state) 'done)
          ((null? state)
           (warning "lost big while changing dynamic context to ~s!~
                    ~%;**     attempting to do the throw anyhow...~%"
                    to-state))
          (else

           (perform-unwind state)
           (loop (state-previous state))))))

(define (unwind-to-top)
  (iterate loop ((state (system-global slink/dynamic-state)))
    (cond ((eq? state nil))
          ((eq? (state-winder state) false)
	   (loop (state-previous state)))
          (else
           (perform-unwind state)
           (loop (state-previous state))))))

(define (get-dynamic-state frame)
  (cond ((null? frame) '())
        ((eq? (frame-header frame) *magic-frame-template*) frame)
        (else (get-dynamic-state (frame-previous frame)))))

(define (reasonable-frame? frame)
  (and (frame? frame)                ; robust?
       (let ((frame (descriptor->fixnum frame)))
         (and (fx> frame (stack-pointer))
              (fx<= frame (system-global slink/stack))))))
