; File bootit.scm / -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; Copyright (c) 1991 Jonathan Rees / See file COPYING

; Booting new Pseudoscheme

; In any Scheme-in-CL implementation, load this file and do
; (bootit).  This compiles and loads the translator, then invokes
; the translator to translate itself.

; A fair amount of hacking will be needed before this can be
; done in a non-Common-Lisp-based Scheme.

(define *pseudoscheme-directory* #f)

(define (bootit . dir-option)
  (cond ((not (null? dir-option))
	 (set! *pseudoscheme-directory* (lisp:pathname (car dir-option))))
	((not *pseudoscheme-directory*)
	 (set! *pseudoscheme-directory*
	       (lisp:make-pathname :name 'lisp:nil
				   :type 'lisp:nil
				   :defaults
				     lisp:*default-pathname-defaults*))))
  (init-hacks)
  (load-untranslated-translator)
  (fix-reader-if-necessary)
  (translate-runtime)
  (translate-translator))

(define (new-package name use nicks)
  (let ((loser (lisp:find-package name)))
    (lisp:if loser
	     (let ((backup (string-append "OLD-" name)))
	       (lisp:if (lisp:find-package backup)
			(begin
			  (lisp:warn "Using existing ~A package."
				     name)
			  (lisp:rename-package loser name nicks)
			  (lisp:use-package use loser)
			  loser)
			(begin
			  (lisp:warn "Renaming existing ~A package to be ~A."
				     name
				     backup)
			  (lisp:rename-package loser backup)
			  (lisp:make-package name :use use :nicknames nicks))))
	     (lisp:make-package name :use use :nicknames nicks))))

(define hacks-package #f)
(define clever-load #f)
(define lisp-package #f)
(define scheme-package (lisp:symbol-package 'askdjfh))

(define (init-hacks)
  (set! hacks-package
	(new-package "SCHEME-HACKS" '("LISP") '("SCHH")))
  (lisp:let ((lisp:*package* hacks-package))
    (lisp:load (pseudo-pathname "clever") :verbose 'lisp:nil)	;Get clever file loader
    (set! clever-load (lisp:symbol-function (hack-symbol 'clever-load)))
    ;; Defines a few things used by the translator
    (clever-load (pseudo-pathname "hacks")
		 :compile-if-necessary #t)
    (lisp:funcall (hack-symbol 'fix-scheme-package-if-necessary)
		  scheme-package)
    (set! lisp-package (lisp:symbol-value (hack-symbol 'lisp-package)))
    ;; Create SCHI package (translator contains quoted schi:foo's)
    (let ((schi-package (lisp:find-package "SCHI")))
      (lisp:if schi-package
	       (begin
		 (lisp:rename-package schi-package "SCHI")
		 (clever-load (pseudo-pathname "schi")))))))

; Make sure that quote and backquote read in properly.
; Careful, this may cause them to stop working in the Scheme from which
; we're bootstrapping.  It should be done after all LOAD's and before any
; READ's.
(define (fix-reader-if-necessary)
  (if (not (eq? (car ''foo) 'quote))
      (lisp:set-macro-character
        #\'
	(lambda (stream c)
	  (list ''quote (lisp:read stream 'lisp:t 'lisp:nil 'lisp:t)))))
  (if (not (eq? (car '`(foo)) 'quasiquote))
      (begin (lisp:set-macro-character
	      #\`
	      (lambda (stream c)
		(list ''quasiquote
		      (lisp:read stream 'lisp:t 'lisp:nil 'lisp:t))))
	     (lisp:set-macro-character
	      #\,
	      (lambda (stream c)
		(let* ((following-char
			(lisp:peek-char 'lisp:nil stream
					'lisp:t 'lisp:nil 'lisp:t))
		       (marker (cond ((char=? following-char #\@)
				      (lisp:read-char stream)
				      'unquote-splicing)
				     (else
				      'unquote))))
		  (list marker
			(lisp:read stream 'lisp:t 'lisp:nil 'lisp:t))))))))

(define (hack-symbol name)
  (lisp:intern (symbol->string name) hacks-package))

(define (pseudo-pathname name)
  (lisp:make-pathname :name (filename-preferred-case name)
		      :defaults *pseudoscheme-directory*))

(define (filename-preferred-case name)
  #+unix (lisp:string-downcase name)
  #-unix (lisp:string-upcase name)
  )

(define *scheme-file-type*     (filename-preferred-case "scm"))
(define *translated-file-type* (filename-preferred-case "pso"))
(define *boot-file-type*       (filename-preferred-case "boot"))

; Make sure the host system understands that files foo.boot are
; compiled.

#+Lucid
(if (not (member *boot-file-type*
		 lucid::*load-binary-pathname-types*))
    (set! lucid::*load-binary-pathname-types*
	  (append lucid::*load-binary-pathname-types*
		  (list *boot-file-type*))))

#+Symbolics
(begin
  (fs:define-canonical-type :boot-bin #,*boot-file-type*)

  (set! fs:*auxiliary-loadable-file-types*
	(cons '(:boot-bin :load-stream-function
			  si:load-binary-file-internal)
	      (lisp:remove :boot-bin fs:*auxiliary-loadable-file-types*
			   :key #'car)))

  (lisp:setf (lisp:get :boot-bin :binary-file-byte-size)
	     (lisp:get :bin :binary-file-byte-size)))

(define translator-files #f)

; ----- Load the translator into a scheme emulation

(define (load-untranslated-translator)
  ;; Make sure we perform integrations!
  (lisp:if (lisp:fboundp 'go)
	   (go 'usual))
  (lisp:if (lisp:fboundp 'benchmark-mode)
	   (benchmark-mode))
  (set! translator-files
	(call-with-input-file (pseudo-pathname "translator.files") read))
  (for-each load-scheme translator-files)
  'done)

(define (load-scheme file)
  (clever-load (pseudo-pathname file)
	       :source-type *scheme-file-type*
	       :object-type *boot-file-type*
	       :compile-if-necessary #t))

; ----- Translating the runtime system

(define (translate-runtime)
  ;; In principle, there could be more stuff here.
  (write-closed-definitions
     revised^4-scheme-module
     (lisp:make-pathname :type *translated-file-type*
			 :defaults (pseudo-pathname "closed")))
  (for-each (lambda (f)
	      (let ((f (pseudo-pathname f)))
		(really-translate-file
		  (lisp:make-pathname :type *scheme-file-type* :defaults f)
		  (lisp:make-pathname :type *translated-file-type* :defaults f)
		  revised^4-scheme-env)))
	    '(;; These are both optional.  Cf. load-runtime in loadit.scm.
	      "read"
	      "write"
	      )))


; ----- Translating the translator

(define scheme-translator-env #f)

(define (translate-translator)
  (set! scheme-translator-env
	(make-program-env 'scheme-translator
			  (list revised^4-scheme-module)))
  (for-each translate-translator-file
	    translator-files)
  (translate-translator-file "reflect")
  'done)

(define (translate-translator-file file)
  (let ((f (pseudo-pathname file)))
    (really-translate-file (lisp:make-pathname :type *scheme-file-type*
					       :defaults f)
			   (lisp:make-pathname :type *translated-file-type*
					       :defaults f)
			   scheme-translator-env)))
