; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File translate.scm / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Translation from Scheme to Common Lisp

; TRANSLATE translates a single Scheme expression into Common Lisp.

(define (translate form macrologies meta-env package)  ;+++ context?
  (with-target-package package
    (lambda ()
      (translate-to-common-lisp (list form) macrologies meta-env))))

; This is used by both the expression compiler and the file compiler.

(define (translate-to-common-lisp forms macrologies meta-env)
  (with-uid-reset
   (lambda ()
     (let* ((g-env (make-global-s-env #f))
	    (s-env (add-macrologies g-env macrologies))
	    (result (map (lambda (form)
			   (alpha-top form s-env))
			 forms)))
       (if meta-env
	   (apply-integrations (accumulated-globals g-env)
			       meta-env))
       (prognify (map (lambda (node)
			(generate-top node g-env))
		      result))))))

#|
(define (translating-to-common-lisp macrologies meta-env proc)
  (with-uid-reset
   (lambda ()
     (let* ((g-env (make-global-s-env #f))
	    (s-env (add-macrologies g-env macrologies)))
       (proc (lambda (form)
	       (accumulating-globals g-env
		 (lambda ()
		   (let ((node (alpha-top form s-env)))
		     (if meta-env
			 (apply-integrations (accumulated-globals g-env)
					     meta-env))
		     (generate-top node g-env))))))))))
|#

; Integrations

(define (apply-integrations vars meta-env)
  (for-each (lambda (var)
	      (let ((probe (meta-env-ref meta-env (variable-name var))))
		(if probe
		    (if (eq? (variable-status var) 'defined)
			(note "DEFINE of an integrated variable" var)
			(apply-integration var probe)))))
	    vars))

(define (apply-integration var int)
  ;; (lisp:format t "~&Integrating ~s~%" (variable-name var))
  (if (eq? (car int) 'integrations)
      ;; int = (integrations <meta-env>)
      (let ((sub (variable-substitution var)))
	(if (and sub (eq? (car sub) 'struct))
	    (apply-integrations ((caddr sub));???+++
				(cadr int))))
      (set-substitution! var int)))

; File transduction

(define (really-translate-file source-file-name
			       translated-file-name
			       macrologies
			       meta-env
			       package)
  (let ((source-code (read-file source-file-name)))
    (compiling-to-file
      translated-file-name
      package
      (lambda (port)
	(display ";  from file " port)
	(display (lisp:namestring (lisp:truename source-file-name)) port)
	(newline port))
      (lambda (port)
	(for-each (lambda (form)
		    (write-flattened form port))
		  (translate-to-common-lisp source-code
					    macrologies
					    meta-env))))))

; The following generates a file CLOSED.PSO from the information we
; have on how to open-code the built-in procedures.

(define (write-closed-definitions package outfile)
  (compiling-to-file outfile
		     package
		     (lambda (port) port)
		     write-revised^3-definitions))

(define (write-revised^3-definitions port)
  (let ((package (fluid @target-package)))
    (write-form
      `(lisp:export
	 (lisp:quote ,(map (lambda (name) (change-package name package))
			   (signature-vars revised^3-scheme-sig))))
      port)
    (let ((funs '())
	  (defs '()))
      (let ((do-it
	     (lambda (name)
	       (let ((info (meta-env-ref revised^3-scheme-integrations name))
		     (sym (change-package name package)))
		 (if info
		     (case (car info)
		       ((val)		;T and NIL
			(write-form `(lisp:locally (lisp:declare (lisp:special ,sym))
				       (lisp:setq ,sym ,(cadr info)))
				    port)
			(write-form `(schi:set-function-from-value
					  (lisp:quote ,sym))
				    port))
		       ((fun)
			(if (not (memq name '(car cdr)))
			    (set! funs (cons (list sym (cadr info))
					     funs))))
		       ((subst lambda)
			(write-form `(lisp:defun ,sym ,@(cdr info)) port)
			(set! defs (cons sym defs))))
		     ;(begin (newline)
			;    (display "Warning: no definition for ")
			;    (write sym))
		   )))))
	(for-each do-it (signature-vars revised^3-scheme-sig))
	(for-each do-it (signature-aux-vars revised^3-scheme-sig)))
      (write-form
        `(lisp:mapc (lisp:function schi:set-value-from-function)
		    (lisp:quote ,(reverse defs)))
	port)
      (write-form
        `(lisp:mapc #'(lisp:lambda (z)
			(lisp:let ((r^3-sym (lisp:car z))
				   (cl-sym (lisp:cadr z)))
			  (lisp:setf (lisp:symbol-function r^3-sym) 
				     (lisp:symbol-function cl-sym))
			  (schi:set-value-from-function r^3-sym)))
		    (lisp:quote ,(reverse funs)))
	port))))

; Utilities

(define (with-target-package package thunk)
  (let-fluid @target-package package
    thunk))

(define (compiling-to-file outfile package write-message proc)
  (let-fluid @translating-to-file? #t
    (lambda ()
      (with-target-package package
	(lambda ()
	  (call-with-output-file outfile
	    (lambda (port)
	      (write-file-identification port)
	      (write-message port)
	      (newline port)
	      (display "(SCHEME-INTERNAL:BEGIN-TRANSLATED-FILE)" port)
	      (newline port)
	      ;; Now do the real work.
	      (proc port)
	      (newline port)
	      outfile)))))))

(define (write-file-identification port)
  (newline)
  (display "Writing ")
  (display (lisp:namestring (lisp:truename port)))
  (display "; -*- Mode: Lisp; Syntax: Common-Lisp; Package: " port)
  (display (lisp:package-name (fluid @target-package)) port) ;Heuristic
  (display "; -*-" port)
  (newline port)
  (newline port)
  (display "; This file was generated by " port)
  (display (translator-version) port)
  (newline port)
  (display ";  running in " port)
  (display (scheme-implementation-version) port)
  (newline port))

(define (write-flattened form port)
  (cond ((not (pair? form))
	 (if (not (or (symbol? form)
		      (number? form)
		      (boolean? form)
		      (string? form)
		      (char? form)))
	     ;; Who knows, it might be important.
	     (write-form form port)))
        ((eq? (car form) 'lisp:quote)
	 )				;do nothing
	((eq? (car form) 'lisp:progn)
	 (for-each (lambda (form)
		     (write-flattened form port))
		   (cdr form)))
	(else
	 (write-form form port))))

(define (write-form form port)
  (write-pretty form port (fluid @target-package)))
