; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File derive.scm / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Macro expanders for standard derived expression types

; The REVISED^3-SCHEME macrology:

(define revised^3-scheme-macrology
  (make-macrology
    (lambda (s-env known-s-env add-macrology)
      (let* ((core-s-env
	      (add-macrology known-s-env
			     core-scheme-macrology
			     'core-scheme))
	     (r^3-s-env (bind-later core-s-env))
	     (a-list `((and    ,expand-and)
		       (case   ,expand-case)
		       (cond   ,expand-cond)
		       (delay  ,expand-delay)
		       (do     ,expand-do)
		       (if     ,expand-if)
		       (lambda ,expand-lambda)
		       (letrec ,expand-letrec)
		       (let    ,expand-let)
		       (let*   ,expand-let*)
		       (or     ,expand-or)
		       (,'quasiquote	;Yikes!
			,expand-quasiquote)
		       ))
	     (keywords (map car a-list))
	     (expanders (map (lambda (z)
			       (make-expander ((cadr z) core-s-env) r^3-s-env))
			     a-list)))
	(bind-now! keywords expanders r^3-s-env)
	(bind keywords expanders
	      (add-macrology s-env core-scheme-macrology 'core-scheme))))))

(define (closer s-env)			;rhymes with "bulldozer"
  (lambda (exp)
    (make-syntactic-closure s-env '() exp)))

; The expanders:

(define (expand-lambda core-s-env)
  (letrec ((flatten-bvl
	    (lambda (bvl)
	      (if (pair? bvl)
		  (cons (car bvl) (flatten-bvl (cdr bvl)))
		  (if (null? bvl)
		      bvl
		      (list bvl))))))
    (lambda (s-env bvl . body)
      (make-syntactic-closure core-s-env '()
	`(lambda ,bvl ,(process-body s-env (flatten-bvl bvl) body))))))

(define (expand-letrec core-s-env)
  (lambda (s-env specs . body)
    (let ((names (map car specs)))
      (make-syntactic-closure core-s-env '()
	`(letrec ,(map (lambda (spec)
			 `(,(car spec) ,(make-syntactic-closure s-env names
					  (cadr spec))))
		       specs)
	   ,(process-body s-env names body))))))

; Deal with internal defines (ugh)
; Returns a single expression

(define (process-body s-env initial-names forms)
  (let loop ((f forms)
	     (names initial-names)
	     (val-exps '()))
    (cond ((null? f)
	   (error "null body" forms))
	  ((definition-form? (car f) s-env names)
	   (loop (cdr f)
		 (cons (definition-form-lhs (car f)) names)
		 (cons (definition-form-rhs (car f)) val-exps)))
	  ((not (eq? names initial-names))
	   `(letrec ,(reverse (map (lambda (name val-exp)
				     `(,name ,(make-syntactic-closure s-env names
								      val-exp)))
				   names
				   val-exps))
	      ,@(map (lambda (exp) (make-syntactic-closure s-env names exp))
		     f)))
	  ((null? (cdr f))		;Peephole
	   (make-syntactic-closure s-env names (car f)))
	  (else
	   ;; No definitions
	   `(begin ,@(map (lambda (exp)
			    (make-syntactic-closure s-env names exp))
			  f))))))

(define (expand-if core-s-env)
  (lambda (s-env test con . maybe-alt)
    (let ((close (closer s-env)))
      (make-syntactic-closure core-s-env '()
	`(if ,(close test)
	     ,(close con)
	     ,(if (null? maybe-alt)
		  `unspecified
		  (close (car maybe-alt))))))))

(define (expand-and core-s-env)
  core-s-env ;ignored
  (lambda (s-env . conjuncts)
    (let ((close (closer s-env)))
      (let recur ((conjuncts conjuncts))
	(if (null? conjuncts)
	    `#t
	    (let ((first (close (car conjuncts))))
	      (if (null? (cdr conjuncts))
		  first
		  `(and-aux ,first
			    (lambda () ,(recur (cdr conjuncts)))))))))))

(define (expand-or core-s-env)
  core-s-env ;ignored
 (lambda (s-env . disjuncts)
  (let ((close (closer s-env)))
    (let recur ((disjuncts disjuncts))
	 (if (null? disjuncts)
	     `,nil			;avoid a literal #f for s48 boot
	     (let ((first (close (car disjuncts))))
	       (if (null? (cdr disjuncts))
		   first
		   `(or-aux ,first
			    (lambda () ,(recur (cdr disjuncts)))))))))))

; (case key ((a b) x) ((c) y) (else z))
;  ==>  (case-aux key
;		  '((a b) (c))
;		  (lambda () z)
;		  (lambda () x)
;		  (lambda () y))

(define (expand-case core-s-env)
  core-s-env ;ignored
 (lambda (s-env key . clauses)
  (let* ((close (closer s-env))
	 (form-result
	     (lambda (else-thunk thunks key-lists)
	       `(case-aux ,(close key)
			  (quote ,(reverse key-lists))
			  ,else-thunk
			  ,@(reverse thunks)))))
       (let loop ((c clauses) (thunks '()) (key-lists '()))
	 (if (null? c)
	     (form-result `(lambda () unspecified)
			  thunks key-lists)
	     (let* ((clause (car c))
		    (key-list (car clause))
		    (body (map close (cdr clause))))
	       (if (eq? key-list 'else)
		   (form-result `(lambda () ,@body) thunks key-lists)
		   (loop (cdr c)
			 (cons `(lambda () ,@body) thunks)
			 (cons key-list key-lists)))))))))

(define (expand-cond core-s-env)
  core-s-env ;ignored
 (lambda (s-env . clauses)
  (let ((close (closer s-env)))
       (let recur ((clauses clauses))
	 (if (null? clauses)
	     `unspecified
	     (process-cond-clause close
				  (car clauses)
				  (recur (cdr clauses))))))))

; Auxiliary also used by DO

(define (process-cond-clause close clause rest-closed)
  (cond ((null? (cdr clause))
	 `(or-aux ,(close (car clause))
		  (lambda () ,rest-closed)))
	((eq? (car clause) 'else)
	 `(begin ,@(map close (cdr clause))))
	((eq? (cadr clause) '=>)
	 `(=>-aux ,(close (car clause))
		  (lambda () ,(close (caddr clause)))
		  (lambda () ,rest-closed)))
	(else
	 `(if ,(close (car clause))
	      (begin ,@(map close (cdr clause)))
	      ,rest-closed))))

(define (expand-delay core-s-env)
  core-s-env ;ignored
 (lambda (s-env thing)
  (let ((close (closer s-env)))
    `(make-promise (lambda () ,(close thing))))))

; This will lose if one of the DO variables is named BEGIN or LOOP!
; Chris Hanson has proposed a solution, but it's not yet implemented
; here.

(define (expand-do core-s-env)
  core-s-env ;ignored
  (lambda (s-env specs end . body)
    (let ((close (closer s-env))
	  (names (map car specs)))
      `(letrec ((loop
		  (lambda ,names
		    ,(let ((close
			    (lambda (exp)
			      (make-syntactic-closure s-env names exp))))
		       (process-cond-clause
			  close
			  end
			  `(begin ,@(map close body)
				  (loop ,@(map (lambda (y)
						 (close
						  (if (null? (cddr y))
						      (car y)
						      (caddr y))))
					       specs))))))))
	(loop ,@(map (lambda (spec)
		       (close (cadr spec)))
		     specs))))))

(define (expand-let core-s-env)
  core-s-env ;ignored
 (lambda (s-env specs . body)
  (cond ((symbol? specs)
	    (process-iterate s-env specs (car body) (cdr body)))
	   (else
	    (process-let s-env specs body)))))

(define (process-let s-env specs body)
  (let ((close (closer s-env))
	(names (map car specs)))
    `((lambda ,names ,(process-body s-env names body))
      ,@(map (lambda (spec) (close (cadr spec)))
	     specs))))

(define (process-iterate s-env tag specs body)
  (let ((close (closer s-env))
	(names (map car specs)))
    `(letrec ((,tag (lambda ,names
		      ,(process-body s-env (cons tag names) body))))
       (,(make-syntactic-closure s-env (list tag) tag)
	,@(map (lambda (spec) (close (cadr spec)))
	       specs)))))

(define (expand-let* core-s-env)
  core-s-env ;ignored
 (lambda (s-env specs . body)
   (let recur ((specs specs) (names '()))
     (if (null? specs)
	 (process-body s-env names body)
	 (let ((name (car (car specs)))
	       (val-exp (cadr (car specs))))
	   `(let ((,name ,(make-syntactic-closure s-env names val-exp)))
	      ,(recur (cdr specs) (cons name names))))))))

;;;; Quasiquote

(define (expand-quasiquote core-s-env)
  core-s-env ;ignored
 (lambda (s-env x)
  (qq-descend x 1 (closer s-env))))

(define (qq-descend x level close)
  (cond ((vector? x)
	 (qq-descend-vector x level close))
	((not (pair? x))
	 (make-quotation x))
	((qq-interesting? x 'quasiquote)
	 (qq-descend-pair x (+ level 1) close))
	((qq-interesting? x 'unquote)
	 (if (= level 1)
	     (close (cadr x))
	     (qq-descend-pair x (- level 1) close)))
	((qq-interesting? x 'unquote-splicing)
	 (if (= level 1)
	     (error ",@ in illegal position" x)
	     (qq-descend-pair x (- level 1) close)))
        (else
	 (qq-descend-pair x level close))))

(define (qq-descend-pair x level close)
  (let ((d-exp (qq-descend (cdr x) level close)))
    (if (and (qq-interesting? (car x) 'unquote-splicing)
	     (= level 1))
	(let ((sc (close (cadr (car x)))))
	  (cond ((and (quotation? d-exp)
		      (null? (quotation-value d-exp)))
		 sc)
		(else
		 `(append ,sc ,d-exp))))
	(let ((a-exp (qq-descend (car x) level close)))
	  (cond ((and (quotation? a-exp)
		      (quotation? d-exp))
		 (make-quotation x))
		((and (quotation? d-exp)
		      (eq? (quotation-value d-exp) '()))
		 `(list ,a-exp))
		((qq-interesting? d-exp 'list)
		 `(list ,a-exp ,@(cdr d-exp)))
		;;+++ Ought to use auxiliary CONS* procedure, for more
		;; readable output
		(else
		 `(cons ,a-exp ,d-exp)))))))

(define (qq-descend-vector x level close)
  (let ((result (qq-descend (vector->list x) level close)))
    (if (quotation? result)
	(make-quotation x)
	`(list->vector ,result))))

(define (qq-interesting? x marker)
  (and (pair? x)
       (eq? (car x) marker)
       (pair? (cdr x))
       (null? (cddr x))))

(define (quotation? x)
  (qq-interesting? x 'quote))

(define quotation-value cadr)

(define (make-quotation value)
  `(quote ,value))
