#| -*-Scheme-*-

$Header: /scheme/src/edwin/RCS/bochser.scm,v 1.1 1991/11/26 20:09:26 markf Exp $

Copyright (c) 1991 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Bochser
;;; package (bochser)

(declare (usual-integrations))

(define (initialize-bochser!)
  (let ((old-syntax-expression hook/syntax-expression))
    (set! hook/syntax-expression
      (lambda (expression syntax-table)
	(let ((syntax-result
	       (old-syntax-expression expression syntax-table)))
	  (if (bochser-record? syntax-result)
	      (bochser-record-result! syntax-result expression syntax-table))
	  syntax-result))))
  (set! *bochs-object-table* (make-1d-table))
  (set! bochser-syntax-table-table (make-1d-table))
  (set! bochser-source-code-table (make-1d-table)))

(define-structure (bochs
		   (conc-name BOCHS/)
		   (constructor %make-bochs (object)))
  object				;the scheme object
  name					;name of bochs
; lookup-number				;index in *bochs-number-table*
  syntax-table				;for procedure definition
  (open-edit-object false)		;the editor representation of
					;open bochs.
  (closed-edit-object false)		;the editor representation of
					;closed bochs.
  (open? false)				;is the bochs open?
  sub-bochses '())			;(e.g. code and env bochses of a
					;procedure bochs)

(define (make-bochs object)
  (let ((bochs (%make-bochs object)))
;    (add-to-bochs-number-table! bochs)
    bochs))

(define (bochs/open bochs)
  (set-bochs/open?! bochs true)
  (editor-open-bochs bochs))

(define (bochs/close bochs)
  (set-bochs/open?! bochs false)
  (editor-close-bochs bochs))

(define (bochs/environment bochs)
  (let ((object (bochs/object bochs)))
    (cond ((procedure? object) (procedure-environment object))
	  ((environment? object) object)
	  (else (nearest-repl/environment)))))

(define (proc-bochs/environment-bochs proc-bochs)
  (car (bochs/sub-bochses proc-bochs)))

(define (proc-bochs/code-bochs proc-bochs)
  (cadr (bochs/sub-bochses proc-bochs)))

(define bochs/lookup-number hash)

#|
(define initial-bochs-number-table-size 50)

(define *bochs-number-table* (make-vector initial-bochs-number-table-size))

(define *bochs-number-table-size* initial-bochs-number-table-size)

(define *bochs-counter* 0)

(define (add-to-bochs-number-table! bochs)
  (if (>= *bochs-counter* *bochs-number-table-size*)
      (let* ((new-table-size (* 2 *bochs-number-table-size*))
	     (new-table (vector-grow *bochs-number-table* new-table-size)))
	(set! *bochs-number-table* new-table)
	(set! *bochs-number-table-size* new-table-size)))
  (vector-set! *bochs-number-table* *bochs-counter* bochs)
  (set-bochs/lookup-number! bochs *bochs-counter*)
  (set! *bochs-counter* (1+ *bochs-counter*)))
	
(define (bochs/lookup number)
  (vector-ref *bochs-number-table* number))
|#
	
(define bochs/lookup unhash)

(define (find-or-create-bochs object)
  (or (find-bochs object)
      (create-bochs object)))

(define (create-bochs object . sub-bochses)
  (let ((bochs
	 (cond
	  ((procedure? object)
	   (create-procedure-bochs object sub-bochses))
	  ((environment? object)
	   (create-environment-bochs object))
	  ((lambda? object)
	   (create-code-bochs object))
	  (else #f))))
    (create-bochs-association! bochs object)
    bochs))

(define (bochsable-object? object)
  (or (procedure? object)
      (and (environment? object) (not (null? object)))
      (lambda? object)))

(define *bochs-object-table*)

(define (find-bochs object)
  (car (1d-table/get *bochs-object-table* object '(#F))))

(define (find-bochses object)
  (1d-table/get *bochs-object-table* object '()))

(define (create-bochs-association! bochs object)
  (1d-table/put! *bochs-object-table*
		 object
		 (cons bochs
		       (1d-table/get *bochs-object-table* object '()))))

(define (create-procedure-bochs procedure env-and-code-bochses)
  (let ((bochs (make-bochs procedure)))
    (set-bochs/syntax-table! bochs (lambda-syntax-table
				    (procedure-lambda procedure)))
    (set-bochs/name! bochs
		     (symbol->string
		      (lambda-name (procedure-lambda procedure))))
    (set-bochs/sub-bochses! bochs env-and-code-bochses)
    (set-bochs/syntax-table! (proc-bochs/environment-bochs bochs)
			     (bochs/syntax-table bochs))
    bochs))

(define (create-environment-bochs environment)
  (let ((bochs (make-bochs environment)))
    (set-bochs/syntax-table! bochs (lambda-syntax-table
				    (environment-lambda environment)))
    (set-bochs/name! bochs
		     (write-to-string (environment-name environment)))
    bochs))

(define (create-code-bochs *lambda)
  (let ((bochs (make-bochs *lambda)))
    (set-bochs/syntax-table! bochs (lambda-syntax-table *lambda))
    (set-bochs/name! bochs (symbol->string (lambda-name *lambda)))
    bochs))

(define bochser-syntax-table-table)

(define (bochser-syntax-table-lookup value)
  (1d-table/get bochser-syntax-table-table value #f))

(define (lambda-syntax-table *lambda)
  (1d-table/get bochser-syntax-table-table *lambda (default-syntax-table)))

(define (bochser-record? object)
  (or (lambda? object)
      (and (definition? object)
	   (lambda? (definition-value object)))))
	
(define (bochser-record-result! value expression syntax-table)
  (let ((value (if (definition? value)
		   (definition-value value)
		   value)))
    (if (not (bochser-syntax-table-lookup value))
	(begin
	  (record-syntax-table! value syntax-table)
	  (record-source-code! value expression)))))

(define (record-syntax-table! value syntax-table)
  (1d-table/put! bochser-syntax-table-table
		 value
		 (or syntax-table (default-syntax-table))))

(define bochser-source-code-table)

(define (lambda-source-code *lambda)
  (1d-table/get bochser-source-code-table *lambda (unsyntax *lambda)))

(define (record-source-code! value code)
  (1d-table/put! bochser-source-code-table value code))

(define (environment-name environment)
  (if environment
      (let ((package (environment->package environment)))
	(if package
	    (package/name package)
	    (environment-procedure-name environment)))
      'GLOBAL))


#| 

;;; This will be for when/if we use regions between marks to delimit
;;; bochses AND if we don't have fields in edwin.

;;;; Un-deletable marks

(define (buffer-undeletable-marks buffer)
  (buffer-get buffer 'UNDELETABLE-MARKS))

(define (mark-undeletable! mark)
  (add-buffer-undeletable-mark! (mark-buffer mark) mark))

(define (add-buffer-undeletable-mark! buffer mark)
  (without-interrupts
   (lambda ()
     (buffer-put! buffer
		  'UNDELETABLE-MARKS
		  (cons mark (buffer-undeletable-marks buffer)))
     (if (not (memq detect-undeletable-marks-daemon
		    (group-delete-daemons (mark-group mark))))
	 (add-group-delete-daemon! (mark-group mark)
				   detect-undeletable-marks-daemon)))))

(define (detect-undeletable-marks-daemon group start end)
  (let loop ((undeletable-marks
	      (buffer-undeletable-marks (group-buffer group))))
    (and (pair? undeletable-marks)
	 (let ((undeletable-mark (car undeletable-marks)))
	   (with-simple-restart 'UNDELETABLE-MARK-CONTINUE "continue?"
	     (lambda ()
	       (if (and (mark>= undeletable-mark start)
			(mark<= undeletable-mark end))
		   (signal-undeletable-mark mark))
	   (loop (cdr undeletable-marks))))))))

(define condition-type:undeletable-mark-error
  (make-condition-type 'UNDELETABLE-MARK-ERROR
		       condition-type:editor-error
		       '(MARK)
    (lambda (condition port)
      (write-string "Attempt to delete undeletable mark:" port)
      (write mark port))))

(define signal-undeletable-mark
  (condition-signaller condition-type:undeletable-mark-error
		       '(MARK)
		       standard-error-handler))

(define (with-temporarily-deletable-mark mark thunk)
  (bind-condition-handler (list condition-type:undeletable-mark-error)
			  (lambda (condition)
			    (let ((undeletable-mark-continue
				   (find-restart 'UNDELETABLE-MARK-CONTINUE)))
			      (invoke-restart undeletable-mark-continue)))
			  (thunk)))

|#