;;; File to build Scheme 7.1 binaries from sources

;;; cd to the "src" directory of the 7.1 release, start scheme with
;;; the "-compiler" option, and load this file.

;;; Compile the runtime system
(with-working-directory-pathname "runtime"
  (lambda ()
    (fluid-let ((sf/default-syntax-table syntax-table/system-internal))
      (sf-directory "."))
    (compile-directory ".")))

;;; Compile and load the CREF subsystem
(with-working-directory-pathname "cref"
  (lambda ()
    (fluid-let ((sf/default-syntax-table system-global-syntax-table))
      (sf-conditionally "object")
      (sf-directory ".")
      (if (not (file-exists? "cref.bcon"))
          (sf "triv.con" "cref.bcon"))
      (if (not (file-exists? "cref.bldr"))
          (sf "triv.ldr" "cref.bldr")))
    (compile-directory ".")
    (if (not (name->package '(CROSS-REFERENCE)))
        (load "make"))))

;;; Generate CREF files for runtime system
(with-working-directory-pathname "runtime"
  (lambda ()
    (if (not (and (file-exists? "runtim.glob")
                  (file-exists? "runtim.con")
                  (file-exists? "runtim.ldr")))
        (begin
          (cref/generate-constructors "runtim")
          (sf "runtim.con" "runtim.bcon")
          (sf "runtim.ldr" "runtim.bldr")))))

;;; Generate CREF files for CREF subsystem
(with-working-directory-pathname "cref"
  (lambda ()
    (if (not (and (file-exists? "cref.glob")
                  (file-exists? "cref.con")
                  (file-exists? "cref.ldr")))
        (begin
          (cref/generate-constructors "cref")
          (sf "cref.con" "cref.bcon")
          (sf "cref.ldr" "cref.bldr")))))

;;; Compile (and generate CREF files for) SF subsystem
(with-working-directory-pathname "sf"
  (lambda ()
    (load "sf.sf")
    (load "sf.cbf")))

;;; Compile (and generate CREF files for) compiler
(let ((types
       '((MC68030 "also 68020")
         (VAX #f)
         (HP-PA "HP Precision Architecture")
         (PMAX "DECStation 3100 or 5100")
         (OTHER "no compiled-code support"))))
  (let loop ()
    (newline)
    (write-string "Enter one of the following machine types:")
    (newline)
    (write-string "Type\t\tNotes")
    (newline)
    (write-string "----\t\t-----")
    (for-each (lambda (type)
                (newline)
                (write (car type))
                (if (cadr type)
                    (begin
                      (write-string "\t\t")
                      (write-string (cadr type)))))
              types)
    (newline)
    (write-string "Enter machine type: ")
    (let ((type (read)))
      (cond ((not (assq type types))
             (loop))
            ((not (eq? type 'OTHER))
             (with-working-directory-pathname "compiler"
               (lambda ()
                 (let ((directory
                        (case type
                          ((MC68030) "bobcat")
                          ((VAX) "vax")
                          ((HP-PA) "spectrum")
                          ((PMAX) "mips")))
                       (file-link-soft
                        (make-primitive-procedure 'FILE-LINK-SOFT)))
                   (let ((prefix (string-append "machines/" directory)))
                     (file-link-soft prefix directory)
                     (file-link-soft (string-append prefix "/comp.cbf") "comp.cbf")
                     (file-link-soft (string-append prefix "/comp.pkg") "comp.pkg")
                     (file-link-soft (string-append prefix "/comp.sf") "comp.sf")))
                 (load "comp.sf")
                 (load "comp.cbf"))))))))

;;; Compile (and generate CREF files for) editor
(with-working-directory-pathname "edwin"
  (lambda ()
    (load "edwin.sf")
    (load "edwin.cbf")))

;;; To make the band "runtime.com":
;;;
;;; cd runtime
;;; scheme -large -fasl make.com
;;; (disk-save "runtime.com")

;;; To make the band "compiler.com" (subsitute appropriate machine
;;; directory for "machines/mips" in the line below):
;;;
;;; scheme -large
;;; (cd "sf")
;;; (load "make")
;;; (cd "../compiler")
;;; (load "machines/mips/make")
;;; (disk-save "compiler.com")

;;; To make the band "edwin.com":
;;;
;;; scheme -large
;;; (cd "edwin")
;;; (load "make")
;;; (disk-save "edwin.com")
