/* macros, scope binding */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include "refine.h"
#include "utils.h"

#ifndef lint
static char SccsId[] = "@(#)refine.c	1.10 2/23/90";
#endif

/*
  Macros are implemented as symbols whose MACRO status bits are set.
  As a result, macros are defined in a global namespace.  They do not
  follow the lexical scoping rules for bindings and override normal
  symbol binding when macro expansion is performed before evaluation.

  A lisp procedure object, held in the symbol's macro field, is applied
  to arguments in an expression of the form (macro-symbol arg1 ...) and
  the result is the macro expansion.

  Two types of expansion are provided.  The first performs at most one
  expansion and is not recursive and should be called on an expression
  before evaluating it.  Special forms are not a problem with this method.

  The second expands all macros (including macros derived from
  expanding macros) and only needs to be called once, except in special
  cases.  First the operator is expanded.  If the operator is a macro
  then the entire expression is macro-expanded.  Then for each argument,
  the process is repeated.  Special forms may need to specify that their
  arguments are not to be expanded with NOEXPAND.  These forms must
  macro-expand their arguments themselves, however.
*/

static Obj scopeLambda();
static Obj _expandQuasi();

/* wrapper that handles vector quasiquotes as well */
static Obj expandQuasi(qq, depth)
     Obj qq;
     int depth;
{
    if (objIsClass(qq, Vector))
	return objOp2(ApplySymb, VectorSymb,
		      _expandQuasi(vectorToList(qq), depth));
    return _expandQuasi(qq, depth);
}
	       
/* Expand the qq of (quasiquote qq) into a form that when evaluated produces
 * the expected result. */
static Obj _expandQuasi(qq, depth)
     Obj qq;
     int depth;
{
    Obj op, splice, done;
    List stack;
    Callback_t cb;

    /* quote atoms */
    if (CLASS(qq) != Pair) return objOp1(QuoteSymb, qq);

    op = objCar(qq);

    /* splices in this context are illegal */
    if (op == SpliceSymb)
	errorPrint(BadSplice, "%O", qq);

    /* unquotations */
    else if (op == UnquoteSymb) {
	if ((done = objGetArg(qq, 1)) == (Obj)NULL)
	    errorPrint(BadArgs, "to unquote");

	if (depth == 0) return done;
	else return objOp2(ListSymb,
			   objOp1(QuoteSymb, UnquoteSymb),
			   expandQuasi(done, depth - 1));
    }

    /* nested quasiquotes */
    else if (op == QuasiSymb) {
	if ((done = objGetArg(qq, 1)) == (Obj)NULL)
	    errorPrint(BadArgs, "to quasiquote");
	return objOp2(ListSymb,
		      objOp1(QuoteSymb, QuasiSymb),
		      expandQuasi(done, depth + 1));
    }

    /* else more complicated stuff */

    stack = gcListNew(&cb); /* stack of splices to append */
    splice = NilSymb; /* current splice */

    for (; CLASS(qq) == Pair; qq = objCdr(qq)) {
	Obj sub;

	op = objCar(qq);
	if (objGetArg(op, 0) == SpliceSymb) {
	    if ((sub = objGetArg(op, 1)) == (Obj)NULL)
		errorPrint(BadArgs, "to unquote-splicing");

	    if (depth == 0) {
		if (splice != NilSymb) {
		    listPush((Ptr)splice, stack);
		    splice = NilSymb;
		}
		listPush((Ptr)sub, stack);
		continue ;
	    }
	    else sub = objOp2(ListSymb,
			      objOp1(QuoteSymb, SpliceSymb),
			      expandQuasi(sub, depth - 1));
	}
	else if (op == SpliceSymb) errorPrint(BadSplice, "%O", qq);
	else if (op == UnquoteSymb) {
	    /* check if qq is (unquote arg) */
	    qq = objCdr(qq);
	    if (CLASS(qq) != Pair || objCdr(qq) != NilSymb)
		errorPrint(BadArgs, "to unquote");
	    sub = objCar(qq);
	    if (depth == 0) {
		if (splice != NilSymb) {
		    listPush((Ptr)splice, stack);
		    splice = NilSymb;
		}
		listPush((Ptr)sub, stack);
		continue ;
	    }
	    else sub = objOp2(ListSymb,
			      objOp1(QuoteSymb, UnquoteSymb),
			      expandQuasi(sub, depth - 1));
	}
	else sub = expandQuasi(op, depth);

	if (splice == NilSymb)
	    splice = objOp1(ListSymb, sub);
	else splice = objAppendObj(splice, sub);
    }

    /* add remaining stuff */
    if (splice != NilSymb) listPush((Ptr)splice, stack);
    if (qq != NilSymb) listPush((Ptr)expandQuasi(qq, depth), stack);

    done = NilSymb;
    while (splice = (Obj)listPop(stack)) {
	if (done == NilSymb)
	    done = splice;
	else {
	    if (CLASS(splice) == Pair && objCar(splice) == ListSymb)
		/* use list* to do the append */
		done = objAppendObj(newPair(gcTemp,
					    ListStarSymb, objCdr(splice)),
				    done);
	    else done = objOp2(AppendSymb, splice, done);
	}
    }
    gcListFree(stack);
    
    return done;
}

/* Expand macros.  A given expression is macro expanded (and re-expanded)
 * while the procedure subexpression is a macro symbol (if the subexpression
 * is a compound expression, macroExpand is recursively applied), unless
 * the macro symbol had been previously expanded during macro expansion
 * at a particular depth. */
Obj macroExpand(expr)
     Obj expr;
{
    Obj temp, nexpr, ptr;
    Boolean expandargs = TRUE;
    List expanded;
    Callback_t cb;

    /* expand operator */

    expanded = gcListNew(&cb);
    for (;;) {
	if (CLASS(expr) != Pair) {
	    expandargs = FALSE;
	    break ;
	}

	temp = objCar(expr);
	if (CLASS(temp) == Pair) {
	    temp = macroExpand(temp);
	    expr = newPair(gcTemp, temp, objCdr(expr));
	}

	/* make quasiquote into a lisp macro? */
	if (temp == QuasiSymb) {
	    if ((CLASS(temp = objCdr(expr)) != Pair)
		|| (objCdr(temp) != NilSymb))
		errorPrint(BadArgs, "to quasiquote (expects 1)");
	    expr = expandQuasi(objCar(temp), 0);
	}
	else if (CLASS(temp) == Symbol) {
	    if (checkCond(temp, MACRO)) {
		if (listFind(expanded, (F_BOOLEAN)NULL, (Ptr)temp))
		    break ; /* short circuit */
		listPush((Ptr)temp, expanded);
		temp = objMacro(temp);
		ASSERT(objIsClass(temp, Proc));

		expr = objApply(temp, objCdr(expr));
	    }
	    else if (checkCond(temp, SFORM)
		     && checkCond(objSForm(temp), NOEXPAND)) {
		expandargs = FALSE;
		break ;
	    }
	    else break ;
	}
	else break ;
    }
    gcListFree(expanded);
    if (!expandargs) return expr;

    /* expand args */

    ptr = nexpr = newPair(gcTemp, objCar(expr), NilSymb);
    expr = objCdr(expr);
    while (CLASS(expr) == Pair) {
	temp = newPair(gcTemp, macroExpand(objCar(expr)), NilSymb);
	expr = objCdr(expr);
	objSetCdr(ptr, temp);
	ptr = temp;
    }
    if (expr != NilSymb) objSetCdr(ptr, expr);

    return nexpr;
}

/* Replace all top-level symbols of expr with bindings from frame. */
Obj scopeBind(expr, frame)
     Obj expr, frame;
{
    Obj temp;

    if (CLASS(expr) == Package && frame)
	return scopeBind(DATA(expr, sym, packageInst),
			 objCdr(DATA(expr, package, packageInst)));
    else if (CLASS(expr) == Symbol) {
	Obj flink; /* follows frame parent links */
	int links; /* number of links followed */
	List nonstable;	/* list of instantiated non-stable frames */

	nonstable = listNew();
	flink = frame;
	links = 0;
	while (flink && (temp = objGetBinding(expr, flink)) == (Obj)NULL) {
	    if (!checkCond(flink, STABLE)) listPush((Ptr)flink, nonstable);
	    flink = DATA(flink, parent, frameInst);
	    links++;
	}

	/* create fixed binding if defined */
	if (temp != (Obj)NULL && CLASS(temp) == FBinding) {
	    if (links != 0)
		/* reset frame links if not zero */
		temp = newFBinding(gcTemp, expr,
				   links, DATA(temp, offset, fbindInst));
	}

	/* make a chain of indirect bindings through non-stable frames */
	else {
	    while (flink = (Obj)listPop(nonstable))
		temp = objPut(expr, temp, flink);
	}
	listFree(nonstable, (F_VOID)NULL);
	return temp;
    }
    
    else if (CLASS(expr) == Pair) {
	Obj nexpr, oexpr, ptr;

	/* scope operator */

	oexpr = expr;
	temp = objCar(expr);

	/* new scope */
	if (temp == LambdaSymb) {
	    /* should check whether the lambda's frame is stable and
	     * scope bind at execution time */
	    return scopeLambda(expr, frame);
	}
	/* define within non-stable frame */
	else if (temp == DefineSymb && !checkCond(frame, STABLE)) {
	    Obj binding, dframe = frame;

	    ptr = objGetArg(expr, 1); /* symbol */
	    nexpr = objGetArg(expr, 2); /* value */

	    if (ptr == (Obj)NULL || nexpr == (Obj)NULL || objGetArg(expr, 3))
		errorPrint(BadArgs, "to special form define (expects 2)");
	    if (CLASS(ptr) == Package) {
		dframe = objCdr(DATA(ptr, package, packageInst));
		ptr = DATA(ptr, sym, packageInst);
	    }
	    if (CLASS(ptr) != Symbol)
		errorPrint(BadClass, "%O is not a symbol", ptr);

	    /* make new binding only if necessary */
	    if ((binding = objGetBinding(ptr, dframe)) == (Obj)NULL)
		binding = objPut(ptr, (Obj)NULL, dframe);

	    /* scope bind in frame not package of symbol */
	    return objOp2(objSForm(DefineSymb), binding,
			  scopeBind(nexpr, frame));
	}

	/* replace symbol for special form with the form evaluator */
	else if (CLASS(temp) == Symbol && checkCond(temp, SFORM)) {
	    temp = objSForm(temp);
	    if (checkCond(temp, NOEXPAND)) {
		/* handle special cases for NOEXPAND forms here */
		return newPair(gcTemp, temp, objCdr(expr));
	    }
	}

	/* else recursively scope bind operator */
	else temp = scopeBind(temp, frame);

	/* scope args */
	ptr = nexpr = newPair(gcTemp, temp, NilSymb);
	expr = objCdr(expr);
	while (CLASS(expr) == Pair) {
	    temp = newPair(gcTemp, scopeBind(objCar(expr), frame), NilSymb);
	    expr = objCdr(expr);
	    objSetCdr(ptr, temp);
	    ptr = temp;
	}
	if (expr != NilSymb) errorPrint(BadSyntax, "in %O", oexpr);
	return nexpr;
    }

    return expr;
}

/* Check if lambda body is stable (i.e., no evals or the-environments) */
static Boolean scopeIsStable(expr)
     Obj expr;
{
    Obj subexpr;

    if (CLASS(expr) == Pair) {
	subexpr = objCar(expr);
	if (subexpr == LambdaSymb || subexpr == QuoteSymb)
	    return TRUE;
	else if (subexpr == TheEnvSymb)
	    return FALSE;

	if (CLASS(subexpr) == Pair && !scopeIsStable(subexpr))
 	    return FALSE;
	for (subexpr = objCdr(expr); CLASS(subexpr) == Pair;
	     subexpr = objCdr(subexpr))
	    if (!scopeIsStable(objCar(subexpr)))
		return FALSE;
    }
    return TRUE;
}

/* Find top-level definitions in expr and add to symlist. */
static void scopeNumDefs(expr, symlist)
     Obj expr;
     List symlist; /* list of definition symbols */
{
    Obj op, temp;

    if (CLASS(expr) == Pair) {
	op = objCar(expr);

	/* define */
	if (op == DefineSymb) {
	    if ((op = objGetArg(expr, 1)) == (Obj)NULL ||
		(temp = objGetArg(expr, 2)) == (Obj)NULL) {
		listFree(symlist, (F_VOID)NULL);
		errorPrint(BadSyntax, "in lambda %O", expr);
	    }
	    if (CLASS(op) != Symbol) {
		listFree(symlist, (F_VOID)NULL);
		errorPrint(BadClass, "%O is not a symbol", op);
	    }
	    if (listFind(symlist, (F_BOOLEAN)NULL, (Ptr)op) == (Ptr)NULL)
		/* ignore any redefinitions */
		listPush((Ptr)op, symlist);
	    scopeNumDefs(temp, symlist);
	    return ;
	}
	else if (CLASS(op) == Symbol && checkCond(op, SFORM)
		 && checkCond(objSForm(op), NOEXPAND))
	    /* Don't examine special forms whose NOEXPAND is set.
	     * If they don't follow normal syntax and can create
	     * local definitions, they are handled specially here */
	    return ;

	if (CLASS(op) == Pair)
	    scopeNumDefs(op, symlist);
	for (op = objCdr(expr); CLASS(op) == Pair; op = objCdr(op))
	    scopeNumDefs(objCar(op), symlist);
    }
}

/* Put the formals into symlist and return TRUE if formals contains
 * an optional rest args parameter.  An error will occur if any of
 * the formals are duplicated or are not symbols. */
static Boolean scopeFormals(formals, symlist)
     Obj formals;
     List symlist;
{
    Obj temp;
    Boolean optarg;

    optarg = FALSE;
    for (temp = formals; ; temp = objCdr(temp)) {
	Obj sym;

	if (CLASS(temp) == Pair) sym = objCar(temp);
	else {
	    if ((sym = temp) == NilSymb) return FALSE;
	    optarg = TRUE;
	}

	if (CLASS(sym) != Symbol) {
	    listFree(symlist, (F_VOID)NULL);
	    errorPrint(BadClass, "formal %O is not a symbol", sym);
	}
	else if (listFind(symlist, (F_BOOLEAN)NULL, (Ptr)sym)) {
	    listFree(symlist, (F_VOID)NULL);
	    errorPrint(BadFormals, "redefinition of %O", sym);
	}
	listPush((Ptr)sym, symlist);

	if (optarg) break ;
    }
    return optarg;
}

/* Scope bind the lambda expression expr.  The body of expr is converted
 * into a sequence if the body contains more than one expression.  Free
 * symbols are replaced by their lexical bindings and local symbols are
 * replaced by the offsets into the stack frame. */
static Obj scopeLambda(expr, cf)
     Obj expr, cf;
{
    int args, numfixed;
    Obj fp, frame, body, temp;
    Boolean stable, optarg;
    List symlist;

    /* extract fp and body */
    if (CLASS(fp = objCdr(expr)) != Pair)
	errorPrint(BadSyntax, "in %O", expr);
    body = objCdr(fp);
    fp = objCar(fp);

    /* convert body into a sequence, if necessary */
    if (CLASS(body) == Pair) {
	if (objCdr(body) == NilSymb)
	    body = objCar(body);
	else body = newPair(gcTemp, SeqSymb, body);
    }
    else if (body != NilSymb)
	errorPrint(BadSyntax, "in lambda");

    body = macroExpand(body);

    symlist = listNew();

    optarg = scopeFormals(fp, symlist);
    args = listLength(symlist);
    if (optarg) --args; /* subtract optarg */

    scopeNumDefs(body, symlist);
    numfixed = listLength(symlist);

    fp = NilSymb;
    while (temp = (Obj)listPop(symlist))
	fp = newPair(gcNew, temp, fp);
    listFree(symlist, (F_VOID)NULL);

    stable = scopeIsStable(body);
    frame = newFrame(gcTemp, cf);
    /* since frame is not really instantiated assume it to be stable */
    setCond(frame, STABLE);
    DATA(frame, formals, frameInst) = fp;
    objLink(fp);
    
    body = scopeBind(body, frame);

    temp = newPair(gcNew, body, NilSymb);
    temp = newPair(gcNew, fp, temp);
#ifdef SAVE_LAMBDA_BODIES
    temp = newPair(gcNew, objCdr(expr), temp);
#else
    temp = newPair(gcNew, NilSymb, temp);
#endif /* SAVE_LAMBDA_BODIES */
    temp = newPair(gcNew,
		   newInteger(gcNew,
			      (long)((optarg ? OPTARG : 0)
				     | (stable ? STABLE : 0))),
		   temp);
    temp = newPair(gcNew, newInteger(gcNew, (long)args), temp);
    temp = newPair(gcNew, newInteger(gcNew, (long)numfixed), temp);

    return newPair(gcTemp, objSForm(LambdaSymb), temp);
}
