;*---------------------------------------------------------------------*/
;*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*---------------------------------------------------------------------*/
;*    serrano/prgm/project/bigloo/runtime1.7/Llib/error.scm ...        */
;*                                                                     */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Jun 11 13:18:15 1992                          */
;*    Last change :  Thu Jan 19 11:09:09 1995 (serrano)                */
;*                                                                     */
;*    La gestion `runtime' des erreurs                                 */
;*---------------------------------------------------------------------*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __error
   (foreign (obj c-exit (bint)                         "bigloo_exit")
	    (include "signal.h")
	    (define obj push-lambda-trace (obj)        "PUSH_LAMBDA_TRACE")
	    (define obj pop-lambda-trace  (obj)        "POP_LAMBDA_TRACE")
	    (define obj c-get-lambda-stack  ()         "GET_LAMBDA_STACK")
	    (define obj c-set-lambda-stack! (obj)      "SET_LAMBDA_STACK")
	    (obj        c-dump-lambda-stack (obj long) "dump_lambda_stack") 
	    (define int sigfpe                         "SIGFPE")
	    (define int sigill                         "SIGILL")
	    (define int sigbus                         "SIGBUS")
	    (define int sigsegv                        "SIGSEGV")
	    (export obj the_failure (obj obj obj)      "the_failure"))
   (export  *warning*
	    (inline exit num)
	    (failure proc message object)
	    (the_failure proc message object)
            (add-error-handler! handler escape)
	    (remove-error-handler!)
	    (inline error proc message object)
	    (notify-error proc mes obj)
	    (warning . args)
	    (warning/location fname location . args)
	    (type-error type name)
	    (inline get-lambda-stack)
	    (inline set-lambda-stack! obj)
	    (inline dump-lambda-stack . depth)
	    (error/location proc message object fname location))
   (static  *error-handler*
	    (default-error-handler proc mes obj)
	    (incorrect-error-handler handler))
   (pragma  (type-error _imbricable_ _no_eval_value_)
	    (failure _no_eval_value_)
	    (the_failure _no_eval_value_)))

;*---------------------------------------------------------------------*/
;*    *warning* ...                                                    */
;*---------------------------------------------------------------------*/
(define *warning* #t)

;*---------------------------------------------------------------------*/
;*    get-lambda-stack ...                                             */
;*---------------------------------------------------------------------*/
(define-inline (get-lambda-stack)
   (c-get-lambda-stack))

;*---------------------------------------------------------------------*/
;*    set-lambda-stack! ...                                            */
;*---------------------------------------------------------------------*/
(define-inline (set-lambda-stack! obj)
   (c-set-lambda-stack! obj))

;*---------------------------------------------------------------------*/
;*    exit ...                                                         */
;*---------------------------------------------------------------------*/
(define-inline (exit n)
   (c-exit n))

;*---------------------------------------------------------------------*/
;*    La valeur par defaut du *error-handler*                          */
;*---------------------------------------------------------------------*/
(define *error-handler* '())

;*---------------------------------------------------------------------*/
;*    dump-lambda-stack ...                                            */
;*---------------------------------------------------------------------*/
(define-inline (dump-lambda-stack . depth)
   (let ((depth (if (or (null? depth) (not (integer? (car depth))))
		    10
		    (car depth))))
      (c-dump-lambda-stack (current-error-port) depth)))
					       
;*---------------------------------------------------------------------*/
;*    add-error-handler! ...                                           */
;*---------------------------------------------------------------------*/
(define (add-error-handler! handler escape)
   (set! *error-handler* (cons (cons handler escape) *error-handler*)))

;*---------------------------------------------------------------------*/
;*    remove-error-handler! ...                                        */
;*---------------------------------------------------------------------*/
(define (remove-error-handler!)
   (if (pair? *error-handler*)
       (set! *error-handler* (cdr *error-handler*)))
   *error-handler*)

;*---------------------------------------------------------------------*/
;*    failure ...                                                      */
;*---------------------------------------------------------------------*/
(define (failure proc message object)
   (failure proc message object))

;*---------------------------------------------------------------------*/
;*    error ...                                                        */
;*---------------------------------------------------------------------*/
(define-inline (error proc message object)
   (failure proc message object))
   
;*---------------------------------------------------------------------*/
;*    warning ...                                                      */
;*---------------------------------------------------------------------*/
(define (warning . args)
   (if *warning*
       (begin
	  (flush-output-port (current-output-port))
	  (newline (current-error-port))
	  (display "*** WARNING:bigloo:" (current-error-port))
	  (if (not (null? args))
	      (begin
		 (fprint (current-error-port) (car args))
		 (for-each (lambda (a)
			      (display a (current-error-port)))
			   (cdr args))))
	  (newline (current-error-port)))))

;*---------------------------------------------------------------------*/
;*    warning/location ...                                             */
;*---------------------------------------------------------------------*/
(define (warning/location fname location . args)
   (if *warning*
       (cond
	  ((string=? fname "[string]")
	   (apply warning args))
	  ((string=? fname "[stdin]")
	   (apply warning args))
	  (else
	   (warning/location-file fname location args)))))

;*---------------------------------------------------------------------*/
;*    the_failure ...                                                  */
;*---------------------------------------------------------------------*/
(define (the_failure proc message object)
   (reader-reset!)
   (if (not (pair? *error-handler*))
       (default-error-handler proc message object)
       (let ((handler (car (car *error-handler*)))
	     (escape  (cdr (car *error-handler*))))
	  (remove-error-handler!)
	  (if (and (procedure? handler)
		   (=fx (procedure-arity handler) 4))
	      (handler escape proc message object)
	      (incorrect-error-handler handler)))))

;*---------------------------------------------------------------------*/
;*    notify-error ...                                                 */
;*---------------------------------------------------------------------*/
(define (notify-error proc mes obj)
   (if (procedure? *error-notifier*)
       (*error-notifier* proc mes obj)
       (begin
	  (flush-output-port (current-output-port))
	  (newline (current-error-port))
	  (let ((old-length (get-write-length)))
	     (set-write-length! 80)
	     (display "*** ERROR:bigloo:" (current-error-port))
	     (fprint (current-error-port) proc ":" #\Newline mes " -- " obj)
	     (dump-lambda-stack)
	     (set-write-length! old-length)))))

;*---------------------------------------------------------------------*/
;*    type-error ...                                                   */
;*---------------------------------------------------------------------*/
(define (type-error type name)
   (let ((type-string (symbol->string type)))
      (string-append "type `" type-string
		     "' expected for variable " (symbol->string name))))

;*---------------------------------------------------------------------*/
;*    *error-notifier* ...                                             */
;*---------------------------------------------------------------------*/
(define *error-notifier* (unspecified))

;*---------------------------------------------------------------------*/
;*    default-error-handler ...                                        */
;*---------------------------------------------------------------------*/
(define (default-error-handler proc mes obj)
   (notify-error proc mes obj)
   (c-exit -1)
   -1)

;*---------------------------------------------------------------------*/
;*    incorrect-error-handler ...                                      */
;*---------------------------------------------------------------------*/
(define (incorrect-error-handler handler)
   (default-error-handler "error" "Not an error handler" handler))

;*---------------------------------------------------------------------*/
;*    sigfpe-error-handler ...                                         */
;*---------------------------------------------------------------------*/
(define (sigfpe-error-handler n)
   (error "arithmetic procedure" "`floating point' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    sigill-error-handler ...                                         */
;*---------------------------------------------------------------------*/
(define (sigill-error-handler n)
   (error "bigloo" "`illegal instruction' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    sigbus-error-handler ...                                         */
;*---------------------------------------------------------------------*/
(define (sigbus-error-handler n)
   (error "bigloo" "`bus error' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    sigsegv-error-handler ...                                        */
;*---------------------------------------------------------------------*/
(define (sigsegv-error-handler n)
   (error "bigloo" "`segmentation violation' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    On installe le ratrappage des exceptions                         */
;*---------------------------------------------------------------------*/
(signal sigfpe  sigfpe-error-handler)
(signal sigill  sigill-error-handler)
(signal sigbus  sigbus-error-handler)
(signal sigsegv sigsegv-error-handler)

;*---------------------------------------------------------------------*/
;*    error/location ...                                               */
;*    -------------------------------------------------------------    */
;*    We print error message understable by emacs. We                  */
;*    print them in the following format:                              */
;*      `File "foobar.ml", lines 5-8, character 20: blah blah'         */
;*---------------------------------------------------------------------*/
(define (error/location proc message object fname location)
   (cond
      ((string=? fname "[string]")
       (error proc message object))
      ((string=? fname "[stdin]")
       (error proc message object))
      (else
       (set! *error-notifier* (error/location-file fname location))
       (error proc message object))))

;*---------------------------------------------------------------------*/
;*    error/location-file ...                                          */
;*---------------------------------------------------------------------*/
(define (error/location-file file-name location)
   ;; we compute the message to print the location
   (let ((port (open-input-file file-name)))
      (if (not (input-port? port))
	  ;; we are enable to re-open the file, we just print a
	  ;; standard error
	  (lambda (proc msg obj)
	     (error proc msg obj))
	  ;; we readlines until we reach location
	  (let loop ((line-string (read-line port))
		     (line-num    1))
	     (if (eof-object? line-string)
		 (begin
		    ;; an error we don't know how to print
		    (close-input-port port)
		    (lambda (proc msg obj)
		       (error proc msg obj)))
		 (if (>=fx (c-input-port-filepos port) location)
		     (begin
			(close-input-port port)
			(make-location-notifier file-name
						line-num
						location
						line-string
						(+fx
						 (-fx (string-length
						       line-string)
						      (-fx
						       (c-input-port-filepos
							port)
						       location))
						 1)))
		     (let ((old-pos (c-input-port-filepos port)))
			(loop (read-line port)
			      (+fx line-num 1)))))))))

;*---------------------------------------------------------------------*/
;*    make-location-notifier ...                                       */
;*---------------------------------------------------------------------*/
(define (make-location-notifier fname line char string marker)
   (lambda (proc msg obj)
      ;; we first re-install default notifier
      (set! *error-notifier* #f)
      ;; we flush error-port
      (flush-output-port (current-output-port))
      (newline (current-error-port))
      (let ((old-length   (get-write-length))
	    (space-string (if (>fx marker 0)
			      (make-string (-fx marker 1) #\space)
			      "")))
	 ;; we ajust tabulation in space string.
	 (fix-tabulation! marker string space-string)
	 ;; we now print the error message
	 (print-cursor fname line char string space-string)
	 ;; we set the write-length before printing the error message.
	 (set-write-length! 80)
	 ;; we display the error message
	 (fprint (current-error-port) "# *** ERROR:bigloo:" proc)
	 (fprint (current-error-port) "# " msg " -- " obj)
	 ;; we restore the current-printer
	 (set-write-length! old-length))))

;*---------------------------------------------------------------------*/
;*    warning/location-file ...                                        */
;*---------------------------------------------------------------------*/
(define (warning/location-file file-name location args)
   ;; we compute the message to print the location
   (let ((port (open-input-file file-name)))
      (if (not (input-port? port))
	  ;; we are enable to re-open the file, we just print a
	  ;; standard warning
	  (apply warning args)
	  ;; we readlines until we reach location
	  (let loop ((line-string (read-line port))
		     (line-num    1))
	     (if (eof-object? line-string)
		 (begin
		    ;; an error we don't know how to print
		    (close-input-port port)
		    (apply warning args))
		 (if (>=fx (c-input-port-filepos port) location)
		     (begin
			(close-input-port port)
			(do-warn/location file-name
					  line-num
					  location
					  line-string
					  (+fx
					   (-fx (string-length
						 line-string)
						(-fx
						 (c-input-port-filepos
						  port)
						 location))
					   1)
					  args))
		     (let ((old-pos (c-input-port-filepos port)))
			(loop (read-line port)
			      (+fx line-num 1)))))))))

;*---------------------------------------------------------------------*/
;*    do-warn/location ...                                             */
;*---------------------------------------------------------------------*/
(define (do-warn/location fname line char string marker args)
   (flush-output-port (current-output-port))
   (newline (current-error-port))
   (let ((old-length   (get-write-length))
	 (space-string (if (>fx marker 0)
			   (make-string (-fx marker 1) #\space)
			   "")))
      ;; we ajust tabulation in space string.
      (fix-tabulation! marker string space-string)
      ;; we now print the warning message
      (print-cursor fname line char string space-string)
      ;; we display the warning message
      (display "# *** WARNING:bigloo:" (current-error-port))
      (if (not (null? args))
	  (begin
	     (fprint (current-error-port) (car args))
	     (for-each (lambda (a)
			  (display a (current-error-port)))
		       (cdr args))))
      (newline (current-error-port))))

;*---------------------------------------------------------------------*/
;*    fix-tabulation! ...                                              */
;*---------------------------------------------------------------------*/
(define (fix-tabulation! marker src dst)
   (let loop ((read (-fx marker 1)))
      (cond
	 ((=fx read -1)
	  'done)
	 ((char=? (string-ref src read) #\tab)
	  (string-set! dst read #\tab))
	 (else
	  (loop (-fx read 1))))))

;*---------------------------------------------------------------------*/
;*    print-cursor ...                                                 */
;*---------------------------------------------------------------------*/
(define (print-cursor fname line char string space-string)
   (fprint (current-error-port)
	   "File \"" fname "\", line " line ", character "
	   char ":"
	   #\Newline
	   "#" string #\Newline
	   "#"
	   space-string
	   "^"))
