;*---------------------------------------------------------------------*/
;*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    .../evcompile.scm ...                                            */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Mar 25 09:09:18 1994                          */
;*    Last change :  Mon Dec 19 10:24:58 1994 (serrano)                */
;*    -------------------------------------------------------------    */
;*    La pre-compilation des formes pour permettre l'interpretation    */
;*    rapide                                                           */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __evcompile
   (include "Eval/byte-code.sch")
   (export  (evcompile exp env where named?)))

;*---------------------------------------------------------------------*/
;*    evcompile ...                                                    */
;*    s-exp x env --> byte-code                                        */
;*    -------------------------------------------------------------    */
;*    La phase d'expansion a genere une syntaxe correcte. On n'a donc  */
;*    plus du tout a la tester maintenant.                             */
;*---------------------------------------------------------------------*/
(define (evcompile exp env where named?)
   (match-case exp
      (()
       (error "eval" "Illegal expression" '()))
      ((module ?- . ?decls)
       (module-declaration! decls)
       (unspecified))
      ((assert ?- ?- ?-)
       (unspecified))
      ((assert ?- ?-)
       (unspecified))
      ((atom ?atom)
       (cond
	  ((symbol? atom)
	   (evcompile-ref (variable atom env)))
	  ((or (vector? atom)
	       (struct? atom))
	   (error "eval" "Ilegal expression (should be quoted)" exp))
	  (else
	   (evcompile-cnst atom))))
      ((quote ?cnst)
       (evcompile-cnst cnst))
      ((if ?si ?alors ?sinon)
       (evcompile-if (evcompile si env where #f)
		     (evcompile alors env where named?)
		     (evcompile sinon env where named?)))
      ((begin . ?rest)
       (evcompile-begin rest env where named?))
      (((or define define-inline) ?var ?val)
       (evcompile-define var (delay (evcompile val '() var #t))))
      ((set! ?var ?val)
       (evcompile-set (variable var env) (evcompile val env var #t)))
      ((bind-exit ?escape ?body)
       (evcompile-bind-exit (evcompile `(lambda ,escape ,body) env escape #t)))
      ((lambda ?formals ?body)
       (evcompile-lambda formals
			 (evcompile body (extend-env formals env) where #f)
			 where
			 named?))
      (((atom ?fun) . ?args)
       (let ((actuals (map (lambda (a) (evcompile a env where #f)) args)))
	  (cond
	     ((symbol? fun)
	      (let ((proc (variable fun env)))
		 (cond
		    ((eval-global? proc)
		     (evcompile-global-application proc actuals))
		    (else
		     (evcompile-application fun
					    (evcompile-ref proc)
					    actuals)))))
	     ((procedure? fun)
	      (evcompile-compiled-application fun actuals))
	     (else
	      (error "eval" "Not a procedure" fun)
	      (make-byte-code -2
			      (list "eval" "Not a procedure" fun))))))
      ((?fun . ?args)
       (let ((actuals (map (lambda (a) (evcompile a env where #f)) args))
	     (proc    (evcompile fun env where #f)))
	  (evcompile-application fun proc actuals)))
      (else
       (error "eval" "Illegal form" exp))))

;*---------------------------------------------------------------------*/
;*    evcompile-cnst ...                                               */
;*---------------------------------------------------------------------*/
(define (evcompile-cnst cnst)
   (cond
      ((pair? cnst)
       (make-byte-code -1 cnst))
      (else
       cnst)))

;*---------------------------------------------------------------------*/
;*    evcompile-ref ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-ref variable)
   (cond
      ((eval-global? variable)
       (make-byte-code (if (eq? (eval-global-tag variable) 1) 5 6)
		       variable))
      ((dynamic? variable)
       (make-byte-code 7 (dynamic-name variable)))
      (else
       (case variable
	  ((0 1 2 3)
	   (make-byte-code variable '()))
	  (else
	   (make-byte-code 4 variable))))))

;*---------------------------------------------------------------------*/
;*    evcompile-set ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-set variable value)
   (cond
      ((eval-global? variable)
       (make-byte-code 8 (cons variable value)))
      ((dynamic? variable)
       (make-byte-code 9 (cons (dynamic-name variable) value)))
      (else
       (case variable
	  ((0 1 2 3)
	   (make-byte-code (+fx 10 variable) value))
	  (else
	   (make-byte-code 14 (cons variable value)))))))

;*---------------------------------------------------------------------*/
;*    evcompile-if ...                                                 */
;*---------------------------------------------------------------------*/
(define (evcompile-if si alors sinon)
   (make-byte-code 15 (vector si alors sinon)))

;*---------------------------------------------------------------------*/
;*    evcompile-begin ...                                              */
;*---------------------------------------------------------------------*/
(define (evcompile-begin body env where named?)
   (if (and (pair? body) (null? (cdr body)))
       (evcompile (car body) env where named?)
       (let ((cbody (let loop ((rest body))
		       (cond
			  ((null? rest)
			   '())
			  ((null? (cdr rest))
			   (cons (evcompile (car rest) env where named?) '()))
			  (else
			   (cons (evcompile (car rest) env where #f)
				 (loop (cdr rest))))))))
	  (make-byte-code 16 (list->vector cbody)))))

;*---------------------------------------------------------------------*/
;*    evcompile-define ...                                             */
;*    -------------------------------------------------------------    */
;*    Le calcul de `val' a ete differe car on ne veut evcompiler la    */
;*    valeur liee d'un define qu'une fois que la variable a ete liee   */
;*    dans l'environment. Si on ne fait pas cela on se tape que des    */
;*    appels dynamics dans les definitions des fonctions               */
;*    auto-recursives !                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-define var val)
   (make-byte-code 17 (cons var val)))
	    
;*---------------------------------------------------------------------*/
;*    evcompile-bind-exit ...                                          */
;*---------------------------------------------------------------------*/
(define (evcompile-bind-exit body)
   (make-byte-code 18 body))

;*---------------------------------------------------------------------*/
;*    evcompile-lambda ...                                             */
;*---------------------------------------------------------------------*/
(define (evcompile-lambda formals body where named?)
   (match-case formals
      ((or () (?-) (?- ?-) (?- ?- ?-) (?- ?- ?- ?-))
       (if named?
	   (make-byte-code (+fx (length formals) 37) (cons where body))
	   (make-byte-code (+fx (length formals) 42) body)))
      ((atom ?-)
       (if named?
	   (make-byte-code 47 (cons where body))
	   (make-byte-code 51 body)))
      (((atom ?-) . (atom ?-))
       (if named?
	   (make-byte-code 48 (cons where body))
	   (make-byte-code 52 body)))
      (((atom ?-) (atom ?-) . (atom ?-))
       (if named?
	   (make-byte-code 49 (cons where body))
	   (make-byte-code 53 body)))
      (((atom ?-) (atom ?-) (atom ?-) . (atom ?-))
       (if named?
	   (make-byte-code 50 (cons where body))
	   (make-byte-code 54 body)))
      (else
       (if named?
	   (make-byte-code 55 (vector where body formals))
	   (make-byte-code 56 (cons body formals))))))

;*---------------------------------------------------------------------*/
;*    evcompile-global-application ...                                 */
;*---------------------------------------------------------------------*/
(define (evcompile-global-application proc actuals)
   (case (length actuals)
      ((0)
       (make-byte-code (if (eq? (eval-global-tag proc) 1) 19 57)
		       (vector (eval-global-name proc)
			       proc)))
      ((1)
       (make-byte-code (if (eq? (eval-global-tag proc) 1) 20 58)
		       (vector (eval-global-name proc)
			       proc
			       (car actuals))))
      ((2)
       (make-byte-code (if (eq? (eval-global-tag proc) 1) 21 59)
		       (vector (eval-global-name proc)
			       proc
			       (car actuals)
			       (cadr actuals))))
      ((3)
       (make-byte-code (if (eq? (eval-global-tag proc) 1) 22 60)
		       (vector (eval-global-name proc)
			       proc
			       (car actuals)
			       (cadr actuals)
			       (caddr actuals))))
      ((4)
       (make-byte-code (if (eq? (eval-global-tag proc) 1) 23 61)
		       (vector (eval-global-name proc)
			       proc
			       (car actuals)
			       (cadr actuals)
			       (caddr actuals)
			       (cadddr actuals))))
      (else
       (make-byte-code (if (eq? (eval-global-tag proc) 1) 24 62)
		       (vector (eval-global-name proc)
			       proc
			       actuals)))))

;*---------------------------------------------------------------------*/
;*    evcompile-compiled-application ...                               */
;*---------------------------------------------------------------------*/
(define (evcompile-compiled-application proc actuals)
   (case (length actuals)
      ((0)
       (make-byte-code 25 proc))
      ((1)
       (make-byte-code 26 (vector proc (car actuals))))
      ((2)
       (make-byte-code 27 (vector proc (car actuals)
				  (cadr actuals))))
      ((3)
       (make-byte-code 28 (vector proc (car actuals)
				  (cadr actuals)
				  (caddr actuals))))
      ((4)
       (make-byte-code 29 (vector proc (car actuals)
				  (cadr actuals)
				  (caddr actuals)
				  (cadddr actuals))))
      (else
       (make-byte-code 30 (cons proc actuals)))))

;*---------------------------------------------------------------------*/
;*    evcompile-application ...                                        */
;*---------------------------------------------------------------------*/
(define (evcompile-application name proc actuals)
   (case (length actuals)
      ((0)
       (make-byte-code 31 (cons name proc)))
      ((1)
       (make-byte-code 32 (vector name proc
				  (car actuals))))
      ((2)
       (make-byte-code 33 (vector name proc
				  (car actuals)
				  (cadr actuals))))
      ((3)
       (make-byte-code 34 (vector name proc
				  (car actuals)
				  (cadr actuals)
				  (caddr actuals))))
      ((4)
       (make-byte-code 35 (vector name proc
				  (car actuals)
				  (cadr actuals)
				  (caddr actuals)
				  (cadddr actuals))))
      (else
       (make-byte-code 36 (vector name proc
				  actuals)))))

;*---------------------------------------------------------------------*/
;*    variable ...                                                     */
;*---------------------------------------------------------------------*/
(define (variable symbol env)
   (let ((offset (let loop ((env   env)
			    (count 0))
		    (cond
		       ((null? env)
			#f)
		       ((eq? (car env) symbol)
			count)
		       (else
			(loop (cdr env) (+fx count 1)))))))
      (if offset
	  offset
	  (let ((global (eval-lookup symbol)))
	     (if (not global)
		 `#(,symbol)
		 global)))))

;*---------------------------------------------------------------------*/
;*    dynamic? ...                                                     */
;*---------------------------------------------------------------------*/
(define-inline (dynamic? variable)
   (and (vector? variable)
	(=fx (vector-length variable) 1)))

;*---------------------------------------------------------------------*/
;*    dynamic-name ...                                                 */
;*---------------------------------------------------------------------*/
(define-inline (dynamic-name dynamic)
   (vector-ref-ur dynamic 0))

;*---------------------------------------------------------------------*/
;*    extend-env ...                                                   */
;*---------------------------------------------------------------------*/
(define (extend-env extend old-env)
   (let _loop_ ((extend extend))
      (cond
	 ((null? extend)
	  old-env)
	 ((not (pair? extend))
	  (cons extend old-env))
	 (else
	  (cons (car extend) (_loop_ (cdr extend)))))))







