/*---------------------------------------------------------------------*/
/*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \    /  '                              */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome Send them to                                           */
/*        <Manuel.Serrano@inria.fr>                                    */
/*        Manuel Serrano                                               */
/*        INRIA -- Rocquencourt                                        */
/*        Domaine de Voluceau, BP 105                                  */
/*        78153 Le Chesnay Cedex                                       */
/*        France                                                       */
/*---------------------------------------------------------------------*/


/*---------------------------------------------------------------------*/
/*    serrano/prgm/project/bigloo/runtime1.7/Clib/string.c ...         */
/*                                                                     */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Fri Jul 17 11:33:47 1992                          */
/*    Last change :  Fri Jan 13 16:46:34 1995 (serrano)                */
/*                                                                     */
/*    La gestion C des chaines de caracteres                           */
/*---------------------------------------------------------------------*/
#include <ctype.h>
#include <stdio.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    make_string ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
make_string( k, c )
obj_t  k;
obj_t  c;
{
   obj_t string , aux;
   long  len = CINT( k );

	string = ALLOCATE( STRING_SIZE + len + 1 );

#if( !defined( TAG_STRING ) )
	string->string_t.header = HEADER_STRING;
#endif	
   string->string_t.length = k;

   memset( (char *)string + STRING_SIZE, CCHAR( c ), len );
   ((char *)string + STRING_SIZE)[ len ] = '\0';

   return BSTRING( string );
}

/*---------------------------------------------------------------------*/
/*    c_constant_string_to_string ...                                  */
/*---------------------------------------------------------------------*/
obj_t
c_constant_string_to_string( c_string )
char *c_string;
{
   return c_string_to_string( c_string );
}

/*---------------------------------------------------------------------*/
/*    c_string_to_string ...                                           */
/*---------------------------------------------------------------------*/
obj_t
c_string_to_string( c_string )
char *c_string;
{
   obj_t string, aux;
   long  i;
   long  len;

   if( !c_string )
      c_string = "";

   len = strlen( c_string );

	string = ALLOCATE( STRING_SIZE + len + 1 );

#if( !defined( TAG_STRING ) )
	string->string_t.header = HEADER_STRING;
#endif	
   string->string_t.length = BINT( len );

   strcpy( (char *)string + STRING_SIZE, c_string );

   return BSTRING( string );
}

/*---------------------------------------------------------------------*/
/*    string_append ...                                                */
/*---------------------------------------------------------------------*/
obj_t
string_append( s1, s2 )
obj_t s1, s2;
{
   long  l1, l2;
   obj_t string;
   obj_t a;
   
   l1 = CINT( STRING( s1 ).length );
   l2 = CINT( STRING( s2 ).length );

	
	string = ALLOCATE( STRING_SIZE + l1 + l2 + 1 );

#if( !defined( TAG_STRING ) )
	string->string_t.header = HEADER_STRING;
#endif	
   string->string_t.length = BINT( l1 + l2 );

	memcpy( (char *)string + STRING_SIZE, &STRING_REF( s1, 0 ), l1 );
	memcpy( (char *)string + STRING_SIZE + l1, &STRING_REF( s2, 0 ), l2 );
	((char *)string + STRING_SIZE)[ l1 + l2 ] = '\0';
	
   return BSTRING( string );
}
 
/*---------------------------------------------------------------------*/
/*    substring ...                                                    */
/*---------------------------------------------------------------------*/
obj_t
c_substring( src_string, min, max )
obj_t src_string, min, max;
{
   long  len;
   obj_t dst_string, a;
   
   len = CINT( SUB_I( max, min ) );

	dst_string = ALLOCATE( STRING_SIZE + len + 1 );

#if( !defined( TAG_STRING ) )
	dst_string->string_t.header = HEADER_STRING;
#endif	
   dst_string->string_t.length = BINT( len );

   memcpy( (char *)dst_string + STRING_SIZE,
			   &STRING_REF( src_string, CINT( min ) ),
            len );
   ((char *)dst_string + STRING_SIZE)[ len ] = '\0';

   return BSTRING( dst_string );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    blit_string ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
blit_string( s1, offset1, s2, offset2, len )
obj_t s1, offset1, s2, offset2, len;
{
	bcopy( &STRING_REF( s1, CINT( offset1 ) ),
			 &STRING_REF( s2, CINT( offset2 ) ),
			 CINT( len ) );
	return BUNSPEC;
}
	
/*---------------------------------------------------------------------*/
/*    integer_to_string ...                                            */
/*---------------------------------------------------------------------*/
char *
integer_to_string( x, radix )
long x, radix;
{
   char *aux;

   aux = (char *)ALLOCATE_ATOMIC( 1024 );

   switch( radix )
   {
      case 8 :
         sprintf( aux, "#o%o", x );
         break;
         
      case 10 :
         sprintf( aux, "%ld", x );
         break;
         
      case 16 :
         sprintf( aux, "#x%lx", x );
         break;
   }

   return aux;
}

/*---------------------------------------------------------------------*/
/*    real_to_string ...                                               */
/*---------------------------------------------------------------------*/
char *
real_to_string( x )
double x;
{
   char *aux;

   aux = (char *)ALLOCATE_ATOMIC( 1024 );

   sprintf( aux, "%#g", x );
   
   return aux;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bigloo_strcmp ...                                                */
/*---------------------------------------------------------------------*/
bool_t        
bigloo_strcmp( o1, o2 )
obj_t o1, o2;
{
	long l1, l2;

	l1 = CINT( STRING_LENGTH( o1 ) );
	l2 = CINT( STRING_LENGTH( o2 ) );

	if( l1 == l2 )
		return !memcmp( BSTRING_TO_CSTRING( o1 ), BSTRING_TO_CSTRING( o2 ), l1 );
	else
	   return 0;
}

/*---------------------------------------------------------------------*/
/*    strcicmp ...                                                     */
/*---------------------------------------------------------------------*/
bool_t        
strcicmp( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return 0;
}

/*---------------------------------------------------------------------*/
/*    string_lt ...                                                    */
/*---------------------------------------------------------------------*/
bool_t       
string_lt( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return *st1 < *st2;
}

/*---------------------------------------------------------------------*/
/*    string_le ...                                                    */
/*---------------------------------------------------------------------*/
bool_t        
string_le( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return *st1 <= *st2;
}

/*---------------------------------------------------------------------*/
/*    string_gt ...                                                    */
/*---------------------------------------------------------------------*/
bool_t        
string_gt( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return *st1 > *st2;
}

/*---------------------------------------------------------------------*/
/*    string_ge ...                                                    */
/*---------------------------------------------------------------------*/
bool_t         
string_ge( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return *st1 >= *st2;
}

/*---------------------------------------------------------------------*/
/*    string_cilt ...                                                  */
/*---------------------------------------------------------------------*/
bool_t        
string_cilt( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return tolower( *st1 ) < tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    string_cile ...                                                  */
/*---------------------------------------------------------------------*/
bool_t       
string_cile( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return tolower( *st1 ) <= tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    string_cigt ...                                                  */
/*---------------------------------------------------------------------*/
bool_t        
string_cigt( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return tolower( *st1 ) > tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    string_cige ...                                                  */
/*---------------------------------------------------------------------*/
bool_t
string_cige( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return tolower( *st1 ) >= tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    compare_strings ...                                              */
/*---------------------------------------------------------------------*/
obj_t
compare_strings( o1, o2 )
obj_t o1, o2;
{
	long  len1, len2, len;
	char *p1, *p2;
	long  i;
	
	
	len1 = CINT( STRING_LENGTH( o1 ) );
	len2 = CINT( STRING_LENGTH( o2 ) );

	p1 = BSTRING_TO_CSTRING( o1 );
	p2 = BSTRING_TO_CSTRING( o2 );

	for( len = (len1 <= len2 ? len1 : len2); len > 0; len--, p1++, p2++ )
	   if (*p1 != *p2)
		   return (*p1 < *p2 ? BINT( -1 ) : BINT( 1 ) );

	if( len1 == len2 )
	   return BINT( 0 );
	else
	   if( len1 < len2 )
		   return BINT( -2 );
	   else
		   return BINT( 2 );
}
	
/*---------------------------------------------------------------------*/
/*    escape_C_string ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine ou la representation des     */
/*    caracteres de controles a ete remplacee par ces caracteres.      */
/*    ex:     +---+---+---+---+          +---+---+---+                 */
/*            | \ | n | a | 0 |    ==>   | \n| a | 0 |                 */
/*            +---+---+---+---+          +---+---+---+                 */
/*    Cette conversion est utile pour l'interprete car les chaines     */
/*    lues ne sont pas parsees. On donne donc la possibilite de le     */
/*    faire avec cette fonction.                                       */
/*---------------------------------------------------------------------*/
obj_t
escape_C_string( src )
unsigned char *src;
{
   /* on supprime un caractere de cette chaine car elle est rendue par le */
   /* lecteur comme etant `"tototo'. Ceci est dut au fait qu'on utilise   */
   /* la fonction `the-small-string' qui supprime le premier et le        */
   /* dernier caractere de la chaine lu. Comme les chaines etrangeres     */
   /* commence par 2 caracteres, on en supprime 1 autre maintenant.       */

   long len = strlen( ++src );
   unsigned char *dst;
   obj_t string, aux;

	
	string = ALLOCATE( STRING_SIZE + len + 1 );

#if( !defined( TAG_STRING ) )
	string->string_t.header = HEADER_STRING;
#endif	

   dst = ((unsigned char *)string + STRING_SIZE);

   while( *src )
   {
      if( *src != '\\' )
         *dst++ = *src++;
      else
      {
			len--;
	 
         switch( *++src ) 
         {
            case '\0' : *dst++ = '\\';
                        break;

            case 'n'  : *dst++ = '\n';
                        break;
                        
            case 't'  : *dst++ = '\t';
                        break;
                        
            case 'b'  : *dst++ = '\b';
                        break;
                        
            case 'r'  : *dst++ = '\r';
                        break;
                        
            case 'f'  : *dst++ = '\f';
                        break;
                        
            case 'v'  : *dst++ = '\v';
                        break;
                        
            case '\\' : *dst++ = '\\';
                        break;
                        
            case '\'' : *dst++ = '\'';
                        break;
                        
            case '"'  : *dst++ = '\"';
                        break;

#if( defined( __STDC___ ) )                          
            case 'a'  : *dst++ = '\a';
                        break;

            case '?'  : *dst++ = '\?';
                        break;
#endif                        

            default   : if( isdigit( *(src) ) &&
									 isdigit( *(src+1) ) &&
									 isdigit( *(src+2) ) )
				            /* C'est une representation alpha-numerique `\???' */
				            {
									unsigned char aux;

									aux = (*src     - '0')*64 +
									      (*(src+1) - '0')*8 +
											(*(src+2) - '0');
									*dst++ = aux;
									src+=2;
									
									len -= 2;
								}
					         else
				               *dst++ = *src;
					  
                        break;
         }
         src++;
      }
   }
   *dst = '\0';
   
   string->string_t.length = BINT( len );

   return BSTRING( string );
}
       
/*---------------------------------------------------------------------*/
/*    escape_scheme_string ...                                         */
/*    -------------------------------------------------------------    */
/*    Cette fonction ressemble a la precedente mais elle filtre moins  */
/*    de caracteres                                                    */
/*---------------------------------------------------------------------*/
obj_t
escape_scheme_string( src )
char *src;
{
   long  len = strlen( src );
   char *dst;
   obj_t string, aux;

	string = ALLOCATE( STRING_SIZE + len + 1 );

#if( !defined( TAG_STRING ) )
	string->string_t.header = HEADER_STRING;
#endif	

   dst = ((char *)string + STRING_SIZE);

   while( *src )
   {
      if( *src != '\\' )
         *dst++ = *src++;
      else
      {
			len--;
	 
         switch( *++src )  
         {
            case '\\' : *dst++ = '\\';
                        break;
                        
            case '"'  : *dst++ = '"';
                        break;

            default   : *dst++ = *src;
                        break;
         }
         src++;
      }
   }
   *dst = '\0';
   
   string->string_t.length = BINT( len );

   return BSTRING( string );
}
       
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    string_for_read ...                                              */
/*---------------------------------------------------------------------*/
obj_t
string_for_read( bstring )
obj_t bstring;
{
	unsigned char *dst;
	unsigned char *src = BSTRING_TO_CSTRING( bstring );
	long  r, w, len = CINT( STRING_LENGTH( bstring ) );
	obj_t res;

	dst = malloc( (len * 4) + 1 );

	for( r = 0, w = 0; r < len; r++ )
	   switch( src[ r ] )
		{
	      case '\n' : dst[ w++ ] = '\\';
			            dst[ w++ ] = 'n';
				         break;

	      case '\t' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 't';
				         break;

	      case '\b' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 'b';
				         break;
				  
	      case '\r' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 'r';
				         break;
				  
	      case '\f' : dst[ w++ ] = '\\';
			            dst[ w++ ] = 'f';
				         break;

	      case '\v' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 'v';
				         break;

	      case '"'  : dst[ w++ ] = '\\';
				         dst[ w++ ] = '"';
				         break;
				  
	      case '\\' : dst[ w++ ] = '\\';
				         dst[ w++ ] = '\\';
				         break;

			default :   if( (src[r] < 128) && (isprint( src[ r ] )) )
			               dst[ w++ ] = src[ r ];
				         else
							{
   				            sprintf( &dst[ w ], "\\%03o",
										  ((unsigned char *)src)[ r ] );
								w += 4;
							}
	}

	dst[ w ] = '\0';
	
   res = c_string_to_string( dst );

	free( dst );

	return res;
}

					  
