;*---------------------------------------------------------------------*/
;*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    .../evmeaning.scm ...                                            */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Wed Aug  4 10:48:41 1993                          */
;*    Last change :  Wed Jun  1 13:51:30 1994 (serrano)                */
;*    -------------------------------------------------------------    */
;*    L'interprete de bigloo (la version `lambda')                     */
;*=====================================================================*/
       
;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __evmeaning
  (export  (inline evmeaning           exp env)
	   (evcompile                  exp env where named?)
	   the-global-environment
	   (init-the-global-environment!)
	   (__unbound! name)
	   (define-primop!      var val)
	   (define-primop-ref!  var addr))
  (foreign (obj funcall-0  (obj obj) "eval_funcall_0")
	   (obj funcall-1  (obj obj obj) "eval_funcall_1")
	   (obj funcall-2  (obj obj obj obj) "eval_funcall_2")
	   (obj funcall-3  (obj obj obj obj obj) "eval_funcall_3")
	   (obj funcall-4  (obj obj obj obj obj obj) "eval_funcall_4")
	   (obj eval-apply (obj obj obj) "eval_apply"))
  (static  (update-global! variable val run-stack)
	   (extend-env extend old-env)
	   (lookup var)))

;*---------------------------------------------------------------------*/
;*    evmeaning ...                                                    */
;*---------------------------------------------------------------------*/
(define-inline (evmeaning exp stack)
   (if (procedure? exp)
       (exp stack)
       exp))

;*---------------------------------------------------------------------*/
;*    the-global-environment                                           */
;*---------------------------------------------------------------------*/
(define the-global-environment the-global-environment)

;*---------------------------------------------------------------------*/
;*    bind-global! ...                                                 */
;*---------------------------------------------------------------------*/
(define (bind-global! name var)
   (putprop! name '0000 var))

;*---------------------------------------------------------------------*/
;*    define-primop! ...                                               */
;*---------------------------------------------------------------------*/
(define (define-primop! var val)
   (let ((cell (lookup var)))
      (if (not (global? cell))
	  (bind-global! var (vector 0 var val))
	  (set-global-value! cell val))))

;*---------------------------------------------------------------------*/
;*    define-primop-ref! ...                                           */
;*---------------------------------------------------------------------*/
(define (define-primop-ref! var addr)
   (if (not (lookup var))
       (bind-global! var (vector 1 var addr))))

;*---------------------------------------------------------------------*/
;*    __unbound! ...                                                   */
;*---------------------------------------------------------------------*/
(define (__unbound! name)
   (remprop! name '0000))

;*---------------------------------------------------------------------*/
;*      update-global! ...                                             */
;*---------------------------------------------------------------------*/
(define (update-global! variable val run-stack)
   (let ((value (evmeaning val run-stack)))
      (if (eq? (global-tag variable) 1)
	  (location-set! (global-value variable) value)
	  (set-global-value! variable value))
      (global-name variable)))

;*---------------------------------------------------------------------*/
;*    extend-env ...                                                   */
;*---------------------------------------------------------------------*/
(define (extend-env extend old-env)
   (let _loop_ ((extend extend))
      (cond
	 ((null? extend)
	  old-env)
	 ((not (pair? extend))
	  (cons extend old-env))
	 (else
	  (cons (car extend) (_loop_ (cdr extend)))))))

;*---------------------------------------------------------------------*/
;*    Les environments ...                                             */
;*---------------------------------------------------------------------*/
(init-the-global-environment!)
       
;*---------------------------------------------------------------------*/
;*    init-the-global-environment! ...                                 */
;*    -------------------------------------------------------------    */
;*    Il faut que cette fonction utilise le symbol `0000' */
;*    pour etre sur qu'il est definit au moment ou on fait les         */
;*    `define-primop'.                                                 */
;*---------------------------------------------------------------------*/
(define (init-the-global-environment!)
   'nothing)

;*---------------------------------------------------------------------*/
;*    lookup ...                                                       */
;*---------------------------------------------------------------------*/
(define (lookup var)
   (let ((prop (getprop var '0000)))
      (if prop
	  prop
	  #f)))

;*---------------------------------------------------------------------*/
;*    evcompile ...                                                    */
;*    s-exp x env --> (lambda () ...)                                  */
;*    -------------------------------------------------------------    */
;*    La phase d'expansion a genere une syntaxe correcte. On n'a donc  */
;*    plus du tout a la tester maintenant.                             */
;*---------------------------------------------------------------------*/
(define (evcompile exp env where named?)
   (match-case exp
      (()
       (error "eval" "Illegal expression" '())
       (lambda (stack)
	 (error "eval" "Illegal expression" '())))
      ((module ?- . ?decls)
       (module-declaration! decls)
       (lambda (stack)
	  (unspecified)))
      ((atom ?atom)
       (cond
	  ((symbol? atom)
	   (evcompile-ref (variable atom env)))
	  (else
	   (evcompile-cnst atom))))
      ((quote ?cnst)
       (evcompile-cnst cnst))
      ((if ?si ?alors ?sinon)
       (evcompile-if (evcompile si env where #f)
		     (evcompile alors env where named?)
		     (evcompile sinon env where named?)))
      ((begin . ?rest)
       (evcompile-begin rest env where named?))
      (((or define define-inline) ?var ?val)
       (evcompile-define var (lambda () (evcompile val '() var #t))))
      ((set! ?var ?val)
       (evcompile-set (variable var env) (evcompile val env var #t)))
      ((bind-exit ?escape ?body)
       (let ((new-body (evcompile `(lambda ,escape ,body) env escape #t)))
	  (lambda (stack)
	     (bind-exit (__dummy__)
			((evmeaning new-body stack)
			 __dummy__)))))
      ((lambda ?formals ?body)
       (evcompile-lambda formals
			 (evcompile body (extend-env formals env) where #f)
			 where
			 named?))
      (((atom ?fun) . ?args)
       (let ((actuals (map (lambda (a) (evcompile a env where #f)) args)))
	  (cond
	     ((symbol? fun)
	      (let ((proc (variable fun env)))
		 (cond
		    ((global? proc)
		     (evcompile-global-application proc actuals))
		    (else
		     (evcompile-application fun
					    (evcompile-ref proc)
					    actuals)))))
	     ((procedure? fun)
	      (evcompile-compiled-application fun actuals))
	     (else
	      (error "eval" "Not a procedure" fun)
	      (lambda (stack) (error "eval" "Not a procedure" fun))))))
      ((?fun . ?args)
       (let ((actuals (map (lambda (a) (evcompile a env where #f)) args))
	     (proc    (evcompile fun env where #f)))
	  (evcompile-application fun proc actuals)))
      (else
       (error "eval" "Illegal form" exp)
       (lambda (stack)
	  (error "eval" "Illegal form" exp)))))


;*---------------------------------------------------------------------*/
;*    variable ...                                                     */
;*---------------------------------------------------------------------*/
(define (variable symbol env)
   (let ((offset (let loop ((env   env)
			    (count 0))
		    (cond
		       ((null? env)
			#f)
		       ((eq? (car env) symbol)
			count)
		       (else
			(loop (cdr env) (+fx count 1)))))))
      (if offset
	  offset
	  (let ((global (lookup symbol)))
	     (if (not global)
		 `#(,symbol)
		 global)))))

;*---------------------------------------------------------------------*/
;*    global? ...                                                      */
;*---------------------------------------------------------------------*/
(define-inline (global? variable)
   (and (vector? variable)
	(=fx (vector-length variable) 3)))

;*---------------------------------------------------------------------*/
;*    global-tag ...                                                   */
;*---------------------------------------------------------------------*/
(define-inline (global-tag global)
   (vector-ref-ur global 0))

;*---------------------------------------------------------------------*/
;*    global-name ...                                                  */
;*---------------------------------------------------------------------*/
(define-inline (global-name global)
   (vector-ref-ur global 1))

;*---------------------------------------------------------------------*/
;*    global-value ...                                                 */
;*---------------------------------------------------------------------*/
(define-inline (global-value global)
   (vector-ref-ur global 2))

;*---------------------------------------------------------------------*/
;*    set-global-value! ...                                            */
;*---------------------------------------------------------------------*/
(define-inline (set-global-value! global value)
   (vector-set-ur! global 2 value))

;*---------------------------------------------------------------------*/
;*    dynamic? ...                                                     */
;*---------------------------------------------------------------------*/
(define-inline (dynamic? variable)
   (and (vector? variable)
	(=fx (vector-length variable) 1)))

;*---------------------------------------------------------------------*/
;*    dynamic-name ...                                                 */
;*---------------------------------------------------------------------*/
(define-inline (dynamic-name dynamic)
   (vector-ref-ur dynamic 0))
     
;*---------------------------------------------------------------------*/
;*    evcompile-ref ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-ref variable)
   (cond
      ((global? variable)
       (if (eq? (global-tag variable) 1)
	   (lambda (stack)
	      (location-ref (global-value variable)))
	   (lambda (stack)
	      (global-value variable))))
      ((dynamic? variable)
       (let ((found #f))
	  (lambda (stack)
	     (if (global? found)
		 (global-value found)
		 (let ((global (lookup (dynamic-name variable))))
		    (if (global? global)
			(begin
			   (set! found global)
			   (global-value global))
			(error "eval"
			       "Unbound variable"
			       (dynamic-name variable))))))))
      (else
       (case variable
	  ((0)
	   (lambda (stack) (c-car stack)))
	  ((1)
	   (lambda (stack) (c-car (c-cdr stack))))
	  ((2)
	   (lambda (stack) (c-car (c-cdr (c-cdr stack)))))
	  ((3)
	   (lambda (stack) (c-car (c-cdr (c-cdr (c-cdr stack))))))
	  (else
	   (lambda (stack)
	      (let ((offset variable))
		 (do ((i 0 (+fx i 1))
		      (env stack (c-cdr env)))
		       ((=fx i offset) (c-car env))))))))))

;*---------------------------------------------------------------------*/
;*    evcompile-set ...                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-set variable value)
   (cond
      ((global? variable)
       (lambda (stack)
	  (update-global! variable value stack)
	  (unspecified)))
      ((dynamic? variable)
       (let ((found #f))
	  (lambda (stack)
	     (if (global? found)
		 (update-global! found value stack)
		 (let ((global (lookup (dynamic-name variable))))
		    (if (global? global)
			(begin
			   (set! found global)
			   (update-global! found value stack))
			(error "eval"
			       "Unbound variable"
			       (dynamic-name variable)))))
	     (unspecified))))
      (else
       (case variable
	  ((0)
	   (lambda (stack)
	      (let ((val (evmeaning value stack)))
		 (set-car! stack val)
		 val)))
	  ((1)
	   (lambda (stack)
	      (let ((val (evmeaning value stack)))
		 (set-car! (c-cdr stack) val)
		 val)))
	  ((2)
	   (lambda (stack)
	      (let ((val (evmeaning value stack)))
		 (set-car! (c-cdr (c-cdr stack)) val)
		 val)))
	  ((3)
	   (lambda (stack)
	      (let ((val (evmeaning value stack)))
		 (set-car! (c-cdr (c-cdr (c-cdr stack))) val)
		 val)))
	  (else
	   (lambda (stack)
	      (let ((val    (evmeaning value stack))
		    (offset variable))
		 (do ((i 0 (+fx i 1))
		      (env stack (c-cdr env)))
		       ((=fx i offset) (set-car! env val)))
		 val)))))))

;*---------------------------------------------------------------------*/
;*    evcompile-cnst ...                                               */
;*---------------------------------------------------------------------*/
(define (evcompile-cnst cnst)
   (if (procedure? cnst)
       (lambda (stack) cnst)
       cnst))

;*---------------------------------------------------------------------*/
;*    evcompile-if ...                                                 */
;*---------------------------------------------------------------------*/
(define (evcompile-if si alors sinon)
   (lambda (stack)
      (if (evmeaning si    stack)
	  (evmeaning alors stack)
	  (evmeaning sinon stack))))

;*---------------------------------------------------------------------*/
;*    evcompile-begin ...                                              */
;*---------------------------------------------------------------------*/
(define (evcompile-begin body env where named?)
   (cond
      ((and (pair? body) (null? (cdr body)))
       ;; le cas degenere
       (evcompile (car body) env where named?))
      (else
       (let ((body (let loop ((rest body))
		      (cond
			 ((null? rest)
			  '())
			 ((null? (cdr rest))
			  (cons (evcompile (car rest) env where named?) '()))
			 (else
			  (cons (evcompile (car rest) env where #f)
				(loop (cdr rest))))))))
	  (lambda (stack)
	     (let _loop_ ((body body))
		(if (null? (cdr body))
		    (evmeaning (car body) stack)
		    (begin
		       (evmeaning (car body) stack)
		       (_loop_ (cdr body))))))))))

;*---------------------------------------------------------------------*/
;*    evcompile-define ...                                             */
;*    -------------------------------------------------------------    */
;*    Le calcul de `val' a ete differe car on ne veut evcompiler la    */
;*    valeur liee d'un define qu'une fois que la variable a ete liee   */
;*    dans l'environment. Si on ne fait pas cela on se tape que des    */
;*    appels dynamics dans les definitions des fonctions               */
;*    auto-recursives !                                                */
;*---------------------------------------------------------------------*/
(define (evcompile-define var val)
   (lambda (stack)
      (let ((cell (lookup var)))
	 (if (global? cell)
	     (begin
		(fprint
		 (current-error-port)
		 "*** WARNING:bigloo:eval"
		 #\Newline
		 "redefinition of variable -- "
		 var)
		(update-global! cell (val) '()))
	     (let ((cell (vector 0 var (unspecified))))
		(bind-global! var cell)
		;; on le fait en deux fois pour etre sur que la liaison
		;; existe.
		(let ((value (evmeaning (val) '())))
		   (set-global-value! cell value))))
	 var)))

;*---------------------------------------------------------------------*/
;*    evcompile-lambda ...                                             */
;*---------------------------------------------------------------------*/
(define (evcompile-lambda formals body where named?)
   (match-case formals
      (()
       (if named?
	   (lambda (stack)
	      (lambda ()
		 (push-lambda-trace where)
		 (pop-lambda-trace (evmeaning body stack))))
	   (lambda (stack)
	      (lambda ()
		 (evmeaning body stack)))))
      ((?-)
       (if named?
	   (lambda (stack)
	      (lambda (x)
		 (push-lambda-trace where)
		 (pop-lambda-trace (evmeaning body (cons x stack)))))
	   (lambda (stack)
	      (lambda (x)
		 (evmeaning body (cons x stack))))))
      ((?- ?-)
       (if named?
	   (lambda (stack)
	      (lambda (x y)
		 (push-lambda-trace where)
		 (pop-lambda-trace (evmeaning body (cons x (cons y stack))))))
	   (lambda (stack)
	      (lambda (x y)
		 (evmeaning body (cons x (cons y stack)))))))
      ((?- ?- ?-)
       (if named?
	   (lambda (stack)
	      (lambda (x y z)
		 (push-lambda-trace where)
		 (pop-lambda-trace
		  (evmeaning body (cons x (cons y (cons z stack)))))))
	   (lambda (stack)
	      (lambda (x y z)
		 (evmeaning body (cons x (cons y (cons z stack))))))))
      ((?- ?- ?- ?-)
       (if named?
	   (lambda (stack)
	      (lambda (x y z t)
		 (push-lambda-trace where)
		 (pop-lambda-trace
		  (evmeaning body (cons x (cons y (cons z (cons t stack))))))))
	   (lambda (stack)
	      (lambda (x y z t)
		 (evmeaning body (cons x (cons y (cons z (cons t stack)))))))))
      ((atom ?-)
       (if named?
	   (lambda (stack)
	      (lambda x
		 (push-lambda-trace where)
		 (pop-lambda-trace (evmeaning body (cons x stack)))))
	   (lambda (stack)
	      (lambda x
		 (evmeaning body (cons x stack))))))
      (((atom ?-) . (atom ?-))
       (if named?
	   (lambda (stack)
	      (lambda (x . y)
		 (push-lambda-trace where)
		 (pop-lambda-trace (evmeaning body (cons x (cons y stack))))))
	   (lambda (stack)
	      (lambda (x . y)
		 (evmeaning body (cons x (cons y stack)))))))
      (((atom ?-) (atom ?-) . (atom ?-))
       (if named?
	   (lambda (stack)
	      (lambda (x y . z)
		 (push-lambda-trace where)
		 (pop-lambda-trace
		  (evmeaning body (cons x (cons y (cons z stack)))))))
	   (lambda (stack)
	      (lambda (x y . z)
		 (evmeaning body (cons x (cons y (cons z stack))))))))
      (((atom ?-) (atom ?-) (atom ?-) . (atom ?-))
       (if named?
	   (lambda (stack)
	      (lambda (x y z . t)
		 (push-lambda-trace where)
		 (pop-lambda-trace
		  (evmeaning body (cons x (cons y (cons z (cons t stack))))))))
	   (lambda (stack)
	      (lambda (x y z . t)
		 (evmeaning body (cons x (cons y (cons z (cons t stack)))))))))
      (else
       (if named?
	   (lambda (stack)
	      (lambda x
		 (push-lambda-trace where)
		 (let ((new-env (let _loop_ ((actuals x)
					     (formals formals))
				   (cond
				      ((null? formals)
				       (if (not (null? actuals))
					   (error "eval"
						  "Too many arguments provided"
						  actuals)
					   stack))
				      ((not (pair? formals))
				       (cons actuals stack))
				      ((null? actuals)
				       (error "eval"
					      "Too few arguments provided"
					      formals))
				      (else
				       (cons (car actuals)
					     (_loop_ (cdr actuals)
						     (cdr formals))))))))
		    (pop-lambda-trace (evmeaning body new-env)))))
	   (lambda (stack)
	      (lambda x
		 (let ((new-env (let _loop_ ((actuals x)
					     (formals formals))
				   (cond
				      ((null? formals)
				       (if (not (null? actuals))
					   (error "eval"
						  "Too many arguments provided"
						  actuals)
					   stack))
				      ((not (pair? formals))
				       (cons actuals stack))
				      ((null? actuals)
				       (error "eval"
					      "Too few arguments provided"
					      formals))
				      (else
				       (cons (car actuals)
					     (_loop_ (cdr actuals)
						     (cdr formals))))))))
		    (evmeaning body new-env))))))))
  
;*---------------------------------------------------------------------*/
;*    evcompile-global-application ...                                 */
;*---------------------------------------------------------------------*/
(define (evcompile-global-application proc actuals)
   (case (length actuals)
      ((0)
       (if (eq? (global-tag proc) 1)
	   (lambda (stack)
	      (funcall-0 (global-name proc)
			 (location-ref (global-value proc))))
	   (lambda (stack)
	      (funcall-0 (global-name proc)
			 (global-value proc)))))
      ((1)
       (if (eq? (global-tag proc) 1)
	   (lambda (stack)
	      (funcall-1 (global-name proc)
			 (location-ref (global-value proc))
			 (evmeaning (c-car actuals) stack)))
	   (lambda (stack)
	      (funcall-1 (global-name proc)
			 (global-value proc)
			 (evmeaning (c-car actuals) stack)))))
      ((2)
       (if (eq? (global-tag proc) 1)
	   (lambda (stack)
	      (funcall-2 (global-name proc)
			 (location-ref (global-value proc))
			 (evmeaning (c-car actuals) stack)
			 (evmeaning (c-car (c-cdr actuals)) stack)))
	   (lambda (stack)
	      (funcall-2 (global-name proc)
			 (global-value proc)
			 (evmeaning (c-car actuals) stack)
			 (evmeaning (c-car (c-cdr actuals)) stack)))))
      ((3)
       (if (eq? (global-tag proc) 1)
	   (lambda (stack)
	      (funcall-3 (global-name proc)
			 (location-ref (global-value proc))
			 (evmeaning (c-car actuals) stack)
			 (evmeaning (c-car (c-cdr actuals)) stack)
			 (evmeaning (c-car (c-cdr (c-cdr actuals))) stack)))
	   (lambda (stack)
	      (funcall-3 (global-name proc)
			 (global-value proc)
			 (evmeaning (c-car actuals) stack)
			 (evmeaning (c-car (c-cdr actuals)) stack)
			 (evmeaning (c-car (c-cdr (c-cdr actuals))) stack)))))
      ((4)
       (if (eq? (global-tag proc) 1)
	   (lambda (stack)
	      (funcall-4 (global-name proc)
			 (location-ref (global-value proc))
			 (evmeaning (c-car actuals) stack)
			 (evmeaning (c-car (c-cdr actuals)) stack)
			 (evmeaning (c-car (c-cdr (c-cdr actuals))) stack)
			 (evmeaning (c-car (c-cdr (c-cdr (c-cdr actuals))))
				    stack)))
	   (lambda (stack)
	      (funcall-4 (global-name proc)
			 (global-value proc)
			 (evmeaning (c-car actuals) stack)
			 (evmeaning (c-car (c-cdr actuals)) stack)
			 (evmeaning (c-car (c-cdr (c-cdr actuals))) stack)
			 (evmeaning (c-car (c-cdr (c-cdr (c-cdr actuals))))
				    stack)))))
      (else
       (if (eq? (global-tag proc) 1)
	   (lambda (stack)
	      (eval-apply (global-name proc)
			  (location-ref (global-value proc))
			  (map (lambda (a) (evmeaning a stack)) actuals)))
	   (lambda (stack)
	      (eval-apply (global-name proc)
			  (global-value proc)
			  (map (lambda (a) (evmeaning a stack)) actuals)))))))

;*---------------------------------------------------------------------*/
;*    evcompile-compiled-application ...                               */
;*---------------------------------------------------------------------*/
(define (evcompile-compiled-application proc actuals)
   (case (length actuals)
      ((0)
       (lambda (stack)
	  (proc)))
      ((1)
       (lambda (stack)
	  (proc (evmeaning (c-car actuals) stack))))
      ((2)
       (lambda (stack)
	  (proc (evmeaning (c-car actuals) stack)
		(evmeaning (c-car (c-cdr actuals)) stack))))
      ((3)
       (lambda (stack)
	  (proc (evmeaning (c-car actuals) stack)
		(evmeaning (c-car (c-cdr actuals)) stack)
		(evmeaning (c-car (c-cdr (c-cdr actuals))) stack))))
      ((4)
       (lambda (stack)
	  (proc (evmeaning (c-car actuals) stack)
		(evmeaning (c-car (c-cdr actuals)) stack)
		(evmeaning (c-car (c-cdr (c-cdr actuals))) stack)
		(evmeaning (c-car (c-cdr (c-cdr (c-cdr actuals)))) stack))))
      (else
       (lambda (stack)
	  (apply proc (map (lambda (a) (evmeaning a stack)) actuals))))))

;*---------------------------------------------------------------------*/
;*    evcompile-application ...                                        */
;*---------------------------------------------------------------------*/
(define (evcompile-application name proc actuals)
  (case (length actuals)
    ((0)
     (lambda (stack)
       (funcall-0 name (evmeaning proc stack))))
    ((1)
     (lambda (stack)
       (funcall-1 name
		  (evmeaning proc stack)
		  (evmeaning (c-car actuals) stack))))
    ((2)
     (lambda (stack)
       (funcall-2 name
		  (evmeaning proc stack)
		  (evmeaning (c-car actuals) stack)
		  (evmeaning (c-car (c-cdr actuals)) stack))))
    ((3)
     (lambda (stack)
       (funcall-3 name
		  (evmeaning proc stack)
		  (evmeaning (c-car actuals) stack)
		  (evmeaning (c-car (c-cdr actuals)) stack)
		  (evmeaning (c-car (c-cdr (c-cdr actuals))) stack))))
    ((4)
     (lambda (stack)
       (funcall-4 name
		  (evmeaning proc stack)
		  (evmeaning (c-car actuals) stack)
		  (evmeaning (c-car (c-cdr actuals)) stack)
		  (evmeaning (c-car (c-cdr (c-cdr actuals))) stack)
		  (evmeaning (c-car (c-cdr (c-cdr (c-cdr actuals)))) stack))))
    (else
     (lambda (stack)
       (eval-apply name
		   (evmeaning proc stack)
		   (map (lambda (v)
			   (evmeaning v stack))
			actuals))))))


