;;; -*-Scheme-*-
;;;
;;; Copyright (c) 1989-91 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Syntactic Closures
;;; written by Alan Bawden
;;; extensively modified by Chris Hanson

;;; See "Syntactic Closures", by Alan Bawden and Jonathan Rees, in
;;; Proceedings of the 1988 ACM Conference on Lisp and Functional
;;; Programming, page 86.

;;;; Classifier
;;;  The classifier maps forms into items.  In addition to locating
;;;  definitions so that they can be properly processed, it also
;;;  identifies keywords and variables, which allows a powerful form
;;;  of syntactic binding to be implemented.

(define (classify/form form environment definition-environment history)
  (cond ((identifier? form)
	 (item/new-history (syntactic-environment/lookup environment form)
			   history))
	((syntactic-closure? form)
	 (let ((form (syntactic-closure/form form))
	       (environment
		(filter-syntactic-environment
		 (syntactic-closure/free-names form)
		 environment
		 (syntactic-closure/environment form))))
	   (classify/form form
			  environment
			  definition-environment
			  (history/replace-reduction form
						     environment
						     history))))
	((pair? form)
	 (let ((item
		(classify/subexpression (car form) environment history
					select-car)))
	   (cond ((keyword-item? item)
		  ((keyword-item/classifier item) form
						  environment
						  definition-environment
						  history))
		 ((list? (cdr form))
		  (let ((items
			 (classify/subexpressions (cdr form)
						  environment
						  history
						  select-cdr)))
		    (make-expression-item
		     history
		     (lambda ()
		       (output/combination
			(compile-item/expression item)
			(map compile-item/expression items))))))
		 (else
		  (syntax-error history
				"combination must be a proper list"
				form)))))
	(else
	 (make-expression-item history (lambda () (output/constant form))))))

(define (classify/subform form environment definition-environment
			  history selector)
  (classify/form form
		 environment
		 definition-environment
		 (history/add-subproblem form environment history selector)))

(define (classify/subforms forms environment definition-environment
			   history selector)
  (select-map (lambda (form selector)
		(classify/subform form environment definition-environment
				  history selector))
	      forms
	      selector))

(define (classify/subexpression expression environment history selector)
  (classify/subform expression environment null-syntactic-environment
		    history selector))

(define (classify/subexpressions expressions environment history selector)
  (classify/subforms expressions environment null-syntactic-environment
		     history selector))

;;;; Compiler
;;;  The compiler maps items into the output language.

(define (compile-item/expression item)
  (let ((illegal
	 (lambda (item name)
	   (let ((history (item/history item)))
	     (syntax-error history
			   (string-append name
					  " may not be used as an expression")
			   (history/original-form history))))))
    (cond ((variable-item? item)
	   (output/variable (variable-item/name item)))
	  ((expression-item? item)
	   ((expression-item/compiler item)))
	  ((body-item? item)
	   (let ((items (flatten-body-items (body-item/components item))))
	     (if (null? items)
		 (illegal item "empty sequence")
		 (output/sequence (map compile-item/expression items)))))
	  ((definition-item? item)
	   (illegal item "definition"))
	  ((keyword-item? item)
	   (illegal item "keyword"))
	  (else
	   (impl-error (if (item? item) "unknown item" "not an item") item)))))

(define (compile/subexpression expression environment history selector)
  (compile-item/expression
   (classify/subexpression expression environment history selector)))

(define (compile/top-level forms environment)
  ;; Top-level syntactic definitions affect all forms that appear
  ;; after them.
  (output/top-level-sequence
   (let ((history (make-top-level-history forms environment)))
     (let forms-loop ((forms forms) (selector select-object))
       (if (null? forms)
	   '()
	   (let items-loop
	       ((items
		 (item->list
		  (classify/subform (car forms)
				    environment
				    environment
				    history
				    (selector/add-car selector)))))
	     (cond ((null? items)
		    (forms-loop (cdr forms) (selector/add-cdr selector)))
		   ((definition-item? (car items))
		    (let ((binding
			   (bind-definition-item! environment (car items))))
		      (if binding
			  (cons (output/top-level-definition
				 (car binding)
				 (compile-item/expression (cdr binding)))
				(items-loop (cdr items)))
			  (items-loop (cdr items)))))
		   (else
		    (cons (compile-item/expression (car items))
			  (items-loop (cdr items)))))))))))

;;;; Syntactic Closures

;;;(module ((exports
;;;	  (#f make-syntactic-closure
;;;	      syntactic-closure?
;;;	      syntactic-closure/environment
;;;	      syntactic-closure/free-names
;;;	      syntactic-closure/form)))

  (define syntactic-closure-type
    (make-record-type "syntactic-closure" '(ENVIRONMENT FREE-NAMES FORM)))

  (define make-syntactic-closure
    (record-constructor syntactic-closure-type '(ENVIRONMENT FREE-NAMES FORM)))

  (define syntactic-closure?
    (record-predicate syntactic-closure-type))

  (define syntactic-closure/environment
    (record-accessor syntactic-closure-type 'ENVIRONMENT))

  (define syntactic-closure/free-names
    (record-accessor syntactic-closure-type 'FREE-NAMES))

  (define syntactic-closure/form
    (record-accessor syntactic-closure-type 'FORM))

;;;  )

(define (make-syntactic-closure-list environment free-names forms)
  (map (lambda (form) (make-syntactic-closure environment free-names form))
       forms))

(define (strip-syntactic-closures object)
  (cond ((syntactic-closure? object)
	 (strip-syntactic-closures (syntactic-closure/form object)))
	((pair? object)
	 (cons (strip-syntactic-closures (car object))
	       (strip-syntactic-closures (cdr object))))
	((vector? object)
	 (let ((length (vector-length object)))
	   (let ((result (make-vector length)))
	     (do ((i 0 (+ i 1)))
		 ((= i length))
	       (vector-set! result i
			    (strip-syntactic-closures (vector-ref object i))))
	     result)))
	(else
	 object)))

(define (identifier? object)
  (or (symbol? object)
      (synthetic-identifier? object)))

(define (synthetic-identifier? object)
  (and (syntactic-closure? object)
       (identifier? (syntactic-closure/form object))))

(define (identifier->symbol identifier)
  (cond ((symbol? identifier)
	 identifier)
	((synthetic-identifier? identifier)
	 (identifier->symbol (syntactic-closure/form identifier)))
	(else
	 (impl-error "not an identifier" identifier))))

(define (identifier=? environment-1 identifier-1 environment-2 identifier-2)
  (let ((item-1 (syntactic-environment/lookup environment-1 identifier-1))
	(item-2 (syntactic-environment/lookup environment-2 identifier-2)))
    (or (item=? item-1 item-2)
	;; This is necessary because an identifier that is not
	;; explicitly bound by an environment is mapped to a variable
	;; item, and the variable items are not cached.  Therefore
	;; two references to the same variable result in two
	;; different variable items.
	(and (variable-item? item-1)
	     (variable-item? item-2)
	     (eq? (variable-item/name item-1)
		  (variable-item/name item-2))))))

;;;; Syntactic Environments

;;;(module ((exports
;;;	  (#f root-syntactic-environment
;;;	      null-syntactic-environment
;;;	      top-level-syntactic-environment
;;;	      internal-syntactic-environment
;;;	      filter-syntactic-environment
;;;	      syntactic-environment?
;;;	      syntactic-environment/lookup
;;;	      syntactic-environment/rename
;;;	      syntactic-environment/define!
;;;	      ;; the next three operations are for debugging:
;;;	      syntactic-environment/assign!
;;;	      syntactic-environment/parent
;;;	      syntactic-environment/bindings)))

  (define syntactic-environment-type
    (make-record-type
     "syntactic-environment"
     '(PARENT
       LOOKUP-OPERATION
       RENAME-OPERATION
       DEFINE-OPERATION
       BINDINGS-OPERATION)))

  (define make-syntactic-environment
    (record-constructor syntactic-environment-type
			'(PARENT
			  LOOKUP-OPERATION
			  RENAME-OPERATION
			  DEFINE-OPERATION
			  BINDINGS-OPERATION)))

  (define syntactic-environment?
    (record-predicate syntactic-environment-type))

  (define syntactic-environment/parent
    (record-accessor syntactic-environment-type 'PARENT))

  (define syntactic-environment/lookup-operation
    (record-accessor syntactic-environment-type 'LOOKUP-OPERATION))

  (define (syntactic-environment/assign! environment name item)
    (let ((binding
	   ((syntactic-environment/lookup-operation environment) name)))
      (if binding
	  (set-cdr! binding item)
	  (impl-error "can't assign unbound identifier" name))))

  (define syntactic-environment/rename-operation
    (record-accessor syntactic-environment-type 'RENAME-OPERATION))

  (define (syntactic-environment/rename environment name)
    ((syntactic-environment/rename-operation environment) name))

  (define syntactic-environment/define!
    (let ((accessor
	   (record-accessor syntactic-environment-type 'DEFINE-OPERATION)))
      (lambda (environment name item)
	((accessor environment) name item))))

  (define syntactic-environment/bindings
    (let ((accessor
	   (record-accessor syntactic-environment-type 'BINDINGS-OPERATION)))
      (lambda (environment)
	((accessor environment)))))

  (define (syntactic-environment/lookup environment name)
    (let ((binding
	   ((syntactic-environment/lookup-operation environment) name)))
      (cond (binding
	     (let ((item (cdr binding)))
	       (if (reserved-name-item? item)
		   (syntax-error (item/history item)
				 "premature reference to reserved name"
				 name)
		   item)))
	    ((symbol? name)
	     (make-variable-item name))
	    ((synthetic-identifier? name)
	     (syntactic-environment/lookup (syntactic-closure/environment name)
					   (syntactic-closure/form name)))
	    (else
	     (impl-error "not an identifier" name)))))

  (define root-syntactic-environment
    (make-syntactic-environment
     #f
     (lambda (name)
       name
       #f)
     (lambda (name)
       name)
     (lambda (name item)
       (impl-error "can't bind name in root syntactic environment" name item))
     (lambda ()
       '())))

  (define null-syntactic-environment
    (make-syntactic-environment
     #f
     (lambda (name)
       (impl-error "can't lookup name in null syntactic environment" name))
     (lambda (name)
       (impl-error "can't rename name in null syntactic environment" name))
     (lambda (name item)
       (impl-error "can't bind name in null syntactic environment" name item))
     (lambda ()
       '())))

  (define (top-level-syntactic-environment parent)
    (let ((bound '()))
      (make-syntactic-environment
       parent
       (let ((parent-lookup (syntactic-environment/lookup-operation parent)))
	 (lambda (name)
	   (or (assq name bound)
	       (parent-lookup name))))
       (lambda (name)
	 name)
       (lambda (name item)
	 (let ((binding (assq name bound)))
	   (if binding
	       (set-cdr! binding item)
	       (set! bound (cons (cons name item) bound)))))
       (lambda ()
	 (alist-copy bound)))))

  (define (internal-syntactic-environment parent)
    (let ((bound '())
	  (free '()))
      (make-syntactic-environment
       parent
       (let ((parent-lookup (syntactic-environment/lookup-operation parent)))
	 (lambda (name)
	   (or (assq name bound)
	       (assq name free)
	       (let ((binding (parent-lookup name)))
		 (if binding (set! free (cons binding free)))
		 binding))))
       (make-name-generator)
       (lambda (name item)
	 (cond ((assq name bound)
		=>
		(lambda (association)
		  (if (and (reserved-name-item? (cdr association))
			   (not (reserved-name-item? item)))
		      (set-cdr! association item)
		      (impl-error "can't redefine name; already bound" name))))
	       ((assq name free)
		(if (reserved-name-item? item)
		    (syntax-error (item/history item)
				  "premature reference to reserved name"
				  name)
		    (impl-error "can't define name; already free" name)))
	       (else
		(set! bound (cons (cons name item) bound)))))
       (lambda ()
	 (alist-copy bound)))))

  (define (filter-syntactic-environment names names-env else-env)
    (if (or (null? names)
	    (eq? names-env else-env))
	else-env
	(let ((make-operation
	       (lambda (get-operation)
		 (let ((names-operation (get-operation names-env))
		       (else-operation (get-operation else-env)))
		   (lambda (name)
		     ((if (memq name names) names-operation else-operation)
		      name))))))
	  (make-syntactic-environment
	   else-env
	   (make-operation syntactic-environment/lookup-operation)
	   (make-operation syntactic-environment/rename-operation)
	   (lambda (name item)
	     (impl-error "can't bind name in filtered syntactic environment"
			 name item))
	   (lambda ()
	     (map (lambda (name)
		    (cons name
			  (syntactic-environment/lookup names-env name)))
		  names))))))

;;;  )

;;;; Items

;;;(module ((exports
;;;	  (#f item?
;;;	      item/history
;;;	      item/new-history
;;;	      item=?
;;;	      item-constructor
;;;	      item-predicate
;;;	      item-accessor)))

  (define item-type
    (make-record-type "item" '(HISTORY RECORD)))

  (define make-item
    (record-constructor item-type '(HISTORY RECORD)))

  (define item?
    (record-predicate item-type))

  (define item/history
    (record-accessor item-type 'HISTORY))

  (define (item/new-history item history)
    (make-item history (item/record item)))

  (define item/record
    (record-accessor item-type 'RECORD))

  (define (item=? x y)
    (eq? (item/record x) (item/record y)))

  (define (item-constructor rtd fields)
    (let ((constructor (record-constructor rtd fields)))
      (lambda (history . arguments)
	(make-item history (apply constructor arguments)))))

  (define (item-predicate rtd)
    (let ((predicate (record-predicate rtd)))
      (lambda (item)
	(predicate (item/record item)))))

  (define (item-accessor rtd field)
    (let ((accessor (record-accessor rtd field)))
      (lambda (item)
	(accessor (item/record item)))))

;;;  )

;;; Reserved name items do not represent any form, but instead are
;;; used to reserve a particular name in a syntactic environment.  If
;;; the classifier refers to a reserved name, a syntax error is
;;; signalled.  This is used in the implementation of LETREC-SYNTAX
;;; to signal a meaningful error when one of the <init>s refers to
;;; one of the names being bound.
;;;(module ((exports (#f make-reserved-name-item reserved-name-item?)))

  (define reserved-name-item-type
    (make-record-type "reserved-name-item" '()))

  (define make-reserved-name-item
    (item-constructor reserved-name-item-type '()))

  (define reserved-name-item?
    (item-predicate reserved-name-item-type))

;;;  )

;;; Keyword items represent macro keywords.
;;;(module ((exports
;;;	  (#f make-keyword-item
;;;	      keyword-item?
;;;	      keyword-item/classifier)))

  (define keyword-item-type
    (make-record-type "keyword-item" '(CLASSIFIER)))

  (define make-keyword-item
    (let ((constructor (item-constructor keyword-item-type '(CLASSIFIER))))
      (lambda (classifier)
	(constructor #f classifier))))

  (define keyword-item?
    (item-predicate keyword-item-type))

  (define keyword-item/classifier
    (item-accessor keyword-item-type 'CLASSIFIER))

;;;  )

;;; Variable items represent run-time variables.
;;;(module ((exports
;;;	  (#f make-variable-item
;;;	      variable-item?
;;;	      variable-item/name)))

  (define variable-item-type
    (make-record-type "variable-item" '(NAME)))

  (define make-variable-item
    (let ((constructor (item-constructor variable-item-type '(NAME))))
      (lambda (name)
	(constructor #f name))))

  (define variable-item?
    (item-predicate variable-item-type))

  (define variable-item/name
    (item-accessor variable-item-type 'NAME))

;;;  )

;;; Expression items represent any kind of expression other than a
;;; run-time variable or a sequence.  The ANNOTATION field is used to
;;; make expression items that can appear in non-expression contexts
;;; (for example, this could be used in the implementation of SETF).
;;;(module ((exports
;;;	  (#f make-special-expression-item
;;;	      expression-item?
;;;	      expression-item/compiler
;;;	      expression-item/annotation)))

  (define expression-item-type
    (make-record-type "expression-item" '(COMPILER ANNOTATION)))

  (define make-special-expression-item
    (item-constructor expression-item-type '(COMPILER ANNOTATION)))

  (define expression-item?
    (item-predicate expression-item-type))

  (define expression-item/compiler
    (item-accessor expression-item-type 'COMPILER))

  (define expression-item/annotation
    (item-accessor expression-item-type 'ANNOTATION))

;;;  )

(define (make-expression-item history compiler)
  (make-special-expression-item history compiler #f))

;;; Body items represent sequences (e.g. BEGIN).
;;;(module ((exports
;;;	  (#f make-body-item
;;;	      body-item?
;;;	      body-item/components)))

  (define body-item-type
    (make-record-type "body-item" '(COMPONENTS)))

  (define make-body-item
    (item-constructor body-item-type '(COMPONENTS)))

  (define body-item?
    (item-predicate body-item-type))

  (define body-item/components
    (item-accessor body-item-type 'COMPONENTS))

;;;  )

;;; Definition items represent definitions, whether top-level or
;;; internal, keyword or variable.
;;;(module ((exports
;;;	  (#f make-definition-item definition-item? bind-definition-item!)))

  (define definition-item-type
    (make-record-type "definition-item" '(BINDING-THEORY NAME VALUE)))

  (define make-definition-item
    (item-constructor definition-item-type '(BINDING-THEORY NAME VALUE)))

  (define definition-item?
    (item-predicate definition-item-type))

  (define definition-item/binding-theory
    (item-accessor definition-item-type 'BINDING-THEORY))

  (define definition-item/name
    (item-accessor definition-item-type 'NAME))

  (define definition-item/value
    (item-accessor definition-item-type 'VALUE))

  (define (bind-definition-item! environment item)
    ((definition-item/binding-theory item)
     environment
     (definition-item/name item)
     (force (definition-item/value item))))

;;;  )

(define (syntactic-binding-theory environment name item)
  (if (or (keyword-item? item)
	  (variable-item? item))
      (begin
	(syntactic-environment/define! environment
				       name
				       (item/new-history item #f))
	#f)
      (let ((history (item/history item)))
	(syntax-error history
		      "syntactic binding value must be a keyword or a variable"
		      (history/original-form history)))))

(define (variable-binding-theory environment name item)
  ;; If ITEM isn't a valid expression, an error will be signalled by
  ;; COMPILE-ITEM/EXPRESSION later.
  (cons (bind-variable! environment name) item))

(define (overloaded-binding-theory environment name item)
  (if (keyword-item? item)
      (begin
	(syntactic-environment/define! environment
				       name
				       (item/new-history item #f))
	#f)
      (cons (bind-variable! environment name) item)))

;;;; Expansion History
;;;  This records each step of the expansion process, separating it
;;;  into subproblems (really, subforms) and reductions.  The history
;;;  is attached to the items that are the result of classification,
;;;  so that meaningful debugging information is available after
;;;  classification has been performed.  The history is NOT preserved
;;;  by the compilation process, although it might be useful to
;;;  extract a small part of the recorded information and store it in
;;;  the output (for example, keeping track of what input form each
;;;  output form corresponds to).

;;;  Note: this abstraction could be implemented in a much simpler
;;;  way, to reduce memory usage.  A history need not remember
;;;  anything other than the original-form for the current reduction,
;;;  plus a bit saying whether that original-form is also the current
;;;  one (for replace-reduction).

(define (make-top-level-history forms environment)
  (list (list (cons forms environment))))

(define (history/add-reduction form environment history)
  (cons (cons (cons form environment)
	      (car history))
	(cdr history)))

(define (history/replace-reduction form environment history)
  ;; This is like ADD-REDUCTION, but it discards the current reduction
  ;; before adding a new one.  This is used when the current reduction
  ;; is not interesting, such as when reducing a syntactic closure.
  (cons (cons (cons form environment)
	      (cdar history))
	(cdr history)))

(define (history/add-subproblem form environment history selector)
  (cons (list (cons form environment))
	(cons (cons selector (car history))
	      (cdr history))))

(define (history/original-form history)
  (caar (last-pair (car history))))

;;;; Selectors
;;;  These are used by the expansion history to record subproblem
;;;  nesting so that debugging tools can show that nesting usefully.
;;;  By using abstract selectors, it is possible to locate the cell
;;;  that holds the pointer to a given subform.

(define (selector/apply selector object)
  (if (null? selector)
      object
      (selector/apply (cdr selector)
		      (if (>= (car selector) 0)
			  (list-ref object (car selector))
			  (list-tail object (- (car selector)))))))

(define (selector/add-car selector)
  (if (or (null? selector) (>= (car selector) 0))
      (cons 0 selector)
      (cons (- (car selector)) (cdr selector))))

(define (selector/add-cdr selector)
  (if (or (null? selector) (>= (car selector) 0))
      (cons -1 selector)
      (cons (- (car selector) 1) (cdr selector))))

(define select-object '())
(define select-car (selector/add-car select-object))
(define select-cdr (selector/add-cdr select-object))
(define select-caar (selector/add-car select-car))
(define select-cadr (selector/add-car select-cdr))
(define select-cdar (selector/add-cdr select-car))
(define select-cddr (selector/add-cdr select-cdr))
(define select-caaar (selector/add-car select-caar))
(define select-caadr (selector/add-car select-cadr))
(define select-cadar (selector/add-car select-cdar))
(define select-caddr (selector/add-car select-cddr))
(define select-cdaar (selector/add-cdr select-caar))
(define select-cdadr (selector/add-cdr select-cadr))
(define select-cddar (selector/add-cdr select-cdar))
(define select-cdddr (selector/add-cdr select-cddr))
(define select-caaaar (selector/add-car select-caaar))
(define select-caaadr (selector/add-car select-caadr))
(define select-caadar (selector/add-car select-cadar))
(define select-caaddr (selector/add-car select-caddr))
(define select-cadaar (selector/add-car select-cdaar))
(define select-cadadr (selector/add-car select-cdadr))
(define select-caddar (selector/add-car select-cddar))
(define select-cadddr (selector/add-car select-cdddr))
(define select-cdaaar (selector/add-cdr select-caaar))
(define select-cdaadr (selector/add-cdr select-caadr))
(define select-cdadar (selector/add-cdr select-cadar))
(define select-cdaddr (selector/add-cdr select-caddr))
(define select-cddaar (selector/add-cdr select-cdaar))
(define select-cddadr (selector/add-cdr select-cdadr))
(define select-cdddar (selector/add-cdr select-cddar))
(define select-cddddr (selector/add-cdr select-cdddr))

(define (selector/add-cadr selector)
  (selector/add-car (selector/add-cdr selector)))

(define (select-map procedure items selector)
  (let loop ((items items) (selector selector))
    (if (null? items)
	'()
	(cons (procedure (car items) (selector/add-car selector))
	      (loop (cdr items) (selector/add-cdr selector))))))

(define (select-for-each procedure items selector)
  (let loop ((items items) (selector selector))
    (if (not (null? items))
	(begin
	  (procedure (car items) (selector/add-car selector))
	  (loop (cdr items) (selector/add-cdr selector))))))

;;;; Classifiers, Compilers, Expanders

(define (sc-expander->classifier expander keyword-environment)
  (lambda (form environment definition-environment history)
    (let ((form (expander form environment)))
      (classify/form form
		     keyword-environment
		     definition-environment
		     (history/add-reduction form
					    keyword-environment
					    history)))))

(define (er-expander->classifier expander keyword-environment)
  (sc-expander->classifier (er->sc-expander expander) keyword-environment))

(define (er->sc-expander expander)
  (lambda (form environment)
    (capture-syntactic-environment
     (lambda (keyword-environment)
       (make-syntactic-closure environment '()
	 (expander form
		   (let ((renames '()))
		     (lambda (identifier)
		       (let ((association (assq identifier renames)))
			 (if association
			     (cdr association)
			     (let ((rename
				    (make-syntactic-closure keyword-environment
					'()
				      identifier)))
			       (set! renames
				     (cons (cons identifier rename)
					   renames))
			       rename)))))
		   (lambda (x y)
		     (identifier=? environment x
				   environment y))))))))

(define (classifier->keyword classifier)
  (make-syntactic-closure
      (let ((environment
	     (internal-syntactic-environment null-syntactic-environment)))
	(syntactic-environment/define! environment
				       'KEYWORD
				       (make-keyword-item classifier))
	environment)
      '()
    'KEYWORD))

(define (classifier->form classifier)
  `(,(classifier->keyword classifier)))

(define (compiler->form compiler)
  (classifier->form (compiler->classifier compiler)))

(define (compiler->classifier compiler)
  (lambda (form environment definition-environment history)
    definition-environment		;ignore
    (make-expression-item
     history
     (lambda () (compiler form environment history)))))

;;;; Macrologies
;;;  A macrology is a procedure that accepts a syntactic environment
;;;  as an argument, producing a new syntactic environment that is an
;;;  extension of the argument.

(define (make-primitive-macrology generate-definitions)
  (lambda (base-environment)
    (let ((environment (top-level-syntactic-environment base-environment)))
      (let ((define-classifier
	      (lambda (keyword classifier)
		(syntactic-environment/define!
		 environment
		 keyword
		 (make-keyword-item classifier)))))
	(generate-definitions
	 define-classifier
	 (lambda (keyword compiler)
	   (define-classifier keyword (compiler->classifier compiler)))))
      environment)))

(define (make-expander-macrology object->classifier generate-definitions)
  (lambda (base-environment)
    (let ((environment (top-level-syntactic-environment base-environment)))
      (generate-definitions
       (lambda (keyword object)
	 (syntactic-environment/define!
	  environment
	  keyword
	  (make-keyword-item (object->classifier object environment))))
       base-environment)
      environment)))

(define (make-sc-expander-macrology generate-definitions)
  (make-expander-macrology sc-expander->classifier generate-definitions))

(define (make-er-expander-macrology generate-definitions)
  (make-expander-macrology er-expander->classifier generate-definitions))

(define (compose-macrologies . macrologies)
  (lambda (environment)
    (do ((macrologies macrologies (cdr macrologies))
	 (environment environment ((car macrologies) environment)))
	((null? macrologies) environment))))

;;;; Utilities

(define (bind-variable! environment name)
  (let ((rename (syntactic-environment/rename environment name)))
    (syntactic-environment/define! environment
				   name
				   (make-variable-item rename))
    rename))

(define (reserve-names! names environment history)
  (let ((item (make-reserved-name-item history)))
    (for-each (lambda (name)
		(syntactic-environment/define! environment name item))
	      names)))

(define (capture-syntactic-environment expander)
  (classifier->form
   (lambda (form environment definition-environment history)
     form				;ignore
     (let ((form (expander environment)))
       (classify/form form
		      environment
		      definition-environment
		      (history/replace-reduction form
						 environment
						 history))))))

(define (capture-expansion-history expander)
  (classifier->form
   (lambda (form environment definition-environment history)
     form				;ignore
     (let ((form (expander history)))
       (classify/form form
		      environment
		      definition-environment
		      (history/replace-reduction form
						 environment
						 history))))))

(define (call-with-syntax-error-procedure expander)
  (capture-expansion-history
   (lambda (history)
     (expander
      (lambda (message irritant)
	(syntax-error history message irritant))))))

(define (unspecific-expression)
  (compiler->form
   (lambda (form environment history)
     form environment history		;ignore
     (output/unspecific))))

(define (unassigned-expression)
  (compiler->form
   (lambda (form environment history)
     form environment history		;ignore
     (output/unassigned))))

(define (syntax-quote expression)
  `(,(classifier->keyword
      (lambda (form environment definition-environment history)
	environment definition-environment ;ignore
	(syntax-check '(KEYWORD DATUM) form history)
	(output/constant (cadr form))))
    ,expression))

(define (flatten-body-items items)
  (append-map item->list items))

(define (item->list item)
  (if (body-item? item)
      (flatten-body-items (body-item/components item))
      (list item)))

(define (output/let names values body)
  (if (null? names)
      body
      (output/combination (output/lambda names body) values)))

(define (output/letrec names values body)
  (if (null? names)
      body
      (output/let names
		  (map (lambda (name) name (output/unassigned)) names)
		  (output/sequence
		   (list (if (null? (cdr names))
			     (output/assignment (car names) (car values))
			     (let ((temps (map (make-name-generator) names)))
			       (output/let
				temps
				values
				(output/sequence
				 (map output/assignment names temps)))))
			 body)))))

(define (output/top-level-sequence expressions)
  (if (null? expressions)
      (output/unspecific)
      (output/sequence expressions)))