;;; -*-Scheme-*-
;;;
;;; Copyright (c) 1991 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; R4RS Evaluator

;;; (module ((exports (#f transformer-eval)))

;;; EVALUATION-ERROR is dependent on the host implementation.

(define (evaluation-error message irritant)
  (error message irritant))

(define (transformer-eval exp env)
  ;; The evaluator assumes the input expression is well-formed,
  ;; because it is output from the macro expander which does that
  ;; checking for us.  We get alot of leverage by using the host
  ;; system's data types, particularly procedures and promises.
  (let new-exp ((exp exp))
    (if (symbol? exp)
	(environment-lookup env exp)
	(case (car exp)
	  ((QUOTE)
	   (cadr exp))
	  ((SET!)
	   (environment-assign! env (cadr exp) (new-exp (caddr exp))))
	  ((IF)
	   (new-exp (if (new-exp (cadr exp)) (caddr exp) (cadddr exp))))
	  ((BEGIN)
	   (let loop ((exps (cdr exp)))
	     (if (null? (cdr exps))
		 (new-exp (car exps))
		 (begin
		   (new-exp (car exps))
		   (loop (cdr exps))))))
	  ((LAMBDA)
	   (lambda args
	     (transformer-eval (caddr exp)
			       (environment-extend env (cadr exp) args))))
	  ((DELAY)
	   (delay (new-exp (cadr exp))))
	  ((DEFINE)
	   (evaluation-error "Definitions not supported:" (cadr exp)))
	  (else
	   (apply (new-exp (car exp)) (map new-exp (cdr exp))))))))

(define (environment-lookup environment name)
  (let loop ((environment environment))
    (if (null? (cdr environment))
	;; Global environment is a pair of vectors.  Vector in car is
	;; a vector of names sorted, vector in cdr is corresponding
	;; values.  Use binary search for fast lookup.
	(let ((names (caar environment))
	      (string (symbol->string name)))
	  (let loop ((low 0) (high (- (vector-length names) 1)))
	    (if (< high low)
		(evaluation-error "Unbound variable:" name)
		(let ((index (quotient (+ high low) 2)))
		  (let ((name* (vector-ref names index)))
		    (cond ((eq? name name*)
			   (vector-ref (cdar environment) index))
			  ((string-ci<? string (symbol->string name*))
			   (loop low (- index 1)))
			  (else
			   (loop (+ index 1) high))))))))
	(let ((association (assq name (car environment))))
	  (if association
	      (cdr association)
	      (loop (cdr environment)))))))

(define (environment-assign! environment name value)
  (let loop ((environment environment))
    (if (null? (cdr environment))
	(evaluation-error "Can't assign global variable:" name)
	(let ((association (assq name (car environment))))
	  (if association
	      (set-cdr! association value)
	      (loop (cdr environment)))))))

(define (environment-extend environment lambda-list arguments)
  (cons (let ((wrong-number-of-arguments
	       (lambda ()
		 (evaluation-error
		  "Wrong number of arguments (given/expected):"
		  (list (length arguments) (length lambda-list))))))
	  (let loop ((lambda-list lambda-list) (arguments arguments))
	    (cond ((null? lambda-list)
		   (if (null? arguments)
		       '()
		       (wrong-number-of-arguments)))
		  ((pair? lambda-list)
		   (if (null? arguments)
		       (wrong-number-of-arguments)
		       (cons (cons (car lambda-list) (car arguments))
			     (loop (cdr lambda-list) (cdr arguments)))))
		  (else
		   (list (cons lambda-list arguments))))))
	environment))

;;; end module
;;;)