;\c	    Copyright (C) 1990 Pertti Kellomaki
;\c	 
;\c	 This file is part of Taurus, a parser generator producing Scheme
;\c	 
;\c	 Taurus is free software; you can redistribute it and/or modify
;\c	 it under the terms of the GNU General Public License as published by
;\c	 the Free Software Foundation; either version 1, or (at your option)
;\c	 any later version.
;\c	 
;\c	 Taurus is distributed in the hope that it will be useful,
;\c	 but WITHOUT ANY WARRANTY; without even the implied warranty of
;\c	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;\c	 GNU General Public License for more details.
;\c	 
;\c	 You should have received a copy of the GNU General Public License
;\c	 along with Taurus; see the file COPYING.  If not, write to
;\c	 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;
;\node Miscellaneous Procedures, , , 
;\comment  node-name,  next,  previous,  up
;\chapter{Miscellaneous Procedures}
;
;These are generally useful auxiliary procedures that are used in many
;places throughout the generator. 
;\c Many of them are procedures that
;\c would be provided directly by a larger language, for example Common
;\c Lisp. 
;Your implementation may include them as primitives, so some
;speedup might be achieved by substituting them here.
;
;These procedures form the module
;
(module tmisc)

;\node set-difference
;\comment  node-name,  next,  previous,  up
;\section{set-difference}
;\c\section{\code{set-difference} }
;
;\code{set-difference} returns \code{set} with all elements matching \code{item} removed.
;
;
;\findex{set-difference}
(define (set-difference item set)
  (cond
    ((null? set) '())
    ((equal? (car set) item)
     (cdr set))
    (else (cons (car set)
                (set-difference item (cdr set))))))

    

;\node set-union
;\comment  node-name,  next,  previous,  up
;\section{union}
;\c\section{\code{union}}
;
;\code{set-union} returns the union of two sets represented as lists.
;
;    
;\findex{set-union}
(define (set-union set1 set2)
  (cond 
   ((null? set1) set2)
   ((member (car set1) set2)
    (set-union (cdr set1) set2))
   (else (cons (car set1)
	       (set-union (cdr set1) set2)))))


;\node set-intersection
;\comment  node-name,  next,  previous,  up
;\section{intersection}
;\c\section{\code{intersection}}
;
;\code{set-intersection} returns the intersection of two sets
;represented as lists.
;
;    
;\findex{set-intersection}
(define (set-intersection set1 set2)
  (cond 
   ((null? set1) '())
   ((member (car set1) set2)
    (cons (car set1)
	  (set-intersection (cdr set1) set2)))
   (else (set-intersection (cdr set1) set2))))

;\node set-adjoin
;\comment  node-name,  next,  previous,  up
;\section{set-adjoin}
;\c\section{\code{set-adjoin}}
;
;\code{set-adjoin} returns a set that includes all the elements of
;\code{set} plus \code{item}.
;
;\findex{set-adjoin}
(define (set-adjoin item set)
  (if (member item set)
      set
      (cons item set)))


;\node up-to
;\comment  node-name,  next,  previous,  up
;\section{up-to}
;\c\section{\code{up-to}}
;
;\code{up-to} returns the head of \code{lyst} up to the first occurence of
;\code{item}. 
;
;
;\findex{up-to}
(define (up-to item lyst)
  (cond ((null? lyst) '())
        ((equal? (car lyst) item) '())
        (else
          (cons (car lyst)
                (up-to item (cdr lyst))))))


;\node rest-from
;\comment  node-name,  next,  previous,  up
;\section{rest-from}
;\c\section{\code{rest-from}}
;
;\code{rest-from} returns the tail of \code{lyst} starting from the
;first occurence of \code{item}
;
;
;
;\findex{rest-from}
(define (rest-from item lyst)
      (cdr (member item lyst)))


;\node , , , 
;\comment  node-name,  next,  previous,  up
;\section remove-duplicates
;
;\code{remove-duplicates} removes all but the last occurence of each
;list item.
;
;\findex{remove-duplicates}
(define (remove-duplicates lst)
  (cond ((null? lst) '())
	((member (car lst) (cdr lst))
	 (remove-duplicates (cdr lst)))
	(else
	 (cons (car lst)
	       (remove-duplicates (cdr lst))))))
  


;\node grammar-rule-for, , , 
;\comment  node-name,  next,  previous,  up
;\section{grammar-rule-for}
;\c\section{\code{grammar-rule-for}}
;
;\code{grammar-rule-for} returns the grammar rule for the given nonterminal.
;
;\findex{grammar-rule-for}
(define (grammar-rule-for nonterminal grammar)
  (let loop ((grammar grammar))
    (cond ((null? grammar)
	   (taurus-error "grammar-rule-for: no rule for "
			 nonterminal))
	  ((equal? nonterminal
		   (rule-nonterminal (car grammar)))
	   (car grammar))
	  (else (loop (cdr grammar))))))

;\node error
;\comment  node-name,  next,  previous,  up
;\section{error}
;
;The \code{error} procedure of \file{scc} differs from that described
;in R3RS. If you plan to run Taurus in some other Scheme, you could
;just define \code{taurus-error} as \code{error}.
;
;\findex{taurus-error}
(define (taurus-error . args)
  (let loop ((args args))
    (cond ((null? args)
	   (error 'taurus-error "~%"))
	  (else
	   (format #t "~S" (car args))
	   (loop (cdr args))))))
