;;;; -*- Scheme -*-
;;;; $Id: avl-tree.scm,v 1.4 1992/01/26 13:24:56 bevan Exp $

;;;+file-overview
;;;
;;; An AVL tree.
;;; Depending on your application, this may be too inefficient as it
;;; does not update the tree in place, rather it re-generates the tree
;;; when changes are made.
;;;
;;; If you'd like to change this to use in-place updates, then copy
;;; the existing functions and rename them to contain ! in the name
;;; e.g. avl-tree:add! would insert and element in place.
;;;
;;; Note this does not contain a delete function.  There are two
;;; reasons for this.
;;;
;;; 1. The application for which this AVL code is used, never deletes
;;;    elements from the tree.
;;;
;;; 2. I don't know how to delete an item!  I haven't seen a
;;;    functional delete yet, and I'm not prepared to wade through the
;;;    state changing versions I've seen in various books.
;;;
;;; Examples of how to use the AVL code is given at the end of the file.
;;;
;;;-file-overview
;;;+copyright/warranty
;;;
;;; Copyright (c) 1991 Department of Computer Science, University of Manchester
;;; All rights reserved.
;;;
;;;-copyright/warranty
;;;+history
;;;
;;; This is based on the add + combine functions given in :-
;;;
;;;   Prolog programming for Artificial Intelligence
;;;   Ivan Bratko
;;;   ISBN 0-201-14224-4
;;;
;;; (Note this is the first edtion, apparently there is a second
;;; edition available now)
;;;
;;; Any changes are to make it use a similar style of interface as that
;;; of btree by Chris Hanson <cph@altdorf.ai.mit.edu>
;;;
;;; Stephen J. Bevan <bevan@cs.man.ac.uk> 910527
;;;
;;;-history
;;;+usage
;;;
;;; Most of the functions take as arguments a comparison function and
;;; a function to unwrap a key from an element.
;;;
;;; The comparison function
;;; -----------------------
;;; This should satisfy the following condition :-
;;;
;;; For all a, b in Elements and c in (ElementxElement -> Bool)
;;;   (not (and (c a b) (c b a)))
;;;
;;; Thus, for example, the following cannot be used as comparisons as
;;; they fail the test :-
;;;
;;;   =   string=?   char=?   string<=?   >=
;;;
;;; whereas the following can be used :-
;;;
;;;   <   string<?   char>? 
;;;
;;; The key unwrapper
;;; -----------------
;;; This is used to extract the key part from a structured element.
;;; For example, consider using an AVL tree to represent a finite map
;;; of strings to integers, say as a telephone directory (oh no, not
;;; another telephone directory example :-)
;;;
;;; The data is of the form :-
;;;
;;; ("bevan" . 6270) ("alan" . 6289) ("nick" . 6171)
;;;
;;; To store this in the tree, the name of the person would be used as
;;; the key.  Therefore, there has to be a way of extracting the name
;;; from each element.  This is what the unwrap-key function is for.
;;; This takes an element and returns the part that should be used for
;;; the comparison.  For example :-
;;;
;;; (list->avl-tree '(("bevan" . 6270) ("alan" . 6289) ("nick" . 6171))
;;;                 (lambda (element) (car element))
;;;                 string<?)
;;; 
;;; builds an avl tree corresponding the the given list.
;;; Here, the unwrap function just returns the first item in the data
;;; i.e. the name of the person.
;;;
;;; If the elements you want to store don't have a key or they aren't
;;; structured at all, then just use the identity function.  For
;;; example :-
;;;
;;; (list->avl-tree '(2 3 8 9 0 4 5) (lambda (id) id) <)
;;;
;;;-usage

;;; The AVL tree is represented as a vector of the form
;;; (left data height right)
;;; If you want to be extra safe, of if you are having some problems
;;; with the AVL tree, then uncomment the lines below.  It will
;;; provide better error trapping.  This of course assumes your Scheme
;;; has an error function.

(define (avl-tree::left avl-tree)
;  (if (not (vector? avl-tree)) (error 'avl-tree::left "oops"))
  (vector-ref avl-tree 0))


(define (avl-tree::data avl-tree)
;  (if (not (vector? avl-tree)) (error 'avl-tree::data "oops"))
  (vector-ref avl-tree 1))


(define (avl-tree::height avl-tree)
  (if (avl-tree:nil? avl-tree) 0 (vector-ref avl-tree 2)))


(define (avl-tree::right avl-tree)
;  (if (not (vector? avl-tree)) (error 'avl-tree::right "oops"))
  (vector-ref avl-tree 3))



;;; Make a new tree containing the given elements.
;;;
(define (avl-tree::make left data height right)
  (vector left data height right))



;;;+vs
;;;
;;; An empty AVL tree.
;;;
;;;-vs
;;;
(define avl-tree:nil '())



;;;+fs
;;;
;;; Returns true if the `avl-tree' is empty i.e. does not contain any
;;; data.
;;;
;;;-fs
;;
(define (avl-tree:nil? avl-tree)
  (null? avl-tree))



;;;+fs
;;;
;;; Add the `element' to the `tree'.
;;;
;;; `if-found' -  (lambda (tree element) ...)
;;;
;;;    a function taking two arguments, an avl-tree and an element.
;;;    This is called if a node containing an element with the same key as the
;;;    `element' was found in the tree.  
;;;    The element passed to it is the element found with the same key.
;;;    It should return an avl-tree.
;;;
;;; `if-not-found' - (lambda (element) ...)
;;;
;;;    a function taking one argument, the element to add.
;;;    This is called if the `element' is not already in the
;;;    tree.  You can use this to do various things, like giving an error
;;;    if an attempt is made to add an element that is not already in
;;;    the tree.  If all you want to do is add the element, then make
;;;    this the identity function.
;;;
;;;-fs

(define avl-tree:add
  (lambda (tree element unwrap-key comparison? if-found if-not-found)
    (let ((unwrapped-key (unwrap-key element)))
      (let loop ((tree tree))
	(if (avl-tree:nil? tree)
	    (avl-tree::make avl-tree:nil (if-not-found element) 1 avl-tree:nil)
	    (let ((current-key (unwrap-key (avl-tree::data tree))))
	      (cond ((comparison? current-key unwrapped-key)
		     (let ((newNode (loop (avl-tree::right tree))))
		       (avl-tree::combine
			(avl-tree::left tree)
			(avl-tree::data tree)
			(avl-tree::left newNode)
			(avl-tree::data newNode)
			(avl-tree::right newNode))))
		    ((comparison? unwrapped-key current-key)
		     (let ((newNode (loop (avl-tree::left tree))))
		       (avl-tree::combine
			(avl-tree::left newNode)
			(avl-tree::data newNode)
			(avl-tree::right newNode)
			(avl-tree::data tree)
			(avl-tree::right tree))))
		    (else
		     (if-found tree element)))))))))



;;; Combines the three trees and the items `a' and `c' and produces
;;; a new balanced tree containing them.

(define avl-tree::combine
  (lambda (left-tree a middle-tree c right-tree)
    (let ((left-tree-height (avl-tree::height left-tree))
	  (middle-tree-height (avl-tree::height middle-tree))
	  (right-tree-height (avl-tree::height right-tree)))
      (cond
       ((and (> middle-tree-height left-tree-height)
	     (> middle-tree-height right-tree-height))
	 (avl-tree::make
	   (avl-tree::make
	     left-tree
	     a
	     (+ 1 left-tree-height)
	     (avl-tree::left middle-tree))
	   (avl-tree::data middle-tree)
	   (+ 2 left-tree-height)
	   (avl-tree::make
	     (avl-tree::right middle-tree)
	     c
	     (+ 1 right-tree-height)
	     right-tree)))
       ((and (>= left-tree-height middle-tree-height)
	     (>= left-tree-height right-tree-height))
	 (let* ((middle-right-max
		  (avl-tree::max middle-tree-height right-tree-height))
		(left-middle-right-max
		  (avl-tree::max middle-right-max left-tree-height)))
	   (avl-tree::make
	     left-tree
	     a
	     left-middle-right-max
	     (avl-tree::make middle-tree c middle-right-max right-tree))))
       (else
	 (let* ((left-middle-max
		 (avl-tree::max left-tree-height middle-tree-height))
		(left-middle-right-max
		 (avl-tree::max left-middle-max right-tree-height)))
	   (avl-tree::make
	     (avl-tree::make left-tree a left-middle-max middle-tree)
	     c
	     left-middle-right-max
	     right-tree)))))))



;;;+fs
;;;
;;; Insert `element' into the `tree'.
;;; If an element with the same key as `element' is in the
;;; `tree', the new one is still added, without altering the
;;; existing value  
;;;
;;;-fs

(define avl-tree:insert-allowing-duplicates
  (lambda (tree element unwrap-key comparison?)
    (avl-tree:add
      tree
      element
      unwrap-key
      comparison?
      (lambda (previous-node new-element)
	(let ((new-node 
	       (avl-tree:insert-allowing-duplicates
		 (avl-tree::left previous-node)
		 new-element
		 unwrap-key
		 comparison?)))
	  (avl-tree::combine
	    (avl-tree::left new-node)
	    (avl-tree::data new-node)
	    (avl-tree::right new-node)
	    (avl-tree::data previous-node)
	    (avl-tree::right previous-node))))
      (lambda (id) id))))



;;;+fs
;;;
;;; Insert `element' into the `tree'.
;;; If an element with the same key as `element' is in the
;;; `tree', the function `merge' is called.  This should take two
;;; arguments the old value and the new element to add.  It should
;;; return a new data value that is in some way a merge of these two.
;;; (Note there is not need to merge them really, you could just
;;; discard one and return the other for example)
;;;
;;;-fs

(define avl-tree:insert-with-merge
  (lambda (tree element unwrap-key comparison? merge)
    (avl-tree:add
     tree
     element
     unwrap-key
     comparison?
     (lambda (node-with-same-key element-trying-to-add)
       (avl-tree::make
	 (avl-tree::left node-with-same-key)
	 (merge (avl-tree::data node-with-same-key) element-trying-to-add)
	 (avl-tree::height node-with-same-key)
	 (avl-tree::right node-with-same-key)))
     (lambda (id) id))))



;;;+fs
;;;
;;; Insert `element' into the `tree'.
;;; If an element with the same key as `element' is in the
;;; `tree', it is replaced by `element' (Note. this is not a
;;; destructive overwrite).
;;;
;;;-fs

(define avl-tree:insert-with-overwrite
  (lambda (tree element unwrap-key comparison?)
    (avl-tree:insert-with-merge
      tree
      element
      unwrap-key
      comparison?
      (lambda (old-element new-element) new-element))))



;;;+fs
;;;
;;; Look for the element with the given `key' in the `tree'.
;;;
;;; `if-found' - (lambda (element-found) ...)
;;;
;;;    a function taking one argument, the element that has the given
;;;    `key'. 
;;;
;;; `if-not-found' - (lambda (key-not-found) ...)
;;;
;;;    a function taking one argument, the `key' that couldn't be found.
;;;
;;;-fs

(define (avl-tree:find tree key unwrap-key comparison? if-found if-not-found)
  (let loop ((tree tree))
    (if (avl-tree:nil? tree)
	(if-not-found key)
	(let ((current-key (unwrap-key (avl-tree::data tree))))
	  (cond ((comparison? key current-key)
		  (loop (avl-tree::left tree)))
		((comparison? current-key key)
		  (loop (avl-tree::right tree)))
		(else
		  (if-found (avl-tree::data tree))))))))



;;;+fs
;;;
;;; Look for the element with the given `key' in the `tree'
;;; If an element with the given key cannot be found, returns #f,
;;; otherwise returns the element.
;;;
;;;-fs

(define (avl-tree:lookup tree key unwrap-key comparison?)
  (avl-tree:find
    tree
    key
    unwrap-key
    comparison?
    (lambda (element-matching-key) element-matching-key)
    (lambda (key-to-not-existent-element) #f)))



(define (avl-tree::max a b)
  (+ (if (> a b) a b) 1))



;;;+fs
;;;
;;; Iterate over the `tree' using an in-order traversal.
;;;
;;; `collector' 
;;;
;;;    (lambda (left-result current-element right-function) ...)
;;;
;;;   `left-result' is the result of processing all the nodes to the
;;;       left of `current-element'
;;;
;;;   `current-element' is the current element to be processed.
;;;
;;;   `right-function' - (lambda (current-result) ...)
;;;
;;;       A function that takes a result of processing the current
;;;       element and then goes onto processing the next element.
;;;
;;; `initial-result' should be the value to use to start the
;;;     collection.
;;;
;;;- +fe
;;;
;;; The following returns a list of all the elements in the tree :-
;;;
;;; > (avl-tree:for-each-in-order-do
;;; >   tree
;;; >   (lambda (left element right)
;;; >     (right (append left (list element))))
;;; >   '())
;;;
;;; This prints out the first 10 elements in the tree.
;;;
;;; > (avl-tree:for-each-in-order-do
;;; >   tree
;;; >   (lambda (count element right)
;;; >   (if (< count 10)
;;; >       (begin (display element) (right (+ count 1)))
;;; >       count))
;;; >   0)
;;;-fe

(define avl-tree:for-each-in-order-do
  (lambda (tree collector initial-result)
    (let loop ((tree tree) (result initial-result))
      (if (avl-tree:nil? tree)
	  result
	  (let ((left-result (loop (avl-tree::left tree) result))
		(right-result (lambda (r) (loop (avl-tree::right tree) r))))
	    (collector left-result (avl-tree::data tree) right-result))))))



;;;+fs
;;;
;;; Apply the `action' to each element in the `tree'
;;;
;;; `action' - (lambda (element) ...)
;;;
;;; Returns : undefined
;;;
;;;-fs

(define (avl-tree:for-each tree action)
  (avl-tree:for-each-in-order-do
    tree
    (lambda (left current-element right)
      (right (action current-element)))
    'dummy-argument))



;;;+fs
;;;
;;; Build an AVL tree from the given `elements'
;;;
;;; `if-duplicate' - (lambda (old-node new-node) ...)
;;;
;;;    a function taking two arguments; a previously
;;;    defined node containing an element with the same key as the
;;;    current element being added.
;;;    This is called if an element with the same key as the
;;;    current element being added was found in the tree.  
;;;    It should return whichever node should actually be in the tree.
;;;
;;;-fs

(define (avl-tree:from-list list unwrap-key comparison? if-duplicate)
  (let loop ((new-tree avl-tree:nil)
	     (data list))
    (if (null? data)
	new-tree
	(loop (avl-tree:add
	        new-tree
		(car data)
		unwrap-key
		comparison?
		if-duplicate
		(lambda (id) id))
	      (cdr data)))))



;;;+fs
;;;
;;; Applies `action' to each element in order
;;; and returns the individual results as a list.
;;;
;;;-

(define avl-tree:map
  (lambda (tree action)
    (avl-tree:for-each-in-order-do
      tree
      (lambda (left element right)
	(right (append left (list (action element)))))
      '())))



;;;+fs
;;;
;;; Returns the number of elements in `tree'.
;;;
;;;-fs

(define avl-tree:size
  (lambda (tree)
    (avl-tree:for-each-in-order-do
      tree
      (lambda (left element right)
	(right (+ 1 left)))
      0)))
	       


;;;+fs
;;;
;;; Sorts the `listToSort' by converting it to an AVL tree and then
;;; converting it back to a list.  In theory I think this should be
;;; faster than a quicksort.  However, unless you have a zippy compiler,
;;; this will most probably be slower.
;;; Having said that, try it out and tell me how it compares.
;;;
;;;-

(define (avl-tree:sort-list list unwrap-key comparison?)
  (avl-tree->list
    (avl-tree:from-list
      list
      unwrap-key
      comparison?
      (lambda (previous-node new-element)
	(avl-tree:insert-allowing-duplicates
	  previous-node
	  new-element
	  unwrap-key
	  comparison?)))))



;;;+fs
;;;
;;; Convert a list into an avl tree.
;;; If the list contains any duplicate elements, the latter is stored
;;; in the tree.
;;;
;;;-fs

(define (list->avl-tree elements unwrap-key comparison?)
  (avl-tree:from-list
    elements
    unwrap-key
    comparison?
    (lambda (previous-element current-element)
      (avl-tree::make
        (avl-tree::left previous-element)
	current-element
	(avl-tree::height previous-element)
	(avl-tree::right previous-element)))))



;;;+fs
;;;
;;; Convert the `tree' to a list using a in-order traversal of the tree.
;;;
;;;-fs

(define (avl-tree->list avl-tree)
  (avl-tree:map avl-tree (lambda (element) element)))



;;;+file-examples
;;;
;;; Some misc. examples of how to use the above code.

;;;+fs
;;;
;;; An example of how to use the `forEachInOrder' function.
;;; This attempts to `display' each element in the tree.
;;; Note depending on the element type this may or may not produce
;;; readable results.
;;;
;;;-fs
;;;+fe
;;;
;;; The following should display the numbers in the list in incresing
;;; numeric value
;;;
;;; > (avl-tree:test:display (list->avl-tree '(7 8 3 4) (lambda (id) id) <))
;;;
;;;-fe

(define (avl-tree:test:display avl-tree)
  (avl-tree:for-each avl-tree (lambda (element) (display element) (newline))))



;;; Sort a given list of integers by converting the list to a AVL tree
;;; and then flattening the tree back into a list.

(define (avl-tree:test:sort)
  (avl-tree:sort-list '(3 4 7 2 8 9 11 4 6 5 8 1 0 2 4 8) (lambda (id) id) <))



;;;+fs
;;;
;;; Creates a simple map using an AVL tree and then shows that
;;; inserting an element with the same key as existing data does
;;; not modify the original tree.
;;;
;;;-fs

(define (avl-tree:test:insertion)
  (let* ((before-tree
	   (list->avl-tree
	     '(("a" . 10) ("b" . 20) ("c" . 30))
	     (lambda (mapplet) (car mapplet))
	     string<?))
	 (before-list (avl-tree->list before-tree))
	 (after-tree
	   (avl-tree:insert-with-overwrite
	     before-tree
	     '("a" . 40)
	     (lambda (mapplet) (car mapplet))
	     string<?))
	 (afterList (avl-tree->list after-tree)))
    (display "Original data :-") (newline)
    (display before-list) (newline)
    (display "New data :-") (newline)
    (display afterList) (newline)
    (display "Original data (again) :-") (newline)
    (display (avl-tree->list before-tree)) (newline)))



(define avl-tree:test:add1
  (lambda ()
    (let loop ((data '(3 2 1 4 5 3 2 8)) (tree avl-tree:nil))
      (if (null? data)
	  (avl-tree->list tree)
	  (loop (cdr data)
		(avl-tree:add
		  tree
		  (car data)
		  (lambda (id) id)
		  <
		  (lambda (tree element)
		    (display element)
		    (display " is already in the tree")
		    (newline)
		    tree)
		  (lambda (element)
		    (display element)
		    (display " is not in the tree")
		    (newline)
		    element)))))))
      
;;;-file-examples
;;; eof (just for ROK)
