;;
;; Some utility procedures
;;

(module util)
(include "cw.sch")

(define (vector-copy! to at from)
  (define (iter addr n l)
	(if (= n l)
		'done
		(begin
		 (vector-set! to addr (vector-ref from n))
		 (iter (scale-to-core (inc addr)) (inc n) l))))
  (iter at 0 (vector-length from)))

(define (foreach-while l f)
  (if (null? l)
	  #t
	  (if (f (car l))
		  (foreach-while (cdr l) f)
		  #f)))

(define (foreach l f)
  (if (null? l)
	  'done
	  (begin
	   (f (car l))
	   (foreach (cdr l) f))))

;;
;; replace the nth thing in list l with x
;;
(define (replace-nth! l n x)
  (define (iter l i)
	(cond ((null? l)
	  	   (error "replace-nth!: list is not long enough:" l n x))
		  ((= i n)
		   (set-car! l x))
		  (else (iter (cdr l) (inc i)))))
	  (iter l 0))

;;
;; delete the nth thing in list l
;;
(define (delete-nth! l n)
  (define (iter l i)
    (cond ((null? l)
	   (error "delete-nth!: list is not long enough:" l n))
	  ((= i n)
	   (set-cdr! l (cddr l))
	   l)
	  (else (iter (cdr l) (inc i)))))
  (cond ((null? l) (error "delete-nth!: list is empty:" l n))
	((null? (cdr l))
	 (error "delete-nth!: can't delete from 1 element list:" l n))
	((= n 0) (set-car! l (cadr l)) (set-cdr! l (cddr l)))
	(else
	 (iter l 1))))

(define (delete-nth l n)
  (define (iter l i)
    (cond ((null? l)
	   (error "delete-nth: list is not long enough:" l n))
	  ((= i n)
	   (cdr l))
	  (else
	   (cons (car l) (iter (cdr l) (inc i))))))
  (iter l 0))

;;
;; delete all x's from l, maintains original order of l
;;
(define (delete x l)
  (define (iter l result)
	(cond ((null? l) result)
		  ((eq? (car l) x) (iter (cdr l) result))
		  (else (iter (cdr l) (cons (car l) result)))))
  (reverse (iter l '())))

(define-c-external (crandom int) longint "random")
(define-c-external (csrandom int) void "srandom")
(define-c-external (ctime int) longint "time")

(csrandom (remainder (ctime 0) 16384))

(define (random n)
(let ((r (crandom 0)))
  (let ((x (remainder r n)))
	(inexact->exact x))))

(define-c-external (fflush pointer) void "fflush")

(define (flush-output-port)
  (fflush (port->stdio-file stdout-port)))

(define-c-external (cclock int) longint "clock")
(define (clock) (/ (cclock 0) 1.e6))
