;;;;
;;;; $Id: environ.scm,v 1.8 1992/11/12 12:29:24 pk Exp $
;;;;
;;;; sclint -- a Scheme lint.
;;;;
;;;; Pertti Kellom\"aki, 1992
;;;;
;;;; $Log: environ.scm,v $
;;;; Revision 1.8  1992/11/12  12:29:24  pk
;;;; Variables now carry information about who uses and sets them.
;;;;
;;;; Revision 1.7  1992/10/30  13:11:48  pk
;;;; PC Scheme does not like make-environment as variable name. Changed to make-env.
;;;;
;;;; Revision 1.6  1992/10/30  07:34:33  pk
;;;; Added a new scope 'builtin, removed a bug affecting redefinition of a builtin.
;;;; Added warning of shadowing a builtin procedure.
;;;;
;;;; Revision 1.5  1992/10/23  13:35:09  pk
;;;; Variables that are defined as procedures now carry information about
;;;; the procedure.
;;;;
;;;; Revision 1.4  1992/10/22  12:58:25  pk
;;;; Added duplicate name detection.
;;;; Variables now store the original name as pexp, so that the location info can be accessed.
;;;;
;;;; Revision 1.3  1992/09/23  15:26:15  pk
;;;; Added sanity checking to make-variable: complains if name is not a symbol.
;;;;
;;;; Revision 1.2  1992/09/22  12:39:06  pk
;;;; Untabified because our reader does not yet understand tabs.
;;;;
;;;; Revision 1.1  1992/09/22  07:19:08  pk
;;;; Initial revision
;;;;
;;;;

;;;;
;;;; Handling of environment (symbol table) information.
;;;;

;;;
;;; make-env returns a new environment with the given
;;; variables. The environment is used for keeping track of procedure
;;; definitions, which varibles are referenced etc.
;;; 

(define (make-env parent variables)

  ;;
  ;; Look up a name in the current lexical scope. Return #f if name is
  ;; not defined.
  ;; 
  (define (lookup name)
    (let loop ((variables variables))
      (cond ((null? variables)
             (if (procedure? parent)
                 (parent 'lookup name)
                 #f))
            ((equal? (variable-name (car variables)) name)
             (car variables))
            (else
             (loop (cdr variables))))))
             

  (define (dispatch op . args)
    (case op
      ((lookup)
       (apply lookup args))
      (else
       (error "make-env: bad message :" op args))))

  (check-duplicate-variables variables)
  (check-builtin-shadowings variables)
  dispatch)

(define (lookup env name)
  (env 'lookup name))

;;;
;;; Variables. Scope must be either 'local, 'global or 'builtin
;;; The fields are:
;;;    name as Scheme symbol
;;;    scope
;;;    name as psd-symbol.    This includes the source file and line.
;;;    parameter info.        If the variable is a procedure, here is a list
;;;                           whose first element is the number of required arguments,
;;;                           and the second tells wether the procedure accepts a
;;;                           rest parameter.
;;;    references	      In what locations is this variable referenced.
;;;    setters		      In what locations is this variable assigned to.
;;;

(define (make-variable name scope)
  (if (psd-symbol? name)
      (if (member scope '(local global builtin))
	  (let ((var (make-vector 6 #f)))
	    (vector-set! var 0 (pexp->sexp name))
	    (vector-set! var 1 scope)
	    (vector-set! var 2 name)
	    var)
	  (error "make-variable: bad scope " scope))
      (error "make-variable: bad name " name)))
  

(define (variable-name var)
  (vector-ref var 0))

(define (variable-source-name var)
  (vector-ref var 2))

(define (variable-set-parameter-info! var info)
  (vector-set! var 3 info))

(define (variable-procedure? var)
  (vector-ref var 3))

(define (variable-parameter-count var)
  (if (vector-ref var 3)
      (car (vector-ref var 3))
      #f))

(define (variable-rest-parameter? var)
  (if (vector-ref var 3)
      (cadr (vector-ref var 3))
      #f))

(define (variable-global? var)
  (eq? (vector-ref var 1) 'global))

(define (variable-local? var)
  (eq? (vector-ref var 1) 'local))

(define (variable-builtin? var)
  (eq? (vector-ref var 1) 'builtin))

(define (variable-add-reference! var expr)
  (vector-set! var 4
	       (if (vector-ref var 4)
		   (cons expr (vector-ref var 4))
		   (list expr))))

(define (variable-add-setter! var expr)
  (vector-set! var 5
	       (if (vector-ref var 5)
		   (cons expr (vector-ref var 5))
		   (list expr))))

(define (variable-references var)
  (vector-ref var 4))

(define (variable-setters var)
  (vector-ref var 5))

;;;
;;; Check that there are no duplicate names. If a builtin is
;;; redefined, a special error message is issued. 
;;; 

(define (check-duplicate-variables variables)

  (define (duplicate? variable)
    (let loop ((this (variable-name variable))
	       (variables variables)
	       (already-seen? #f))
      (cond ((null? variables)
	     #f)
	    ((equal? this (variable-name (car variables)))
	     (if already-seen?
		 #t
		 (loop this
		       (cdr variables)
		       #t)))
	    (else
	     (loop this
		   (cdr variables)
		   already-seen?)))))
		       

  (let loop ((variables variables))
    (cond ((null? variables)
	   #f)
	  ((duplicate? (car variables))

	   (if (variable-builtin? (car variables))
	       (if (psd-expr-start (variable-source-name (car variables)))

		   ;; this is the actual redefinition
		   (warning (variable-source-name (car variables))
			    "Redefinition of builtin prodecure "
			    (variable-name (car variables)))

		   ;; this is the binding in the initial environment,
		   ;; do not report it
		   #f)

	       ;; this was a user defined procedure
	       (warning (variable-source-name (car variables))
			"Duplicate definition of "
			(variable-name (car variables))))

	   (loop (cdr variables)))
	  (else
	   (loop (cdr variables))))))
	  
;;;
;;; Check that the variables do not shadow any builtin procedures.
;;;

(define (check-builtin-shadowings variables)
  (map (lambda (var) (if (and (variable-local? var)
			      (member (variable-name var) r4rs-names))
			 (warning (variable-source-name var)
				  "This declaration shadows the builtin procedure "
				  (variable-name var))
			 #f))
       variables))
