Object subclass: #KnowledgeBaseCommentator
	instanceVariableNames: 'knowledgeBaseName globalComment objectComments slotComments '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
KnowledgeBaseCommentator comment:
'I encapsulate all comments for a particular knowledge base.

Instance Variables:
	knowledgeBaseName	<Symbol>		The knowledge base I am commenting
	globalComment			<RemoteString>	Comment for the knowledge base as a whole
	objectComments		<Dictionary>	Comments for individual objects
	slotComments			<Dictionary>	Comments for particular slots of classes'!


!KnowledgeBaseCommentator methodsFor: 'accessing'!

classNamed: className slot: slotName comment: aString
	"Set the comment for slot slotName of class className to be the argument, aString."
	| aStream key |
	slotComments == nil ifTrue: [slotComments _ Dictionary new].
	key _ (className, '>', slotName).
	aString size = 0
		ifTrue: 
			[slotComments removeKey: key ifAbsent: []]
		ifFalse: 
			["double internal quotes of the comment string"
			aStream _ WriteStream on: (String new: aString size).
			aStream nextPutAll: knowledgeBaseName , ' comments classNamed: ',
					className storeString, ' slot: ', slotName storeString, ' comment:'; cr.
			aString storeOn: aStream.
			slotComments
				at: key
				put: (RemoteString newString: aStream contents onFileNumber: 2)]!

commentForClass: className slot: slotName
	"Return the comment for slot slotName of class className."
	| remoteString key string |
	slotComments == nil ifTrue: [^ ''].
	key _ (className, '>', slotName).
	remoteString _ slotComments at: key ifAbsent: [^ ''].
	string _ remoteString string.
	"Skip beyond the slotName, since it will otherwise be mistaken for the comment."
	string _ string copyFrom: (string findString: 'comment:' startingAt: 1) to: string size.
	^ String readFromString: string!

commentForObjectAt: aSymbol
	"Return the comment for the object at aSymbol."
	| remoteString key |
	objectComments == nil ifTrue: [^ ''].
	remoteString _ objectComments at: aSymbol ifAbsent: [^ ''].
	^ String readFromString: remoteString string!

globalComment
	"Return the global comment for my KnowledgeBase."
	globalComment == nil ifTrue: [^ ''].
	^ String readFromString: globalComment string!

globalComment: aString
	"Set the global comment for my KnowledgeBase to be the argument, aString."
		| aStream |
	aString size = 0
		ifTrue: 
			[globalComment _ nil]
		ifFalse: 
			["double internal quotes of the comment string"
			aStream _ WriteStream on: (String new: aString size).
			aStream nextPutAll: knowledgeBaseName , ' comments globalComment:'; cr.
			aString storeOn: aStream.
			globalComment _ RemoteString newString: aStream contents onFileNumber: 2]!

knowledgeBaseName: aSymbol
	knowledgeBaseName _ aSymbol!

objectAt: aSymbol comment: aString
	"Set the comment for the object at aSymbol to be the argument, aString."
	| aStream |
	objectComments == nil ifTrue: [objectComments _ Dictionary new].
	aString size = 0
		ifTrue: 
			[objectComments removeKey: aSymbol ifAbsent: []]
		ifFalse: 
			["double internal quotes of the comment string"
			aStream _ WriteStream on: (String new: aString size).
			aStream nextPutAll: knowledgeBaseName , ' comments objectAt: ',
									aSymbol storeString, ' comment:'; cr.
			aString storeOn: aStream.
			objectComments
				at: aSymbol
				put: (RemoteString newString: aStream contents onFileNumber: 2)]! !

!KnowledgeBaseCommentator methodsFor: 'fileIn/Out'!

fileOutOn: aFileStream moveSource: moveSource toFile: fileIndex 
	"Store all comments about my KnowledgeBase onto the file, aFileStream."

	| newRemoteString |
	globalComment ~~ nil ifTrue: 
		[aFileStream cr.
		newRemoteString _ RemoteString
								newString: globalComment string
								onFileNumber: fileIndex
								toFile: aFileStream.
		moveSource ifTrue: [globalComment _ newRemoteString]].

	objectComments ~~ nil ifTrue: [objectComments associationsDo:
		[:assoc | aFileStream cr.
		newRemoteString _ RemoteString
								newString: assoc value string
								onFileNumber: fileIndex
								toFile: aFileStream.
		moveSource ifTrue: [assoc value: newRemoteString]]].

	slotComments ~~ nil ifTrue: [slotComments associationsDo:
		[:assoc | aFileStream cr.
		newRemoteString _ RemoteString
								newString: assoc value string
								onFileNumber: fileIndex
								toFile: aFileStream.
		moveSource ifTrue: [assoc value: newRemoteString]]]! !

Object subclass: #Binder
	instanceVariableNames: 'constraint '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
Binder comment:
'I represent a kind of temporary variable in a Rule, to which a sequence of values may be bound.

Instance Variables:
	constraint			<Symbol>	The name of the class to which I am constrained'!


!Binder methodsFor: 'accessing'!

constraint
	^ Smalltalk at: constraint!

constraint: aClass
	"Store the name and not the class itself, in case aClass is later redefined."
	constraint _ aClass name!

printOn: aStream
	aStream nextPut: $<.
	constraint printOn: aStream! !

Dictionary variableSubclass: #BindingList
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
BindingList comment:
'I am a dictionary of (binderNames -> values), used to remember what values are bound to binders during execution of a Rule.'!


!BindingList methodsFor: 'binding'!

bind: name to: instance
	self at: name put: instance!

unbind: name
	self removeKey: name! !

Object subclass: #Clause
	instanceVariableNames: 'block localBinders '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
Clause comment:
'I represent one compiled statement in a Rule.

Instance Variables:
	block			<BlockContext>		The statement, converted to a block and compiled
	localBinders		<Dictionary>		Binders accessed in the statement'!


!Clause methodsFor: 'initialize-release'!

block: aBlockContext localBinders: aDictionary
	block _ aBlockContext.
	localBinders _ aDictionary! !

!Clause methodsFor: 'evaluating'!

concludeWithBindings: bindings
	"Evaluate myself in the context of the current bindings.  Since this is the conclusion of a rule, there should be no unbound binders."
	| arguments |
	arguments _ localBinders collect:
		[:assoc | bindings at: assoc key ifAbsent:
			[self error: 'Unbound binder in rule conclusion: ', assoc key]].
	^ block valueWithArguments: arguments asArray!

valueWithBindings: bindings
	"Evaluate myself in the context of the current bindings.  An unbound argument is given the default value in localBinders.  Only one such value is allowed."

	| arguments binderName val |
	binderName _ nil.
	arguments _  localBinders collect:
		[:assoc | bindings at: assoc key ifAbsent:
			[binderName == nil
				ifTrue: [binderName _ assoc key]
				ifFalse: [self error: 'More than one unbound binder in rule'].
			assoc value]].
	^ ClauseResult new
		binderName: binderName
		value: (block valueWithArguments: arguments asArray)! !

!Clause methodsFor: 'private'!

literals
	^ block method literals! !

Dictionary variableSubclass: #KnowledgeBase
	instanceVariableNames: 'name rules comments classNames cutBlock hiddenSlots suggestions '
	classVariableNames: 'DefaultCollectionClass ListableCollections '
	poolDictionaries: ''
	category: 'Stress-Framework'!
KnowledgeBase comment:
'I represent a Stress knowledge base.  Objects (representing facts) are stored in my indexable fields, and can be added and removed using ordinary Dictionary protocol.

Instance Variables:
	name			<Symbol>
	rules			<OrderedCollection>
	comments		<KnowledgeBaseCommentator>
	cutBlock		<BlockContext>			Used to terminate rules with a ''cut''
	hiddenSlots		<OrderedCollection>		Slots which should not be displayed in a browser
	suggestions		<OrderedCollection>		Records past suggestions to avoid repetition

Class Variables:
	DefaultCollectionClass		<Class>		The default class for constructing collections defined
											using one-item-per-line format in browser
	ListableCollections			<Array>		Collections that should be formatted as above'!


!KnowledgeBase methodsFor: 'initialize-release'!

initializeWithName: aSymbol
	name _ aSymbol.
	rules _ OrderedCollection new.
	comments _ KnowledgeBaseCommentator new knowledgeBaseName: aSymbol.
	self classNames: SortedCollection new!

release
	"Break dependencies and cycles."
	self rules do: [:rule | rule release].
	self do: [:object | object release].
	self associationsDo: [:assoc | assoc release].
	self classes do: [:class | self releaseClass: class].
	rules _ nil.!

releaseClass: aClass
	"Remove rule dependencies from aClass.  Don't remove rules that belong to another knowledge base!!"
	aClass dependents do:
		[:dep | (rules includes: dep) ifTrue: [aClass removeDependent: dep]]! !

!KnowledgeBase methodsFor: 'accessing'!

name
	^ name!

rename: aSymbol
	name _ aSymbol.
	comments ~~ nil ifTrue: [comments knowledgeBaseName: aSymbol]!

scopeHas: varName ifTrue: assocBlock 
	"Look up varName in this knowledgeBase.  If it is there, pass the association to assocBlock, and answer true; else answer false.  Used by StressEncoder during parsing.  This method imitates Behavior>scopeHas:ifTrue:."
	| assoc |
	assoc _ self associationAt: varName ifAbsent: [].
	assoc == nil
		ifFalse: 
			[assocBlock value: assoc.
			^true].
	^false! !

!KnowledgeBase methodsFor: 'object access'!

at: key put: anObject
	self initialize: anObject key: key.
	^ super at: key put: anObject!

nameOf: anObject
	^ self keyAtValue: anObject ifAbsent: [anObject printString]!

removeAssociation: anAssociation ifAbsent: anExceptionBlock
	"Release association and value before removing."
	| assoc | 
	assoc _ super removeAssociation: anAssociation ifAbsent: anExceptionBlock.
	assoc release.
	assoc value release.
	^ assoc!

removeKey: key ifAbsent: aBlock
	"Release association and value before removing."
	| assoc |
	assoc _ super removeKey: key ifAbsent: aBlock.
	assoc release.
	assoc value release.
	^ assoc!

renameObject: oldName to: newName
	| assoc | 
	(self includesKey: newName) ifTrue: [^ false].
	assoc _ self removeKey: oldName.
	self at: newName put: assoc value.
	^ true! !

!KnowledgeBase methodsFor: 'slot access'!

evaluateSlot: aString oldValue: oldValue notifying: requestor ifFail: failBlock
	| range source value oldClass converter |

	(range _ self outerBrackets: aString) == nil
		ifTrue:
			[^ StressCompiler new
						evaluate: aString
						inKB: self
						notifying: requestor
						ifFail: failBlock]
		ifFalse:
			[source _ ReadStream on: aString from: range first + 1 to: range last.
			value _ StressCompiler new 
						evaluateCollection: source
						inKB: self
						notifying: requestor
						ifFail: failBlock.
			(oldValue isKindOf: Collection) ifTrue:
				[oldClass _ oldValue class name.
				converter _ ('as', oldClass) asSymbol.
				(value respondsTo: converter)
					ifTrue: [value _ value perform: converter]
					ifFalse: [self error:
					'Could not reconstruct as ', oldClass, ' - Proceed for OrderedCollection']].
			^ value]!

fillSlot: slotName of: anObject fromString: aString notifying: requestor
	| value | 
	value _ self evaluateSlot: aString
			oldValue: (anObject slot: slotName)
			notifying: requestor
			ifFail: [^ false].
	"Use a general message which works for classes and instances."
	anObject slot: slotName put: value in: self.
	^ true!

hiddenSlots
	hiddenSlots == nil ifTrue: [hiddenSlots _ OrderedCollection new].
	^ hiddenSlots!

hideSlot: slotName
	self hiddenSlots add: slotName!

suggest
	"Return an Association of (className->slotName), as a suggestion of what information should be supplied next."
	| slotClass | 
	suggestions == nil ifTrue: [suggestions _ OrderedCollection new].
	rules do: [:rule |
		rule slotDependencies do: [:slot |
			(suggestions includes: slot)
				ifFalse:
					[suggestions add: slot.
					slotClass _ self classes detect: [:class | class slots includes: slot].
					slotClass == nil
						ifTrue: [slotClass _ self classes detect:
							[:class | class allSlots includes: slot]].
					slotClass == nil
						ifTrue: ["Try again"  ^ self suggest].
					^ Association key: slotClass name value: slot]]].
	^ nil!

unhideSlots
	hiddenSlots _ OrderedCollection new! !

!KnowledgeBase methodsFor: 'rule access'!

compileRuleBlock: aText rule: aRule notifying: aController 
	| ruleMethod selector rule |
	Cursor execute
		showWhile: 
			[selector _ RuleCompiler new
						compile: aText
						into: aRule
						inKB: self
						notifying: aController].
	selector == nil ifTrue: [^ false].
	aRule
		putSource: aText asString
		selector: selector
		inFile: 2.
	^ true!

cut
	"Terminate the rule currently executing."
	^ cutBlock value!

forwardChain: rule
	| transcript | 
	rule ready: false.
	Stress startRuleTranscript: rule knowledgeBase: self.
	self prepareCutAndForwardChain: rule.
	Stress endTranscript.
	cutBlock _ nil.  "Break cycles"!

ifSourceForRule: rule
	"Answer the string corresponding to the source code for the if-part of rule."
	| newSource index |
	newSource _ rule getSourceFor: #if.
	newSource == nil 
		ifFalse:	[((newSource at: newSource size) isSeparator)
				ifTrue:	[index _ newSource size. "tidy up for file out"
						[((newSource at: index) isSeparator)
							and: [index > 1]]
							whileTrue:	[index _ index - 1].
						newSource _ newSource copyFrom: 1 to: index]].
	^newSource!

includesRuleNamed: aSymbol 
	^ (self ruleNamed: aSymbol ifAbsent: [nil]) ~~ nil!

initiateChaining
	| tryAgain readyRule |
	Cursor execute showWhile:
		[tryAgain _ true.
		[tryAgain] whileTrue:
			[readyRule _ self rules detect: [:rule | rule ready] ifNone: [nil].
			readyRule == nil
				ifTrue:
					[tryAgain _ false]
				ifFalse:
					[self forwardChain: rule.
					tryAgain _ true]]]!

newRuleNamed: aSymbol
	"Add a new rule called aSymbol as the last in the rule collection."
	^ self newRuleNamed: aSymbol before: nil!

newRuleNamed: aSymbol before: aRule
	| newRule index | 
	(self includesRuleNamed: aSymbol)
		ifTrue: [self error: 'Rule ', aSymbol, ' already exists'].
	newRule _ Rule named: aSymbol in: self.
	aRule == nil
		ifTrue: [^ rules addLast: newRule]
		ifFalse: [^ rules add: newRule before: aRule]!

readyRules
	"Return a string with the names of all rules that are 'ready', one name per line."
	| stream | 
	stream _ ReadWriteStream on: String new.
	rules do: [:rule |
		rule ready ifTrue: [stream nextPutAll: rule name; cr]].
	^ stream contents!

removeRule: aRule
	aRule release.
	rules remove: aRule!

renameRule: aRule to: aSymbol
	(self includesRuleNamed: aSymbol)
		ifTrue: [^ false].
	aRule rename: aSymbol.
	^ true!

ruleNamed: aSymbol
	^ self
		ruleNamed: aSymbol
		ifAbsent: [self error: 'No rule named ', aSymbol, ' in knowledge base']!

ruleNamed: aSymbol ifAbsent: aBlock
	rules do: [:rule | rule name == aSymbol ifTrue: [^ rule]].
	^ aBlock value!

ruleOrdering
	"Return a string showing the ordering of rules, with one rule name per line."
	| aStream | 
	aStream _ ReadWriteStream on: String new.
	rules do:
		[:rule | rule printOn: aStream.
		aStream cr].
	^ aStream contents!

ruleOrdering: aString
	| newCollection ruleNames rule | 
	"Reorder rules according to aString, which contains one rule name per line."
	newCollection _ OrderedCollection new.
	ruleNames _ Scanner new scanTokens: aString.
	ruleNames do: [:ruleName |
		(rule _ self ruleNamed: ruleName ifAbsent: [nil]) ~~ nil
			ifTrue: [newCollection addLast: rule]].
	"Don't lose any, even if names are missing!!"
	rules do: [:rule |
		(newCollection includes: rule)
			ifFalse: [newCollection addLast: rule]].
	rules _ newCollection!

rules
	^ rules!

thenSourceForRule: rule
	"Answer the string corresponding to the source code for the then-part of rule."
	| newSource index |
	newSource _ rule getSourceFor: #then.
	newSource == nil 
		ifFalse:	[((newSource at: newSource size) isSeparator)
				ifTrue:	[index _ newSource size. "tidy up for file out"
						[((newSource at: index) isSeparator)
							and: [index > 1]]
							whileTrue:	[index _ index - 1].
						newSource _ newSource copyFrom: 1 to: index]].
	^newSource! !

!KnowledgeBase methodsFor: 'classes'!

addClass: aClass
	(self includesClass: aClass)
		ifFalse: [classNames add: aClass name].
	^ aClass!

allKindsOf: aClass
	"Answer a Dictionary consisting of all of my elements for which aClass is a superclass or class."
	^ self select: [:each | each isKindOf: aClass]!

allMembersOf: aClass
	"Answer a Dictionary consisting of all of my elements which are instances of the class, aClass."
	^ self select: [:each | each isMemberOf: aClass]!

allSlotNames
	"Return all the names of all slots of objects in the knowledge base."

	| names | 
	names _ Set new.  "Remove repetitions by using a Set"
	self classes do: [:class | names addAll: class slots].
	^ names!

allSubclassesOf: aClass
	^ aClass allSubclasses select: [:each | self includesClass: each]!

categoryForClasses
	^ ('Stress-' , self name) asSymbol!

checkClasses
	"Check that each instance is represented by some class.  If not, add that instance's class."
	self do: [:anObject | self classForDisplaying: anObject]!

classes
	| removes classes class |
	classes _ OrderedCollection new.
	removes _ OrderedCollection new.
	classNames do: [:className |
		class _ Smalltalk at: className ifAbsent:
			[(self confirm: 'Class ', className, ' does not exist!! 
Remove key from the knowledge base?')
			ifTrue: 
				[removes add: className].
			nil].
		class ~~ nil ifTrue: [classes add: class]].
	removes do: [:each | classNames remove: each].
	^ classes!

classForDisplaying: anObject
	| classes | 
	"Search the classes in the knowledge base for the 'lowest' class or superclass of anObject.  If none exists, add the class of the object."
	classes _ self classes.
	anObject class withAllSuperclasses do: [:class |
		(classes includes: class) ifTrue: [^ class]].
	^ self addClass: anObject class!

classNames
	^ classNames!

classNames: aSortedCollection
	classNames _ aSortedCollection!

defaultCollectionClass
	^ DefaultCollectionClass!

includesClass: aClass
	^ classNames includes: aClass name!

removeClass: aClass
	classNames
		remove: aClass name
		ifAbsent: [self error: 'Class ', aClass name, ' not in knowledge base'].
	^ aClass! !

!KnowledgeBase methodsFor: 'comments'!

comments
	"Return the KnowledgeBaseCommentator encapsulating all of my comments."
	^ comments! !

!KnowledgeBase methodsFor: 'printing'!

isListableCollection: anObject
	"Answer whether anObject is a Collection which can be printed in a one-item-per-line format."
	^ ListableCollections includes: anObject class name!

printOn: aStream
	aStream nextPutAll: self name!

printSlot: slotName of: anObject
	^ self printSlot: slotName of: anObject format: true!

printSlot: slotName of: anObject format: format
	"Return a String representing the value of the specified slot.  If format is true and the value is a collection, then format the collection, one item per line."

	| aStream decoder value first | 
	decoder _ Decoder with: self.
	aStream _ WriteStream on: String new.
	value _ anObject slot: slotName.
	(format and: [self isListableCollection: value])
		ifFalse:
			[value printOn: aStream decoding: decoder]
		ifTrue:
			[aStream nextPut: $(.
			first _ true.
			value do: [:each |
				first
					ifTrue: [first _ false]
					ifFalse: [aStream cr].
				each printOn: aStream decoding: decoder].
			aStream nextPut: $)].
	^ aStream contents!

printString: anObject
	| aStream | 
	aStream _ ReadWriteStream on: String new.
	anObject printOn: aStream decoding: (Decoder with: self).
	^ aStream contents!

storeOn: aStream
	aStream nextPutAll: self name!

storeSlot: slotName of: anObject
	| aStream | 
	aStream _ WriteStream on: String new.
	(anObject slot: slotName)
			storeOn: aStream
			decoding: (Decoder with: self).
	^ aStream contents! !

!KnowledgeBase methodsFor: 'fileIn/Out'!

classesAndInstances
	^ StressObjectReader knowledgeBase: self!

codeForRule: aSymbol
	^RuleReader
		knowledgeBase: self
		rule: (self ruleNamed: aSymbol ifAbsent: [self newRuleNamed: aSymbol])!

contentsOfSlots
	^ StressCompiler new knowledgeBase: self!

fileOutClassesOn: aFileStream
	aFileStream cr; cr.
	SystemOrganization fileOutCategory: self categoryForClasses on: aFileStream!

fileOutCommentsOn: aFileStream
	aFileStream cr.
	comments fileOutOn: aFileStream moveSource: false toFile: nil!

fileOutFactsOn: aFileStream
	self fileOutClassesOn: aFileStream.
	self fileOutHiddenSlotsOn: aFileStream.
	self fileOutNamesOn: aFileStream.
	self fileOutCommentsOn: aFileStream.
	self fileOutSlotsOn: aFileStream!

fileOutHiddenSlotsOn: aFileStream
	| first | 
	self hiddenSlots isEmpty ifTrue: [^ self].
	aFileStream cr; cr.
	first _ true.
	self hiddenSlots do: [:slotName |
		first
			ifTrue:
				[aFileStream nextPutAll: self name.
				first _ false]
			ifFalse:
				[aFileStream nextPut: $;].
		aFileStream nextPutAll: ' hideSlot: '.
		slotName storeOn: aFileStream].
	aFileStream nextChunkPut: ''!

fileOutNamesOn: aFileStream
	aFileStream cr; cr; nextPut: $!!.
	aFileStream nextChunkPut: self name, ' classesAndInstances'.
	self classes do:
		[: class | aFileStream nextChunkPut: (self printClassAndInstances: class)].
	aFileStream nextChunkPut: ' '!

fileOutOn: aFileStream
	aFileStream nextChunkPut: 'Stress newKnowledgeBase: ', self name storeString.
	self fileOutFactsOn: aFileStream.
	self fileOutRulesOn: aFileStream!

fileOutRulesOn: aFileStream
	rules do: [:rule | rule fileOutOn: aFileStream]!

fileOutSlotsOf: anObject on: aStream
	| value | 
	anObject allSlots do:
		[:slotName |
		"Don't store dependents - restore by recompiling rules."
		slotName = 'dependents' ifFalse:
			[value _ self storeSlot: slotName of: anObject.
			value = 'nil' ifFalse:
				[aStream cr.
				aStream nextChunkPut: (self nameOf: anObject), ' slot: ', slotName storeString, ' put: ', value]]]!

fileOutSlotsOn: aFileStream
	aFileStream cr; cr; nextPut: $!!.
	aFileStream nextChunkPut: self name, ' contentsOfSlots'.
	aFileStream cr.
	self classes do:
		[:class | self fileOutSlotsOf: class on: aFileStream].
	self do:
		[:object | self fileOutSlotsOf: object on: aFileStream].
	aFileStream nextChunkPut: ' '!

moveChangesTo: newFile
	"Used in the process of condensing changes, this message requests that everything saved in the '.changes' file (rules and comments) should be moved to newFile.  "

	rules do: [:rule |
		rule fileOutOn: newFile moveSource: true toFile: 2].
	comments fileOutOn: newFile moveSource: true toFile: 2!

printClassAndInstances: aClass
	| str | 
	str _ WriteStream on: String new.
	str cr; cr.
	str nextPutAll: aClass name.
	(self allMembersOf: aClass) keysDo:
		[:instanceName | str crtab; nextPutAll: instanceName].
	^ str contents!

printRuleNameChunk: aSymbol on: aFileStream
	"print rule definition on aFileStream"

	aFileStream cr; cr; nextPut: $!!.
	aFileStream nextChunkPut:
				self name , ' codeForRule: ', aSymbol storeString! !

!KnowledgeBase methodsFor: 'private'!

addClassDependenciesAt: aClass to: newValue
	"The class of newValue, or one of its superclasses, may be referred to in one or more rules.  Look for such dependencies, and add them to newValue."

	aClass withAllSuperclasses do:
		[:class | class dependents do:  "Most classes will have no dependents"
			[:dep | (self rules includes: dep)  "Could be a rule in another knowledge base!!"
				ifTrue:
					[(newValue dependents includes: dep)
						ifFalse: [newValue addDependent: dep]]]]!

inheritClassSlots: anObject
	"Fill any nil slots of anObject with the default values in its class."

	| classVal |
	anObject allSlots do: [:aSlot |
		(anObject slot: aSlot) == nil
			ifTrue: [(classVal _ anObject class slot: aSlot) == nil
				ifFalse: [anObject slot: aSlot put: classVal]]]!

initialize: anObject key: key
	self inheritClassSlots: anObject.
	self transferDependenciesAt: key to: anObject.
	self addClassDependenciesAt: anObject class to: anObject.!

isModifiable: anObject
	"Answer whether anObject can be modified.  For example, nil, true and 10 cannot."
	| class | 
	class _ anObject class.
	^ (class == Symbol or: [class instSize == 0 & class isFixed]) not!

outerBrackets: aString
	| left right stream level ch start end |
	left _ $(.
	right _ $).
	stream _ ReadStream on: aString.
	stream skipSeparators.
	stream next == left ifFalse: [^ nil].
	start _ stream position + 1.
	level _ 1.
	"Scan until end, or when first left bracket is matched."
	[stream atEnd or: [level = 0]] whileFalse:
		[(ch _ stream next) == left
			ifTrue: [level _ level + 1]
			ifFalse: [ch == right
				ifTrue: [level _ level - 1]]].
	end _ stream position - 1.
	level = 0 ifFalse: [^ nil].		"No matching right bracket"
	stream skipSeparators.		"There should be nothing left but separators"
	stream atEnd
		ifFalse: [^ nil]
		ifTrue: [^ start to: end]!

prepareCutAndForwardChain: rule
	cutBlock _ [^ self].
	rule forwardChainWithBindings: BindingList new!

transferDependenciesAt: key to: newValue
	"If this is not a new key, then the name, key, may already be referred to in one or more rules.  Look for such dependencies, and transfer them from the old value at key, to the new."
	| oldAssoc oldValue | 
	oldAssoc _ self associationAt: key ifAbsent: [^ self].
	oldValue _ oldAssoc value.
	oldAssoc dependents do: [:dep |
		oldValue removeDependent: dep.  "No error if absent"
		(self isModifiable: newValue) ifTrue: [newValue addDependent: dep]]! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

KnowledgeBase class
	instanceVariableNames: ''!


!KnowledgeBase class methodsFor: 'class initialization'!

initialize
	"KnowledgeBase initialize"
	DefaultCollectionClass _ OrderedCollection.
	ListableCollections _ #(Bag Set IdentitySet Array OrderedCollection SortedCollection)! !

!KnowledgeBase class methodsFor: 'instance creation'!

called: aSymbol
	^ super new initializeWithName: aSymbol! !

!KnowledgeBase class methodsFor: 'defaults'!

evaluatorClass
	"For DoIts in the StressBrowser text view."
	^ StressCompiler! !

KnowledgeBase initialize!


Object subclass: #ClauseResult
	instanceVariableNames: 'binderName value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
ClauseResult comment:
'I represent the value returned after evaluating one statement (Clause) in a rule.

Instance Variables:
	binderName		<Symbol>	The name of an unbound binder, if there was one.
	value						Either a collection of bindings, or a simple Boolean result.'!


!ClauseResult methodsFor: 'accessing'!

binderName
	^ binderName!

binderName: aSymbol value: anObject
	binderName _ aSymbol.
	value _ anObject!

value
	^ value! !

!ClauseResult methodsFor: 'printing'!

printOn: aStream
	value printOn: aStream! !

Object subclass: #Rule
	instanceVariableNames: 'name knowledgeBase ifPart thenPart binders sourceFiles sourcePositions ready slotDependencies '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
Rule comment:
'I represent a rule within a Stress knowledge base.  I provide access to both source code and compiled form of rule.

Instance Variables:
	name				<Symbol>
	knowledgeBase 	<KnowledgeBase>		In which I live
	ifPart				<OrderedCollection>		Collection of Clauses representing compiled
												preconditions
	thenPart			<OrderedCollection>		Collection of Clauses representing compiled
												conclusions
	binders				<Dictionary>			Binders that I access
	sourceFiles			<Dictionary>			(#if->aFileStream #then->aFileStream)
	sourcePositions		<Dictionary>			(#if->anInteger    #then->anInteger)
	ready				<Boolean>				Should I execute?
	slotDependencies	<Set>					Slots that I access'!


!Rule methodsFor: 'initialise-release'!

release
	"Remove dependencies and break cycles."
	self removeOldDependencies.
	knowledgeBase _ nil!

setName: aSymbol knowledgeBase: aKnowledgeBase
	name _ aSymbol.
	knowledgeBase _ aKnowledgeBase.
	ifPart _ OrderedCollection new.
	thenPart _ OrderedCollection new.
	binders _ Dictionary new.
	sourceFiles _ Dictionary new.
	sourcePositions _ Dictionary new.
	ready _ false! !

!Rule methodsFor: 'accessing'!

binders
	^ binders!

binders: aDictionary
	binders _ aDictionary!

ifPart
	^ ifPart!

ifPart: ifCollection 
	ifPart _ ifCollection!

ifPart: ifCollection thenPart: thenCollection 
	ifPart _ ifCollection.
	thenPart _ thenCollection!

knowledgeBase
	^ knowledgeBase!

name
	^name!

rename: aSymbol
	name _ aSymbol!

selectors
	^ #(if then)!

thenPart
	^ thenPart!

thenPart: thenCollection 
	thenPart _ thenCollection! !

!Rule methodsFor: 'executing'!

forwardChainWithBindings: bindings
	| theRest reply value binderName | 

	ifPart isEmpty ifTrue:
		[Stress ifTranscriptActive: [:transcript | transcript ruleSucceeded].
		thenPart do: [:each | each concludeWithBindings: bindings].
		^ self].

	theRest _ self copy.
	theRest ifPart: (ifPart copyFrom: 2 to: ifPart size).
	reply _ ifPart first valueWithBindings: bindings.
	value _ reply value.
	(binderName _ reply binderName) == nil 
		ifTrue: [value
			ifTrue: [^ theRest forwardChainWithBindings: bindings]
			ifFalse: [^ false]]
		ifFalse: ["Result is an OrderedCollection of valid instanciations for binderName"
			value do: [:eachInstance |
				Stress ifTranscriptActive: [:transcript | transcript levelUp; bind: binderName to: eachInstance].
				bindings bind: binderName to: eachInstance.
				theRest forwardChainWithBindings: bindings.
				Stress ifTranscriptActive: [:transcript | transcript levelDown].
				bindings unbind: binderName]]!

ready
	^ ready!

ready: aBoolean
	| transcript | 
	"If aBoolean is true, then I should execute at the next opportunity."
	ready _ aBoolean.
	ready ifTrue: [Stress ifTranscriptActive: [:transcript | transcript ruleReady: self]]!

update: aSlot
	"An object on which I depend has changed its slot aSlot.  If I also depend on that slot, I should note that I am ready to execute."
	(ready not and: [slotDependencies includes: aSlot])
		ifTrue: [self ready: true]! !

!Rule methodsFor: 'dependencies'!

addNewDependencies
	| allSlotNames | 
	self objectsIDependOn do: [:obj | obj addDependent: self].
 	allSlotNames _ knowledgeBase allSlotNames.
	slotDependencies _	self literals select:
		"First test quickly eliminates most values."
		[:literal | literal class == String and: [allSlotNames includes: literal]]!

classesAccessed
	| classes | 
	classes _ Set new.  "Remove repetitions by using a Set"
	binders do: [:binder | classes add: binder constraint].
	^ classes!

literals
	| literals | 
	literals _ Set new.  "Remove repetitions by using a Set"
	ifPart do: [:clause | literals addAll: clause literals].
	^ literals!

objectsIDependOn
	| objects val | 
	objects _ Set new.  "Remove repetitions by using a Set"
	self literals do:
		[:literal | (knowledgeBase includesAssociation: literal)
			ifTrue:
				["Be dependent on value, except for things like Booleans, Symbols
				or nil, since they can never change."
				(knowledgeBase isModifiable: (val _ literal value)) ifTrue:
					[objects add: val].
				"Be dependent on association - if value is changed, dependency
				can be restored more reliably from the association."
				objects add: literal]].
	self classesAccessed do:
		[:class |
			objects addAll: (knowledgeBase allKindsOf: class).
			"Be dependent on class, so that new instances can be informed."
			objects add: class].
	^ objects!

removeOldDependencies
	self objectsIDependOn do: [:obj | obj removeDependent: self].
	slotDependencies _ Set new!

slotDependencies
	^ slotDependencies! !

!Rule methodsFor: 'printing'!

ifPartTemplate
	^
'if
	"comment stating purpose of rule"
	| binder names and constraints |
	statements'!

ifSource
	| string |
	string _ ''.
	ifPart do: [:clause | string _ string, clause sourceCode].
	^string!

printOn: aStream
	name == nil
		ifFalse: [^ name printOn: aStream]
		ifTrue: [^ super printOn: aStream]!

thenPartTemplate
	^
'then
	statements'!

thenSource
	| string |
	string _ ''.
	thenPart do: [:clause | string _ string, clause sourceCode].
	^string! !

!Rule methodsFor: 'source code management'!

getSourceFor: selector 
	"Selector will be #if or #then.  Answer the appropriate source code.  
	Answer nil if there are no source files specified in the global 
	SourceFiles. "

	| source file pos |
	SourceFiles == nil ifTrue: [^nil].
	Cursor read
		showWhile: 
			[pos _ sourcePositions at: selector ifAbsent: [0].
			pos = 0
				ifTrue: [source _ nil]
				ifFalse: [source _ (RemoteString newFileNumber: (sourceFiles at: selector) position: pos) string]].
	^source!

putSource: sourceString selector: selector inFile: fileIndex 
	"Print an expression that is a message to my knowledgeBase, asking it to accept the source code, sourceString, as code for the 'selector' (#if or #then) part of myself.  This is an imitation of the method for writing descriptions of methods on files.  If no sources are specified, i.e., SourceFile is nil, then do nothing.  If the fileIndex is 1, print on *.sources;  if it is 2, print on *.changes."

	| file remoteString |
	SourceFiles == nil ifTrue: [^self].
	file _ SourceFiles at: fileIndex.
	file setToEnd; readWriteShorten.
	knowledgeBase printRuleNameChunk: self name on: file.
	file cr.
	remoteString _ 
		RemoteString
			newString: sourceString
			onFileNumber: fileIndex
			toFile: file.
	file nextChunkPut: ' '; readOnly.
	sourceFiles at: selector put: fileIndex.
	sourcePositions at: selector put: remoteString position! !

!Rule methodsFor: 'fileIn/Out'!

fileOutOn: aFileStream
	^ self
		fileOutOn: aFileStream
		moveSource: false
		toFile: 0!

fileOutOn: aFileStream moveSource: moveSource toFile: fileIndex
	| position | 
	knowledgeBase printRuleNameChunk: name on: aFileStream.
	Cursor write showWhile:
		[moveSource
			ifTrue:
				[self selectors do: [:sel |
					position _ aFileStream position.
					aFileStream cr; cr; nextChunkPut: (self getSourceFor: sel).
					sourceFiles at: sel put: fileIndex.
					sourcePositions at: sel put: position]]
			ifFalse:
				[self selectors do: [:sel |
					aFileStream cr; cr; nextChunkPut: (self getSourceFor: sel)]]].
	aFileStream nextChunkPut: ' '! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Rule class
	instanceVariableNames: ''!


!Rule class methodsFor: 'instance creation'!

named: aSymbol in: aKnowledgeBase
	^ super new setName: aSymbol knowledgeBase: aKnowledgeBase! !

Dictionary variableSubclass: #StressDictionary
	instanceVariableNames: 'transcripts activeTranscript '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Framework'!
StressDictionary comment:
'I represent a dictionary which contains all Stress knowledge bases, and performs general administrative tasks.  My sole instance is called Stress.

Instance Variables:
	transcripts			<Dictionary>		all stress transcripts currently open
	activeTranscript	<StressTranscript>'!


!StressDictionary methodsFor: 'initialization'!

initialize
	transcripts _ Dictionary new! !

!StressDictionary methodsFor: 'knowledge base access'!

newKnowledgeBase: aSymbol
	| newKnowledgeBase | 
	self validateNewKey: aSymbol.
	newKnowledgeBase _ KnowledgeBase called: aSymbol.
	self at: aSymbol put: newKnowledgeBase.
	Smalltalk at: aSymbol put: newKnowledgeBase!

removeKnowledgeBase: aKnowledgeBase
	aKnowledgeBase release.
	self removeKey: aKnowledgeBase name.
	Smalltalk removeKey: aKnowledgeBase name!

renameKnowledgeBase: oldName to: newName
	| kb |
	self validateNewKey: newName.
	kb _ self at: oldName.
	kb rename: newName.
	self removeKey: oldName.
	self at: newName put: kb.
	Smalltalk removeKey: oldName.
	Smalltalk at: newName put: kb!

validateNewKey: aSymbol
	(self includesKey: aSymbol)
		ifTrue:
			[self error: aSymbol, ' already exists!!  Proceed will store over it'.
			self removeKnowledgeBase: (self at: aSymbol)]
		ifFalse:
			[(Smalltalk includesKey: aSymbol)
				ifTrue:
					[self error:
		'A global called ', aSymbol , ' already exists!!  Proceed will store over it']]! !

!StressDictionary methodsFor: 'classes'!

classOrganization
	"Return a string listing each knowledge base and the classes it contains."
	| quote aStream |
	quote _ $'.
	aStream _ ReadWriteStream on: (Text new: 64).
	self associationsDo:
		[:kb | aStream nextPut: $(; nextPut: quote.
		kb key printOn: aStream.
		aStream nextPut: quote.
		kb value classNames do:
			[:className | aStream nextPut: $ .
			aStream nextPutAll: className].
		aStream nextPut: $); nextPut: Character cr].
	^ aStream contents!

classOrganizationFromString: aString
	"Update the classes in each knowledge base according to aString."
	| scanner anArray knowledgeBase oldClassNames newClassNames newClasses | 
	scanner _ Scanner new scanTokens: aString.
	1 to: scanner size do:
		[:i | anArray _ scanner at: i.
		knowledgeBase _ self at: (anArray first asSymbol).
		oldClassNames _ knowledgeBase classNames.
		newClassNames _ ((anArray copyFrom: 2 to: anArray size) select:
			[:className | Smalltalk includesKey: className]) asSortedCollection.
		knowledgeBase classNames: newClassNames.
		knowledgeBase checkClasses]! !

!StressDictionary methodsFor: 'transcript'!

endTranscript
	activeTranscript ~~ nil
		ifTrue:
			[activeTranscript endEntry.
			activeTranscript _ nil]!

ifTranscriptActive: aBlock
	"Evaluate aBlock if there is a transcript active."
	activeTranscript ~~ nil
		ifTrue: [^ aBlock value: activeTranscript]!

openTranscriptFor: aKnowledgeBase
	| transcript key |
	key _ aKnowledgeBase name.
	(transcripts includesKey: key)
		ifFalse:
			[transcripts
				at: key
				put: (transcript _ StressTranscript on: aKnowledgeBase).
			StressTranscriptView openOn: transcript]
		ifTrue:
			"It already exists - bring it to the top."
			[(transcripts at: key) changed: #update]!

releaseTranscriptFor: knowledgeBaseName
	transcripts removeKey: knowledgeBaseName ifAbsent: []!

startBrowserTranscript: aKnowledgeBase
	activeTranscript _ transcripts at: aKnowledgeBase name ifAbsent: [^ self].
	activeTranscript on
		ifTrue: [activeTranscript startBrowserTranscript]
		ifFalse: [activeTranscript _ nil]!

startRuleTranscript: aRule knowledgeBase: aKnowledgeBase
	activeTranscript _ transcripts at: aKnowledgeBase name ifAbsent: [^ self].
	activeTranscript on
		ifTrue: [activeTranscript startRuleTranscript: aRule]
		ifFalse: [activeTranscript _ nil]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

StressDictionary class
	instanceVariableNames: ''!


!StressDictionary class methodsFor: 'installation'!

installStress
	"StressDictionary installStress"
	(Smalltalk includesKey: #Stress)
		ifTrue: [self notify: 'Stress is already installed']
		ifFalse: [Smalltalk at: #Stress put: self new initialize]! !Parser subclass: #Splitter
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Compilers'!
Splitter comment:
'I am a Parser used to split a fragment of Smalltalk code into its component outer-level statements.  I am used by RuleCompilers to break rules up into separate statements.'!


!Splitter methodsFor: 'public access'!

split: sourceStream notifying: req ifFail: aBlock
	 "Answer with an OrderedCollection of integers, representing the end positions of the outer-level statments."

	 | meth stmtLimits startPos |
	startPos _ sourceStream position.
	self init: sourceStream notifying: req failBlock: [^aBlock value].
	encoder _ AnythingGoesEncoder new init: (nil class) context: nil notifying: req.
	stmtLimits _ self outerStatements.
	encoder _ failBlock _ requestor _ parseNode _ nil. "break cycles & mitigate refct overflow"
	sourceStream position: startPos.
	^ stmtLimits! !

!Splitter methodsFor: 'expression types'!

outerStatements
	| inner stmts returns start more blockComment stmtLimits startOfStmt |
	inner _ false.
	stmtLimits _ OrderedCollection new.
	stmts _ OrderedCollection new.
	"give initial comment to block, since others trail statements"
	blockComment _ currentComment.
	currentComment _ nil.
	returns _ false.
	more _ hereType ~~ #rightBracket.
	[more]
		whileTrue: 
			[start _ self startOfNextToken.
			(returns _ self match: #upArrow)
				ifTrue: 
					[self expression
						ifFalse: [^self expected: 'Expression to return'].
					self addComment.
					stmts addLast: 
						(ReturnNode new
							expr: parseNode
							encoder: encoder
							sourceRange: (start to: self endOfLastToken)).
					stmtLimits addLast: self startOfNextToken]
				ifFalse: 
					[self expression
						ifTrue: 
							[self addComment.
							stmts addLast: parseNode.
							stmtLimits addLast: self startOfNextToken]
						ifFalse: 
							[self addComment.
							stmts size = 0
								ifTrue: 
									[stmts addLast: 
										(encoder encodeVariable:
											(inner ifTrue: ['nil'] ifFalse: ['self']))]]].
			returns 
				ifTrue: 
					[(hereType == #rightBracket or: [hereType == #doIt])
						ifFalse: [^self expected: 'End of block']].
			more _ returns not and: [self match: #period]].
	hereType == #doIt ifFalse: [^self expected: 'Nothing more'].
	^ stmtLimits!

primaryExpression
	hereType == #word
		ifTrue: 
			[parseNode _ encoder encodeVariable: self advance.
			^true].
	hereType == #leftBracket
		ifTrue: 
			[self advance.
			self blockExpression.
			^true].
	hereType == #leftParenthesis
		ifTrue: 
			[self advance.
			self expression ifFalse: [^self expected: 'expression'].
			(self match: #rightParenthesis)
				ifFalse: [^self expected: 'right parenthesis'].
			^true].
	(hereType == #string or: [hereType == #number or: [hereType == #literal]])
		ifTrue: 
			[parseNode _ encoder encodeLiteral: self advance.
			^true].
	(here == #- and: [tokenType == #number])
		ifTrue: 
			[self advance.
			parseNode _ encoder encodeLiteral: self advance negated.
			^true].
	hereType == #leftCurly
		ifTrue:
			[parseNode _ encoder encodeVariable: self advance.
			hereType == #word ifFalse: [^ self expected: 'Binder name'].
			self advance.
			(self match: #rightCurly)
				ifFalse: [^self expected: 'right curly bracket'].
			^true].
	^false! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Splitter class
	instanceVariableNames: ''!


!Splitter class methodsFor: 'class initialization'!

initialize
	TypeTable at: ${ asciiValue put: #leftCurly.
	TypeTable at: $} asciiValue put: #rightCurly! !

Splitter initialize!


Parser subclass: #StressParser
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Compilers'!
StressParser comment:
'I am a Parser for parsing in the context of a KnowledgeBase.  I am used by instances of class StressCompiler.'!


!StressParser methodsFor: 'public access'!

parse: sourceStream class: class noPattern: noPattern context: ctxt
	notifying: req notifyCorrection: correction knowledgeBase: aKnowledgeBase ifFail: aBlock
	 "Answer with a parse tree.  noPattern is true for doIts (Compiler evaluate)"

	 | meth |
	self init: sourceStream notifying: req failBlock: [^aBlock value].
	self initialCorrection: correction.
	encoder _ StressEncoder new initKnowledgeBase: aKnowledgeBase notifying: self.
	failBlock_ aBlock.
	meth _ self method: noPattern context: ctxt.
	encoder _ failBlock _ requestor _ parseNode _ nil. "break cycles & mitigate refct overflow"
	^meth! !

!StressParser methodsFor: 'rule parsing'!

binders
	"Parse binder declarations and return a dictionary of (names->binders)."
	"[ '|' (variable['<'(class)] )* '|']"

	| binders class name |
	binders _ Dictionary new.
	(self match: #verticalBar)
		ifFalse: [lastTempMark _ self bareEndOfLastToken. ^binders].	"no temps"
	(self match: #verticalBar)
		ifTrue: [lastTempMark _ self endOfLastToken. ^binders].	"empty temps"
	[hereType == #word]
		whileTrue:
			[name _ (encoder bindTemp: self advance).
			(self matchToken: #<)
				ifFalse: [class _ Object]
				ifTrue: [(class _ Smalltalk at: here asSymbol ifAbsent: [nil]) == nil
					ifFalse: [self advance]
					ifTrue: [^ self expected: 'Class name']].
			binders at: name put: (Binder new constraint: class)].

	(self match: #verticalBar)
		ifTrue: [lastTempMark _ self endOfLastToken. ^binders].
	^self expected: 'Vertical bar'!

initialCorrection: correction
	"Allow for an overall correction to match original source to the text actually being parsed.  Used by RuleCompiler when compiling clauses, for error notification."
	correction == nil ifFalse: [correctionDelta _ correction]!

parseSelectorAndBinders: aString notifying: req
	"Answer with the selector and the binders for this rule method."

	| sel |  
	^self
		initPattern: aString
		notifying: req
		return: [:pattern |
			(sel _ pattern at: 1) == #if	"Only if-part should have any binder declarations."
				ifTrue: [Array with: sel with: self binders]
				ifFalse: [Array with: sel with: nil]]! !

Encoder subclass: #StressEncoder
	instanceVariableNames: 'knowledgeBase '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Compilers'!
StressEncoder comment:
'I am an Encoder for encoding in the context of a KnowledgeBase.  I am able to encode the name of a knowledge base object by returning an association referencing the object.  I am used by instances of class StressParser.

Instance Variables:
	knowledgeBase		<KnowledgeBase>'!


!StressEncoder methodsFor: 'initialize-release'!

initKnowledgeBase: aKnowledgeBase notifying: req 
	knowledgeBase _ aKnowledgeBase.
	super init: (nil class) context: nil notifying: req! !

!StressEncoder methodsFor: 'encoding'!

encodeVariable: name
	name = 'self' ifTrue: [^ self knowledgeBaseNode].
	^scopeTable at: name
		ifAbsent: 
			[self lookupInPools: name 
				ifFound: [:assoc | ^ self global: assoc name: name].
			self lookupInKnowledgeBase: name
				ifFound: [:assoc | ^ self global: assoc name: name].
			requestor editor notNil
				ifTrue: [self undeclared: name]
				ifFalse: [self declareUndeclared: name]]! !

!StressEncoder methodsFor: 'undeclared variables'!

declareCorrect: name
	"Attempt to correct the spelling of an undeclared variable."

	| old lc names score bestScore guess |
	old _ name.
	lc _ old first asLowercase.
	names _
			(scopeTable keys select: [:key | key first asLowercase = lc]).
	names addAll:
			(class allVarNamesSelect: [:key | key first asLowercase = lc]).
	names addAll:
			(knowledgeBase keys select: [:key | key first asLowercase = lc]).

	bestScore _ 0.
	names do:
		[:aName |
		(score _ aName spellAgainst: old) > bestScore ifTrue:
			[bestScore _ score. guess _ aName]].

	bestScore > 50 ifFalse: [^ self notify: 'Couldn''t correct'].
	(self confirm: 'Confirm correction to ' , guess)
		ifTrue:
			[requestor replaceEditSelectionWith: guess.
			^ self encodeVariable: guess]
		ifFalse:
			[^ self notify: 'Undeclared']!

declareKnowledgeBaseObject: name
	| sym | 
	sym _ name asSymbol.
	knowledgeBase at: sym put: nil.
	^ self global: (knowledgeBase associationAt: sym) name: sym!

undeclared: name
	| menu index |
	Cursor normal show.
	requestor selectVariable: name.
	index _ (menu _ ActionMenu
			labels: 'knowledge base object\global\undeclared\correct it\abort' withCRs
			lines: #(4)
			selectors: #(declareKnowledgeBaseObject: declareGlobal: declareUndeclared: declareCorrect: declareFailed:))
				startUp: #anyButton
				withHeading: ' declare ' asText , (name contractTo: 20) asText allBold , ' as           ' asText.
	index = 0 ifTrue: [^ self declareFailed: name].
	^ self perform: (menu selectorAt: index) with: name! !

!StressEncoder methodsFor: 'private'!

knowledgeBaseNode
	^self
		name: 'self'
		key: (Smalltalk associationAt: knowledgeBase name)
		class: VariableNode
		type: LdLitIndType
		set: litIndSet!

lookupInKnowledgeBase: name ifFound: assocBlock 

	Symbol 
		hasInterned: name 
		ifTrue: [:sym | ^knowledgeBase scopeHas: sym ifTrue: assocBlock].
	^false! !

Compiler subclass: #RuleCompiler
	instanceVariableNames: 'notifyCorrection '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Compilers'!
RuleCompiler comment:
'I am a Compiler used to compile Stress Rules.  Source text is separated into outer-level statements, compiled into clauses, and then installed in the Rule.

InstanceVariables:
	notifyCorrection	<Integer>	Correction to match original source to text being compiled.
									Allows error notification at correct position in source text.'!


!RuleCompiler methodsFor: 'public access'!

compile: aText into: aRule inKB: aKnowledgeBase notifying: aRequestor
	| parser selectorAndBinders selector binders clauses start stmtLimits clauseStream |
	self from: aText
		class: nil class
		context: nil
		notifying: aRequestor.
	start _ sourceStream position max: 1.
	parser _ StressParser new.
	(selectorAndBinders _ (parser parseSelectorAndBinders: aText notifying: self)) == nil
		ifTrue: [^ nil].
	selector _ selectorAndBinders at: 1.
	(self validateSelector: selector at: start) ifFalse: [^ nil].
	selector == #if
		ifTrue: [binders _ selectorAndBinders at: 2.
				aRule removeOldDependencies]
		ifFalse: [binders _ aRule binders].

	start _ parser startOfNextToken.
	sourceStream _ ReadStream on: aText asString from: start to: aText size.
	stmtLimits _ Splitter new
			split: sourceStream
			notifying: self
			ifFail: [^ nil].
	clauses _ OrderedCollection new.
	stmtLimits do: [:limit |
		clauseStream _ ReadStream on: aText asString from: start to: limit-1.
		clauses addLast: (self
				compileClause: clauseStream
				in: aKnowledgeBase
				binders: binders
				ifFail: [^ nil]).
		start _ limit+1].
	
	selector == #if
		ifTrue:
			[aRule ifPart: clauses.
			aRule binders: (selectorAndBinders at: 2).
			aRule addNewDependencies]
		ifFalse:
			[aRule thenPart: clauses].
	^ selector! !

!RuleCompiler methodsFor: 'clause compiling'!

blockPrefix: bindersThisClause
	| aStream |
	aStream _ WriteStream on: (String new: 16).
	aStream nextPut: $[.
	bindersThisClause do:
		[:assoc | aStream nextPut: $:.
		aStream nextPutAll: assoc key.
		aStream nextPut: Character space].
	bindersThisClause size > 0 ifTrue: [aStream nextPutAll: '| '].
	^ aStream contents!

blockSuffix
	^ ']'!

compileClause: source in: aKnowledgeBase binders: binders ifFail: failBlock
	| dest block bindersThisClause blockPrefix correction |

	correction _ source position.
	dest _ ReadWriteStream on: (String new: 32).
	(bindersThisClause _ self preprocess: source into: dest binders: binders) == nil
		ifTrue: [^ failBlock value].
	blockPrefix _ self blockPrefix: bindersThisClause.
	correction _ correction - blockPrefix size.
	block _ blockPrefix, dest contents, self blockSuffix.
	^ Clause new
		block: (StressCompiler new
			evaluate: block
			in: nil
			inKB: aKnowledgeBase
			notifying: requestor
			notifyCorrection: correction
			ifFail: failBlock)
		localBinders: bindersThisClause!

preprocess: source into: dest binders: binders
	"Scan source for binderNames in {} pairs, and remove the brackets from each of these.  Return an OrderedCollection of all binderNames found in this clause."

	|  leftBracket rightBracket binderName bindersThisClause gripePos | 
	leftBracket _ ${.
	rightBracket _ $}.
	bindersThisClause _ OrderedCollection new.  "Must be ordered for argument substitution to work"
	[source atEnd] whileFalse:
		"Copy from source to dest, substituting sections between { and }."
		[dest nextPutAll: (source upTo: leftBracket).
		source atEnd ifFalse:
			[gripePos _ source position+1.
			binderName _ source upTo: rightBracket.  "Splitter checked that there was one"
			(binders includesKey: binderName) ifFalse:
				[self notify: 'Undeclared binder ->' at: gripePos.
				^ nil].
			dest nextPutAll: ' ', binderName, ' '.    "Blanks keep in step for error notifications"
			(bindersThisClause includes: (binders associationAt: binderName))
				ifFalse: [bindersThisClause add: (binders associationAt: binderName)]]].
	^ bindersThisClause! !

!RuleCompiler methodsFor: 'error handling'!

validateSelector: aSymbol at: position
	(aSymbol == #if) | (aSymbol == #then)
		ifTrue: [^ true]
		ifFalse: [self notify: 'if or then expected ->' at: position.  ^ false]! !

Encoder subclass: #AnythingGoesEncoder
	instanceVariableNames: ''
	classVariableNames: 'DefaultVariableNode '
	poolDictionaries: ''
	category: 'Stress-Compilers'!
AnythingGoesEncoder comment:
'I am an Encoder used by instances of class Splitter.  Since Splitters don''t care about whether variables are declared, I simply encode any variable with a default node.

Class Variables:
	DefaultVariableNode	<VariableNode>'!


!AnythingGoesEncoder methodsFor: 'encoding'!

encodeVariable: name 
	"Don't check whether name is declared, just return DefaultVariableNode."
	^ DefaultVariableNode! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AnythingGoesEncoder class
	instanceVariableNames: ''!


!AnythingGoesEncoder class methodsFor: 'initialization'!

initialize
	"Set up the DefaultVariableNode"
	DefaultVariableNode _ VariableNode new
		name: #default
		key: (Association new)
		index: nil
		type: LdLitIndType! !

AnythingGoesEncoder initialize!


Compiler subclass: #StressCompiler
	instanceVariableNames: 'knowledgeBase notifyCorrection '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Compilers'!
StressCompiler comment:
'I represent a Compiler for compiling and evaluating text in the context of a KnowledgeBase.  The text may therefore contain references to knowledge base objects.  I am used by StressBrowsers, and by RuleCompilers in the process of compiling rules.

Instance Variables:
	knowledgeBase		<KnowledgeBase>
	notifyCorrection	<Integer>	Correction to match original source to text being compiled.
									Used by RuleCompiler to allow error notification at correct
									position in source text.'!


!StressCompiler methodsFor: 'public access'!

evaluate: textOrStream in: aContext inKB: aKnowledgeBase notifying: aRequestor notifyCorrection: correction ifFail: failBlock
	knowledgeBase _ aKnowledgeBase.
	notifyCorrection _ correction.
	^ super evaluate: textOrStream
		in: aContext
		to: nil
		notifying: aRequestor
		ifFail: failBlock!

evaluate: textOrStream in: ignoreContext to: aKnowledgeBase notifying: aRequestor ifFail: failBlock
	"A somewhat kludgey way of receiving DoIt messages in the text view of a StressBrowser."
	^ self
		evaluate: textOrStream
		inKB: aKnowledgeBase
		notifying: aRequestor
		ifFail: failBlock!

evaluate: textOrStream inKB: aKnowledgeBase notifying: aRequestor ifFail: failBlock
	^ self evaluate: textOrStream
		in: nil
		inKB: aKnowledgeBase
		notifying: aRequestor
		notifyCorrection: 0
		ifFail: failBlock!

evaluateCollection: textOrStream inKB: aKnowledgeBase notifying: aRequestor ifFail: failBlock
	"The text consists of zero or more expressions, each on a separate line.  Create and return a collection containing the values of these expressions."

	| source string cr previous nextLine collection |
	(textOrStream isKindOf: PositionableStream)
		ifTrue: [source _ textOrStream]
		ifFalse: [source _ ReadStream on: textOrStream asString].
	string _ source contents.
	cr _ Character cr.
	previous _ source position.
	collection _ aKnowledgeBase defaultCollectionClass new.
	[source atEnd] whileFalse:
		[(source skipTo: cr)
			ifTrue: [nextLine _ ReadStream on: string from: previous to: source position - 1]
			ifFalse: [nextLine _ ReadStream on: string from: previous to: source position].
		previous _ source position.
		nextLine skipSeparators atEnd ifFalse:
			[collection addLast:
				(self
					evaluate: nextLine
					inKB: aKnowledgeBase
					notifying: aRequestor
					ifFail: failBlock)]].
	^ collection! !

!StressCompiler methodsFor: 'file in/out'!

scanFrom: aStream
	| string | 
	[string _ aStream nextChunk.
	string size > 0]			"done when double terminators"
		whileTrue:
			[super evaluate: string
				in: nil
				to: nil
				notifying: nil
				ifFail: []]! !

!StressCompiler methodsFor: 'private'!

knowledgeBase: aKnowledgeBase
	knowledgeBase _ aKnowledgeBase!

translate: aStream noPattern: noPattern ifFail: failBlock 
	| tree |
	tree _ 
		StressParser new
			parse: aStream
			class: class
			noPattern: noPattern
			context: context
			notifying: self
			notifyCorrection: notifyCorrection
			knowledgeBase: knowledgeBase
			ifFail: [^failBlock value].
	^tree! !StandardSystemView subclass: #RuleBrowserView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Browsers'!
RuleBrowserView comment:
'I am a StandardSystemView for viewing a RuleBrowser.  I provide initialization methods (messages to myself) to create and schedule such a browser.'!


!RuleBrowserView methodsFor: 'subview creation'!

addIfTextView: area on: aRuleBrowser initialSelection: sel
	| view | 
	view _ (CodeView on: aRuleBrowser aspect: #ifText change: #acceptIfText:from:
				menu: #textMenu initialSelection: sel).
	self addSubView: view in: area borderWidth: 1.
	view controller: AlwaysAcceptCodeController new!

addKnowledgeBaseView: area on: aRuleBrowser readOnly: RO
	self addSubView:
		(SelectionInListView on: aRuleBrowser printItems: false oneItem: RO
			aspect: #knowledgeBaseName change: #knowledgeBaseName: list: #knowledgeBaseList
			menu: #knowledgeBaseMenu initialSelection: #knowledgeBaseName)
		in: area borderWidth: 1!

addRuleView: area on: aRuleBrowser readOnly: RO
	self addSubView:
		(SelectionInListView on: aRuleBrowser printItems: false oneItem: RO
			aspect: #ruleName change: #ruleName: list: #ruleList
			menu: #ruleMenu initialSelection: #ruleName)
		in: area borderWidth: 1!

addThenTextView: area on: aRuleBrowser initialSelection: sel
	| view | 
	view _ (CodeView on: aRuleBrowser aspect: #thenText change: #acceptThenText:from:
				menu: #textMenu initialSelection: sel).
	self addSubView: view in: area borderWidth: 1.
	view controller: AlwaysAcceptCodeController new! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

RuleBrowserView class
	instanceVariableNames: ''!


!RuleBrowserView class methodsFor: 'instance creation'!

openOn: aStressDictionary
	"RuleBrowserView openOn: Stress."
	| aRuleBrowser topView |
	aRuleBrowser _ RuleBrowser new on: aStressDictionary.
	(topView _ self model: aRuleBrowser label: 'Rule Browser' minimumSize: 400@250)
		addKnowledgeBaseView: (0@0 extent: 0.5@0.3) on: aRuleBrowser readOnly: false;
		addRuleView: (0.5@0 extent: 0.5@0.3) on: aRuleBrowser readOnly: false;
		addIfTextView: (0@0.3 extent: 1@0.35) on: aRuleBrowser initialSelection: nil;
		addThenTextView: (0@0.65 extent: 1@0.35) on: aRuleBrowser initialSelection: nil.
	topView controller open! !

StandardSystemView subclass: #StressBrowserView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Browsers'!
StressBrowserView comment:
'I am a StandardSystemView for viewing a StressBrowser.  I provide initialization methods (messages to myself) to create and schedule such a browser.'!


!StressBrowserView methodsFor: 'subview creation'!

addClassView: area on: aStressBrowser readOnly: RO
	self addSubView:
		(SelectionInListView on: aStressBrowser printItems: false oneItem: RO
			aspect: #className change: #className: list: #classList
			menu: #classMenu initialSelection: #className)
		in: area borderWidth: 1!

addFormatView: area on: aStressBrowser readOnly: ignored
	| mid |
	mid _ (area left + area right) * 0.5.
	self addSubView: (BooleanView on: aStressBrowser aspect: #format
			label: 'formatted' asText change: #format: value: true)
		in: (area copy right: mid) borderWidth: 1.
	self addSubView: (BooleanView on: aStressBrowser aspect: #format
			label: 'raw' asText change: #format: value: false)
		in: (area copy left: mid) borderWidth: 1!

addInstanceView: area on: aStressBrowser readOnly: RO
	self addSubView:
		(SelectionInListView on: aStressBrowser printItems: false oneItem: RO
			aspect: #instanceName change: #instanceName: list: #instanceList
			menu: #instanceMenu initialSelection: #instanceName)
		in: area borderWidth: 1!

addKnowledgeBaseView: area on: aStressBrowser readOnly: RO
	self addSubView:
		(SelectionInListView on: aStressBrowser printItems: false oneItem: RO
			aspect: #knowledgeBaseName change: #knowledgeBaseName: list: #knowledgeBaseList
			menu: #knowledgeBaseMenu initialSelection: #knowledgeBaseName)
		in: area borderWidth: 1!

addSlotView: area on: aStressBrowser readOnly: RO
	self addSubView:
		(SelectionInListView on: aStressBrowser printItems: false oneItem: RO
			aspect: #slot change: #slot: list: #slotList
			menu: #slotMenu initialSelection: #slot)
		in: area borderWidth: 1!

addTextView: area on: aStressBrowser initialSelection: sel
	| textView | 
	textView _ (CodeView on: aStressBrowser aspect: #text
				change: #acceptAndTranscript:from:
				menu: #textMenu initialSelection: sel).
	self addSubView: textView in: area borderWidth: 1.
	textView controller: AlwaysAcceptCodeController new! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

StressBrowserView class
	instanceVariableNames: ''!


!StressBrowserView class methodsFor: 'instance creation'!

openOn: aStressDictionary
	"StressBrowserView openOn: Stress."
	| aStressBrowser topView topY bottomY formatY |
	aStressBrowser _ StressBrowser new on: aStressDictionary.
	topY _ 0.35.  "Change this to re-proportion stress browser"
	bottomY _ 1 - topY.
	formatY _ 0.05.  "Change this to re-proportion stress browser"
	(topView _ self model: aStressBrowser label: 'Stress Browser' minimumSize: 400@250)
		addKnowledgeBaseView: (0 @ 0 extent: 0.25 @ topY) on: aStressBrowser readOnly: false;
		addClassView: (0.25 @ 0 extent: 0.25 @ topY) on: aStressBrowser readOnly: false;
		addInstanceView: (0.5 @ 0 extent: 0.25 @ topY) on: aStressBrowser readOnly: false;
		addSlotView: (0.75 @ 0 extent: 0.25 @ (topY - formatY)) on: aStressBrowser readOnly: false;
		addFormatView: (0.75 @ (topY - formatY) extent: 0.25 @ formatY) on: aStressBrowser readOnly: false;
		addTextView: (0 @ 0.35 extent: 1 @ bottomY) on: aStressBrowser initialSelection: nil.
	topView controller open! !

Model subclass: #RuleBrowser
	instanceVariableNames: 'stressDictionary knowledgeBaseName ruleName textMode '
	classVariableNames: 'KnowledgeBaseMenu RuleMenu TextMenu '
	poolDictionaries: ''
	category: 'Stress-Browsers'!
RuleBrowser comment:
'I represent a hierarchical browser for examining Stress knowledge bases, and the rules they contain.

Instance Variables:
	stressDictionary		Stress
	knowledgeBaseName	a selection from all knowledge bases in stressDictionary
	ruleName				a selection from all rules in the knowledge base
	textMode				symbol indicating the nature of the currently viewed text'!


!RuleBrowser methodsFor: 'initialization'!

on: aStressDictionary
	stressDictionary _ aStressDictionary! !

!RuleBrowser methodsFor: 'knowledge base list'!

knowledgeBase
	knowledgeBaseName == nil ifTrue: [^nil].
	^stressDictionary at: knowledgeBaseName!

knowledgeBaseList
	^stressDictionary keys asOrderedCollection asArray!

knowledgeBaseMenu
	"RuleBrowser flushMenus"
	knowledgeBaseName == nil ifTrue: [^ActionMenu
			labels: 'add knowledge base\update\reorder rules' withCRs
			lines: #()
			selectors: #(addKnowledgeBase updateKnowledgeBases reorderRules)].
	KnowledgeBaseMenu == nil ifTrue: [KnowledgeBaseMenu _ ActionMenu
					labels: 'file out\print out\file out rules
add knowledge base\rename\remove
open transcript
update\reorder rules
show ''ready'' rules\initiate chaining' withCRs
					lines: #(3 6 7 9)
					selectors: #(fileOutKnowledgeBase printOutKnowledgeBase fileOutRules
addKnowledgeBase renameKnowledgeBase removeKnowledgeBase
openTranscript
updateKnowledgeBases reorderRules
showReadyRules initiateChaining)].
	^KnowledgeBaseMenu!

knowledgeBaseName
	^knowledgeBaseName!

knowledgeBaseName: selection 
	knowledgeBaseName _ selection.
	selection isNil ifTrue: [self textMode: nil].
	self newRuleList: ruleName!

newKnowledgeBaseList: initialSelection
	knowledgeBaseName _ initialSelection.
	self changed: #knowledgeBaseName! !

!RuleBrowser methodsFor: 'knowledge base functions'!

addKnowledgeBase
	| aString newName | 
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new knowledge base name' initially: 'knowledge base name'.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	stressDictionary newKnowledgeBase: newName.
	self newKnowledgeBaseList: newName!

fileOutKnowledgeBase
	|fileName aFileStream|
	fileName _ FillInTheBlank request: 'File out on' initialAnswer: (knowledgeBaseName, '.stress'). 
	fileName = '' ifTrue: [^nil].
	aFileStream _ FileStream fileNamed: fileName.
	self knowledgeBase fileOutOn: aFileStream.
	aFileStream shorten; close!

fileOutRules
	|fileName aFileStream|
	fileName _ FillInTheBlank request: 'File out on' initialAnswer: (knowledgeBaseName, '.rules'). 
	fileName = '' ifTrue: [^nil].
	aFileStream _ FileStream fileNamed: fileName.
	self knowledgeBase fileOutRulesOn: aFileStream.
	aFileStream shorten; close!

initiateChaining
	self knowledgeBase initiateChaining!

openTranscript
	Stress openTranscriptFor: self knowledgeBase!

printOutKnowledgeBase
		"Default to being the same as file out."
	
		self fileOutKnowledgeBase!

removeKnowledgeBase
	self changeRequest ifFalse: [^self].
	(self confirm: 'Are you certain that you want to
remove the knowledge base ', knowledgeBaseName, '?')
		ifTrue: 
		[Stress removeKnowledgeBase: self knowledgeBase.
		self newKnowledgeBaseList: nil]!

renameKnowledgeBase
	| aString newName |
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new knowledge base name' initially: knowledgeBaseName.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	stressDictionary renameKnowledgeBase: knowledgeBaseName to: newName.
	self newKnowledgeBaseList: newName!

reorderRules
	self changeRequest ifFalse: [^self].
	self textMode: #ruleOrdering!

showReadyRules
	self changeRequest ifFalse: [^self].
	self textMode: #readyRules!

updateKnowledgeBases
	self changeRequest ifFalse: [^self].
	self newKnowledgeBaseList: knowledgeBaseName! !

!RuleBrowser methodsFor: 'rule functions'!

addRule
	| aString newName | 
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter the name for the new rule' initially: 'rule name'.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	(self knowledgeBase includesRuleNamed: newName) ifTrue:
		[self notify: 'Rule ', newName, ' already exists!!'].
	self knowledgeBase newRuleNamed: newName before: self rule.
	self newRuleList: newName!

fileOutRule
	|fileName fileStream|
	fileName _ FillInTheBlank request: 'File out on' initialAnswer: (knowledgeBaseName, '-', ruleName, '.rule'). 
	fileName = '' ifTrue: [^nil].
	fileStream _ FileStream fileNamed: fileName.
	fileStream timeStamp.
	self rule fileOutOn: fileStream.
	fileStream close!

printOutRule
	self fileOutRule!

readyRule
	self rule ready: true!

removeRule
	self changeRequest ifFalse: [^self].
	(self confirm: 'Are you certain that you want to
remove the rule ', ruleName, '?')
		ifTrue: 
			[self knowledgeBase removeRule: self rule.
			self newRuleList: nil]!

renameRule
	| aString newName | 
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new name for rule' initially: ruleName.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	(self knowledgeBase renameRule: self rule to: newName)
		ifTrue: [self newRuleList: ruleName]
		ifFalse: [self notify: 'Rule ', newName, ' already exists!!']! !

!RuleBrowser methodsFor: 'rule list'!

newRuleList: initialSelection
	ruleName _ initialSelection.
	self changed: #ruleName!

rule
	ruleName == nil ifTrue: [^ nil].
	^ self knowledgeBase ruleNamed: ruleName!

ruleList
	knowledgeBaseName == nil ifTrue: [^nil].
	^ (self knowledgeBase rules collect: [:rule | rule name]) asArray!

ruleMenu
	"RuleBrowser flushMenus"
	ruleName == nil ifTrue:
		[^ ActionMenu labels: 'add rule' selectors: #(addRule)].
	RuleMenu == nil ifTrue:
		[RuleMenu _ ActionMenu
			labels: 'file out\print out\add rule\rename\remove\mark as ''ready''' withCRs
			lines: #(2 5)
			selectors: #(fileOutRule printOutRule addRule renameRule removeRule readyRule)].
	^ RuleMenu!

ruleName
	^ ruleName!

ruleName: selection
	ruleName _ selection.
	ruleName == nil
		ifTrue: [self textMode: #knowledgBase]
		ifFalse: [self textMode: #definition]! !

!RuleBrowser methodsFor: 'text'!

ifText
	| source |
	textMode == #knowledgeBase ifTrue: [^ Text new].
	textMode == #ruleOrdering ifTrue:
		[^ self knowledgeBase ruleOrdering asText].
	textMode == #readyRules ifTrue:
		[^ self knowledgeBase readyRules asText].
	textMode == #definition ifTrue:
		[source _ self knowledgeBase ifSourceForRule: self rule.
			source == nil
				ifTrue: [^ self rule ifPartTemplate asText]
				ifFalse: [^ source asText makeSelectorBoldIn: nil class]].
	^Text new!

prompt: promptString initially: initialString
	| aString |
	FillInTheBlank
		request: promptString , '
then accept or CR'
		displayAt: Sensor cursorPoint
		 centered: true
		action: [:aString]
		initialAnswer: initialString.
	^ aString!

textMenu
	TextMenu == nil ifTrue: [TextMenu _ ActionMenu
					labels: 'again\undo\copy\cut\paste\do it\print it\inspect\accept\cancel\format\spawn\explain' withCRs
					lines: #(2 5 8 10 )
					selectors: #(again undo copySelection cut paste doIt printIt inspectIt accept cancel format spawnEdits:from: explain )].
	^TextMenu!

textMode: aSymbol 
	textMode _ aSymbol.
	self changed: #ifText.
	self changed: #thenText!

thenText
	| source |
	textMode == #knowledgeBase ifTrue: [^ Text new].
	textMode == #definition ifTrue:
		[source _ self knowledgeBase thenSourceForRule: self rule.
			source == nil
				ifTrue: [^ self rule thenPartTemplate asText]
				ifFalse: [^ source asText makeSelectorBoldIn: nil class]].
	^Text new! !

!RuleBrowser methodsFor: 'doIt/accept/explain'!

acceptIfText: aText from: aController
	textMode == #ruleOrdering ifTrue:
		[self knowledgeBase ruleOrdering: aText asString.
		self newRuleList: ruleName.
		^ true].
	textMode == #definition ifTrue:
		[^ self knowledgeBase
				compileRuleBlock: aText
				rule: self rule
				notifying: aController].
	^ false!

acceptThenText: aText from: aController
	textMode == #definition ifTrue:
		[^ self knowledgeBase
				compileRuleBlock: aText
				rule: self rule
				notifying: aController].
	^ false!

changeRequestFrom: aView
	"Never refuse a request from one of the code views."
	(aView isMemberOf: CodeView)
		ifTrue: [^ true]
		ifFalse: [^ super changeRequestFrom: aView]!

doItContext
	^nil!

doItReceiver
	^nil!

doItValue: ignore 
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

RuleBrowser class
	instanceVariableNames: ''!


!RuleBrowser class methodsFor: 'class initialization'!

flushMenus  "RuleBrowser flushMenus."
	"Causes all menus to be newly created (so changes appear)"
	KnowledgeBaseMenu _ nil.
	RuleMenu _ nil.
	TextMenu _ nil! !

Model subclass: #StressBrowser
	instanceVariableNames: 'stressDictionary knowledgeBaseName className instanceName slot format textMode '
	classVariableNames: 'ClassMenu InstanceMenu KnowledgeBaseMenu SlotMenu TextMenu '
	poolDictionaries: ''
	category: 'Stress-Browsers'!
StressBrowser comment:
'I represent a hierarchical browser for examining Stress knowledge bases, and the classes and instances they contain.

Instance Variables:
	stressDictionary		Stress
	knowledgeBaseName	a selection from all knowledge bases in stressDictionary
	className				a selection from all classes in the knowledge base
	instanceName			a selection from all instances belonging to the class
	slot						a selection from all slots of the class
	format					false for viewing collections raw, true for viewing them formatted
	textMode				symbol indicating the nature of the currently viewed text'!


!StressBrowser methodsFor: 'initialization'!

on: aStressDictionary
	stressDictionary _ aStressDictionary.
	format _ true! !

!StressBrowser methodsFor: 'knowledge base list'!

knowledgeBase
	knowledgeBaseName == nil ifTrue: [^ nil].
	^ stressDictionary at: knowledgeBaseName!

knowledgeBaseList
	^ stressDictionary keys asOrderedCollection asArray!

knowledgeBaseMenu
	"StressBrowser flushMenus"
	knowledgeBaseName == nil ifTrue:
		[^ ActionMenu labels: 'add knowledge base\update\edit classes' withCRs
			lines: #(1)
			selectors: #(addKnowledgeBase updateKnowledgeBases editClasses)].
	KnowledgeBaseMenu == nil ifTrue:
		[KnowledgeBaseMenu _ ActionMenu
			labels: 'file out\print out\file out facts
add knowledge base\rename\remove
open transcript
update\edit classes
comment\suggest' withCRs
			lines: #(3 6 7  9)
			selectors: #(fileOutKnowledgeBase printOutKnowledgeBase fileOutFacts
addKnowledgeBase renameKnowledgeBase removeKnowledgeBase
openTranscript
updateKnowledgeBases editClasses
knowledgeBaseComment suggest)].
	^ KnowledgeBaseMenu!

knowledgeBaseName
	^ knowledgeBaseName!

knowledgeBaseName: selection
	knowledgeBaseName _ selection.
	selection isNil ifTrue: [self textMode: nil].
	self newClassList: className!

newKnowledgeBaseList: initialSelection
	knowledgeBaseName _ initialSelection.
	self changed: #knowledgeBaseName! !

!StressBrowser methodsFor: 'knowledge base functions'!

addKnowledgeBase
	| aString newName | 
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new knowledge base name' initially: 'knowledge base name'.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	stressDictionary newKnowledgeBase: newName.
	self newKnowledgeBaseList: newName!

editClasses
	self changeRequest ifFalse: [^self].
	self textMode: #classes!

fileOutFacts
	|fileName aFileStream|
	fileName _ FillInTheBlank request: 'File out on' initialAnswer: (knowledgeBaseName, '.facts'). 
	fileName = '' ifTrue: [^nil].
	aFileStream _ FileStream fileNamed: fileName.
	self knowledgeBase fileOutFactsOn: aFileStream.
	aFileStream shorten; close!

fileOutKnowledgeBase
	|fileName aFileStream|
	fileName _ FillInTheBlank request: 'File out on' initialAnswer: (knowledgeBaseName, '.stress'). 
	fileName = '' ifTrue: [^nil].
	aFileStream _ FileStream fileNamed: fileName.
	self knowledgeBase fileOutOn: aFileStream.
	aFileStream shorten; close!

knowledgeBaseComment
	self changeRequest ifFalse: [^ self].
	self textMode: #knowledgeBaseComment.
	self newClassList: nil!

openTranscript
	Stress openTranscriptFor: self knowledgeBase!

printOutKnowledgeBase
		"Default to being the same as file out."
	
		self fileOutKnowledgeBase!

removeKnowledgeBase
	self changeRequest ifFalse: [^self].
	(self confirm: 'Are you certain that you want to
remove the knowledge base ', knowledgeBaseName, '?')
		ifTrue: 
		[Stress removeKnowledgeBase: self knowledgeBase.
		self newKnowledgeBaseList: nil]!

renameKnowledgeBase
	| aString newName |
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new knowledge base name' initially: knowledgeBaseName.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	stressDictionary renameKnowledgeBase: knowledgeBaseName to: newName.
	self newKnowledgeBaseList: newName!

suggest
	| suggestion | 
	suggestion _ self knowledgeBase suggest.
	suggestion == nil ifTrue: [^ self].
	className _ suggestion key.
	instanceName _ nil.
	slot _ suggestion value.
	self textMode: #classSlotValue.
	self changed: #className!

updateKnowledgeBases
	self changeRequest ifFalse: [^self].
	stressDictionary do: [:kb | kb checkClasses].
	self newKnowledgeBaseList: knowledgeBaseName! !

!StressBrowser methodsFor: 'class list'!

classList
	knowledgeBaseName == nil ifTrue: [^nil].
	^ ChangeSet hierarchicalOrder: self knowledgeBase classes!

classMenu
	"StressBrowser flushMenus"
	className == nil ifTrue:
		[^ ActionMenu labels: 'add existing class' selectors: #(addClass)].
	ClassMenu == nil ifTrue:
		[ClassMenu _ ActionMenu
			labels: 'file out\print out\spawn\spawn hierarchy
definition\comment
add existing class\rename\remove\remove instances' withCRs
			lines: #(4 6)
			selectors: #(fileOutClass printOutClass spawnClass spawnHierarchy  
editClass classComment
addClass renameClass removeClass removeInstances)].
	^ ClassMenu!

className
	^ className!

className: selection
	className _ selection isNil
		ifTrue: [nil]
		ifFalse: [(selection copyWithout: $ ) asSymbol].
	(className == nil and: [self commentMode & (textMode ~~ #classComment)])
			ifTrue: [^ self newInstanceList: nil].
	self newInstanceList: instanceName.
	selection isNil & knowledgeBaseName notNil
		ifTrue: [self textMode: #classDefinition]
		ifFalse: [self classMode ifTrue: [self changed: #text]]!

newClassList: initialSelection
	className _ initialSelection.
	self changed: #className!

selectedClass
	className == nil ifTrue: [^ nil].
	^ Smalltalk at: className! !

!StressBrowser methodsFor: 'class functions'!

acceptClass: aText from: aController 
	| oldClass class |
	oldClass _ className == nil ifTrue: [Object] ifFalse: [self selectedClass].
	class _ oldClass subclassDefinerClass
				evaluate: aText string
				notifying: aController
				logged: true.
	(class isKindOf: Behavior)
		ifTrue: [self knowledgeBase addClass: class.
				self newClassList: class name.  ^true]
		ifFalse: [^false]!

addClass
	| aString newClass | 
	self changeRequest ifFalse: [^self].
	aString _ self
			prompt: 'Enter the name of the class (must already exist)'
			initially: 'class name'.
	aString isEmpty ifTrue: [^ self].
	newClass _ Smalltalk at: aString asSymbol
		ifAbsent: [self notify: 'Class ', aString, ' does not exist.'].
	self knowledgeBase addClass: newClass.
	self newClassList: aString!

browseClassReferences
	Smalltalk browseAllCallsOn: (Smalltalk associationAt: className)!

classComment
	self changeRequest ifFalse: [^self].
	self textMode: #classComment.
	self newInstanceList: nil!

classMode
	^#(classDefinition hierarchy classComment protocols) includes: textMode!

editClass
	self changeRequest ifFalse: [^self].
	self textMode: #classDefinition.
	self newInstanceList: nil!

fileOutClass
	|fileName fileStream|
	fileName _ FillInTheBlank request: 'File out on' initialAnswer: (className, '.stress'). 
	fileName = '' ifTrue: [^nil].
	fileStream _ FileStream fileNamed: fileName.
	fileStream timeStamp.
	self selectedClass fileOutOn: fileStream.
	fileStream close
	"The following has been deemed silly."
	"self selectedClass removeFromChanges"!

printOutClass
	"Default to filing out the class."
	
	self fileOutClass!

removeClass
	self changeRequest ifFalse: [^self].
	(self confirm: 'Are you certain that you want to remove
class ', className, ' from the knowledge base?')
		ifTrue: 
			[self knowledgeBase removeClass: self selectedClass.
			self knowledgeBase checkClasses.
			self newClassList: nil]!

removeInstances
	self changeRequest ifFalse: [^self].
	(self confirm: 'Are you certain that you want to
remove all objects of class ', className, '?')
		ifTrue: 
			[(self knowledgeBase allKindsOf: self selectedClass) associationsDo:
				[:inst | self knowledgeBase removeAssociation: inst].
			self newInstanceList: instanceName]
			"Gives no selection if instanceName has been removed."!

renameClass
	| aString newName cleanString |
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new ClassName' initially: className.
	aString isEmpty ifTrue: [^self].
	cleanString _ aString select: [:char | char isAlphaNumeric].
	cleanString = aString
		ifFalse: [aString _ cleanString.
				(self confirm: 'Invalid characters in class name.  Should I use ', aString, '?')
					ifFalse: [^self]].
	aString first isUppercase
		ifFalse: [aString at: 1 put: (aString at: 1) asUppercase.
				(self confirm: 'Class names must be capitalized.  Should I use ', aString, '?')
					ifFalse: [^self]].
	newName _ aString asSymbol.
	aString = self selectedClass name
		ifFalse:
			[self selectedClass rename: newName.
			self knowledgeBase classNames remove: className; add: newName.
			self newClassList: newName.
			Transcript cr; show: 'Searching for references to this class...'.
			self browseClassReferences]!

spawnClass
	Browser newOnClass: (self selectedClass)!

spawnHierarchy
	| browser | 
	browser _ (Browser new on: SystemOrganization).
	browser className: className.
	BrowserView openCategoryBrowserOn:  browser browseHierarchy! !

!StressBrowser methodsFor: 'instance list'!

instance
	instanceName == nil ifTrue: [^ nil].
	^ self knowledgeBase at: instanceName!

instanceList
	| instances |
	className == nil ifTrue: [^nil].
	^ (self knowledgeBase allKindsOf: self selectedClass) keys asOrderedCollection asArray!

instanceMenu
	"StressBrowser flushMenus"
	instanceName == nil ifTrue:
		[^ ActionMenu labels: 'create instance' selectors: #(createInstance)].
	InstanceMenu == nil ifTrue:
		[InstanceMenu _ ActionMenu
			labels: 'create instance\rename\remove\comment' withCRs
			lines: #(3)
			selectors: #(createInstance renameInstance removeInstance instanceComment)].
	^ InstanceMenu!

instanceName
	^ instanceName!

instanceName: selection
	instanceName _ selection.
	(instanceName == nil and:
		[(self classMode | self commentMode) & (textMode ~~ #instanceComment)])
			ifTrue: [^ self newSlotList: nil].
	self newSlotList: slot.
	className ~~ nil
		ifTrue: [self decideTextMode]!

newInstanceList: initialSelection
	instanceName _ initialSelection.
	self changed: #instanceName! !

!StressBrowser methodsFor: 'instance functions'!

createInstance
	| aString newName newValueString newValue | 
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter name for new instance' initially: 'instanceName'.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	(self knowledgeBase includesKey: newName) ifTrue:
		[self notify: 'This knowledge base already contains an object called ', newName].
	newValueString _ self 
			prompt: 'Enter the initial value of ', newName
			initially: className, ' new'.
	newValue _ StressCompiler new
			evaluate: newValueString
			inKB: self knowledgeBase
			notifying: nil
			ifFail: [].
	self knowledgeBase at: newName put: newValue.
	self knowledgeBase checkClasses.
	self newInstanceList: newName!

instanceComment
	self changeRequest ifFalse: [^self].
	self textMode: #instanceComment.
	self newSlotList: nil!

removeInstance
	self changeRequest ifFalse: [^self].
	(self confirm: 'Are you certain that you want
to remove the object ', instanceName, '?')
		ifTrue: 
			[self knowledgeBase removeKey: instanceName.
			self newInstanceList: nil]!

renameInstance
	| aString newName |
	self changeRequest ifFalse: [^self].
	aString _ self prompt: 'Enter new name for object' initially: instanceName.
	aString isEmpty ifTrue: [^ self].
	newName _ aString asSymbol.
	(self knowledgeBase renameObject: instanceName to: newName)
		ifTrue: [self newInstanceList: newName]
		ifFalse: [self notify: 'Object ', newName, ' already exists!!']! !

!StressBrowser methodsFor: 'slot list'!

newSlotList: initialSelection
	slot _ initialSelection.
	self changed: #slot!

slot
	^ slot!

slot: aSelection
	slot _ aSelection.
	(slot == nil and:
		[(self classMode | self commentMode) & (textMode ~~ #slotComment)])
			ifTrue: [^ self].
	className ~~ nil
		ifTrue: [self decideTextMode]!

slotList
	| list | 
	className == nil ifTrue: [^ nil].
	instanceName == nil
		ifTrue: [list _ self selectedClass allSlots]
		ifFalse: [list _ self instance allSlots].
	self knowledgeBase hiddenSlots do: [:hidden | list remove: hidden ifAbsent: []].
	^ list!

slotMenu
	"StressBrowser flushMenus"
	slot == nil ifTrue:
		[^ ActionMenu labels: 'show all' selectors: #(showAll)].
	SlotMenu == nil ifTrue:
		[SlotMenu _ ActionMenu
			labels: 'hide slot\show all\comment' withCRs
			lines: #(2)
			selectors: #(hideSlot showAll slotComment)].
	^ SlotMenu! !

!StressBrowser methodsFor: 'slot functions'!

hideSlot
	self changeRequest ifFalse: [^ self].
	self knowledgeBase hideSlot: slot.
	self newSlotList: nil!

showAll
	self changeRequest ifFalse: [^ self].
	self knowledgeBase unhideSlots.
	self newSlotList: slot!

slotComment
	self changeRequest ifFalse: [^self].
	self textMode: #slotComment! !

!StressBrowser methodsFor: 'format switch'!

format
	^ format!

format: aBoolean
	self changeRequest ifFalse: [^ self changed: #format].
	format _ aBoolean.
	self changed: #format.
	self changed: #text! !

!StressBrowser methodsFor: 'text'!

acceptAndTranscript: aText from: aController
	"Alert the transcript (if active) to expect notification of browser interaction, then accept the text."
	| value |
	knowledgeBaseName == nil  "Eg when editing classes"
		ifTrue: [^ self acceptText: aText from: aController].
	Stress startBrowserTranscript: self knowledgeBase.
	value _ self acceptText: aText from: aController.
	Stress endTranscript.
	^ value!

acceptText: aText from: aController
	| newValue commentClass |
	textMode == #classDefinition ifTrue:
		[^ self acceptClass: aText from: aController].
	textMode == #knowledgeBaseComment ifTrue:
		[self knowledgeBase comments globalComment: aText string.
		^ true].
	textMode == #classComment ifTrue:
		[self selectedClass comment: aText string.
		^ true].
	textMode == #instanceComment ifTrue:
		[self knowledgeBase comments objectAt: instanceName comment: aText string.
		^ true].
	textMode == #slotComment ifTrue:
		[commentClass _ (self selectedClass whichSuperHasSlot: slot) name.
		self knowledgeBase comments classNamed: commentClass slot: slot comment: aText string.
		^ true].
	textMode == #instanceValue ifTrue:
		[newValue _ StressCompiler new
				evaluate: aText
				inKB: self knowledgeBase
				notifying: aController
				ifFail: [^ false].
		self knowledgeBase at: instanceName put: newValue.
		self newClassList: (self knowledgeBase classForDisplaying: newValue) name.
		^ true].
	textMode == #slotValue ifTrue:
		[^ self knowledgeBase fillSlot: slot of: self instance fromString: aText asString notifying: aController].
	textMode == #classSlotValue ifTrue:
		[^ self knowledgeBase fillSlot: slot of: self selectedClass fromString: aText asString notifying: aController].
	textMode == #classes ifTrue:
		[stressDictionary classOrganizationFromString: aText asString.
		self newKnowledgeBaseList: knowledgeBaseName.
		^ true].
	^ false!

commentMode
	^#(knowledgeBaseComment classComment instanceComment slotComment) includes: textMode!

decideTextMode
	"Determine whether the text represents a slot value, an instance value, a class slot value, or a class definition."
	instanceName == nil
		ifFalse: [slot == nil
			ifFalse: [self textMode: #slotValue]
			ifTrue: [self textMode: #instanceValue]]
		ifTrue: [slot == nil
			ifFalse: [self textMode: #classSlotValue]
			ifTrue: [self textMode: #classDefinition]]!

prompt: promptString initially: initialString
	| aString |
	FillInTheBlank
		request: promptString , '
then accept or CR'
		displayAt: Sensor cursorPoint
		centered: true
		action: [:aString]
		initialAnswer: initialString.
	^ aString!

text
	| text string commentClass |
	textMode == #classDefinition ifTrue:
		[className == nil
			ifTrue: [^ (Class stressTemplate: self knowledgeBase categoryForClasses) asText]
			ifFalse: [^ self selectedClass stressDefinition asText]].
	textMode == #classes ifTrue:
		[^ stressDictionary classOrganization].
	textMode == #knowledgeBaseComment ifTrue:
		[text _ self knowledgeBase comments globalComment asText.
		text isEmpty ifFalse: [^ text].
		^ 'This knowledge base has no comment' asText].
	textMode == #classComment ifTrue:
		[text _ self selectedClass comment asText.
		text isEmpty ifFalse: [^ text].
		^ 'This class has no comment' asText].
	textMode == #instanceComment ifTrue:
		[text _ (self knowledgeBase comments commentForObjectAt: instanceName) asText.
		text isEmpty ifFalse: [^ text].
		^ 'This object has no comment' asText].
	textMode == #slotComment ifTrue:
		[commentClass _ (self selectedClass whichSuperHasSlot: slot) name.
		text _ (self knowledgeBase comments commentForClass: commentClass slot: slot) asText.
		text isEmpty ifFalse: [^ text].
		^ 'This slot has no comment' asText].
	textMode == #instanceValue ifTrue:
		[^ self instance printString asText].
	textMode = #slotValue ifTrue:
		[string _ (self knowledgeBase printSlot: slot of: self instance format: format).
		string = 'nil'
			ifTrue: [^ Text new]
			ifFalse: [^ string asText]].
	textMode == #classSlotValue ifTrue:
		[string _ (self knowledgeBase printSlot: slot of: self selectedClass format: format).
		string = 'nil'
			ifTrue: [^ Text new]
			ifFalse: [^ string asText]].
	^ Text new!

textMenu
	"StressBrowser flushMenus"
	TextMenu == nil ifTrue:
		[TextMenu _ ActionMenu
			labels: 'again\undo\copy\cut\paste\do it\print it\inspect\accept\cancel\format\spawn\explain' withCRs
			lines: #(2 5 8 10)
			selectors: #(again undo copySelection cut paste doIt printIt inspectIt accept cancel format spawnEdits:from: explain)].
	^ TextMenu!

textMode: aSymbol
	textMode _ aSymbol.
	self changed: #text! !

!StressBrowser methodsFor: 'doIt/accept/explain'!

doItContext
	^ nil!

doItReceiver
	^ self knowledgeBase!

doItValue: ignored! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

StressBrowser class
	instanceVariableNames: ''!


!StressBrowser class methodsFor: 'class initialization'!

flushMenus  "StressBrowser flushMenus."
	"Causes all menus to be newly created (so changes appear)"
	KnowledgeBaseMenu _ nil.
	ClassMenu _ nil.
	InstanceMenu _ nil.
	SlotMenu _ nil.
	TextMenu _ nil! !TextCollector subclass: #StressTranscript
	instanceVariableNames: 'knowledgeBaseName on terse running list texts index level savedMessages '
	classVariableNames: 'RuleExecutingMenu RunningMenu '
	poolDictionaries: ''
	category: 'Stress-Transcript'!
StressTranscript comment:
'I represent a transcript into which text can be gathered describing the progress of Stress rules and browser interactions.

Instance Variables:
	knowledgeBaseName	<Symbol>
	on						<Boolean>	True if active and receiving text, false otherwise
	terse					<Boolean>	True if text is to be summarised.
	running					<String>	Identifies rule curently executing (or browser interaction)
	list						<OrderedCollection>		All the ''running'' strings
	texts					<OrderedCollection>		All texts (matches ''list'')
	index					<Integer>	Current index to ''list'' and ''texts''
	level					<Integer>	Current level of indentation
	savedMessages			<Array>		Messages held back in terse mode'!


!StressTranscript methodsFor: 'initialize-release'!

clear
	"Clear all entries to give a fresh transcript."
	running _ 'Browser Interaction'.
	list _ OrderedCollection with: 'Browser Interaction'.
	texts _ OrderedCollection with: Text new.
	index _ 1.
	super initialize.
	self changed: #running!

knowledgeBase: aKnowledgeBase
	"Set the knowledge base and initialize"
	knowledgeBaseName _ aKnowledgeBase name. 
	running _ 'Browser Interaction'.
	list _ OrderedCollection with: 'Browser Interaction'.
	texts _ OrderedCollection with: Text new.
	on _ true.
	terse_ true.
	index _ 1.
	super initialize!

release
	Stress releaseTranscriptFor: knowledgeBaseName! !

!StressTranscript methodsFor: 'accesing'!

knowledgeBase
	^ Stress at: knowledgeBaseName! !

!StressTranscript methodsFor: 'running list'!

contentsHasChanged
	"Answer whether the current value of 'contents' is different from the saved text."
	^ index > 0 and: [contents ~= (texts at: index)]!

running
	^ running!

running: selection
	self endEntry.
	self contentsHasChanged ifTrue: [texts at: index put: contents copy].
	savedMessages _ Array new: 20.
	selection == nil
		ifTrue:
			[index _ 0.
			contents _ Text new]
		ifFalse:
			[index _ list indexOf: selection.
			contents _ texts at: index].
	self changed: #update!

runningList
	^ list!

runningMenu
	"StressTranscript flushMenus"
	RunningMenu == nil ifTrue:
		[RunningMenu _ ActionMenu
			labels: 'clear' withCRs
			lines: #()
			selectors: #(clear)].
	^ RunningMenu! !

!StressTranscript methodsFor: 'switches'!

on
	^ on!

on: aBoolean
	on _ aBoolean.
	self changed: #on!

terse
	^ terse!

terse: aBoolean
	terse _ aBoolean.
	self changed: #terse! !

!StressTranscript methodsFor: 'transcript sessions'!

startBrowserTranscript
	running _ 'Browser Interaction'.
	level _ 0.
	self changed: #running!

startRuleTranscript: aRule
	running _ (list size) printString, '.	 ', aRule name.
	list add: running.
	texts add: Text new.
	level _ 0.
	self changed: #running! !

!StressTranscript methodsFor: 'transcript messages'!

bind: binderName to: instance
	| string |
	(level = 1 and: [contents isEmpty not])
		ifTrue: [string _ String with: Character cr]
		ifFalse: [string _ ''].
	string _ string, self indent, '{', binderName, '} _ ', (self knowledgeBase printString: instance).
	terse
		ifFalse: [self show: string; cr]
		ifTrue: [savedMessages at: level put: string]!

conclude: aString
	| msg | 
	terse ifTrue:
		[1 to: level+1 do:
			[:i | (msg _ savedMessages at: i) ~~ nil
				ifTrue: [self show: msg; cr].
			savedMessages at: i put: nil]].
	self show: self indent, '>>>> ', aString; cr!

object: anObject slot: slotName add: aValue
	| kb | 
	kb _ self knowledgeBase.
	self conclude:
		(kb printString: anObject), ' slot: ',
		slotName printString, ' add: ',
		(kb printString: aValue)!

object: anObject slot: slotName put: aValue
	| kb | 
	kb _ self knowledgeBase.
	self conclude:
		(kb printString: anObject), ' slot: ',
		slotName printString, ' put: ',
		(kb printString: aValue)!

object: anObject slot: slotName remove: aValue
	| kb | 
	kb _ self knowledgeBase.
	self conclude:
		(kb printString: anObject), ' slot: ',
		slotName printString, ' remove: ',
		(kb printString: aValue)!

ruleReady: aRule
	self show: self indent, '	 (Rule ', aRule name, ' will execute)'; cr!

ruleSucceeded
	| string | 
	string _ self indent, 'Rule succeeded - executing conclusion'.
	terse
		ifFalse: [self show: string; cr]
		ifTrue: [savedMessages at: level + 1 put: string]! !

!StressTranscript methodsFor: 'indenting'!

indent
	"Answer a string representing the current indent, ie 'level' number of tabs."
	level > 1
		ifTrue: [^ String new: level - 1 withAll: Character tab]
		ifFalse: [^ '']!

levelDown
	level _ level - 1!

levelUp
	level _ level + 1! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

StressTranscript class
	instanceVariableNames: ''!


!StressTranscript class methodsFor: 'instance creation'!

on: aKnowledgeBase
	^ self new knowledgeBase: aKnowledgeBase! !

!StressTranscript class methodsFor: 'class initialization'!

flushMenus  "StressTranscript flushMenus."
	"Causes all menus to be newly created (so changes appear)"
	RunningMenu _ nil! !

SelectionInListView subclass: #SafeSelectionInListView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Transcript'!
SafeSelectionInListView comment:
'I am a SelectionInListView that uses ''displaySafe'', so that I can be used in a transcript without making a mess of the display.'!


!SafeSelectionInListView methodsFor: 'updating'!

update: aSymbol 
	aSymbol == partMsg
		ifTrue: [self displaySafe: [self list: self getList; displayView]]! !

StandardSystemView subclass: #StressTranscriptView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Stress-Transcript'!
StressTranscriptView comment:
'I am a StandardSystemView for viewing the contents of a StressTranscript.'!


!StressTranscriptView methodsFor: 'initialize-release'!

release
	"Release my model (a StressTranscript) and then myself."
	model release.
	super release! !

!StressTranscriptView methodsFor: 'subview creation'!

addOnOffView: area on: aStressTranscript readOnly: ignored
	| mid |
	mid _ (area left + area right) * 0.5.
	self addSubView: (BooleanView on: aStressTranscript aspect: #on
			label: 'on' asText change: #on: value: true)
		in: (area copy right: mid) borderWidth: 1.
	self addSubView: (BooleanView on: aStressTranscript aspect: #on
			label: 'off' asText change: #on: value: false)
		in: (area copy left: mid) borderWidth: 1!

addRunningView: area on: aStressTranscript readOnly: RO
	self addSubView:
		(SafeSelectionInListView on: aStressTranscript printItems: false oneItem: RO
			aspect: #running change: #running: list: #runningList
			menu: #runningMenu initialSelection: #running)
		in: area borderWidth: 1!

addTerseView: area on: aStressTranscript readOnly: ignored
	| mid |
	mid _ (area left + area right) * 0.5.
	self addSubView: (BooleanView on: aStressTranscript aspect: #terse
			label: 'terse' asText change: #terse: value: true)
		in: (area copy right: mid) borderWidth: 1.
	self addSubView: (BooleanView on: aStressTranscript aspect: #terse
			label: 'verbose' asText change: #terse: value: false)
		in: (area copy left: mid) borderWidth: 1!

addTextView: area on: aStressTranscript
	| textView | 
	textView _ TextCollectorView new model: aStressTranscript.
	textView controller turnLockingOff.
	textView insideColor: Form white.
	self addSubView: textView in: area borderWidth: 1! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

StressTranscriptView class
	instanceVariableNames: ''!


!StressTranscriptView class methodsFor: 'instance creation'!

openOn: aStressTranscript
	| label topView |
	label _ 'Transcript for ', aStressTranscript knowledgeBase printString.
	(topView _ self model: aStressTranscript label: label minimumSize: 400@250)
		addRunningView: (0@0 extent: 1@0.25) on: aStressTranscript readOnly: false;
		addOnOffView: (0@0.25 extent: 0.5@0.05) on: aStressTranscript readOnly: false;
		addTerseView: (0.5@0.25 extent: 0.5@0.05) on: aStressTranscript readOnly: false;
		addTextView: (0@0.3 extent: 1@0.7) on: aStressTranscript.
	topView controller open! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 21 February 1989 at 2:57:04 pm'!

Object subclass: #RuleReader
	instanceVariableNames: 'rule knowledgeBase '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Kernel-Support'!
RuleReader comment:
'An instance of class RuleReader is created during file in, in response to a "knowledgeBase codeForRule: ruleName" message.  The instance subsequently scans consecutive "chunks" from the file in stream and asks the knowledge base to compile them in this rule.  It continues in this way until an empty chunk is found.  There will normally be two chunks, one for ''if'' and one for ''then''.

Instance Variables:
	knowledgeBase		<KnowledgeBase>
	rule					<Rule>'!


!RuleReader methodsFor: 'fileIn/Out'!

scanFrom: aStream 
	"Files in rules from the stream, aStream.  Prints the name and category of the methods in the transcript view."

	| string |
	[string _ aStream nextChunk.
	string size > 0]						"done when double terminators"
		whileTrue: [knowledgeBase compileRuleBlock: string rule: rule notifying: nil].
	Transcript show: knowledgeBase name printString, '<' , rule printString , '
'! !

!RuleReader methodsFor: 'private'!

setKnowledgeBase: aKnowledgeBase rule: aRule
	knowledgeBase _ aKnowledgeBase.
	rule _ aRule! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

RuleReader class
	instanceVariableNames: ''!


!RuleReader class methodsFor: 'instance creation'!

knowledgeBase: aKnowledgeBase rule: aRule
	"Answer a new instance of RuleReader for aRule, in the knowledge base, aKnowledgeBase."

	^self new setKnowledgeBase: aKnowledgeBase rule: aRule! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 21 February 1989 at 2:57:05 pm'!

Object subclass: #StressObjectReader
	instanceVariableNames: 'knowledgeBase '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Kernel-Support'!
StressObjectReader comment:
'An instance of class StressObjectReader is created during file in, in response to a "knowledgeBase classesAndInstances" message.  The instance subsequently scans consecutive "chunks" of {class {instance instance instance . . . }} descriptions from the file, and adds these classes and ''basicNew'' instances to the knowledge base.  It continues in this way until an empty chunk is found.

Instance Variables:
	knowledgeBase		<KnowledgeBase>'!


!StressObjectReader methodsFor: 'private'!

setKnowledgeBase: aKnowledgeBase
	knowledgeBase _ aKnowledgeBase! !

!StressObjectReader methodsFor: 'fileIn/Out'!

scanFrom: aStream 
	"Files in a series of {class {instance instance instance . . . }} descriptions.  Adds the classes and their specified instances to knowledgeBase."

	| string names class |
	Cursor execute showWhile: [
	[string _ aStream nextChunk.
	string size > 0]						"done when double terminators"
		whileTrue:
			[names _ Scanner new scanTokens: string.
			class _ Smalltalk at: names first ifAbsent: [nil].
			class notNil ifTrue:
				[knowledgeBase addClass: class.
				2 to: names size do:
					[:i | knowledgeBase at: (names at: i) put: class basicNew]]]]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

StressObjectReader class
	instanceVariableNames: ''!


!StressObjectReader class methodsFor: 'instance creation'!

knowledgeBase: aKnowledgeBase
	"Answer a new instance of StressObjectReader for the knowledge base, aKnowledgeBase."

	^self new setKnowledgeBase: aKnowledgeBase! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 28 February 1989 at 2:13:16 pm'!



!Object methodsFor: 'stress operations'!

, aCollection
	^ self coerceToCollection, aCollection!

allSlots
	^ self class allInstVarNames!

coerceToCollection
	"Answer as if I was a one-element collection."
	^ OrderedCollection with: self!

has: anObject
	"Coerce myself into a Collection, and send the result the message collectionHas: anObject."
	^ self coerceToCollection collectionHas: anObject!

indexOfSlot: aSlot
	^ self
		indexOfSlot: aSlot
		ifAbsent: [self error: 'This object has no slot ', aSlot]!

indexOfSlot: aSlot ifAbsent: aBlock
	^ self allSlots
		indexOf: aSlot
		ifAbsent: [^ aBlock value]!

slot: aSlot
	^ self slot: aSlot ifAbsent:
		[self error: self class name, ' has no slot ', aSlot printString]!

slot: aSlot add: anObject
	"Add anObject to the values of aSlot."

	| index contents transcript | 
	index _ self indexOfSlot: aSlot.
	contents _ (self instVarAt: index) coerceToCollection.
	contents add: anObject.
	self instVarAt: index put: contents.
	Stress ifTranscriptActive: [:transcript | transcript object: self slot: aSlot add: anObject].
	self changed: aSlot.
	^ anObject!

slot: aSlot addIfAbsent: anObject
	"Add anObject to the values of aSlot, unless it is already present."
	(self slot: aSlot has: anObject)
		ifFalse: [^ self slot: aSlot add: anObject]!

slot: aSlot has: anObject
	^ (self slot: aSlot) has: anObject!

slot: aSlot ifAbsent: aBlock
	^ self instVarAt: (self indexOfSlot: aSlot ifAbsent: [^ aBlock value])!

slot: aSlot put: anObject
	^ self slot: aSlot put: anObject ifAbsent:
		[self error: self class name, ' has no slot ', aSlot printString]!

slot: aSlot put: anObject ifAbsent: aBlock
	| index transcript |
	index _ self indexOfSlot: aSlot ifAbsent: [^ aBlock value].
	self instVarAt: index put: anObject.
	Stress ifTranscriptActive: [:transcript | transcript object: self slot: aSlot put: anObject].
	self changed: aSlot!

slot: aSlot put: anObject in: aKnowledgeBase
	^ self slot: aSlot put: anObject!

slot: aSlot remove: anObject
	"Remove anObject from the values of aSlot."

	| index contents transcript |
	index _ self indexOfSlot: aSlot.
	contents _ (self instVarAt: index) coerceToCollection.
	contents remove: anObject.
	Stress ifTranscriptActive: [:transcript | transcript object: self slot: aSlot remove: anObject].
	self changed: aSlot!

slot: aSlot removeIfPresent: anObject
	"If aSlot includes anObject, then remove it.  Otherwise do nothing."
	(self slot: aSlot has: anObject)
		ifTrue: [^ self slot: aSlot remove: anObject]!

slots
	^ self class instVarNames! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 28 February 1989 at 2:13:14 pm'!



!Collection methodsFor: 'stress operations'!

coerceToCollection
	"I am already a Collection, so just return self."
	^ self!

collectionHas: anObject
	"If anObject is a Binder, answer with an OrderedCollection of all my elements which fit the class constraint.  If anObject is not a Binder, answer whether it is one of my elements."

	| qualifiers |  
	(anObject class == Binder)
		ifTrue:
			[qualifiers _ OrderedCollection new.
			self do: [:each | (each isKindOf: anObject constraint) ifTrue: [qualifiers add: each]].
			^ qualifiers]
		ifFalse:
			[^ self includes: anObject]! !
!Collection reorganize!
('accessing' size)
('testing' includes: isEmpty occurrencesOf:)
('adding' add: addAll:)
('removing' remove: remove:ifAbsent: removeAll:)
('enumerating' collect: detect: detect:ifNone: do: inject:into: reject: select:)
('printing' printOn: storeOn:)
('converting' asBag asOrderedCollection asSet asSortedCollection asSortedCollection:)
('private' emptyCheck errorEmptyCollection errorNoMatch errorNotFound errorNotKeyed growSize maxPrint)
('stress operations' coerceToCollection collectionHas:)
!

'From Smalltalk-80, Version 2.2 of July 4, 1987 on 13 February 1989 at 4:27:32 pm'!



!UndefinedObject methodsFor: 'stress operations'!

coerceToCollection
	"Answer as if I was an empty collection."
	^ OrderedCollection new! !
!UndefinedObject reorganize!
('initialize-release' release)
('copying' deepCopy shallowCopy)
('printing' printOn: storeOn:)
('testing' isNil notNil)
('dependents access' addDependent:)
('instance creation' new)
('stress operations' coerceToCollection)
!

'From Smalltalk-80, Version 2.2 of July 4, 1987 on 13 February 1989 at 4:27:31 pm'!



!String methodsFor: 'stress operations'!

coerceToCollection
	"Act like a single object, not like a collection."
	^ OrderedCollection with: self! !
!String reorganize!
('accessing' at: at:put: basicAt: basicAt:put: findString:startingAt: replaceFrom:to:with:startingAt: replaceFrom:to:withByteArray:startingAt: size string)
('comparing' < <= = > >= hash match: match:ignoreCase: sameAs: spellAgainst:)
('copying' copyUpTo: deepCopy)
('printing' isLiteral printOn: storeOn:)
('converting' asByteArray asDisplayText asFileName asLowercase asNumber asParagraph asString asSymbol asText asUppercase contractTo: oldRunDecodeOn: oldRunEncoded withCRs)
('displaying' displayAt: displayOn:at:)
('private' compare: primReplaceFrom:to:with:startingAt: stringhash)
('stress operations' coerceToCollection)
!

'From Smalltalk-80, Version 2.2 of July 4, 1987 on 20 February 1989 at 1:09:59 pm'!



!Behavior methodsFor: 'private'!

updateDependencies: oldSelf
	"I have replaced an old behavior or class.  Transfer any dependents from oldSelf."
	oldSelf dependents do:
		[:dep | self addDependent: dep].
	oldSelf breakDependents! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 28 February 1989 at 2:13:18 pm'!



!ClassDescription methodsFor: 'fileIn/Out'!

fileOutOn: aFileStream moveSource: moveSource toFile: fileIndex
	"File me out on aFileStream.  If aFileStream is a Stress file, use stressDefinition."
	| fileName extn | 
	aFileStream emphasis: 5.		"Meant to be 12 point bold font."
	fileName _ aFileStream fileName.
	extn _ (fileName reverse copyUpTo: $.) reverse.
	((extn size ~= fileName size) and: [#('stress' 'facts') includes: extn])
		ifTrue: [aFileStream nextChunkPut: self stressDefinition]
		ifFalse: [aFileStream nextChunkPut: self definition].
	self organization
		putCommentOnFile: aFileStream
		numbered: fileIndex
		moveSource: moveSource.
	aFileStream cr.
	self organization categories do: 
		[:heading |
		self
			fileOutCategory: heading
			on: aFileStream
			moveSource: moveSource
			toFile: fileIndex]! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 13 February 1989 at 4:27:33 pm'!



!ClassDescription methodsFor: 'stress operations'!

allSlots
	^ self allInstVarNames!

slot: aSlot ifAbsent: aBlock
	| index | 
	index _ self class allInstVarNames indexOf: aSlot ifAbsent:
		[((self whichSuperHasSlot: aSlot) == nil)
			ifTrue: [^ aBlock value]
			ifFalse: [^ nil]].
	^ self instVarAt: index!

slot: aSlot put: anObject ifAbsent: aBlock
	| index aClass | 
	index _ self class allInstVarNames indexOf: aSlot ifAbsent:
		[((aClass _ self whichSuperHasSlot: aSlot) == nil)
			ifTrue: [^ aBlock value]
			ifFalse: [aClass class addInstVarName: aSlot.
				self class allInstVarNames indexOf: aSlot]].
	^ self instVarAt: index put: anObject!

slot: aSlot put: value in: aKnowledgeBase
	"Put the value in my slot, aSlot.  Do the same for all of my instances (in the knowledge base), all of my subclasses, and all of their instances.  Send copies of the value, unless it is an object in the knowledge base, in which case send the value itself."

	| putBlock |
	Cursor execute showWhile:
		[self slot: aSlot put: value.
		(aKnowledgeBase includes: value)
			ifTrue: [putBlock _ [:each | each slot: aSlot put: value]]
			ifFalse: [putBlock _ [:each | each slot: aSlot put: value copy]].
		(aKnowledgeBase allSubclassesOf: self) do: putBlock.
		(aKnowledgeBase allKindsOf: self) do: putBlock]!

slots
	^ self instVarNames!

stressDefinition
	"Answer a string that defines the receiver."
	| aStream simple |
	aStream _ WriteStream on: (String new: 300).
	self hasMultipleSuperclasses
		ifTrue:
			[aStream nextPutAll: 'Class named: '.
			self name storeOn: aStream.
			aStream cr; tab; nextPutAll: 'superclasses: '.
			aStream store: self superclassesString.
			aStream cr; tab; nextPutAll: 'slots: '.
			aStream store: self instanceVariablesString.
			aStream cr; tab; nextPutAll: 'classVariableNames: '.
			aStream store: self classVariablesString]
		ifFalse:
			[simple _ (self kindOfSubclass = ' subclass: ') and:
				[(self classVariablesString = '') & (self sharedPoolsString = '')].
			aStream nextPutAll: (superclass == nil ifTrue: ['nil'] ifFalse: [superclass name]).
			aStream nextPutAll: self kindOfSubclass.
			self name storeOn: aStream.
			aStream cr; tab; nextPutAll: 'slots: '.
			aStream store: self instanceVariablesString.
			simple ifFalse:
				[aStream cr; tab; nextPutAll: 'classVariableNames: '.
				aStream store: self classVariablesString.
				aStream cr; tab; nextPutAll: 'poolDictionaries: '.
				aStream store: self sharedPoolsString]].
	aStream cr; tab; nextPutAll: 'category: '.
	(SystemOrganization categoryOfElement: self name) asString storeOn: aStream.
	^aStream contents!

whichSuperHasSlot: aSlot
	| val | 
	(self instVarNames includes: aSlot)
		ifTrue: [^ self].
	self superclasses do: [:class |
		((val _ class whichSuperHasSlot: aSlot) ~~ nil)
			ifTrue: [^ val]].
	^ nil! !
!ClassDescription reorganize!
('initialize-release' obsolete subclassOf:oldClass:instanceVariableNames:variable:words:pointers:ifBad: updateInstancesFrom: validateFrom:in:instanceVariableNames:methods:)
('accessing' comment comment: commentTemplate name)
('copying' copy:from: copy:from:classified: copyAll:from: copyAll:from:classified: copyAllCategoriesFrom: copyCategory:from: copyCategory:from:classified:)
('testing' isMeta)
('printing' classVariablesString definition instanceVariablesString printOn: sharedPoolsString storeOn: superclassesString)
('instance variables' addInstVarName: instVarNames removeInstVarName:)
('method dictionary' removeCategory: removeSelector:)
('organization' category category: logOrganizationChange organization reorganize whichCategoryIncludesSelector:)
('compiling' compile:classified: compile:classified:notifying: compile:notifying:trailer:ifFail:)
('fileIn/Out' fileOutCategory: fileOutCategory:on:moveSource:toFile: fileOutChangedMessages:on: fileOutChangedMessages:on:moveSource:toFile: fileOutMessage: fileOutMessage:fileName: fileOutMessage:on:moveSource:toFile: fileOutOn: fileOutOn:moveSource:toFile: fileOutOrganizationOn: kindOfSubclass methodsFor: moveChangesTo: printCategoryChunk:on: printMethodChunk:on:moveSource:toFile: printOutCategory: printOutMessage:)
('private' errorCategoryName)
('stress operations' allSlots fillSlot:fromString:in:notifying: slot:ifAbsent: slot:put:ifAbsent: stressDefinition)
('change trees' changesToClass changesToMethodCat:)
!

'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:27:51 pm'!



!Class class methodsFor: 'instance creation'!

named: newClassName superclasses: newSuperNames slots: mySlots classVariableNames: classVarNames category: cat
	^ self named: newClassName
		superclasses: newSuperNames
		instanceVariableNames: mySlots
		classVariableNames: classVarNames
		category: cat! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:27:52 pm'!



!Class class methodsFor: 'instance creation'!

stressTemplate: category
	"Answer an expression that can be edited and evaluated in order to define
	 a new class."

	^'NameOfSuperclass subclass: #NameOfClass
	slots: ''slot1 slot2''
	category: ''' , category , ''''! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:27:49 pm'!



!Class methodsFor: 'stress subclass creation'!

subclass: t slots: f category: cat 
	^ self subclass: t
		instanceVariableNames: f
		classVariableNames: ''
		poolDictionaries: ''
		category: cat!

subclass: t slots: f classVariableNames: d poolDictionaries: s category: cat 
	^ self subclass: t
		instanceVariableNames: f
		classVariableNames: d
		poolDictionaries: s
		category: cat!

variableByteSubclass: t slots: f classVariableNames: d poolDictionaries: s category: cat
	^ self variableByteSubclass: t
		instanceVariableNames: f 
		classVariableNames: d
		poolDictionaries: s
		category: cat!

variableSubclass: t slots: f classVariableNames: d poolDictionaries: s category: cat
	^ self variableSubclass: t
		instanceVariableNames: f 
		classVariableNames: d
		poolDictionaries: s
		category: cat!

variableWordSubclass: t slots: f classVariableNames: d poolDictionaries: s category: cat
	^ self variableWordSubclass: t
		instanceVariableNames: f 
		classVariableNames: d
		poolDictionaries: s
		category: cat! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 20 February 1989 at 1:10:00 pm'!



!Class methodsFor: 'initialize-release'!

validateFrom: oldClass in: environ instanceVariableNames: invalidFields methods: invalidMethods 
	"Recompile the receiver and redefine its subclasses if necessary."

	super
		validateFrom: oldClass
		in: environ
		instanceVariableNames: invalidFields
		methods: invalidMethods.
	self ~~ oldClass
		ifTrue: 
			[environ at: name put: self.
			self updateInheritanceTables: oldClass.
			self updateDependencies: oldClass.
			oldClass obsolete]! !
!Class reorganize!
('initialize-release' declare: obsolete removeFromSystem sharing: superclass:methodDict:format:name:organization:instVarNames:classPool:sharedPools: validateFrom:in:instanceVariableNames:methods:)
('accessing' classPool name)
('accessing class hierarchy' hasMultipleSuperclasses isObsolete)
('testing method dictionary' hasMethods)
('copying' copy copyForValidation)
('class name' rename:)
('instance variables' addInstVarName: removeInstVarName:)
('class variables' addClassVarName: allClassVarNames classVarNames initialize removeClassVarName:)
('pool variables' addSharedPool: allSharedPools removeSharedPool: sharedPools)
('compiling' compileAllFrom: poolHas:ifTrue:)
('subclass creation' subclass:instanceVariableNames:classVariableNames:poolDictionaries:category: subclass:otherSupers:instanceVariableNames:classVariableNames:category: variableByteSubclass:instanceVariableNames:classVariableNames:poolDictionaries:category: variableSubclass:instanceVariableNames:classVariableNames:poolDictionaries:category: variableWordSubclass:instanceVariableNames:classVariableNames:poolDictionaries:category:)
('stress subclass creation' subclass:slots:category: subclass:slots:classVariableNames:poolDictionaries:category: variableByteSubclass:slots:classVariableNames:poolDictionaries:category: variableSubclass:slots:classVariableNames:poolDictionaries:category: variableWordSubclass:slots:classVariableNames:poolDictionaries:category:)
('fileIn/Out' fileOut fileOutOn:moveSource:toFile: printOut removeFromChanges)
!

'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:27:57 pm'!



!Metaclass methodsFor: 'printing'!

stressDefinition
	"Answer with a string that defines me"

	| aStream names |
	aStream _ WriteStream on: (String new: 300).
	self printOn: aStream.
	aStream nextPutAll: '
	instanceVariableNames: '''.
	names _ self instVarNames.
	1 to: names size do: [:i | aStream nextPutAll: (names at: i); space].
	aStream nextPut: $'.
	^ aStream contents! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 13 February 1989 at 4:27:50 pm'!



!SystemDictionary methodsFor: 'system compression'!

condenseChanges
	"Move all the changes onto a compacted sources file.
		Smalltalk condenseChanges."

	| f fileName |
	f _ FileStream fileNamed: 'temp.changes'.
	f timeStamp.
	Smalltalk allBehaviorsDo: [:class | class moveChangesTo: f].
	(self includesKey: #Stress) ifTrue:
		[Stress do: [:knowledgeBase | knowledgeBase moveChangesTo: f]].
	f close.
	f readOnly.
	fileName _ (SourceFiles at: 2) name.
	(SourceFiles at: 2) close.
	SourceFiles at: 2 put: f.
	FileDirectory removeKey: fileName.
	f file rename: fileName! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 13 February 1989 at 5:39:57 pm'!



!ScreenController class methodsFor: 'class initialization'!

initialize
	"Initialize the System Menu."

	"ScreenController initialize.
	ScreenController allInstancesDo: [:c | c initializeYellowButtonMenu]"

	ScreenYellowButtonMenu _
		PopUpMenu
			labels: 
'restore display
garbage collect
exit project
browser
workspace
file list
file editor
terminal
project
system transcript
system workspace
stress browser
rule browser
suspend
save
quit'
			lines: #(3 9 11 13).
	ScreenYellowButtonMessages _
			#(restoreDisplay garbageCollect exitProject
			openBrowser openWorkspace openFileList openFileEditor 
			openCshView  openProject 
			openTranscript openSystemWorkspace 
			openStressBrowser openRuleBrowser 
			suspend save quit).! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:28:01 pm'!



!ScreenController methodsFor: 'menu messages'!

openRuleBrowser
	RuleBrowserView openOn: Stress! !'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:28:02 pm'!



!ScreenController methodsFor: 'menu messages'!

openStressBrowser
	StressBrowserView openOn: Stress! !


ScreenController initialize.
ScreenController allInstancesDo: [:c | c initializeYellowButtonMenu].
StressDictionary installStress! 
