; ========= fns to control search of a SearchProblem ========
;

(define Search
  (lambda (aSearchProblem aSearchMethod evalInitial?)
    ; attempt to find a solution to aSearchProblem using aSearchMethod
    ; to control the generation of new state spaces If evalInitial?
    ; is #t then it is necessary to score the initial state
    
    (define failSearch
      (lambda (aStateSpace)
        (if (GetAnnounceFlag aSearchProblem)
            (begin (DisplayLine "Search failed after"
                                (GetCount aSearchProblem)
                                "move(s)")
                   (if aStateSpace
                       (begin
                        (display "State space at failure: ")
                        (PrintStateSpace aStateSpace aSearchProblem)
                        (newline))
                       #f))
            #f)))    
    
    (define announceSolution
      (lambda (aStateSpace)
        (let* ((finalLevel (CurrentSearchLevel aStateSpace))
               (finalPath (CurrentPath finalLevel))
               (depth (GetLevelDepth finalLevel))
               (result (list  depth
                              (GetCount aSearchProblem)
                              finalPath)))
          (if (GetAnnounceFlag aSearchProblem)
              (begin (DisplayLine "Solution found after"
                                  (GetCount aSearchProblem)
                                  "move(s) at depth"
                                  depth)
                     (DisplayLine "Successful path (length"
                                  (PathLength finalPath) ") is:")
                     (PrintPath finalPath aSearchProblem)
                     (newline))
              #f)
          result)))
    
    (define searchAux
      (lambda (aStateSpace tracing?)
        ; Look for a solution to aSearchProblem using aSearchMethod to
        ; control the generation of new state spaces. The  search
        ; fails if aStateSpace contains no search levels. If the
        ; current search level is empty it is stripped from the state
        ; space and the search continued else the procedure
        ; aSearchMethod is called to generate a new state space.
        
        (define finished?
          (lambda (aStateTriple)
            (apply (GetGoalFN aSearchProblem)
                   (list (GetTripleState aStateTriple)))))
        
        (define foundSolution?
          (lambda ()
            (let ((currLevel (CurrentSearchLevel aStateSpace)))
              (cond ((EmptySearchLevel? currLevel) #f)
		    ((InvalidPath? (CurrentPath currLevel)) #f)
		    ((finished? (CurrentTriple (CurrentPath currLevel))) #t)
		    (else #f)))))
        
        (define pruneTopLevel
          (lambda ()
            (if (GetTraceFlag aSearchProblem)
                (DisplayLine "PruneTopLevel pruning empty"
                             "level in state space!!")
                #f)
            (OldSearchLevels aStateSpace)))
        
        (define makeNewStateSpace
          (lambda ()
            (if (EmptySearchLevel? (CurrentSearchLevel aStateSpace))
                (pruneTopLevel)
                (apply aSearchMethod (list aStateSpace aSearchProblem)))))
        
        (if tracing?
            (begin (newline)
                   (DisplayLine "*** After"
                                (GetCount aSearchProblem)
                                "moves the state space is:")
                   (PrintStateSpace aStateSpace aSearchProblem))
            #f)
        (cond ((EmptyStateSpace? aStateSpace)
               (failSearch aStateSpace))
              ((foundSolution?)
               (announceSolution aStateSpace))
              (else
               (searchAux (makeNewStateSpace) tracing?)))))
    
    (ZeroCount! aSearchProblem)
    (let ((initialTriple (MakeTriple (GetInitialState aSearchProblem) 'ok)))
      (if evalInitial?
          (ScoreTripleValue! initialTriple (GetEvalFN aSearchProblem))
          #f)
      (let* ((initialPath (MakePath initialTriple))
             (initialLevel (MakeSearchLevel 1 (list initialPath)))
             (initialStateSpace (MakeStateSpace initialLevel)))
        (searchAux initialStateSpace (GetTraceFlag aSearchProblem))))))

; ===========================================================
; *************** S E A R C H   M E T H O D S ***************
; ===========================================================

; ===== Depth First Search =====
; Method to generate a newStateSpace is as follows: at each node all
; successor nodes are expanded and the first of these is always selected for
; further exploration, pushing a search path deeply into the state/action
; tree. Backtracking occurs whenever a node terminates in a dead-end
; or a specified maximum level of recursion (aMaxDepth) has been
; reached. This strategy makes no attempt to classify states into more or
; less "promising" ones. Its efectiveness largely depends on the "intel-
; ligent" ordering of action-functions in "Actions". If no constraint on
; the search depth is made the strategy may "loose itself" in an infinite
; search space and may not terminate, even if solutions might be found
; at quite shallow levels of the state-action tree.

(define DFSearch
  (lambda (aSearchProblem aMaxDepth)
    ; The following method is used to generate a new state space by
    ; expanding the first node if the current depth is less than the
    ; max depth to be explored. If this is not the case then the first
    ; node is removed from the state space
    
    (define aDFMethod
      (lambda (aStateSpace aSearchProblem)
        (let* ((currLevel (CurrentSearchLevel aStateSpace))
               (currDepth (GetLevelDepth currLevel)))
          (append (if (= currDepth aMaxDepth)
                      (list (RestOfSearchLevel currLevel))
                      (list (SpawnSearchLevel (CurrentPath currLevel)
                                              aSearchProblem
                                              (+ currDepth 1))
                            (RestOfSearchLevel currLevel)))
                  (OldSearchLevels aStateSpace)))))
    
    (Search aSearchProblem aDFMethod #f)))

; ===== Breadth First Search =====
; Method to generate a newStateSpace is as follows: this is (like depth first
; search) another "uninformed" search strategy. Instead of growing
; a search path downwards into a state/action tree, (as is done for
; depth first search), it first explores ALL alternatives at a given
; level. This is guaranteed to find any possible solution eventually, but it can
; be VERY slow if it is hidden deeply inside the tree. Its effectiveness
; can again be influenced by the order in which "Actions" are evaluated.

(define BFSearch
  (lambda (aSearchProblem)
    ; The following method is used to generate a new state space by deal-
    ; ing with a new level only if the top level is empty. The first path
    ; of the top level is taken and any new levels spawned by it are
    ; added to the end of all the remaining partially explored paths.
    
    (define appendLevel
      (lambda (aStateSpace aLevel)
        ; Add aLevel to end of aStateSpace and merge with
        ; last level if they are of the same depth
        
        (define appendLevelAux
          (lambda (aStateSpace aLevel)
            (let ((head (CurrentSearchLevel aStateSpace))
                  (tail (OldSearchLevels aStateSpace)))
              (if (EmptyStateSpace? tail)
                  (if (and (not (EmptySearchLevel? head))
                           (not (EmptySearchLevel? aLevel))
                           (= (GetLevelDepth head)
                              (GetLevelDepth aLevel)))
                      (list (cons (GetLevelDepth head)
				  (append (GetPathList head)
					  (GetPathList aLevel))))
                      (list head aLevel))
                  (cons head (appendLevelAux tail aLevel))))))
        
        (if (EmptyStateSpace? aStateSpace)
            (list aLevel)
            (appendLevelAux aStateSpace aLevel))))
    
    (define aBFMethod
      (lambda (aStateSpace aSearchProblem)
        (let ((currLevel (CurrentSearchLevel aStateSpace)))
          (if (EmptySearchLevel? currLevel)
              (OldSearchLevels aStateSpace)
              (append (list (RestOfSearchLevel currLevel))
                      (appendLevel
                       (OldSearchLevels aStateSpace)
                       (SpawnSearchLevel (CurrentPath currLevel)
                                         aSearchProblem
                                         (+ (GetLevelDepth currLevel)
                                            1))))))))
   
    (Search aSearchProblem aBFMethod #f)))

; ===== Hill Climbing Search =====
; Method to generate a newStateSpace is as follows:
; this strategy introduces the idea of a "scoring function" which is
; used to rank states according to their expected distance from a
; solution. A hill climbing search will always select the "next"
; alternative on a search path (essentially "depth first") as long as
; its value is non-decreasing. As soon as it encounters a "downhill"
; path, this node is pruned from the tree and the next alternative
; is selected, backtracking whenever a search-level has been exhausted.
; Effectiveness of this strategy is strongly dependent on the "quality"
; of the scoring function. It may not always find a (pruned) solution
; even though one exists. For well behaved search spaces it may,
; however, quite quickly converge to a goal state.

(define HillSearch
  (lambda (aSearchProblem aMaxDepth)
    ; The following method is used to generate a new state space by
    ; expanding the first node only if the current depth is less than
    ; the max depth. Paths produced by this node are pruned if their
    ; scores are less than the score of the expanded node.

    (define aHillMethod
      (lambda (aStateSpace aSearchProblem)
        
        (define pruneDownhillPaths
          (lambda (aSearchLevel aScore)
            (if (EmptySearchLevel? aSearchLevel)
                aSearchLevel
                (let ((aDepth   (GetLevelDepth aSearchLevel))
                      (oldPaths (GetPathList aSearchLevel))
                      (newPaths ()))
                  (for-each
                    (lambda (aPath)
		      (if (>= (ScoreTripleValue!
			       (CurrentTriple aPath)
			       (GetEvalFN aSearchProblem))
			      aScore)
			  (set! newPaths (append newPaths (list aPath)))
			  #f))
		    oldPaths)
                  (if (null? newPaths)
                      ()
                      (MakeSearchLevel aDepth newPaths))))))
        
        (let* ((currLevel (CurrentSearchLevel aStateSpace))
               (currDepth (GetLevelDepth currLevel))
               (nodeScore (GetTripleValue (CurrentTriple
                                           (CurrentPath currLevel)))))
          (append
           (if (= currDepth aMaxDepth)
               (list (RestOfSearchLevel currLevel))
               (list (pruneDownhillPaths (SpawnSearchLevel
                                          (CurrentPath currLevel)
                                          aSearchProblem
                                          (+ currDepth 1))
                                         nodeScore)
                     (RestOfSearchLevel currLevel)))
           (OldSearchLevels aStateSpace)))))
    
    (Search aSearchProblem aHillMethod #t)))

; ===== Steepest Ascent Search =====
; Method to generate a newStateSpace is as follows:
; this strategy carries the idea of scoring states even further. As
; each node is expanded the whole list of its successors can be
; scanned for their scores. At each level, therefore, the "best"
; alternative is always chosen next. This approach suffers from the
; same difficulty as the "hill climbing" approach, in as much as its
; performance is very sensitive to the predictions made by the
; evaluation function. It may converrge more quickly to a solution,
; but the additional computational effort of scoring and finding the
; "best" alternative of all the successors of a state may be
; significant.

(define SteepestSearch
  (lambda (aSearchProblem)
    ; The following method is used to generate a new state space. The
    ; state space will consist only of a single level and a single
    ; "best" path. This path is expanded and then the the next level
    ; constructed by selecting the best path form those paths that result.
    
    (define aSteepestMethod
      (lambda (aStateSpace aSearchProblem)
        
        (define findBestPath
          (lambda (aPathList bestValue bestPath)
            (if (null? aPathList)
                bestPath
                (let* ((head      (car aPathList))
		       (tail      (cdr aPathList))
		       (currValue (ScoreTripleValue!
				    (CurrentTriple head)
				    (GetEvalFN aSearchProblem))))
                  ; Note the next line is >= rather than > to ensure that even if all
		  ; paths have a value of NEG-INFINITY then one will be chosen.
                  (if (>= currValue bestValue)
                      (findBestPath tail currValue head)
                      (findBestPath tail bestValue bestPath))))))
        
        (let* ((currLevel (CurrentSearchLevel aStateSpace))
               (currPath (CurrentPath currLevel)))
          (if (EmptyPath? currPath)
              () ; return empty state space; i.e. failure
              (let*
                ((newDepth (+ (GetLevelDepth currLevel) 1))
                 (newLevel (SpawnSearchLevel currPath
                                             aSearchProblem
                                             newDepth)))
                (if (EmptySearchLevel? newLevel)
                    ()
                    (MakeStateSpace (MakeSearchLevel
                                     newDepth
                                     (list (findBestPath
                                             (GetPathList newLevel)
					     NEG-INFINITY
					     #f))))))))))
    
    (Search aSearchProblem aSteepestMethod #t)))
