;;;
;;;   S E A R C H   T O O L B O X
;;;   = = = = = =   = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;
;;; This toolbox provides the tools to define and explore a
;;; search space.
;;;
;;;  STATE          - a list of symbols encoding a state description 
;;;  STATETRIPLE    - a vector of <status value (state)> triples
;;;  PATH           - a list of statetriples
;;;  SEARCHLEVEL    - a list of (partially) explored paths of
;;;                   the same length (<depth> <path> ...)
;;;  STATESPACE	    - a list of searchlevels
;;;  ACTIONLIST	    - a list of function names
;;;  SEARCH-PROBLEM - a vector of variables controlling a search
;;;                   problem.
;;;

; ========== operations on SEARCHPROBLEMS ===================

;
; A search problem is represented by a vector whose elements
; store information about a search.

;   traceFlag    - determines the amount of information
;                  displayed during the search
;   announceFlag - determines the amount of information
;                  displayed after the search
;   count        - number of moves performed to date
;   initialState - state in state space from which the search began
;   printStateFN - a procedure used to display a state
;   evalFN       - a function that provides a measure of
;                  closeness to a goal state. The closer the
;                  state the larger the returned value.
;   goalFN       - a function that tests for goal states
;   actions      - a list of transformer functions that generate a new
;                  state from an old state. An action returns #f if
;                  it can not be applied to the state.
;   sameStateFN? - a procedure used to compare states for equality.
;

(define *traceIndex      0)
(define *announceIndex   1)
(define *countIndex      2)
(define *stateIndex      3)
(define *printStateIndex 4)
(define *evalIndex       5)
(define *goalIndex       6)
(define *actionIndex     7)
(define *sameStateIndex  8)

(define *lengthSearchVector  9)

(define MakeSearchProblem
 (lambda (aState aGoalFN anEvalFN someActions)
   (let ((aProblem (make-vector *lengthSearchVector)))
     (vector-set! aProblem *traceIndex      #f)
     (vector-set! aProblem *announceIndex   #f)
     (vector-set! aProblem *countIndex      0)
     (vector-set! aProblem *stateIndex      aState)
     (vector-set! aProblem *printStateIndex display)
     (vector-set! aProblem *evalIndex       anEvalFN)
     (vector-set! aProblem *goalIndex       aGoalFN)
     (vector-set! aProblem *actionIndex     someActions)
     (vector-set! aProblem *sameStateIndex  equal?)
     aProblem)))

(define ZeroCount!
  (lambda (aSearchProblem)
    (vector-set! aSearchProblem *countIndex 0)))

(define GetCount
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *countIndex)))

(define IncCount!
  (lambda (aSearchProblem)
    (vector-set! aSearchProblem
                 *countIndex
                 (+ (vector-ref aSearchProblem *countIndex) 1))))

(define GetInitialState
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *stateIndex)))

(define PrintState
  (lambda (aState aSearchProblem)
    ((vector-ref aSearchProblem *printStateIndex) aState)))

(define SetPrintState!
  (lambda (aSearchProblem aValue)
    (vector-set! aSearchProblem *printStateIndex aValue)))

(define GetEvalFN
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *evalIndex)))

(define GetGoalFN
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *goalIndex)))

(define GetActions
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *actionIndex)))

(define SameState?
  (lambda (aState1 aState2 aSearchProblem)
    ((vector-ref aSearchProblem *sameStateIndex) aState1 aState2)))

(define SetSameState!
  (lambda (aSearchProblem aValue)
    (vector-set! aSearchProblem *sameStateIndex aValue)))

(define GetAnnounceFlag
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *announceIndex)))

(define SetAnnounceFlag!
  (lambda (aSearchProblem)
    (vector-set! aSearchProblem *announceIndex #t)))

(define ResetAnnounceFlag!
  (lambda (aSearchProblem)
    (vector-set! aSearchProblem *announceIndex #f)))

(define GetTraceFlag
  (lambda (aSearchProblem)
    (vector-ref aSearchProblem *traceIndex)))

(define SetTraceFlag!
  (lambda (aSearchProblem)
    (vector-set! aSearchProblem *traceIndex #t)))

(define ResetTraceFlag!
  (lambda (aSearchProblem)
    (vector-set! aSearchProblem *traceIndex #f)))

(define PrintSearchProblem
  (lambda (aSearchProblem)
    (newline)
    (display "Initial state: ")
    (PrintState (GetInitialState aSearchProblem) aSearchProblem)
    (newline)
    (DisplayLine "Move(s)      :" (GetCount aSearchProblem))
    (DisplayLine "Goal function:" (GetGoalFN aSearchProblem))
    (DisplayLine "Eval function:" (GetEvalFN aSearchProblem))
    (DisplayLine "Action(s)    :" (GetActions aSearchProblem))
    (DisplayLine "Tracing      :"
                 (if (GetTraceFlag aSearchProblem) "on" "off"))
    (DisplayLine "Announcing   :"
                 (if (GetAnnounceFlag aSearchProblem) "on" "off"))))

; ========== fns on STATES ==================================
; These functions may be of use to the user if a state
; consists of a simple list of symbols.

(define FirstElement
  (lambda (aState) (car  aState)))

(define RestOfElements
  (lambda (aState) (cdr  aState)))

(define EmptyState?
  (lambda (aState) (null? aState)))

(define FindSymbolInSlot
  (lambda (aSlotNumber aState)
    (if (EmptyState? aState)
        (Fatal-Error "FindSymbolInSlot:" "Slot-Index in State out of Range")
        (if (eq? aSlotNumber 1)
            (FirstElement aState)
            (FindSymbolInSlot (- aSlotNumber 1)
                              (RestOfElements aState))))))

(define FindFirstSlotOfSymbol
  (lambda (aSymbol aState)
    (let ((subList (memq aSymbol aState)))
      (if (not subList)
          (Fatal-Error "FindFirstSlotOfSymbol:"
                       "No symbol" aSymbol "in state" aState)
          (- (length aState) (length (RestOfElements subList)))))))

(define FillSlot
  (lambda (aSlotNumber aSymbol aState)
    (cond ((or (<= aSlotNumber 0) (EmptyState? aState))
           (Fatal-Error "FillSlot:" "Slot-Index in State out of Range"))
          ((eq? aSlotNumber 1)
           (cons aSymbol (RestOfElements aState)))
          (else
           (cons (FirstElement aState)
                 (FillSlot (- aSlotNumber 1)
                           aSymbol
                           (RestOfElements aState)))))))

; ========== fns on ACTIONLISTS =============================
; (lists of function names)

(define MakeActionList (lambda aList aList))

(define FirstAction (lambda (anActionList) (car anActionList)))

(define RestOfActions (lambda (anActionList) (cdr anActionList)))

(define EmptyActions? (lambda (anActionList) (null? anActionList)))

(define PrintActionList (lambda (anActionList) (display anActionList)))

; ========== fns on STATETRIPLES ============================
; Vectors of <status> <value> <state>

;  - The "status" slot will record the fact that a state is "ok", has
;    been "seen-before" or has some other reason for being invalid;
;  - The "value"  slot records the state's score according to
;    some user-defined EvalFN. It defaults to NEG-INFINITY, a large
;    negative score representing a state of little relevance.
;  - The "state" slot holds a list of symbols encoding the
;    state's instantiation.
;

(define *statusSlot 0)
(define *valueSlot  1)
(define *stateSlot  2)

(define MakeTriple
  (lambda (aState aStatus) (vector aStatus NEG-INFINITY aState)))
  
(define GetTripleStatus
  (lambda (aStateTriple) (vector-ref aStateTriple *statusSlot)))
  
(define SetTripleStatus!
  (lambda (aStateTriple aStatus)
    (vector-set! aStateTriple *statusSlot aStatus)
    aStateTriple))

(define ValidTripleState?
  (lambda (aStateTriple) (eq? 'ok (GetTripleStatus aStateTriple))))
  
(define InvalidTripleState?
  (lambda (aStateTriple) (not (ValidTripleState? aStateTriple))))       

(define GetTripleValue
  (lambda (aStateTriple)
    ; return -INFINITY if state not 'ok'
    (if (ValidTripleState? aStateTriple)
        (vector-ref aStateTriple *valueSlot)
        NEG-INFINITY)))
  
(define SetTripleValue!
  (lambda (aStateTriple aValue)
    (vector-set! aStateTriple *valueSlot aValue)
    aStateTriple))

(define ScoreTripleValue!
  (lambda (aStateTriple anEvalFN)
    (if (or (not anEvalFN) (InvalidTripleState? aStateTriple))
        NEG-INFINITY
        (let* ((state (GetTripleState aStateTriple))
               (value (apply anEvalFN (list state))))
          (SetTripleValue! aStateTriple value)
          value))))

(define GetTripleState
  (lambda (aStateTriple) (vector-ref aStateTriple *stateSlot)))

(define EmptyTriple?
  (lambda (aStateTriple) (null? aStateTriple)))

(define SameTriple?
  (lambda (aTriple1 aTriple2 aSearchProblem)
    (if (EmptyTriple? aTriple1)
        (EmptyTriple? aTriple2)
        (SameState? (GetTripleState aTriple1)
                    (GetTripleState aTriple2)
                    aSearchProblem))))

(define PrintTriple
  (lambda (aStateTriple aSearchProblem)
    (if (EmptyTriple? aStateTriple)
        (display "empty triple")
        (begin
         (DisplayList "#<" (GetTripleStatus aStateTriple)
                      (let ((val (GetTripleValue aStateTriple)))
                        (if (= val NEG-INFINITY) "-INF" val)))
         (PrintState (GetTripleState aStateTriple) aSearchProblem)
         (display ">")))))

(define ExploreAll
  (lambda (aStateTriple aSearchProblem)
    ; Return a new search-level (list of all successors of a given statetriple),
    ; obtained by applying all transformations described in "anActionList"
    
    (define exploreTriple
      (lambda (anAction)
        ; Return a new statetriple, by applying a specified
        ; transformation to the current triple 
        (if (or (EmptyTriple? aStateTriple) (null? anAction))
            (Fatal-Error "ExploreAll:"
                         "Unable to Explore state triple" aStateTriple)
            (let* ((oldState (GetTripleState aStateTriple))
                   (newState (apply anAction (list oldState))))
              (if (not newState)                     
                  (MakeTriple #f 'invalid-state)
                  (MakeTriple newState 'ok))))))
    
    (define exploreAllAux
      (lambda (anActionList)
        (if (EmptyActions? anActionList)
            ()
            (begin (IncCount! aSearchProblem)
                   (cons (exploreTriple (FirstAction anActionList))
                         (exploreAllAux (RestOfActions anActionList)))))))
    
    (cond ((EmptyTriple? aStateTriple)
           (Fatal-Error "ExploreAll: Illegal empty triple"))
          (else
           (exploreAllAux (GetActions aSearchProblem))))))


; ========= fns on PATHS ====================================
; A list of statetriples giving the path from the initial
; state. The first triple of the path is the current triple
; and the remaining are ancestors of the current triple

(define MakePath
  (lambda aStateTriple aStateTriple))

(define CurrentTriple (lambda (aPath) (car aPath)))

(define AncestorTriples (lambda (aPath) (cdr aPath)))

(define PathLength (lambda (aPath) (length aPath)))

(define EmptyPath? (lambda (aPath) (null? aPath)))

(define ValidPath?
  (lambda (aPath)
    (equal? (GetTripleStatus (CurrentTriple aPath)) 'ok)))

(define InvalidPath? (lambda (aPath) (not (ValidPath? aPath))))

(define PrintPath
  (lambda (aPath aSearchProblem)
    
    (define printPathAux
      (lambda (aPath)
        (if (EmptyPath? aPath)
            #f
            (begin (PrintTriple (CurrentTriple aPath)
                                aSearchProblem)
                   (display " ")
                   (printPathAux (AncestorTriples aPath))))))
    
    (cond ((EmptyPath? aPath) (display "Empty path  !"))
          ((InvalidPath? aPath) (display "Invalid path: ")
                                (printPathAux aPath))         
	  (else (printPathAux aPath)))))

(define SpawnSearchLevel
  (lambda (aPath aSearchProblem aDepth)
    ; Return a new search-level by building a list of a valid,
    ; partially  explored  state sequences (paths). This is achieved
    ; by finding all successors of the state at the end of a given
    ; path and deleting all cycles (those sequences leading to states
    ; which were previously encountered on the same path). This
    ; function returns () if "aPath" is invalid or a list of new
    ; alternative paths (with next nodes tacked on). "cycle" is stored
    ; in the terminal path node if that path leads to a cycle and can
    ; therefore not  be extended any further.
    
    (define FlagCyclicPaths
      (lambda (aPath)
        ; If aPath contains a cycle then return aPath with
        ; the status of the head node of a path as "cycle",
        ; otherwise return original path.
        
        (define containsTriple?
          (lambda (aTriple aPath)
            ; Does path contain triple?
            (cond ((EmptyPath? aPath)
                   #f)
                  ((SameTriple? aTriple (CurrentTriple aPath)
                                        aSearchProblem)
                   #t)
                  (else
                   (containsTriple? aTriple
                                    (AncestorTriples aPath))))))
        
        (if (EmptyPath? aPath)
            ()
            (if (containsTriple? (CurrentTriple aPath)
                                 (AncestorTriples aPath))
                (cons (SetTripleStatus! (CurrentTriple aPath) 'cycle)
                      (AncestorTriples aPath))
                aPath))))
    
    (let ((newPath aPath))
      (if (InvalidPath? newPath)  ; if 'status' is not 'ok'         
          	 ()                    ; collapse path to () 
            (MakeSearchLevel
             aDepth
             (map (lambda (newState)
                    (FlagCyclicPaths (cons newState newPath)))
                  (ExploreAll (CurrentTriple newPath) aSearchProblem)))))))

; ========= fns on SEARCHLEVELS =============================
;  (a list of <depth> <path> ...)

(define MakeSearchLevel
  (lambda (aLevelNo aPathList) (cons aLevelNo aPathList)))

(define GetLevelDepth (lambda (aSearchLevel) (car aSearchLevel)))

(define GetPathList (lambda (aSearchLevel) (cdr aSearchLevel)))

(define EmptySearchLevel? (lambda (aSearchLevel) (null? aSearchLevel)))

(define CurrentPath
  (lambda (aSearchLevel)
    (if (not (EmptySearchLevel? aSearchLevel))
        (car (GetPathList aSearchLevel))
        ())))

(define RestOfPaths
  (lambda (aSearchLevel) 
    (if (EmptySearchLevel? aSearchLevel)
        (Fatal-Error "RestOfPaths: Illegal empty search level")
        (cdr (GetPathList aSearchLevel)))))

(define RestOfSearchLevel
  (lambda (aSearchLevel)
    (if (or (EmptySearchLevel? aSearchLevel) 
            (null? (RestOfPaths aSearchLevel)))
        ()
        (MakeSearchLevel (GetLevelDepth aSearchLevel) 
                         (RestOfPaths aSearchLevel)))))

(define PrintSearchLevel
  (lambda (aSearchLevel aSearchProblem)
    
    (define printSearchLevelAux
      (lambda (aPathList)
        (if (null? aPathList)
            #f
            (begin (display "....")
                   (PrintPath (car aPathList) aSearchProblem)
                   (newline)
                   (printSearchLevelAux (cdr aPathList))))))
            
    (if (EmptySearchLevel? aSearchLevel)
        	 (DisplayLine "Empty search level")
          (begin (DisplayLine "Level at depth"
                              (GetLevelDepth aSearchLevel)
                              "contains"
                              (length (GetPathList aSearchLevel))
                              "path(s):")
                 (printSearchLevelAux (GetPathList aSearchLevel))))
    #f))

; ========= fns on STATESPACES  =============================
;  (list of search-levels)

(define MakeStateSpace
  (lambda someSearchLevels someSearchLevels))

(define CurrentSearchLevel (lambda (aStateSpace) (car aStateSpace)))

(define OldSearchLevels (lambda (aStateSpace) (cdr aStateSpace)))

(define EmptyStateSpace? (lambda (aStateSpace) (null? aStateSpace)))

(define PrintStateSpace
  (lambda (aStateSpace aSearchProblem) 
    (if (EmptyStateSpace? aStateSpace)
        #f
        (begin (PrintSearchLevel (CurrentSearchLevel aStateSpace)
                                 aSearchProblem)
               (PrintStateSpace (OldSearchLevels aStateSpace)
                                aSearchProblem)))))
