; ****** D O D G E M ********
;
; This game, invented by Colin Vout, is played on a 3x3
; board, two white counters and two black counters. The
; initial state of the board where B represents a black
; counter and W a white counter is:
;
;          white off here
;          +---+---+---+
;          | B |   |   |
;   |      +---+---+---+ black
;   B -    | B |   |   | off
;   |      +---+---+---+ here
;          |   | W | W |
;          +---+---+---+
;                |
;               -W-
;
; The black counters can move one position up, down, or
; right while the white counters can move one position left,
; right or up. The aim is to move both your counters off the
; board, black to the right and white off the top. Only one
; counter is permitted on each square and you lose if you
; prevent your opponent from moving.

(define DodgemGameProblem #f)

(define SetUpDodgem
  (lambda ()
    ; Represent the board as a list of four values, the
    ; position of the two black counters and then the two
    ; white counters. The positions are numbered as follows
    ; will 0 representing a counter moved off the board.
    ; Board positions:
    ;          +---+---+---+
    ;          | 1 | 2 | 3 |
    ;          +---+---+---+
    ;          | 4 | 5 | 6 |
    ;          +---+---+---+
    ;          | 7 | 8 | 9 |
    ;          +---+---+---+
    ;
    ; Thus the initial state is represented by:
    (define initial '(1 4 8 9))
    
    (define Black1 0)
    (define Black2 1)
    (define White1 2)
    (define White2 3)
    
    (define getPos
      ; Return the square occupied by counter 'which'
      (lambda (aState which) (list-ref aState which)))
    
    (define setPos
      (lambda (aState which pos)
        ; Return state with position of counter 'which'
        ; replaced by 'pos'. If this results in two counters
        ; in same position then return #f
        
        (define setList
          (lambda (list k obj)
            (cond ((null? list) ())
                  ((zero? k) (cons obj (cdr list)))
                  (else (cons (car list)
                              (setList (cdr list)
                                       (- k 1) obj))))))
        
        (define recurrs
          (lambda (list)
            ; does the first in list recurr latter in list?
            ; allow zero (off board) to recurr
            (and (positive? (car list))
                 (memq (car list) (cdr list)))))
        
        (let ((newState (setList aState which pos)))
          (cond ((null? newState) #f)
                ((recurrs (list-tail newState Black1)) #f)
                ((recurrs (list-tail newState Black2)) #f)
                ((recurrs (list-tail newState White1)) #f)
                (else newState)))))
    
    
    (define blackPos?
      (lambda (position) (<= position Black2)))
    
    (define whitePos?
      (lambda (position) (>= position White1)))
    
    (define offBoard? zero?)
    
    (define evalFN
      (lambda (aState machineMove?)
        ; Encourage black to move right and up and white to move up and right
	; by assigning appropriate values to each board position. Also encourange
	; blocking an opponent by positioning a counter between one of his and
	; the row from which they must move off the board.
        
        ; Value of squares:    0   1   2   3 ...etc
        (define blackValues '( 40  10  25  40
                                   5   20  35
                                   0   15  30))
        (define whiteValues '(-40 -30 -35 -40
                                  -15 -20 -25
                                   0  -5  -10))
    
        (define blockingFactor
          (lambda (whitePos blackPos)
            ; White is blocking black if its number is one greater.
	    ; White is (partially) blocking black if its number is two greater.
            ; Black is blocking white if its number is three less.
	    ; Black is (partially) blocking white if its number is six less.
            ; Must ignore pieces off the board
            (cond ((or (= whitePos 0)
                       (= blackPos 0)) 0)
                  ((= whitePos (+ blackPos 1)) -40)
                  ((= whitePos (+ blackPos 2)) -30)
                  ((= blackPos (- whitePos 3)) +40)
                  ((= blackPos (- whitePos 6)) +30)
                  (else 0))))
        
        ; sum across all counter positions and poss. blocks
        (+ (list-ref blackValues (getPos aState Black1))
         (+ (list-ref blackValues (getPos aState Black2))
          (+ (list-ref whiteValues (getPos aState White1))
           (+ (list-ref whiteValues (getPos aState White2))
            (+ (blockingFactor (getPos aState White1)
                               (getPos aState Black1))
             (+ (blockingFactor (getPos aState White1)
                                (getPos aState Black2))
              (+ (blockingFactor (getPos aState White2)
                                 (getPos aState Black1))
                 (blockingFactor (getPos aState White2)
                                 (getPos aState Black2)))))))))))
    
    ; Moves from:   0  1  2  3  4  5  6  7  8  9
    (define left  '(#f #f 1  2  #f 4  5  #f 7  8))
    (define right '(#f 2  3  b  5  6  b  8  9  b))
    (define up    '(#f w  w  w  1  2  3  4  5  6))
    (define down  '(#f 4  5  6  7  8  9  #f #f #f))
    
    (define moveCounter
      (lambda (which moves)
        ; Return a function which moves the counter in 'which' position of the state
	; according to the "coded instructions" in 'moves'. It is assumed that the
	; program always plays black. The moves consist of a list accessed by a counter's
	; current position. This list specifies a new state if legal, #f if not, and 'b'
	; or 'w' if the state is only legal for a black or white counter. It is also
	; necessary to ensure that the target position is free.
        (lambda (aState)
          (let* ((currPos (getPos aState which))
                 (newPos  (list-ref moves currPos))
                 (newState #f))
            (cond ((not newPos) #f)
                  ((eq? newPos 'b)
                   (if (blackPos? which)
                       (setPos aState which 0)
                       #f))
                  ((eq? newPos 'w)
                   (if (blackPos? which)
                       #f
                       (setPos aState which 0)))
                  (else (setPos aState which newPos)))))))
    
    (define b1u (moveCounter Black1 up))
    (define b1r (moveCounter Black1 right))
    (define b1d (moveCounter Black1 down))
    (define b2u (moveCounter Black2 up))
    (define b2r (moveCounter Black2 right))
    (define b2d (moveCounter Black2 down))
    (define w1l (moveCounter White1 left))
    (define w1u (moveCounter White1 up))
    (define w1r (moveCounter White1 right))
    (define w2l (moveCounter White2 left))
    (define w2u (moveCounter White2 up))
    (define w2r (moveCounter White2 right))
  
    (define specialStatus
      (lambda (aState machineMove?)
        
        (define noBlackMove?
          (lambda (aState)
            ; returns true if no black move can be made
            (not (or (b1u aState)
                     (b1r aState)
                     (b1d aState)
                     (b2u aState)
                     (b2r aState)
                     (b2d aState)))))
        
        (define noWhiteMove?
          (lambda (aState)
            ; returns true if no white move can be made
            (not (or (w1l aState)
                     (w1u aState)
                     (w1r aState)
                     (w2l aState)
                     (w2u aState)
                     (w2r aState)))))
        
        (cond ( ; white wins if both its counters are off
               (and (offBoard? (getPos aState Black1))
                    (offBoard? (getPos aState Black2))) "machine")
              ( ; black wins if both its counters are off
                (and (offBoard? (getPos aState White1))
                     (offBoard? (getPos aState White2))) "opponent")
              ( ; black wins if white can't move
               (and machineMove?
                    (noBlackMove? aState)) "machine")
              ( ; white wins if black can't move
               (and (not machineMove?)
                    (noWhiteMove? aState)) "opponent")
              (else #f))))
    
    (define printDodgem
      (lambda (aState)
        ; Build a vector and then print in rows of three
        
        (define in3
          (lambda (list)
            (if (null? list)
                (newline)
                (begin (newline)
                       (display (car list))
                       (display (cadr list))
                       (display (caddr list))
                       (in3 (cdddr list))))))
        
        (if (null? aState)
            (display "Invalid board state")
            (begin
             (let ((board (make-vector 10 ".")))
               (vector-set! board (getPos aState Black1) "b")
               (vector-set! board (getPos aState Black2) "b")
               (vector-set! board (getPos aState White1) "w")
               (vector-set! board (getPos aState White2) "w")
               (in3 (cdr (vector->list board)))
               (newline))))))
    
    (define printDodgem-MacVersion
      (lambda (aState)
        ; Use MacScheme graphics is available. Use must
        ; ensure that the graphics heap is used and the
        ; graphics window is open by first invoking
        ; (BeginGraphics)
        
        (define paintSym
          (lambda (place asym apos)
            (let ((x (car apos))
                  (y (cadr apos)))
              (cond ((eq? asym #\b)
                     (paint-circle (+ x 20) (+ y 20) 10))
                    ((eq? asym #\w)
                     (frame-circle (+ x 20) (+ y 20) 10))
                    ((eq? asym #\.)
                     (move-to (+ x 20) (+ y 20))
                     (draw-char (number->string place)))))))
        
        (define paintBoard
          (lambda (place symList posList)
            (if (null? symList)
                #f
                (begin (paintSym place (car symList)
                                       (car posList))
                       (paintBoard (+ place 1)
                                   (cdr symList)
                                   (cdr posList))))))
        
        (clear-graphics)
        (move-to 40 0) (line-to 40 120)
        (move-to 80 0) (line-to 80 120)
        (move-to 0 40) (line-to 120 40)
        (move-to 0 80) (line-to 120 80)
        (if (null? aState)
            (display "Invalid board state")
            (begin
             (let ((board (make-vector 10 #\.)))
               (vector-set! board (getPos aState Black1) #\b)
               (vector-set! board (getPos aState Black2) #\b)
               (vector-set! board (getPos aState White1) #\w)
               (vector-set! board (getPos aState White2) #\w)
               (paintBoard 1
                           (cdr (vector->list board))
                           '( (0   0) (40  0) (80  0)
                              (0  40) (40 40) (80 40)
                              (0  80) (40 80) (80 80))))))))
    
    (define getMove
      (lambda (aState)
        (let* ((which
                (do ((ok #f)
                     (pos 0))
                    (ok pos) ; return pos once it is "ok"
                    (newline)
                    (display "Give postion (1-9): ")
                    (newline)
                    (set! pos (read))
                    (cond ((not (number? pos))
                           (display "invalid number"))
                          ((or (< pos 0)
                               (> pos 9))
                           (display "invalid position "))
                          ((= pos (getPos aState White1))
                           (set! pos White1)
                           (set! ok #t))
                          ((= pos (getPos aState White2))
                           (set! pos White2)
                           (set! ok #t))
                          (else
                           (display "Not a white counter")))))
               (direction
                (do ((ok #f)
                     (dir #f))
                    (ok dir) ; return direction once it is "ok"
                    (newline)
                    (display "Give direction ")
                    (display "(u(p), (l)eft,  (r)ight): ")
                    (newline)
                    (set! dir (read))
                    (set! ok #t)
                    (cond
                     ((eq? dir 'u) (set! dir up))
                     ((eq? dir 'l) (set! dir left))
                     ((eq? dir 'r) (set! dir right))
                     (else (display "Invalid direction! ") (set! ok #f))))))
          ; make the move!
          ((moveCounter which direction) aState))))
    
    (set! DodgemGameProblem
      (MakeGameProblem initial
                       specialStatus
                       evalFN
                       (MakeActionList
                          b1u b1r b1d b2u b2r b2d)
                       (MakeActionList
                          w1l w1u w1r w2l w2u w2r)))
    
    (SetPrintState! DodgemGameProblem
                    (if (and AllowGraphics
                             (string=? *SchemeVersion*
                                       "MacScheme"))
                        printDodgem-MacVersion
                        printDodgem))
    
    (SetGetAMove! DodgemGameProblem getMove)))

(begin (SetUpDodgem)
       (newline)
       (DisplayLine "Type (PlayGame DodgemGameProblem) to"
                    "play Dodgem"))
