/*
 * PCN Abstract Machine Emulator
 * Authors:     Steve Tuecke and Ian Foster
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * md_nx.c  -  Machine dependent file for Intel nX parallel operating
 *		system, for use on the iPSC/860 and the Delta
 *
 * This assumes ASYNC_KS==0 and ASYNC_MSG==0 -- they are defined this
 * way in defs.h when ipsc860 or delta is defined.
 */

#include "pcn.h"

#include <signal.h>
#include <fcntl.h>

static int keyboard_fd;

static void abort_handler();
static void expc6_handler();
static void expc13_handler();
static void expc14_handler();


/*
 * void _p_init_machine_dep()
 *
 * Initialize any machine dependencies.
 */
void _p_init_machine_dep()
{
    if (_p_host)
    {
	/* Set up everything to handle keyboard i/o cleanly */
#ifdef delta
	keyboard_fd = 0;
#endif /* delta */	

#ifdef ipsc860	
	keyboard_fd = open("/dev/tty", O_RDONLY);
#endif /* ipsc860 */
    }
    
    /* Set up the signal handlers for controlled crashes */
    signal(SIGQUIT, abort_handler);
    signal(SIGILL,  abort_handler);
    signal(SIGFPE,  abort_handler);
    signal(SIGBUS,  abort_handler);
    signal(SIGTERM, abort_handler);
    signal(SIGINT,  abort_handler);
    signal(SIGSEGV, abort_handler);
}


/*
 * void _p_shutdown_machine_dep()
 *
 * Shutdown any machine dependencies.
 */
void _p_shutdown_machine_dep()
{
    if (_p_host)
    {
#ifdef ipsc860	
	close(keyboard_fd);
#endif /* ipsc860 */	
    }
}


/*
 * void _p_abort_machine_dep()
 *
 * Shutdown any machine dependencies for an abortive exit.
 */
void _p_abort_machine_dep()
{
    _p_shutdown_machine_dep();
}


static void abort_handler(i)
int i;
{
    char buf[256];

    sprintf(buf, "Aborting on signal %d", i);
    _p_fatal_error(buf);

    signal(i, SIG_DFL);
    sigrelse(i);
    kill(getpid(), i);
    exit(1);
}

static void expc6_handler(etype)
long etype;
{
    _p_fatal_error("Exception 6 raised -- Invalid Opcode");
}
static void expc13_handler(etype)
long etype;
{
    _p_fatal_error("Exception 13 raised -- General Protection Fault");
}
static void expc14_handler(etype)
long etype;
{
    _p_fatal_error("Exception 14 raised -- Page Fault");
}


#ifdef PDB_HOST
/*
 * char *_p_fgets(char *buf, int buf_len)
 *
 * This function implements a blocking read of a single
 * line of text from the keyboard (stdin normally).
 * It is used by PDB to get an input line.
 *
 * It reads n-1 characters, or up through a newline character,
 * whichever comes first, from the keyboard stream into the buffer
 * 'buf'.  The last character read into s is followed by a null
 * character.  (In other words, this function is like fgets()...)
 *
 * If EOF is encountered, then reset the keyboard_fp so that a
 * future read from it will not automatically return EOF.  In other
 * words, this routine should return EOF -- but after returning
 * EOF then it should go on as normal as if the EOF did not occur.
 *
 * Return:	NULL upon EOF, otherwise 'buf'.
 */
char *_p_fgets(buf, buf_len)
char *buf;
int buf_len;
{
    if (!_p_host)
    {
	return ((char *) NULL);
    }
    else
    {
	char *rc = fgets(buf, buf_len, stdin);
	if (rc == (char *) NULL)
            fseek(stdin, 2, 0L);	/* Reset keyboard_fp */
	return (rc);
    }
} /* _p_fgets() */

#endif /* PDB_HOST */
