/*
 * PCN System
 * Author:      Steve Tuecke
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * pcn_structs.h
 *
 * Structure declarations for stuff that pcnlink outputs into
 * the custom info file, and used is by the runtime system.
 */

#ifndef _PCN_INCLUDE_PCN_STRUCTS_H
#define _PCN_INCLUDE_PCN_STRUCTS_H

#include "pcn_tags.h"
#include "pcn_types.h"

/*
#ifdef alliant
#define void int
#endif
*/

typedef u_int_t	code_t;
typedef int_t	foreign_t;

#ifndef NULL
#define NULL		((void *) 0)
#endif

/*
 * CELL_SIZE should be sizeof(cell_t), but it is used in spots that
 * require an integer and not a sizeof(cell_t)
 */
#ifndef CELL_SIZE
#define CELL_SIZE	4
#endif
#define CELL_BITS	CELL_SIZE*8

#if CELL_SIZE == 4
#define CELLS_PER_DOUBLE	2
#define CHARS_PER_CELL		4
#else /* CELL_SIZE == 8 */
#define CELLS_PER_DOUBLE	1
#define CHARS_PER_CELL		8
#endif /* CELL_SIZE */

#define StringSizeToCells(N)  (((N) + CHARS_PER_CELL - 1) / CHARS_PER_CELL)


#if defined(PDB) && !defined(DEBUG)
#define DEBUG 0
#endif

#ifdef PCN_PROFILE
#ifndef GAUGE
#define GAUGE
#endif
#ifndef UPSHOT
#define UPSHOT
#endif
#endif

#ifdef PCNT_FILE
/*
 * If we are compiling a pcnt file, then we want to declare some
 * some variables depending on if its profiling and/or pdb build.
 * The purpose of these variables is to cause a link to fail if
 * the pcnt file is compiled with one set of flags, but is linked
 * with an emulator version for a different set of flags.
 */
#ifdef PDB
int_t _p_compiled_with_pdb;
#else
int_t _p_not_compiled_with_pdb;
#endif
#ifdef PCN_PROFILE
int_t _p_compiled_with_profile;
#else
int_t _p_not_compiled_with_profile;
#endif
#endif /* PCNT_FILE /

/*
 * data_header_t
 *
 * All data that is accessible from PCN is tagged. This structure
 * defines the layout of the tag word.  The 'size' field
 * gives the number of elements in this data structure (i.e., number
 * of array or tuple elements).
 *
 * Note: It is important that the tag reside in the low order
 * 	bits of the word.  This is because a tag word must be
 *	distinguishable from a pointer.  Pointers are always
 *	word aligned, so their two low order bits are always
 *	0.  A tag never has both of the two low order bits set
 *	to 0.  This allows us to follow pointers without having
 *	to first mask off a tag.
 */
typedef struct data_header_struct
{
#ifdef PCN_BIT_FIELDS_R_TO_L
    u_int_t	tag		: TAG_SIZE;
    u_int_t	mark		: 1;
    u_int_t	size		: (CELL_BITS - TAG_SIZE - 1);
#else  /* PCN_BIT_FIELDS_R_TO_L */
    u_int_t	size		: (CELL_BITS - TAG_SIZE - 1);
    u_int_t	mark		: 1;
    u_int_t	tag		: TAG_SIZE;
#endif /* PCN_BIT_FIELDS_R_TO_L */
} data_header_t;


/*
 * byte_filled_word_t
 *
 * The PCN abstract machine instructions divide some words into
 * byte sized chunks.  This byte_filled_word_t structure defines
 * how those bytes are laid out in the word.  b0 (or the pad in
 * a 64 bit machine) should always be the high order byte, and b3 
 * should always be the low order byte.  
 */
typedef struct byte_filled_word_struct {
#ifdef PCN_BIT_FIELDS_R_TO_L
    u_int_t	b3	: 8;
    u_int_t	b2	: 8;
    u_int_t	b1	: 8;
    u_int_t	b0	: 8;
#if CELL_BITS == 64
    u_int_t	pad	: 32;
#endif
#else  /* PCN_BIT_FIELDS_R_TO_L */
#if CELL_BITS == 64
    u_int_t	pad	: 32;
#endif
    u_int_t	b0	: 8;
    u_int_t	b1	: 8;
    u_int_t	b2	: 8;
    u_int_t	b3	: 8;
#endif /* PCN_BIT_FIELDS_R_TO_L */
} byte_filled_word_t;


/*
 * instruction_t
 *
 * PCN abstract machine instruction format.  There are two different
 * type of instructions, t1 and t2:
 *	t1 : Uses the first word for byte size values, and second and
 *		third words for pointers.
 *	t2 : Uses the first two words for byte size values, and
 *		the third word for a pointer.
 */
typedef union instruction_struct {
    struct {
	byte_filled_word_t	op0;
	cell_t *		p1;
	cell_t *		p2;
    } t1;
    struct {
	byte_filled_word_t	op0;
	byte_filled_word_t	op1;
	cell_t *		p2;
    } t2;
} instruction_t;


/*
 * proc_header_t
 *
 * PCN procedure header structure.
 *
 * Every PCN procedure that is linked in has one of these.
 */
typedef struct proc_header_struct
{
    char_t *module_name;/* String containing this procedure's module's name */
    char_t *proc_name;	/* String containing this procedure's name */
    u_int_t arity;	/* The arity of this procedure */
    code_t *code;	/* Pointer to the code array */
    struct proc_header_struct *next; /* Next proc_header in this hash bucket */
#ifdef GAUGE    
    int_t  *idle_timer;	/* Pointer to this procedure's idle timer */
    u_int_t n_counters;	/* Number of counters */
    int_t  *counters;	/* Pointer to counters array */
    u_int_t n_timers;	/* Number of timers */
    int_t  *timers;	/* Pointer to timers array */
    char_t **model;
#endif /* GAUGE */
#ifdef PDB
    bool_t  debugable;	/* procedure compiled with debugging on */
    bool_t  debug;	/* PDB currently debugging this procedure */
    int_t   break_num;	/* Breakpoint number, or 0 if not set as breakpoint */
			/*    If negative, then this breakpoint is disabled */
    bool_t  exported;	/* Is this procedure exported? */
#endif /* PDB */    
} proc_header_t;


/*
 * foreign_table_t
 *
 * A table of all foreign procedures that are linked in and
 * accessible from PCN.  This is only used for debugging
 * purposes.
 */
typedef struct foreign_table_struct
{
    char_t *foreign_name;	/* Name of foreign procedure */
    foreign_t *foreign_ptr;	/* Pointer to foreign procedure */
} foreign_table_t;


/*
 * static_double_value_t
 *
 * A structure defining how doubles that are accessible from
 * PCN are layed out.  (i.e., location of tag, data, etc.)
 *
 * This is used by the linker via the DOUBLE_VALUE_T macro
 * below to create double constants for PCN programs.
 */
typedef struct static_double_value_struct
{
    int_t		n_cells;
    data_header_t	h;
    double_t		d;
} static_double_value_t;

/*
 * static_int_value_t
 *
 * A structure defining how integers that are accessible from
 * PCN are layed out.  (i.e., location of tag, data, etc.)
 *
 * This is used by the linker via the INT_VALUE_T macro
 * below to create integer constants for PCN programs.
 */
typedef struct static_int_value_struct
{
    int_t		n_cells;
    data_header_t	h;
    int_t		i;
} static_int_value_t;

/*
 * static_string_value_t
 *
 * A structure defining how strings that are accessible from
 * PCN are layed out.  (i.e., location of tag, data, etc.)
 *
 * This is used by the linker via the STRING_VALUE_T macro
 * below to create string constants for PCN programs.
 *
 * (Well, actually it is not, because arbitrary length
 * strings need to be layed out in memory immediately after
 * the data_header.  Therefore, the 'str' field array size
 * needs to be different for each string constant.  However,
 * this structure does give show you how string constants
 * are layed out...)
 */
typedef struct static_string_value_struct
{
    int_t		n_cells;
    data_header_t	h;
    char_t		str[4];
} static_string_value_t;


/*
 * Macros which the linker uses to lay out constants in memory...
 */
#ifdef PCN_BIT_FIELDS_R_TO_L

#define DV(StrSize,VarName,Value) \
    static static_double_value_t VarName = {(1+CELLS_PER_DOUBLE),{DOUBLE_TAG,0,1},Value}
#define IV(StrSize,VarName,Value) \
    static static_int_value_t VarName = {(2),{INT_TAG,0,1},Value}
#define SV(StrSize,VarName,Value) \
    static struct { int_t n_cells; data_header_t h; char_t str[StrSize]; } VarName = {(1+StringSizeToCells(StrSize)),{STRING_TAG,0,StrSize},Value}

#else  /* PCN_BIT_FIELDS_R_TO_L */

#define DV(StrSize,VarName,Value) \
    static static_double_value_t VarName = {(1+CELLS_PER_DOUBLE),{1,0,DOUBLE_TAG},Value}
#define IV(StrSize,VarName,Value) \
    static static_int_value_t VarName = {(2),{1,0,INT_TAG},Value}
#define SV(StrSize,VarName,Value) \
    static struct { int_t n_cells; data_header_t h; char_t str[StrSize]; } VarName = {(1+StringSizeToCells(StrSize)),{StrSize,0,STRING_TAG},Value}

#endif /* PCN_BIT_FIELDS_R_TO_L */

/* Some macros to shorten up the pcnt file */
#define EPH	extern proc_header_t
#define EF	extern foreign_t
#define FC	(foreign_t *)
#define SI	static int_t
#define SC	static code_t
#define SCH	static char_t
#define PH	proc_header_t
#define PHC	(proc_header_t *)

#endif /* _PCN_INCLUDE_PCN_STRUCTS_H */
