/*
 * PCN System Assembler
 * Author:      Steve Tuecke
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * pcn_instr.h
 *
 * This file contains the PCN Abstract Machine instruction set opcodes,
 * instruction sizes, etc.
 */

#ifndef _PCN_INCLUDE_PCN_INSTR_H
#define _PCN_INCLUDE_PCN_INSTR_H

/*
 * PCN abstract machine instruction opcodes
 */
#define I_FORK			0
#define I_RECURSE		1
#define I_HALT			2
#define I_DEFAULT		3
#define I_TRY			4
#define I_RUN			5
#define I_BUILD_STATIC		6
#define I_BUILD_DYNAMIC		7
#define I_BUILD_DEF		8
#define I_PUT_DATA		9
#define I_PUT_VALUE		10
#define I_COPY			11
#define I_GET_TUPLE		12
#define I_EQUAL			13
#define I_NEQ			14
#define I_TYPE			15
#define I_LE			16
#define I_LT			17
#define I_DATA			18
#define I_UNKNOWN		19
#define I_DEFINE		20
#define I_GET_ELEMENT		21
#define I_PUT_ELEMENT		22
#define I_ADD			23
#define I_SUB			24
#define I_MUL			25
#define I_DIV			26
#define I_MOD			27
#define I_LENGTH		28
#define I_COPY_MUT		29
#define I_PUT_FOREIGN		30
#define I_CALL_FOREIGN		31
#define I_EXIT			32
#define I_PRINT_TERM		33
#define I_INIT_SEND		34
#define I_INIT_RECV		35
#define I_CLOSE_STREAM		36
#define I_STREAM_SEND		37
#define I_STREAM_RECV		38


/*
 * Instructions sizes, in cells.
 *
 * Some of the instructions have pointers to Gauge counters or timer
 * at the end of the instruction.  If we are linking without Gauge,
 * then these pointers need not be included in the instruction, and
 * thus the instruction smaller.  For these instruction, gauge_size_*
 * is #ifdef's to the proper value.
 */
#define SIZE_FORK		2
#define SIZE_RECURSE		3
#define SIZE_HALT		2
#define SIZE_DEFAULT		2
#define SIZE_TRY		2
#define SIZE_RUN		2
#define SIZE_BUILD_STATIC	2
#define SIZE_BUILD_DYNAMIC	1
#define SIZE_BUILD_DEF		1
#define SIZE_PUT_DATA		2
#define SIZE_PUT_VALUE		1
#define SIZE_COPY		1
#define SIZE_GET_TUPLE		1
#define SIZE_EQUAL		1
#define SIZE_NEQ		1
#define SIZE_TYPE		1
#define SIZE_LE			1
#define SIZE_LT			1
#define SIZE_DATA		1
#define SIZE_UNKNOWN		1
#define SIZE_DEFINE		1
#define SIZE_GET_ELEMENT	1
#define SIZE_PUT_ELEMENT	1
#define SIZE_ADD		1
#define SIZE_SUB		1
#define SIZE_MUL		1
#define SIZE_DIV		1
#define SIZE_MOD		1
#define SIZE_LENGTH		1
#define SIZE_COPY_MUT		2
#define SIZE_PUT_FOREIGN	1
#define SIZE_CALL_FOREIGN	3
#define SIZE_EXIT		1
#define SIZE_PRINT_TERM		1
#define SIZE_INIT_SEND		1
#define SIZE_INIT_RECV		1
#define SIZE_CLOSE_STREAM	1
#define SIZE_STREAM_SEND	2
#define SIZE_STREAM_RECV	2


/*
 * Instructions classes that are used by Gauge in modeling PCN programs.
 */
#define INSTRUCTION_CLASSES 2

#define CLASS_NOCOST		-1
#define CLASS_FORK		0
#define CLASS_RECURSE		0
#define CLASS_HALT		0
#define CLASS_DEFAULT		0
#define CLASS_TRY		0
#define CLASS_RUN		0
#define CLASS_BUILD_STATIC	0
#define CLASS_BUILD_DYNAMIC	0
#define CLASS_BUILD_DEF		0
#define CLASS_PUT_DATA		0
#define CLASS_PUT_VALUE		0
#define CLASS_COPY		0
#define CLASS_GET_TUPLE		0
#define CLASS_EQUAL		0
#define CLASS_NEQ		0
#define CLASS_TYPE		0
#define CLASS_LE		0
#define CLASS_LT		0
#define CLASS_DATA		0
#define CLASS_UNKNOWN		0
#define CLASS_DEFINE		0
#define CLASS_GET_ELEMENT	0
#define CLASS_PUT_ELEMENT	0
#define CLASS_ADD		0
#define CLASS_SUB		0
#define CLASS_MUL		0
#define CLASS_DIV		0
#define CLASS_MOD		0
#define CLASS_LENGTH		0
#define CLASS_COPY_MUT		0
#define CLASS_PUT_FOREIGN	0
#define CLASS_CALL_FOREIGN	0
#define CLASS_EXIT		-1
#define CLASS_PRINT_TERM	0
#define CLASS_INIT_SEND		0
#define CLASS_INIT_RECV		0
#define CLASS_CLOSE_STREAM	0
#define CLASS_STREAM_SEND	0
#define CLASS_STREAM_RECV	0


#ifndef PCN

/*
 * Instruction components.  These #defines give nice symbolic
 * names to the parts of the various abstract machine instructions.
 */
#define I_OPCODE		t1.op0.b0
#define I_FORK_ARITY		t1.op0.b1
#define I_FORK_PROC		t1.p1
#define I_RECURSE_ARITY		t1.op0.b1
#define I_RECURSE_PROC		t1.p1
#define I_RECURSE_COUNTER	t1.p2
#define I_HALT_COUNTER		t1.p1
#define I_DEFAULT_ARITY		t1.op0.b1
#define I_DEFAULT_COUNTER	t1.p1
#define I_TRY_LOCATION		t1.p1
#define I_RUN_VT_R		t2.op0.b1
#define I_RUN_TARGET_R		t2.op0.b2
#define I_RUN_MOD_R		t2.op0.b3
#define I_RUN_PROC_R		t2.op1.b0
#define I_RUN_SYNC_R		t2.op1.b1
#define I_BUILD_STATIC_DEST_R	t1.op0.b1
#define I_BUILD_STATIC_TAG	t1.op0.b2
#define I_BUILD_STATIC_SIZE	t1.p1
#define I_BUILD_DYNAMIC_TAG	t1.op0.b1
#define I_BUILD_DYNAMIC_SIZE_R	t1.op0.b2
#define I_BUILD_DYNAMIC_DEST_R	t1.op0.b3
#define I_BUILD_DEF_DEST_R	t1.op0.b1
#define I_PUT_DATA_DEST_R	t1.op0.b1
#define I_PUT_DATA_TAG		t1.op0.b2
#define I_PUT_DATA_PTR		t1.p1
#define I_PUT_VALUE_SRC_R	t1.op0.b1
#define I_COPY_SRC_R		t1.op0.b1
#define I_COPY_DEST_R		t1.op0.b2
#define I_GET_TUPLE_SRC_R	t1.op0.b1
#define I_GET_TUPLE_ARITY	t1.op0.b2
#define I_GET_TUPLE_DEST_R	t1.op0.b3
#define I_EQUAL_SRC1_R		t1.op0.b1
#define I_EQUAL_SRC2_R		t1.op0.b2
#define I_NEQ_SRC1_R		t1.op0.b1
#define I_NEQ_SRC2_R		t1.op0.b2
#define I_TYPE_SRC_R		t1.op0.b1
#define I_TYPE_TAG		t1.op0.b2
#define I_COMPARE_LSRC_R	t1.op0.b1
#define I_COMPARE_RSRC_R	t1.op0.b2
#define I_DATA_SRC_R		t1.op0.b1
#define I_UNKNOWN_SRC_R		t1.op0.b1
#define I_DEFINE_TO_R		t1.op0.b1
#define I_DEFINE_FROM_R		t1.op0.b2
#define I_GET_ELEMENT_SINDEX_R	t1.op0.b1
#define I_GET_ELEMENT_SRC_R	t1.op0.b2
#define I_GET_ELEMENT_DEST_R	t1.op0.b3
#define I_PUT_ELEMENT_DINDEX_R	t1.op0.b1
#define I_PUT_ELEMENT_DEST_R	t1.op0.b2
#define I_PUT_ELEMENT_SRC_R	t1.op0.b3
#define I_ARITH_LSRC_R		t1.op0.b1
#define I_ARITH_RSRC_R		t1.op0.b2
#define I_ARITH_DEST_R		t1.op0.b3
#define I_LENGTH_SRC_R		t1.op0.b1
#define I_LENGTH_DEST_R		t1.op0.b2
#define I_COPY_MUT_SRC_R	t1.op0.b1
#define I_COPY_MUT_DEST_R	t1.op0.b2
#define I_COPY_MUT_COUNTER	t1.p1
#define I_PUT_FOREIGN_SRC_R	t1.op0.b1
#define I_CALL_FOREIGN_N_ARGS	t1.op0.b1
#define I_CALL_FOREIGN_FOR	t1.p1
#define I_CALL_FOREIGN_TIMER	t1.p2
#define I_EXIT_CODE_R		t1.op0.b1
#define I_PRINT_TERM_SRC_R	t1.op0.b1
#define I_INIT_SEND_CON_ARRAY_R	t1.op0.b1
#define I_INIT_SEND_STREAM_R	t1.op0.b2
#define I_INIT_SEND_SINDEX_R	t1.op0.b3
#define I_INIT_RECV_CON_ARRAY_R	t1.op0.b1
#define I_INIT_RECV_STREAM_R	t1.op0.b2
#define I_INIT_RECV_SINDEX_R	t1.op0.b3
#define I_CLOSE_STREAM_STREAM_R	t1.op0.b1
#define I_CLOSE_STREAM_SINDEX_R	t1.op0.b2
#define I_STREAM_SEND_STREAM_R	t2.op0.b1
#define I_STREAM_SEND_SINDEX_R	t2.op0.b2
#define I_STREAM_SEND_ARRAY_R	t2.op0.b3
#define I_STREAM_SEND_OFFSET_R	t2.op1.b0
#define I_STREAM_SEND_SIZE_R	t2.op1.b1
#define I_STREAM_RECV_STREAM_R	t2.op0.b1
#define I_STREAM_RECV_SINDEX_R	t2.op0.b2
#define I_STREAM_RECV_ARRAY_R	t2.op0.b3
#define I_STREAM_RECV_OFFSET_R	t2.op1.b0
#define I_STREAM_RECV_SIZE_R	t2.op1.b1
#define I_STREAM_RECV_STATUS_R	t2.op1.b2


#ifdef DEFINE_INSTR_NAMES

#ifndef GLOBAL
#define GLOBAL extern
#undef DEFINE_GLOBALS
#endif

/*
 * Instruction names
 */
GLOBAL char *_p_instr_names[]
#ifdef DEFINE_GLOBALS
    =	{
	"fork",
	"recurse",
	"halt",
	"default",
	"try",
	"run",
	"build_static",
	"build_dynamic",
	"build_def",
	"put_data",
	"put_value",
	"copy",
	"get_tuple",
	"equal",
	"neq",
	"type",
	"le",
	"lt",
	"data",
	"unknown",
	"define",
	"get_element",
	"put_element",
	"add",
	"sub",
	"mul",
	"div",
	"mod",
	"length",
	"copy_mut",
	"put_foreign",
	"call_foreign",
	"exit",
	"print_term",
	"init_send",
	"init_recv",
	"close_stream",
	"stream_send",
	"stream_recv"
	}
#endif /* DEFINE_GLOBALS */
;

#endif /* DEFINE_INSTR_NAMES */

#endif /* !PCN */

#endif /* _PCN_INCLUDE_PCN_INSTR_H */
