;;; $Header: /usr/baobab/u/gauge/PCN/IF/el/RCS/pcn-mode.el,v 1.1 90/10/05 09:47:42 carl Exp $

;; PCN code editing commands for Emacs
;; Copyright (C) 1985, 1986, 1987 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;;
;; To use this mode, put it somewhere in the emacs search path and
;; put the following emacs lisp code into your .emacs file:

;; (setq pcn-mode-hook '(lambda ()
;;                       (local-set-key "\n" 'newline)
;;                       (local-set-key "\r" 'newline-and-indent)))
;; (autoload 'pcn-mode "pcn-mode" nil t)
;; (setq auto-mode-alist (cons '("\\.pcn$" . pcn-mode)
;;                       (cons '("\\.ptn$" . pcn-mode)
;;                             auto-mode-alist)))
;;


(provide 'pcn-mode)

(defvar pcn-mode-abbrev-table nil
  "Abbrev table in use in Pcn-mode buffers.")
(define-abbrev-table 'pcn-mode-abbrev-table ())

(defun pcn-mode-commands (map)
  (define-key pcn-mode-map "{" 'electric-pcn-brace)
  (define-key pcn-mode-map "}" 'electric-pcn-brace)
  (define-key pcn-mode-map ";" 'electric-pcn-semi)
  (define-key pcn-mode-map ":" 'electric-pcn-terminator)
  (define-key pcn-mode-map "\e\C-h" 'mark-pcn-function)
  (define-key pcn-mode-map "\e\C-q" 'indent-pcn-exp)
  (define-key pcn-mode-map "\ef" 'forward-pcn-word)
  (define-key pcn-mode-map "\eb" 'backward-pcn-word)
  (define-key pcn-mode-map "\ed" 'kill-pcn-word)
  (define-key pcn-mode-map "\e." 'find-pcn-definition)
  (define-key pcn-mode-map "\177" 'backward-delete-char-untabify)
  (define-key pcn-mode-map "\t" 'pcn-indent-command)
  )

(defun pcn-mode-initialize ()
  (use-local-map pcn-mode-map)
  (setq major-mode 'pcn-mode)
  (setq mode-name "PCN")
)

(defun pcn-mode-variables ()
  (setq local-abbrev-table pcn-mode-abbrev-table)
  (set-syntax-table pcn-mode-syntax-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'pcn-indent-line)
  (make-local-variable 'require-final-newline)
  (setq require-final-newline t)
  (make-local-variable 'comment-start)
  (setq comment-start "/* ")
  (make-local-variable 'comment-end)
  (setq comment-end " */")
  (make-local-variable 'comment-column)
  (setq comment-column 32)
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "/\\*+ *")
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'pcn-comment-indent)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments t)
)

(defvar pcn-mode-map ()
  "Keymap used in C mode.")
(if pcn-mode-map
    ()
  (setq pcn-mode-map (make-sparse-keymap))
  (pcn-mode-commands pcn-mode-map)
)

(autoload 'pcn-macro-expand "cmacexp"
  "Display the result of expanding all C macros occurring in the region.
The expansion is entirely correct because it uses the C preprocessor."
  t)

(defvar pcn-mode-syntax-table nil
  "Syntax table in use in Pcn-mode buffers.")

(if pcn-mode-syntax-table
    ()
  (setq pcn-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\_ "_   " pcn-mode-syntax-table)
  (modify-syntax-entry ?\\ "\\" pcn-mode-syntax-table)
  (modify-syntax-entry ?/ ". 14" pcn-mode-syntax-table)
  (modify-syntax-entry ?* ". 23" pcn-mode-syntax-table)
  (modify-syntax-entry ?+ "." pcn-mode-syntax-table)
  (modify-syntax-entry ?- "." pcn-mode-syntax-table)
  (modify-syntax-entry ?= "." pcn-mode-syntax-table)
  (modify-syntax-entry ?% "." pcn-mode-syntax-table)
  (modify-syntax-entry ?< "." pcn-mode-syntax-table)
  (modify-syntax-entry ?> "." pcn-mode-syntax-table)
  (modify-syntax-entry ?& "." pcn-mode-syntax-table)
  (modify-syntax-entry ?| "." pcn-mode-syntax-table)
  (modify-syntax-entry ?\' "." pcn-mode-syntax-table))

(defconst pcn-indent-level 4
  "*Indentation of PCN statements with respect to containing block.")
(defconst pcn-brace-imaginary-offset 0
  "*Imagined indentation of a PCN open brace that actually follows a statement.")
(defconst pcn-brace-offset 0
  "*Extra indentation for braces, compared with other text in same context.")
(defconst pcn-argdecl-indent 5
  "*Indentation level of declarations of PCN function arguments.")

(defconst pcn-continued-statement-offset 2
  "*Extra indent for lines not starting new statements.")
(defconst pcn-continued-brace-offset 0
  "*Extra indent for substatements that start with open-braces.
This is in addition to pcn-continued-statement-offset.")

(defconst pcn-auto-newline nil
  "*Non-nil means automatically newline before and after braces,
and after colons and semicolons, inserted in PCN code.")

(defconst pcn-tab-always-indent t
  "*Non-nil means TAB in PCN mode should always reindent the current line,
regardless of where in the line point is when the TAB command is used.")


(defun forward-pcn-word (arg)
  "Just like word forward, but it treats `_' like a word
if it is used in the position that it is a anynomous
variable"
  (interactive "p")
  (or arg (setq arg 1))
  (cond
   ((zerop arg) t)
   ((> arg 0)
    (if (looking-at "[(,]?\\s-*_\\s-*[),]")
        (progn
          (forward-char 1)
          (re-search-forward "[),]")
          (backward-char 1))
      (forward-word 1))
    (forward-pcn-word (1- arg)))
   (t
    (if (looking-at "_") (forward-char -1))
    (if (looking-at "\\s-\\|$") (re-search-backward "\\S-"))
    (cond
     ((looking-at "[,)]")
      (forward-char -1)
      (if (looking-at "\\s-") (re-search-backward "\\S-"))
      (if (not (looking-at "_"))
          (progn
            (forward-char 1)
            (forward-word -1))))
     ((looking-at "_") t)
     (t (forward-word -1)))
    (forward-pcn-word (1+ arg)))))

(defun backward-pcn-word (arg)
  "Move backward until encountering the end of a word.
With argument, do this that many times.
In programs, it is faster to call forward-word with negative arg."
  (interactive "p")
  (forward-pcn-word (- arg)))

(defun kill-pcn-word (arg)
  "Kill characters forward until encountering the end of a word.
Treats underscore as a word if it appears as an argument.
With argument, do this that many times."
  (interactive "*p")
  (kill-region (point) (progn (forward-pcn-word arg) (point))))

(defun backward-kill-pcn-word (arg)
  "Kill characters backward until encountering the end of a word.
With argument, do this that many times."
  (interactive "*p")
  (kill-pcn-word (- arg)))

(defun pcn-mode ()
  "Major mode for editing PCN code.
Expression and list commands understand all PCN brackets.
Tab indents for PCN code.
Comments are delimited with /* ... */.
Paragraphs are separated by blank lines only.
Delete converts tabs to spaces as it moves back.
\\{pcn-mode-map}
Variables controlling indentation style:
 pcn-tab-always-indent
    Non-nil means TAB in PCN mode should always reindent the current line,
    regardless of where in the line point is when the TAB command is used.
 pcn-auto-newline
    Non-nil means automatically newline before and after braces,
    and after colons and semicolons, inserted in PCN code.
 pcn-indent-level
    Indentation of PCN statements within surrounding block.
    The surrounding block's indentation is the indentation
    of the line on which the open-brace appears.
 pcn-continued-statement-offset
    Extra indentation given to a substatement, such as the
    then-clause of an if or body of a while.
 pcn-continued-brace-offset
    Extra indentation given to a brace that starts a substatement.
    This is in addition to pcn-continued-statement-offset.
 pcn-brace-offset
    Extra indentation for line if it starts with an open brace.
 pcn-brace-imaginary-offset
    An open brace following other text is treated as if it were
    this far to the right of the start of its line.
 pcn-argdecl-indent
    Indentation level of declarations of PCN function arguments.
 pcn-label-offset
    Extra indentation for line that is a label, or case or default.

Settings for K&R and BSD indentation styles are
  pcn-indent-level                5    8
  pcn-continued-statement-offset  5    8
  pcn-brace-offset               -5   -8
  pcn-argdecl-indent              0    8
  pcn-label-offset               -5   -8

Turning on PCN mode calls the value of the variable pcn-mode-hook with no args,
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (pcn-mode-initialize)
  (pcn-mode-variables)
  (run-hooks 'pcn-mode-hook))

;; This is used by indent-for-comment
;; to decide how much to indent a comment in C code
;; based on its context.
(defun pcn-comment-indent ()
  (if (looking-at "^/\\*")
      0				;Existing comment at bol stays there.
    (save-excursion
      (skip-chars-backward " \t")
      (max (1+ (current-column))	;Else indent at comment column
	   comment-column))))	; except leave at least one space.

(defun electric-pcn-brace (arg)
  "Insert character and correct line's indentation."
  (interactive "P")
  (let (insertpos)
    (if (and (not arg)
	     (eolp)
	     (or (save-excursion
		   (skip-chars-backward " \t")
		   (bolp))
		 (if pcn-auto-newline (progn (pcn-indent-line) (newline) t) nil)))
	(progn
	  (insert last-command-char)
	  (pcn-indent-line)
	  (if pcn-auto-newline
	      (progn
		(newline)
		;; (newline) may have done auto-fill
		(setq insertpos (- (point) 2))
		(pcn-indent-line)))
	  (save-excursion
	    (if insertpos (goto-char (1+ insertpos)))
	    (delete-char -1))))
    (if insertpos
	(save-excursion
	  (goto-char insertpos)
	  (self-insert-command (prefix-numeric-value arg)))
      (self-insert-command (prefix-numeric-value arg)))))

(defun electric-pcn-semi (arg)
  "Insert character and correct line's indentation."
  (interactive "P")
  (if pcn-auto-newline
      (electric-pcn-terminator arg)
    (self-insert-command (prefix-numeric-value arg))))

(defun electric-pcn-terminator (arg)
  "Insert character and correct line's indentation."
  (interactive "P")
  (let (insertpos (end (point)))
    (if (and (not arg) (eolp)
	     (not (save-excursion
		    (beginning-of-line)
		    (skip-chars-forward " \t")
		    (or (= (following-char) ?#)
			;; Colon is special only after a label, or case ....
			;; So quickly rule out most other uses of colon
			;; and do no indentation for them.
			(and (eq last-command-char ?:)
			     (not (looking-at "case[ \t]"))
			     (save-excursion
			       (forward-word 1)
			       (skip-chars-forward " \t")
			       (< (point) end)))
			(progn
			  (beginning-of-defun)
			  (let ((pps (parse-partial-sexp (point) end)))
			    (or (nth 3 pps) (nth 4 pps) (nth 5 pps))))))))
	(progn
	  (insert last-command-char)
	  (pcn-indent-line)
	  (and pcn-auto-newline
	       (not (pcn-inside-parens-p))
	       (progn
		 (newline)
		 (setq insertpos (- (point) 2))
		 (pcn-indent-line)))
	  (save-excursion
	    (if insertpos (goto-char (1+ insertpos)))
	    (delete-char -1))))
    (if insertpos
	(save-excursion
	  (goto-char insertpos)
	  (self-insert-command (prefix-numeric-value arg)))
      (self-insert-command (prefix-numeric-value arg)))))

(defun pcn-inside-parens-p ()
  (condition-case ()
      (save-excursion
	(save-restriction
	  (narrow-to-region (point)
			    (progn (beginning-of-defun) (point)))
	  (goto-char (point-max))
	  (= (char-after (or (scan-lists (point) -1 1) (point-min))) ?\()))
    (error nil)))

(defun pcn-indent-command (&optional whole-exp)
  (interactive "P")
  "Indent current line as C code, or in some cases insert a tab character.
If pcn-tab-always-indent is non-nil (the default), always indent current line.
Otherwise, indent the current line only if point is at the left margin
or in the line's indentation; otherwise insert a tab.

A numeric argument, regardless of its value,
means indent rigidly all the lines of the expression starting after point
so that this line becomes properly indented.
The relative indentation among the lines of the expression are preserved."
  (if whole-exp
      ;; If arg, always indent this line as C
      ;; and shift remaining lines of expression the same amount.
      (let ((shift-amt (pcn-indent-line))
	    beg end)
	(save-excursion
	  (if pcn-tab-always-indent
	      (beginning-of-line))
	  (setq beg (point))
	  (forward-sexp 1)
	  (setq end (point))
	  (goto-char beg)
	  (forward-line 1)
	  (setq beg (point)))
	(if (> end beg)
	    (indent-code-rigidly beg end shift-amt "#")))
    (if (and (not pcn-tab-always-indent)
	     (save-excursion
	       (skip-chars-backward " \t")
	       (not (bolp))))
	(insert-tab)
      (pcn-indent-line))))

(defun pcn-indent-line ()
  "Indent current line as PCN code.
Return the amount the indentation changed by."
  (let ((indent (calculate-pcn-indent nil))
	beg shift-amt
	(case-fold-search nil)
	(pos (- (point-max) (point))))
    (beginning-of-line)
    (setq beg (point))
    (cond ((eq indent nil)
	   (setq indent (current-indentation)))
	  ((eq indent t)
	   (setq indent (calculate-pcn-indent-within-comment)))
	  ((looking-at "[ \t]*#")
	   (setq indent 0))
	  (t
	   (skip-chars-forward " \t")
	   (if (listp indent) (setq indent (car indent)))
	   (cond ((and (looking-at "else\\b")
		       (not (looking-at "else\\s_")))
		  (setq indent (save-excursion
				 (pcn-backward-to-start-of-if)
				 (current-indentation))))
		 ((= (following-char) ?})
		  (setq indent (save-excursion
				 (beginning-of-line)
				 (let ((indent-point (point))
				       state)
				   (beginning-of-defun)
				   (setq state (parse-partial-sexp (point) indent-point 0))
				   (goto-char (car (cdr state)))
				   (current-column)))))
		 ((= (following-char) ?{)
		  (setq indent (+ indent pcn-brace-offset))))))
    (skip-chars-forward " \t")
    (setq shift-amt (- indent (current-column)))
    (if (zerop shift-amt)
	(if (> (- (point-max) pos) (point))
	    (goto-char (- (point-max) pos)))
      (delete-region beg (point))
      (indent-to indent)
      ;; If initial point was within line's indentation,
      ;; position after the indentation.  Else stay at same point in text.
      (if (> (- (point-max) pos) (point))
	  (goto-char (- (point-max) pos))))
    shift-amt))

(defun calculate-pcn-indent (&optional parse-start)
  "Return appropriate indentation for current line as PCN code.
In usual case returns an integer: the column to indent to.
Returns nil if line starts inside a string, t if in a comment."
  (save-excursion
    (beginning-of-line)
    (let ((indent-point (point))
	  (case-fold-search nil)
	  state
	  containing-sexp)
      (if parse-start
	  (goto-char parse-start)
	(beginning-of-defun))
      (while (< (point) indent-point)
	(setq parse-start (point))
	(setq state (parse-partial-sexp (point) indent-point 0))
	(setq containing-sexp (car (cdr state))))
      (cond ((or (nth 3 state) (nth 4 state))
	     ;; return nil or t if should not change this line
	     (nth 4 state))
	    ((null containing-sexp)
	     ;; Line is at top level.  May be data or function definition,
	     ;; or may be function argument declaration.
	     ;; Indent like the previous top level line
	     ;; unless that ends in a closeparen without semicolon,
	     ;; in which case this line is the first argument decl.
	     (goto-char indent-point)
	     (skip-chars-forward " \t")
	     (if (= (following-char) ?{)
		 0			; Unless it starts a function body
	       (pcn-backward-to-noncomment (or parse-start (point-min)))
	       ;; Look at previous line that's at column 0
	       ;; to determine whether we are in top-level decls
	       ;; or function's arg decls.  Set basic-indent accordinglu.
	       (let ((basic-indent
		      (save-excursion
			(re-search-backward "^[^ \^L\t\n#]" nil 'move)
			(if (and (looking-at "\\sw\\|\\s_")
				 (looking-at ".*(")
				 (progn
				   (goto-char (1- (match-end 0)))
				   (forward-sexp 1)
				   (and (< (point) indent-point)
					(not (memq (following-char)
						   '(?\, ?\;))))))
			    pcn-argdecl-indent 0))))
		 ;; Now add a little if this is a continuation line.
		 (+ basic-indent (if (or (bobp)
					 (memq (preceding-char) '(?\) ?\; ?\})))
				     0 pcn-continued-statement-offset)))))
	    ((/= (char-after containing-sexp) ?{)
	     ;; line is expression, not statement:
	     ;; indent to just after the surrounding open.
	     (goto-char (1+ containing-sexp))
	     (current-column))
	    (t
	     ;; Statement level.  Is it a continuation or a new statement?
	     ;; Find previous non-comment character.
	     (goto-char indent-point)
	     (pcn-backward-to-noncomment containing-sexp)
	     ;; Now we get the answer.
	     (if (not (or (memq (preceding-char) '(nil ?\, ?\; ?\} ?\{ ?\?))
			  (and (eq (preceding-char) ?\|) 
			       (eq (char-after (1- (point))) ?\|))))
		 ;; This line is continuation of preceding line's statement;
		 ;; indent  pcn-continued-statement-offset  more than the
		 ;; previous line of the statement.
		 (progn
		   (pcn-backward-to-start-of-continued-exp containing-sexp)
		   (+ pcn-continued-statement-offset (current-column)
		      (if (save-excursion (goto-char indent-point)
					  (skip-chars-forward " \t")
					  (eq (following-char) ?{))
			  pcn-continued-brace-offset 0)))
	       ;; This line starts a new statement.
	       ;; Position following last unclosed open.
	       (goto-char containing-sexp)
	       ;; Is line first statement after an open-brace?
	       (or
		;; If no, find that first statement and indent like it.
		(save-excursion
		  (forward-char 1)
		  (while (progn (skip-chars-forward " \t\n")
				(looking-at "#\\|/\\*\\|?\\|;\\|\|\|"))
		    ;; Skip over comments and labels following openbrace.
		    (cond ((= (following-char) ?\#)
			   (forward-line 1))
			  ((= (following-char) ?\/)
			   (forward-char 2)
			   (search-forward "*/" nil 'move))
			  ;; composition:
			  (t (re-search-forward "\|\|\\|;\\|?"))))
		  ;; The first following code counts
		  ;; if it is before the line we want to indent.
		  (and (< (point) indent-point)
			 (current-column)))
	       ;; If no previous statement,
	       ;; indent it relative to line brace is on.
	       ;; For open brace in column zero, don't let statement
	       ;; start there too.  If pcn-indent-level is zero,
	       ;; use pcn-brace-offset + pcn-continued-statement-offset instead.
	       ;; For open-braces not the first thing in a line,
	       ;; add in pcn-brace-imaginary-offset.
	       (+ (if (and (bolp) (zerop pcn-indent-level))
		      (+ pcn-brace-offset pcn-continued-statement-offset)
		    pcn-indent-level)
		  ;; Move back over whitespace before the openbrace.
		  ;; If openbrace is not first nonwhite thing on the line,
		  ;; add the pcn-brace-imaginary-offset.
		  (progn (skip-chars-backward " \t")
			 (if (bolp) 0 pcn-brace-imaginary-offset))
		  ;; If the openbrace is preceded by a parenthesized exp,
		  ;; move to the beginning of that;
		  ;; possibly a different line
		  (progn
		    (if (eq (preceding-char) ?\))
			(forward-sexp -1))
		    ;; Get initial indentation of the line we are on.
		    (current-indentation))))))))))

(defun calculate-pcn-indent-within-comment ()
  "Return the indentation amount for line, assuming that
the current line is to be regarded as part of a block comment."
  (let (end star-start)
    (save-excursion
      (beginning-of-line)
      (skip-chars-forward " \t")
      (setq star-start (= (following-char) ?\*))
      (skip-chars-backward " \t\n")
      (setq end (point))
      (beginning-of-line)
      (skip-chars-forward " \t")
      (and (re-search-forward "/\\*[ \t]*" end t)
	   star-start
	   (goto-char (1+ (match-beginning 0))))
      (current-column))))


(defun pcn-backward-to-noncomment (lim)
  (let (opoint stop)
    (while (not stop)
      (skip-chars-backward " \t\n\f" lim)
      (setq opoint (point))
      (if (and (>= (point) (+ 2 lim))
	       (save-excursion
		 (forward-char -2)
		 (looking-at "\\*/")))
	  (search-backward "/*" lim 'move)
	(beginning-of-line)
	(skip-chars-forward " \t")
	(setq stop (or (not (looking-at "#")) (<= (point) lim)))
	(if stop (goto-char opoint)
	  (beginning-of-line))))))

(defun pcn-backward-to-start-of-continued-exp (lim)
  (if (= (preceding-char) ?\))
      (forward-sexp -1))
  (beginning-of-line)
  (if (<= (point) lim)
      (goto-char (1+ lim)))
  (skip-chars-forward " \t?|;"))

(defun pcn-backward-to-start-of-if (&optional limit)
  "Move to the start of the last ``unbalanced'' if."
  (or limit (setq limit (save-excursion (beginning-of-defun) (point))))
  (let ((if-level 1)
	(case-fold-search nil))
    (while (not (zerop if-level))
      (backward-sexp 1)
      (cond ((looking-at "else\\b")
	     (setq if-level (1+ if-level)))
	    ((looking-at "if\\b")
	     (setq if-level (1- if-level)))
	    ((< (point) limit)
	     (setq if-level 0)
	     (goto-char limit))))))


(defun mark-pcn-function ()
  "Put mark at end of PCN function, point at beginning."
  (interactive)
  (push-mark (point))
  (end-of-defun)
  (push-mark (point))
  (beginning-of-defun)
  (backward-paragraph))

(defun indent-pcn-exp ()
  "Indent each line of the PCN grouping following point."
  (interactive)
  (let ((indent-stack (list nil))
	(contain-stack (list (point)))
	(case-fold-search nil)
	restart outer-loop-done inner-loop-done state ostate
	this-indent last-sexp
	at-else at-brace
	(opoint (point))
	(next-depth 0))
    (save-excursion
      (forward-sexp 1))
    (save-excursion
      (setq outer-loop-done nil)
      (while (and (not (eobp)) (not outer-loop-done))
	(setq last-depth next-depth)
	;; Compute how depth changes over this line
	;; plus enough other lines to get to one that
	;; does not end inside a comment or string.
	;; Meanwhile, do appropriate indentation on comment lines.
	(setq innerloop-done nil)
	(while (and (not innerloop-done)
		    (not (and (eobp) (setq outer-loop-done t))))
	  (setq ostate state)
	  (setq state (parse-partial-sexp (point) (progn (end-of-line) (point))
					  nil nil state))
	  (setq next-depth (car state))
	  (if (and (car (cdr (cdr state)))
		   (>= (car (cdr (cdr state))) 0))
	      (setq last-sexp (car (cdr (cdr state)))))
	  (if (or (nth 4 ostate))
	      (pcn-indent-line))
	  (if (or (nth 3 state))
	      (forward-line 1)
	    (setq innerloop-done t)))
	(if (<= next-depth 0)
	    (setq outer-loop-done t))
	(if outer-loop-done
	    nil
	  ;; If this line had ..))) (((.. in it, pop out of the levels
	  ;; that ended anywhere in this line, even if the final depth
	  ;; doesn't indicate that they ended.
	  (while (> last-depth (nth 6 state))
	    (setq indent-stack (cdr indent-stack)
		  contain-stack (cdr contain-stack)
		  last-depth (1- last-depth)))
	  (if (/= last-depth next-depth)
	      (setq last-sexp nil))
	  ;; Add levels for any parens that were started in this line.
	  (while (< last-depth next-depth)
	    (setq indent-stack (cons nil indent-stack)
		  contain-stack (cons nil contain-stack)
		  last-depth (1+ last-depth)))
	  (if (null (car contain-stack))
	      (setcar contain-stack (or (car (cdr state))
					(save-excursion (forward-sexp -1)
							(point)))))
	  (forward-line 1)
	  (skip-chars-forward " \t")
	  (if (eolp)
	      nil
	    (if (and (car indent-stack)
		     (>= (car indent-stack) 0))
		;; Line is on an existing nesting level.
		;; Lines inside parens are handled specially.
		(if (/= (char-after (car contain-stack)) ?{)
		    (setq this-indent (car indent-stack))
		  ;; Line is at statement level.
		  ;; Is it a new statement?  Is it an else?
		  ;; Find last non-comment character before this line
		  (save-excursion
		    (setq at-else (looking-at "else\\W"))
		    (setq at-brace (= (following-char) ?{))
		    (pcn-backward-to-noncomment opoint)
		    (if (not (memq (preceding-char) '(nil ?\, ?\; ?} ?: ?{)))
			;; Preceding line did not end in comma or semi;
			;; indent this line  pcn-continued-statement-offset
			;; more than previous.
			(progn
			  (pcn-backward-to-start-of-continued-exp (car contain-stack))
			  (setq this-indent
				(+ pcn-continued-statement-offset (current-column)
				   (if at-brace pcn-continued-brace-offset 0))))
		      ;; Preceding line ended in comma or semi;
		      ;; use the standard indent for this level.
		      (if at-else
			  (progn (pcn-backward-to-start-of-if opoint)
				 (setq this-indent (current-indentation)))
			(setq this-indent (car indent-stack))))))
	      ;; Just started a new nesting level.
	      ;; Compute the standard indent for this level.
	      (let ((val (calculate-pcn-indent
			   (if (car indent-stack)
			       (- (car indent-stack))))))
		(setcar indent-stack
			(setq this-indent val))))
	    ;; Adjust line indentation according to its contents
	    (if (or (looking-at "case[ \t]")
		    (and (looking-at "[A-Za-z]")
			 (save-excursion
			   (forward-sexp 1)
			   (looking-at ":"))))
		(setq this-indent (max 1 (+ this-indent pcn-label-offset))))
	    (if (= (following-char) ?})
		(setq this-indent (- this-indent pcn-indent-level)))
	    (if (= (following-char) ?{)
		(setq this-indent (+ this-indent pcn-brace-offset)))
	    ;; Put chosen indentation into effect.
	    (or (= (current-column) this-indent)
		(= (following-char) ?\#)
		(progn
		  (delete-region (point) (progn (beginning-of-line) (point)))
		  (indent-to this-indent)))
	    ;; Indent any comment following the text.
	    (or (looking-at comment-start-skip)
		(if (re-search-forward comment-start-skip (save-excursion (end-of-line) (point)) t)
		    (progn (indent-for-comment) (beginning-of-line)))))))))
; (message "Indenting C expression...done")
  )
