/*
 * PCN Abstract Machine Emulator
 * Authors:     Steve Tuecke and Ian Foster
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * gauge.h	- Macros for accessing various structures, etc.
 */

/*
  Gauge assumes that clock values are stored into 64 bit structures.
  On some machines, such as the symmetry, only 32 bits worth of real data 
  can be obtained.  Therefore, if you timing events are larger than about
  one hour, you are in trouble.
  */
#ifndef _GAUGE_H
#define _GAUGE_H

/* Leave the following #define in -- it is used in em_gauge.c */
#define PrintStartupMessage()

#if (sun || s2010 || NeXT || rs6000 || intel386 || hpux || dec5000)
#define GAUGE_PRIMITIVES_DEFINED 1

#include <sys/time.h>

typedef unsigned long gauge_counter;

#define TICKS_PER_SEC 1000000

extern unsigned long UnixBaseTime;

/* This must be 64 bits wide */
typedef struct { unsigned long low_word, hi_word;} gauge_timer;

#define LOW_WORD(t) (t).low_word
#define HI_WORD(t) (t).hi_word

#define TIMER(time) \
{\
  struct timeval tval; \
  struct timezone zone; \
  gettimeofday(&tval,&zone); \
  time.hi_word = 0L; \
  time.low_word = (tval.tv_sec - UnixBaseTime) * 1000000 + tval.tv_usec; \
}

#define LTIMER(time) TIMER(time)

/* tv3 = tv1 + tv2 */
#define timer_add(tv1,tv2,tv3) \
{\
   unsigned long temp = (tv2).low_word; \
   (tv3).low_word = (tv1).low_word + (tv2).low_word; \
   (tv3).hi_word = (tv1).hi_word + (tv2).hi_word + (((tv3).low_word < temp) ? 1 : 0); \
}

/* tv3 = tv2 - tv1 */
#define timer_sub(tv1,tv2,tv3) \
{\
   unsigned long temp = (tv2).low_word; \
   (tv3).low_word = (tv2).low_word - (tv1).low_word; \
   (tv3).hi_word = (tv2).hi_word - (tv1).hi_word - (((tv3).low_word > temp) ? 1 : 0); \
}

#define scale_timer(tv,scale) (tv).low_word /= size;

#define lapse_time(tvS,tvF,tvL)  timer_sub(tvS,tvF,tvL)

#endif /* sun || s2010 || NeXT || ... */


#ifdef symmetry

#define GAUGE_PRIMITIVES_DEFINED 1

#include <sys/time.h>
#include <usclkc.h>

typedef unsigned long gauge_counter;

#define TICKS_PER_SEC 1000000

extern unsigned long UnixBaseTime;

/* This must be 64 bits wide */
typedef struct { usclk_t low_word, hi_word;} gauge_timer;

#define LOW_WORD(t) (t).low_word
#define HI_WORD(t) (t).hi_word

#define TIMER(time) { (time).hi_word = 0L; (time).low_word = getusclk();}

/* tv3 = tv1 + tv2 */
#define timer_add(tv1,tv2,tv3) \
{\
   usclk_t temp = (tv2).low_word; \
   (tv3).low_word = (tv1).low_word + (tv2).low_word; \
   (tv3).hi_word = (tv1).hi_word + (tv2).hi_word + (((tv3).low_word < temp) ? 1 : 0); \
}

#define LTIMER(time) \
{\
  struct timeval tval; \
  struct timezone zone; \
  gettimeofday(&tval,&zone); \
  time.hi_word = 0L; \
  time.low_word = (tval.tv_sec - UnixBaseTime) * 1000000 + tval.tv_usec; \
}

/* tv3 = tv2 - tv1 */
#define timer_sub(tv1,tv2,tv3) \
{\
   usclk_t temp = (tv2).low_word; \
   (tv3).low_word = (tv2).low_word - (tv1).low_word; \
   (tv3).hi_word = (tv2).hi_word - (tv1).hi_word - (((tv3).low_word > temp) ? 1 : 0); \
}

#define scale_timer(tv,scale) (tv).low_word /= size;

/* Counter could have rolled over, but we assume that
   this could have happend at most once. */
#define lapse_timer(tvS,tvF,tvL) \
{\
 usclk_t diff;\
 if ((tvF).low_word >= (tvS).low_word) \
   (tvL).low_word = (tvF).low_word - (tvS).low_word; \
 else { \
   diff = 0xFFFFFFFF - (tvS).low_word; \
   (tvL).low_word = (tvF).low_word + diff; \
 } \
}

#endif /* symmetry */

#if defined(ipsc860) || defined(delta)

#define GAUGE_PRIMITIVES_DEFINED 1

#include <sys/time.h>
#include <sys/types.h>
#include <estat.h>
#if defined(ipsc860)
#include <cube.h>
#else
#include <mesh.h>
#endif

typedef unsigned long gauge_counter;

/* This must be 64 bits wide */
/* This is compatible with the  esize_t type */
typedef esize_t gauge_timer;

#define TICKS_PER_SEC (HWHZ)

#define LOW_WORD(t) (t).slow
#define HI_WORD(t) (t).shigh

#define TIMER(time) { hwclock(&time);}
#define LTIMER(time) TIMER(time)

/* tv3 = tv1 + tv2 */
#define timer_add(tv1,tv2,tv3)  tv3 = eadd(tv1, tv2)

/* tv3 = tv2 - tv1 */
#define timer_sub(tv1,tv2,tv3) tv3 = esub(tv2, tv1)

/* Assume that 2^32 * .1 microseconds are enough */
#define scale_timer(tv,scale) (tv).slow /= size;

/* Since counter gets reset on reboot, assume that rollover
   will never occur */
#define lapse_time(tvS,tvF,tvL) tvL = esub(tvF,tvS)

#endif /* ipsc860 || delta*/

#ifdef encore
#define GAUGE_PRIMITIVES_DEFINED 1

#include <sys/time.h>
extern unsigned *encore_counter;

typedef unsigned long gauge_counter;

#define TICKS_PER_SEC 1000000

/* This must be 64 bits wide */
typedef struct {unsigned long low_word, hi_word;} gauge_timer;

#define LOW_WORD(t) (t).low_word
#define HI_WORD(t) (t).hi_word

#define TIMER(time) { (time).hi_word = 0L; (time).low_word = (*EncoreTimer); }

#define LTIMER(time) \
{\
  struct timeval tval; \
  struct timezone zone; \
  gettimeofday(&tval,&zone); \
  time.hi_word = 0L; \
  time.low_word = (tval.tv_sec - UnixBaseTime) * 1000000 + tval.tv_usec; \
}

/* tv3 = tv1 + tv2 */
#define timer_add(tv1,tv2,tv3) \
{\
   usclk_t temp = (tv2).low_word; \
   (tv3).low_word = (tv1).low_word + (tv2).low_word; \
   (tv3).hi_word = (tv1).hi_word + (tv2).hi_word + (((tv3).low_word < temp) ? 1 : 0); \
}

/* tv3 = tv2 - tv1 */
#define timer_sub(tv1,tv2,tv3) \
{\
   usclk_t temp = (tv2).low_word; \
   (tv3).low_word = (tv2).low_word - (tv1).low_word; \
   (tv3).hi_word = (tv2).hi_word - (tv1).hi_word - (((tv3).low_word > temp) ? 1 : 0); \
}

#define scale_timer(tv,scale) (tv).low_word /= size;

/* Counter could have rolled over, but we assume that
   this could have happend at most once. */
#define lapse_time(tvS,tvF,tvL) \
{\
   unsigned long diff;\
 if ((tvF).low_word >= (tvS).low_word) \
   (tvL).low_word = (tvF).low_word - (tvS).low_word; \
 else { \
   diff = 0xFFFFFFFF - (tvS).low_word; \
   (tvL).low_word = (tvF).low_word + diff; \
 } \
}

#endif /* encore */

#ifdef DONT_INCLUDE /* tc2000 */
#define GAUGE_PRIMITIVES_DEFINED 1

#include <sys/time.h>
#include <usclkc.h>

typedef unsigned long gauge_counter;

typedef struct { long tv_sec, tv_usec;} gauge_ltime;
/* This must be 64 bits wide */
typedef struct { usclk_t low_word, hi_word;} gauge_timer;

#define LOW_WORD(t) (t).low_word
#define HI_WORD(t) (t).hi_word

#define TIMER(time) {(time).hi_word = 0L;(time).low_word = getusclk();}

/* tv3 = tv1 + tv2 */
#define timer_add(tv1,tv2,tv3) \
{\
   usclk_t temp = (tv2).low_word; \
   (tv3).low_word = (tv1).low_word + (tv2).low_word; \
   (tv3).hi_word = (tv1).hi_word + (tv2).hi_word + (((tv3).low_word < temp) ? 1 : 0); \
}

/* tv3 = tv2 - tv1 */
#define timer_sub(tv1,tv2,tv3) \
{\
   usclk_t temp = (tv2).low_word; \
   (tv3).low_word = (tv2).low_word - (tv1).low_word; \
   (tv3).hi_word = (tv2).hi_word - (tv1).hi_word - (((tv3).low_word > temp) ? 1 : 0); \
}

#define scale_timer(tv,scale) stop_time.low_word /= size;

/* Counter could have rolled over, but we assume that
   this could have happend at most once. */
#define lapse_time(tvS,tvF,tvL) \
{\
   usclk_t diff;\
 if ((tvF).low_word >= (tvS).low_word) \
   (tvL).low_word = (tvF).low_word - (tvS).low_word; \
 else { \
   diff = 0xFFFFFFFF - (tvS).low_word; \
   (tvL).low_word = (tvF).low_word + diff; \
 } \
}

/* Convert times into a unix timeval struct */
#define normalize_ltime(t,nt) (nt) = (t)
#define normalize_stime(t,nt) \
{\
    (nt).tv_sec = (0x100000000/1000000); \
    (nt).tv_usec = (0x100000000%1000000); \
    (nt).tv_sec *= (t).hi_word; \
    (nt).tv_usec *= (t).hi_word; \
    (nt).tv_sec += (t).low_word / 1000000; \
    (nt).tv_usec += (t).low_word % 1000000; \
}

#endif /* tc2000 */


#ifndef GAUGE_PRIMITIVES_DEFINED

#undef PrintStartupMessage
#define PrintStartupMessage() \
{ printf("Gauge's timing primities are not available\n"); fflush(stdout); }
    
typedef struct
{
    unsigned long low_word;
    unsigned long hi_word;
} gauge_timer;

typedef unsigned long gauge_counter;

#define TICKS_PER_SEC 1000000

#define LOW_WORD(t) (t).low_word
#define HI_WORD(t) (t).hi_word

#define TIMER(time) \
{ time.hi_word = 100; time.low_word = 100; }

#define LTIMER(time) TIMER(time)

#define timer_add(tv1,tv2,tv3) 
#define timer_sub(tv1,tv2,tv3) 
#define scale_timer(tv,scale) 
#define lapse_time(tvS,tvF,tvL) 

#endif /* others */

extern gauge_timer gc_start, gc_stop;

/* This is in Cells, not bytes */
#define SMALL_MSG_THRESHOLD	8

extern struct s_stats {
  gauge_timer gc_time;
  gauge_timer run_time;
  gauge_timer idle_time;
  int timer;
  int gc_calls;
  long ssmallmsgs;
  long sbigmsgs;
  long sbigmsgslen;
  long rsmallmsgs;
  long rbigmsgs;
  long rbigmsgslen;
} GaugeStats;

#endif /* _GAUGE_H */

