/* $Id: ghash.c,v 1.9 1993/11/05 11:56:56 bd Exp $ */

#include "include.h"
#include "term.h"
#include "tree.h"
#include "predicate.h"
#include "exstate.h"
#include "engine.h"
#include "storage.h"
#include "unify.h"
#include "copy.h"
#include "gc.h"
#include "initial.h"
#include "config.h"
#include "names.h"
#include "display.h"
#include "error.h"
#include "hash.h"
#include "ghash.h"


/* ######################################################################## */
/* 			HASH DEFINITION FOR TERMS 			    */
/* ######################################################################## */

/* Initial settings */

#define DEFAULT_TABLE_SIZE 32


/* Define a hash table for terms. The key has to be ground */
/* Set hash_exs before calling this functions */

typedef struct {
  exstate *exs;
  bool result;
} HashFuncState;

#ifndef sequent
typedef unsigned long ulong;
#endif

/* TO BE DELETED LATER */
static ulong ground_hash_func(HashFuncState *,Term);

static bool hash_compare(exstate *exs, Term key0, Term key1);


/* ######################################################################## */

#define HashFuncINT(_T) IntVal((_T)) /* A function of the value maybe? */
#define HashFuncFLT(_T) (ulong)FltVal(_T) /* Just cast, not so good */
#define HashFuncATM(_T) AtmHash((_T))
#define HashFuncSTR(_T) (AtmHash(StrName((_T))) + StrArity((_T)))
#define HashFuncGEN(_T) (ulong)((_T)->method) /* What to use? */

#define HashFunc(TERM,HVAL) \
{ \
  if (IsINT(TERM)) \
    (HVAL) = HashFuncINT(Int(TERM)); \
\
  if (IsFLT(TERM)) \
    (HVAL) = HashFuncFLT(Flt(TERM)); \
\
  if (IsATM(TERM)) \
    (HVAL) = HashFuncATM(Atm(TERM)); \
\
  if (IsGEN(TERM)) \
    (HVAL) = HashFuncGEN(Gen(TERM)); \
\
  if (IsLST(TERM)) \
    (HVAL) = 1;			/* Or what?? */ \
\
  if (IsSTR(TERM)) \
    (HVAL) = HashFuncSTR(Str(TERM)); \
}



static ulong ground_hash_func(state, term)
     HashFuncState *state;
     Term term;
{
  exstate *exs = state->exs;

  state->result = SUSPEND;	/* Assume for now */

  Deref(term, term);
  IfVarSuspend(term);

  state->result = TRUE;		/* No, we passed */

  if (IsINT(term))
    return HashFuncINT(Int(term));

  if (IsFLT(term))
    return HashFuncFLT(Flt(term));

  if (IsATM(term))
    return HashFuncATM(Atm(term));

  if (IsGEN(term))
    return HashFuncGEN(Gen(term));

  if (IsLST(term))
    {
      ulong hval;
      Term car, cdr;
      
      GetLstCar(car,Lst(term));
      
      hval = ground_hash_func(state, car);
      if (state->result != TRUE)
	return 0;		/* Give up because suspended */
	  
      GetLstCdr(cdr,Lst(term));

      return hval + ground_hash_func(state, cdr);
    }

  
  if (IsSTR(term))
    {
      int i;
      ulong hval = 0;
      Term tmp_term;
      
      for (i = 0; i < StrArity(Str(term)); i++)
	{
	  GetStrArg(tmp_term,Str(term),i);
	  hval += ground_hash_func(state, tmp_term);
	  if (state->result != TRUE)
	    return 0;		/* Give up because suspended */
	}
      
      return hval + HashFuncSTR(Str(term)); /* On functor and arity */
    }
  
  USAGE_FAULT("Not a term in key calculation");
}

#define GHashValue(Key) \
{ \
  state.exs = exs; \
  state.result = TRUE; \
\
  hash = ground_hash_func(&state, Key); \
\
  if (state.result == SUSPEND) \
    return SUSPEND;			/* suspend */ \
}

#define HashValue(Key) \
{ \
  IfVarSuspend(Key); \
  HashFunc(Key,hash); \
}


/* ######################################################################## */

/* Check if two ground terms are equal */

static Functor functor_cons;

#define HashCompare(X0,X1,RES) \
{ \
  if (IsINT(X0)) \
    (RES) = IsINT(X1) ? (IntVal(Int(X0)) == IntVal(Int(X1))) : FALSE; \
  else \
    if (IsFLT(X0)) \
      (RES) = IsFLT(X1) ? (FltVal(Flt(X0)) == FltVal(Flt(X1))) : FALSE; \
    else \
      if (IsATM(X0)) \
	(RES) = Eq(X0,X1); \
      else \
	if (IsLST(X0)) \
	  (RES) = IsLST(X1); \
	else \
	  if (IsSTR(X0)) \
	    (RES) = (IsSTR(X1) && \
		     (StrFunctor(Str(X0)) == StrFunctor(Str(X1)))); \
}


static bool hash_compare(exs, X0, X1)
     exstate *exs;
     Term X0, X1;
{
  Deref(X0, X0);
  Deref(X1, X1);

  if (IsINT(X0))
    if (IsINT(X1))
      return (IntVal(Int(X0)) == IntVal(Int(X1)));
    else
      return FALSE;

  if (IsFLT(X0))
    if (IsFLT(X1))
      return (FltVal(Flt(X0)) == FltVal(Flt(X1)));
    else
      return FALSE;

  if (IsATM(X0))
    return Eq(X0,X1);

  if (IsLST(X0))
    {
      if (IsLST(X1))
	{
	  Term temp0, temp1;

	  GetLstCar(temp0,Lst(X0));
	  GetLstCar(temp1,Lst(X1));
	  if (!hash_compare(exs, temp0, temp1))
	    return FALSE;

	  GetLstCdr(temp0,Lst(X0));
	  GetLstCdr(temp1,Lst(X1));
	  return hash_compare(exs, temp0, temp1);
	}
      
      if (IsSTR(X1) && (StrFunctor(Str(X1)) == functor_cons))
	{
	  Term temp0, temp1;
	  GetLstCar(temp0,Lst(X0));
	  GetStrArg(temp1,Str(X1),0);
	  if (!hash_compare(exs, temp0, temp1))
	    return FALSE;

	  GetLstCdr(temp0,Lst(X0));
	  GetStrArg(temp1,Str(X1),1);
	  return hash_compare(exs, temp0, temp1);
	}

      return FALSE;
    }

  if (IsSTR(X0))
    {
      if (IsSTR(X1) &&
	  (StrFunctor(Str(X0)) == StrFunctor(Str(X1))))
	{
	  int i;
	  Term temp0, temp1;
	  
	  for (i = 0; i < StrArity(Str(X0)); i++)
	    {
	      GetStrArg(temp0,Str(X0),i);
	      GetStrArg(temp1,Str(X1),i);
	      if (!hash_compare(exs, temp0, temp1))
		return FALSE;
	    }
	  return TRUE;
	}

      if (IsLST(X1) && (StrFunctor(Str(X0)) == functor_cons))
	{
	  Term temp0, temp1;
	  GetStrArg(temp0,Str(X0),0);
	  GetLstCar(temp1,Lst(X1));
	  if (!hash_compare(exs, temp0, temp1))
	    return FALSE;

	  GetStrArg(temp0,Str(X0),1);
	  GetLstCdr(temp1,Lst(X1));
	  return hash_compare(exs, temp0, temp1);
	}

      return FALSE;
    }

  return FALSE;
}

/* ######################################################################## */

#define HashAllocate \
{ \
  if (size > HLIMIT) \
    memory = malloc(size); \
  else \
    { \
      memory = (char *) heapcurrent; \
      heapcurrent += size; \
      if(heapcurrent >= heapend) \
	{\
	  SaveHeapRegisters(); \
	  reinit_heap(memory,size); \
	  RestoreHeapRegisters(); \
	  memory = (char *) heapcurrent; \
	  heapcurrent += size; \
	}\
    } \
}

#define HashFree \
{ \
  if (size > HLIMIT) \
    free(memory);		/* Then allocated by malloc */ \
}


/* ######################################################################## */
/* 			GENERIC HASH OBJECT	 			    */
/* ######################################################################## */


typedef struct {
  gvainfo	header;
  int		ground_key;	/* If require ground key */
  HashTable	*table;		/* Pointer to actual table and info */
} GenHTable;

static Gvainfo	newhash(GenHTable *old);
static bool 	unifyhash(Term x0, Term x1, andbox *andb, exstate *exs);
static int 	printhash(FILE *file, GenHTable *table, int tsiz);
static Gvainfo	copyhash(GenHTable *old,GenHTable *new);
static Gvainfo	gchash(GenHTable *old,GenHTable *new, gcstatep gcst);
static bool	sendhash(Term message, Term self, exstate *exs);
static int 	uncopiedhash(GenHTable *gtable);
static envid	*deallocatehash(GenHTable *gtable);
static int	killhash(GenHTable *gtable);


static gvamethod hashmethods = {
  newhash,
  unifyhash,
  printhash,
  copyhash,
  gchash,
  uncopiedhash,			/* Only used if in close list */
  deallocatehash,		/* Only used if in close list */
  killhash,			/* Only used if in close list */
  NULL,
  sendhash,
  NULL,
  NULL
};



/* If a hash table bigger than a limit it is */
/* allocated with malloc() outside the heap */
/* The limit is in number of cells in table */

#define HLIMIT 10000		/* Limit in bytes of heap hash table */

#define IsGenHTable(Prt) (IsCvaTerm(Prt) && \
			  RefCvaMethod(Ref(Prt)) == &(hashmethods))

#define GvaHash(g)	((GenHTable*)(g))
#define Hash(x)		GvaHash(RefGva(Ref(x)))

/* ######################################################################## */

/* LOW LEVEL OBJECT METHODS */

/* Called before copy and gc */
/* Allocates space for the generic object itself, NOT the array */

static Gvainfo newhash(old)
     GenHTable *old;
{
  GenHTable *new;

  NEW(new,GenHTable);

  return (Gvainfo) new;
}


/* ------------------------------------------------------------------------ */

#define HashElemCopy \
{ \
  newentry->key = entry->key; \
  newentry->value = entry->value; \
  copy_location(&newentry->key); \
  copy_location(&newentry->value); \
}

/* Copy the actual data */

static Gvainfo copyhash(old,new)
     GenHTable *old, *new;
{
  HashTable *oldtab, *newtab;

  oldtab = old->table;

  CopyHashTable(newtab,oldtab,HashAllocate,HashElemCopy)

  if (newtab == NULL)
    FatalError("Could not malloc() for copy of hash table");

  new->table = newtab;
  new->ground_key = old->ground_key;

  return (Gvainfo) new;
}


/* ------------------------------------------------------------------------ */

#define HashElemGc \
{ \
  newentry->key = entry->key; \
  newentry->value = entry->value; \
  gc_location(&newentry->key,gcst); \
  gc_location(&newentry->value,gcst); \
}

/* GC: Don't copy if allocated with malloc() */

static Gvainfo gchash(old,new,gcst)
     GenHTable *old, *new;
     gcstatep gcst;
{
  HashTable *oldtab, *newtab;

  oldtab = old->table;

  CopyHashTable(newtab,oldtab,HashAllocate,HashElemGc)

  new->table = newtab;
  new->ground_key = old->ground_key;

  return (Gvainfo) new;
}


static bool unifyhash(x0,x1,andb,exs)
     Term x0, x1;
     andbox *andb;
     exstate *exs;
{
  if (Eq(x0, x1))		/* Not as simple as I thought */
    return TRUE;		/* so just test for equality */
  return FALSE;
}


static int printhash(file,table,tsiz)
     FILE *file;
     GenHTable *table;
     int tsiz;
{
  fprintf(file,"{hash[%d]}",table->table->size);

  return 1;
}


/* ------------------------------------------------------------------------ */
/* LOW LEVEL OBJECT SEND METHOD */

static Atom atom_nil;
static Atom atom_yes;
static Atom atom_no;
static Atom atom_hash_table;
static Functor functor_typeof;
static Functor functor_sizeof;
static Functor functor_indomain_2;
static Functor functor_get_2;
static Functor functor_set_2;
static Functor functor_set_3;
static Functor functor_delete_1;
static Functor functor_delete_2;
static Functor functor_domain_1;
static Functor functor_range_1;
static Functor functor_attributes_1;
static bool method_indomain_hash();
static bool method_get_hash();
static bool method_set_hash();
static bool method_swap_hash();
static bool method_delete_hash();
static bool method_domain();
static bool method_range();
static bool method_attributes();
    

static bool sendhash(message, self, exs)
     Term message, self;
     exstate *exs;
{
  Deref(message, message);
  
  if(IsSTR(message))
    {
      Functor op = StrFunctor(Str(message));
      Term X0, X1, X2;
      
      if (op == functor_indomain_2)
	{
	  GetStrArg(X0,Str(message),0);
	  GetStrArg(X1,Str(message),1);
	  return method_indomain_hash(X0,X1,self,exs);
	}
      else
	if (op == functor_get_2)
	  {
	    GetStrArg(X0,Str(message),0);
	    GetStrArg(X1,Str(message),1);
	    return method_get_hash(X0,X1,self,exs);
	  }
	else
	  if(op == functor_set_2)
	    {
	      GetStrArg(X0,Str(message),0);
	      GetStrArg(X1,Str(message),1);
	      return method_set_hash(X0,X1,self,exs);
	    }
	  else
	    if(op == functor_set_3)
	      {
		GetStrArg(X0,Str(message),0);
		GetStrArg(X1,Str(message),1);
		GetStrArg(X2,Str(message),2);
		return method_swap_hash(X0,X1,X2,self,exs);
	      }
	    else
	      if(op == functor_delete_1)
		{
		  GetStrArg(X0,Str(message),0);
		  return method_delete_hash(X0,NULL,self,exs);
		}
	      else
		if(op == functor_delete_2)
		  {
		    GetStrArg(X0,Str(message),0);
		    GetStrArg(X1,Str(message),1);
		    return method_delete_hash(X0,X1,self,exs);
		  }
		else
		  if(op == functor_domain_1)
		    {
		      GetStrArg(X0,Str(message),0);
		      GetStrArg(X1,Str(message),1);
		      GetStrArg(X2,Str(message),2);
		      return method_domain(self,X0,exs);
		    }
		  else
		    if(op == functor_range_1)
		      {
			GetStrArg(X0,Str(message),0);
			GetStrArg(X1,Str(message),1);
			GetStrArg(X2,Str(message),2);
			return method_range(self,X0,exs);
		      }
		    else
		      if(op == functor_attributes_1)
			{
			  GetStrArg(X0,Str(message),0);
			  GetStrArg(X1,Str(message),1);
			  GetStrArg(X2,Str(message),2);
			  return method_attributes(self,X0,exs);
			}
		      else
			if(op == functor_typeof)
			  {
			    GetStrArg(X0,Str(message),0);
			    Deref(X0,X0);
			    return unify(X0, TagAtm(atom_hash_table), exs->andb, exs);
			  }
			else
			  if(op == functor_sizeof)
			    {
			      Term tmp;
			      
			      GetStrArg(X0,Str(message),0);
			      MakeIntegerTerm(tmp, Hash(self)->table->size);
			      Deref(X0,X0);
			      return unify(X0, tmp, exs->andb, exs);
			    }
      
      USAGE_FAULT("Wrong message sent to hash table");
    }
  
  IfVarSuspend(message);
  USAGE_FAULT("Wrong message sent to hash table");
}

/* ######################################################################## */

/* These are used if in close list */

static int uncopiedhash(gtable)
     GenHTable *gtable;
{
  return 0;
}

static envid *deallocatehash(gtable)
     GenHTable *gtable;
{
  free(gtable->table);
  return NULL;
}

static int killhash(gtable)
     GenHTable *gtable;
{
  free(gtable->table);
  return 1;
}

/* ######################################################################## */

/* hash(+GenHTable) */

static bool akl_hash_table(Arg)
     Argdecl;
{
  Term X0;
  Deref(X0,A(0));
  
  if(IsGenHTable(X0))
    return TRUE;
  IfVarSuspend(X0);
  return FALSE;
}
  
/* ######################################################################## */

static bool new_ghash_table(exs,Size,New,ground_key)
     exstate *exs;
     Term Size, New;
     int ground_key;
{
  int size;
  GenHTable *new;
  Term tmpv;

  if (Size == NullTerm)
    size = DEFAULT_TABLE_SIZE;
  else
    {
      Deref(Size,Size);		/* The size of the new hash */
      IfVarSuspend(Size);

      if (!IsINT(Size))
	USAGE_FAULT("new_hash_table/2: Size is not a number");

      if ((size = IntVal(Int(Size))) < 1)
	USAGE_FAULT("new_hash_table/2: Size is less than minimum 1");
    }

  MakeGvainfo(new,GenHTable,&hashmethods,exs->andb);

  new->ground_key = ground_key;

  NewHashTable(new->table,size,HashAllocate);

  if (new->table == NULL)
    USAGE_FAULT("new_hash/2: Could not allocate hash table");    

  if (SizeOfHashTable(size) > HLIMIT)
    add_gvainfo_to_close((Gvainfo)new,exs);

  Deref(New,New);
  MakeCvaTerm(tmpv, (Gvainfo)new);
  return unify(New,tmpv,exs->andb,exs);
}

/* ------------------------------------------------------------------------ */


static bool akl_new_hash_table_1(Arg)
     Argdecl;
{
  return new_ghash_table(exs,NullTerm,A(0),FALSE);
}


static bool akl_new_hash_table_2(Arg)
     Argdecl;
{
  return new_ghash_table(exs,A(0),A(1),FALSE);
}


static bool akl_new_ghash_table_1(Arg)
     Argdecl;
{
  return new_ghash_table(exs,NullTerm,A(0),TRUE);
}


static bool akl_new_ghash_table_2(Arg)
     Argdecl;
{
  return new_ghash_table(exs,A(0),A(1),TRUE);
}

/* ######################################################################## */

/* indomain(+Key, -YesOrNo)@ +GenHTable */

static bool method_indomain_hash(Key, Value, Self, exs)
     Term Key, Value, Self;
     exstate *exs;
{
  HashTable *table;
  GenHTable *gen;
  HashFuncState state;
  ulong val;
  Atom res = atom_yes;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  Deref(Key, Key);

  gen = Hash(Self);
  table = gen->table;

  if (gen->ground_key)
    HashFind(table, Key, val, GHashValue(Key),
	     {result = hash_compare(exs, Key, key);},
	     {res = atom_no;
	      goto ending;})
  else
    HashFind(table, Key, val, HashValue(Key),
	     {HashCompare(Key,key,result);},
	     {res = atom_no;
      	      goto ending;});
 ending:
  Deref(Value, Value);
  return unify(Value,TagAtm(res),exs->andb,exs);
}

/* ######################################################################## */

/* get(+Key, -Term)@ +GenHTable */

static bool method_get_hash(Key, Value, Self, exs)
     Term Key, Value, Self;
     exstate *exs;
{
  HashTable *table;
  GenHTable *gen;
  HashFuncState state;
  ulong val;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  Deref(Key, Key);

  gen = Hash(Self);
  table = gen->table;

  if (gen->ground_key)
    HashFind(table, Key, val, GHashValue(Key),
	     {result = hash_compare(exs, Key, key);},
	     {return FALSE;})
  else
    HashFind(table, Key, val, HashValue(Key),
	     {HashCompare(Key,key,result);},
	     {return FALSE;})

  Deref(Value, Value);
  return unify(Value,tad_to_term(val),exs->andb,exs);
}

/* ######################################################################## */

/* set(+Key, +Term)@ +GenHTable */

static bool method_set_hash(Key, Value, Self, exs)
     Term Key, Value, Self;
     exstate *exs;
{
  HashTable *table;
  GenHTable *gen;
  HashFuncState state;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  Deref(Key, Key);

  gen = Hash(Self);
  table = gen->table;

  if (gen->ground_key)
    HashEnter(table, Key, Value, GHashValue(Key),
	      {result = hash_compare(exs, Key, key);},
	      HashAllocate,
	      HashFree)
  else
    HashEnter(table, Key, Value, HashValue(Key),
	      {HashCompare(Key,key,result);},
	      HashAllocate,
	      HashFree)

  return TRUE;
}

/* ######################################################################## */

/* set(+Key, -OldTerm, +Term)@ +GenHTable */

static bool method_swap_hash(Key, OldValue, Value, Self, exs)
     Term Key, OldValue, Value, Self;
     exstate *exs;
{
  HashTable *table;
  GenHTable *gen;
  HashFuncState state;
  HashEntry *entry;
  ulong hval;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  Deref(Key, Key);

  gen = Hash(Self);
  table = gen->table;

  if (gen->ground_key)
    {
      state.exs = exs;
      state.result = TRUE;

      hval = ground_hash_func(&state, Key);

      if (state.result == SUSPEND)
	return SUSPEND;			/* suspend */

      HashLookupLoop(table, Key, hval, entry, 
		     {HashCompare(Key,key,result);},
		     {},
		     {return FALSE;});
    }
  else
    {
      IfVarSuspend(Key);
      HashFunc(Key,hval);
      HashLookupLoop(table, Key, hval, entry, 
		     {result = hash_compare(exs, Key, key);},
		     {},
		     {return FALSE;});
    }

  Deref(OldValue, OldValue);

  if (!unify(OldValue,tad_to_term(entry->value),exs->andb,exs))
    return FALSE;

  entry->value = (ulong)Value;	/* destructive set value */
  return TRUE;
}

/* ######################################################################## */

/* delete(+Key)@ +GenHTable */
/* delete(+Key, -Term)@ +GenHTable */

static bool method_delete_hash(Key, Value, Self, exs)
     Term Key, Value, Self;
     exstate *exs;
{
  HashTable *table;
  GenHTable *gen;
  HashFuncState state;
  HashEntry *entry;
  ulong hval;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  Deref(Key, Key);

  gen = Hash(Self);
  table = gen->table;

  if (gen->ground_key)
    {
      state.exs = exs;
      state.result = TRUE;

      hval = ground_hash_func(&state, Key);

      if (state.result == SUSPEND)
	return SUSPEND;			/* suspend */

      HashLookupLoop(table, Key, hval, entry, 
		     {HashCompare(Key,key,result);},
		     {},
		     {return FALSE;});
    }
  else
    {
      IfVarSuspend(Key);
      HashFunc(Key,hval);
      HashLookupLoop(table, Key, hval, entry, 
		     {result = hash_compare(exs, Key, key);},
		     {},
		     {return FALSE;});
    }

  if (Value != NULL)
    {
      Deref(Value, Value);
      if (!unify(Value,tad_to_term(entry->value),exs->andb,exs))
	return FALSE;
    }

  entry->key = DELETED_HASH_ENTRY;	/* clear entry */

  return TRUE;
}

/* ######################################################################## */

/* domain(-DomainList)@Self */

static bool method_domain(Self,X0,exs)
     Term Self,X0;
     exstate *exs;
{
  HashTable *table;
  Term lst = NIL;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  table = Hash(Self)->table;

  HashEntryLoop(table,
		{
		  Term l;

		  MakeListTerm(l);
		  LstCdr(Lst(l)) = lst;
		  lst = l;
		  LstCar(Lst(lst)) = tad_to_term(entry->key);
		});

  Deref(X0, X0);
  return unify(X0,lst,exs->andb,exs);
}

/* ######################################################################## */

/* range(-RangeList)@Self */

static bool method_range(Self,X0,exs)
     Term Self,X0;
     exstate *exs;
{
  HashTable *table;
  Term lst = NIL;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  table = Hash(Self)->table;

  HashEntryLoop(table,
		{
		  Term l;

		  MakeListTerm(l);
		  LstCdr(Lst(l)) = lst;
		  lst = l;
		  LstCar(Lst(lst)) = tad_to_term(entry->value);
		});

  Deref(X0, X0);
  return unify(X0,lst,exs->andb,exs);
}

/* ######################################################################## */

static Functor functor_comma_2;

/* attributes(-AttributeList)@Self */

static bool method_attributes(Self,X0,exs)
     Term Self,X0;
     exstate *exs;
{
  HashTable *table;
  Term lst = NIL;

  /* Self is already dereferenced and checked */

  if(!IsLocalGVA(Ref(Self),exs->andb))
    return SUSPEND;

  table = Hash(Self)->table;

  HashEntryLoop(table,
		{
		  Term l;
		  Structure s;

		  MakeStruct(s,functor_comma_2,exs->andb);
		  StrArgument(s,0) = tad_to_term(entry->key);
		  StrArgument(s,1) = tad_to_term(entry->value);

		  MakeListTerm(l);
		  LstCdr(Lst(l)) = lst;
		  lst = l;
		  LstCar(Lst(lst)) = TagStr(s);
		});

  Deref(X0, X0);
  return unify(X0,lst,exs->andb,exs);
}

/* ######################################################################## */

void initialize_hash() {

  atom_nil = store_atom("[]");
  atom_yes = store_atom("yes");
  atom_no = store_atom("no");
  atom_hash_table = store_atom("hash_table");

  functor_typeof = store_functor(store_atom("typeof"),1);
  functor_sizeof = store_functor(store_atom("sizeof"),1);

  functor_indomain_2 = store_functor(store_atom("indomain"),2);
  functor_get_2 = store_functor(store_atom("get"),2);
  functor_set_2 = store_functor(store_atom("set"),2);
  functor_set_3 = store_functor(store_atom("set"),3);
  functor_set_3 = store_functor(store_atom("set"),3);
  functor_delete_1 = store_functor(store_atom("delete"),1);
  functor_delete_2 = store_functor(store_atom("delete"),2);
  functor_domain_1 = store_functor(store_atom("domain"),1);
  functor_range_1 = store_functor(store_atom("range"),1);
  functor_attributes_1 = store_functor(store_atom("attributes"),1);
  functor_comma_2 = store_functor(store_atom(","),2);
  functor_cons = store_functor(store_atom("."),2);

  define("new_hash_table",akl_new_hash_table_1,1);
  define("new_hash_table",akl_new_hash_table_2,2);
  define("new_ghash_table",akl_new_ghash_table_1,1);
  define("new_ghash_table",akl_new_ghash_table_2,2);
  define("hash_table",akl_hash_table,1);
}
