% module.nl
%
% Contains the code to translate a file written as a 'module' in some
% Prolog flavour, which has 'local' predicates.
% Assuming that 'local' predicates may clash with predicates of the same
% name in another module, they need to be renamed in order to run
% the code in a Prolog without module facilities.
% for example, a local predicate  sort/2, in a module called 'sorting' will be
% renamed sorting$sort - as will every reference to it in the file.
% sort (A, B)  => sorting$sort( A, B)
%
% The grammar is in module_gr.gr and a test files are test1.mod and test2.mod


%module( +Basename).
%

module(Basename) :-
	name( Basename, Chars),
        append( Chars, ".mod", InputFile),
	append( Chars, ".nl", OutputFile),
		% tokenize and parse the file <Basename>.mod
        get_tokens( InputFile, Toks),
	program( Toks, [], Node),
		% unify the program parse tree with a sub representing 
		% the parse tree for the name of the module and the locals
 	unify( Node, pplus( T, sub( moduleX, Toks1, Pt), 1)),
		% extract the modulename from the parse tree
	find_modname( Pt, Modname),
		% extract the local predicate name/aritys
	find_locals( Pt, Locals), 
		% construct a list of new predicate names
	locals_newnames( Locals, Modname, Names),
		% substitute those names in the clauses 
		% and return a parse tree representing
		% the new program
	substitute_names( T, Names, Node1), !,
		% print the new program to the <Basename>.nl file
	extract_tokens( Node1, Tokens2), 
	print_tokens( OutputFile, Tokens2).

%find_locals( +ParseTree, -ListofName/Arity)
%
% unify the parse tree with a sub representing the locals
% already in Name/Arity form
find_locals( Node, L) :-
	S = mult( local, L, Pts),
	unify( Node, pplus( T, S, 2)).

% find_modname( +ParseTree, -Modname)
%
find_modname( Node, Modname) :-
	Sub =  ["module", sub( atom, [Modname], S2)],
	unify( Node, pplus( _, Sub, 3)).

locals_newnames( [], _, []).

locals_newnames( [[L, _, Arity|_]|Rest], Mod, [[L, Arity, New]|News]) :-
	name_modname( L, Arity, Mod, New),
	locals_newnames( Rest, Mod, News).

name_modname( Name, Arity, Modname, Newname) :-
	name( Name, Chars1),
	name( Modname, Chars2),
	append( Chars2, "$", Chars3),
	append( Chars3, Chars1, NewChars),
	name( Newname, NewChars).

% substitute_names( +ParseTree, +Names, -ParseTree)
%
% extract all the goal Parsetrees, process each goal and return the
% new goal parse trees to the Clause
substitute_names( Node, Names, Node1) :-
	Plus= pplus( T, mult( goal, _, GlPts), 4),
	unify( Node, Plus),
	process_goals( GlPts, Names, NewGlPts),	
	Plus1 = pplus( T, mult( goal, _, NewGlPts), 4),
	unify( Node1, Plus1).

process_goals( [], _, []).
process_goals( [GlPt|GlPts], Names, [NewGlPt|NewGlPts]) :-
	p_goal( GlPt, Names, NewGlPt), 
	process_goals( GlPts, Names, NewGlPts).

% p_goal( +GoalParseTree, +Names, -NewGoalParseTree)
%
% unify Goal Parse tree with  a sequence of subs representing the
% atom, and arguments
% if the goal atom is in the list of locals, then the new name is
% substituted and the New Goal Parse tree returned
% else - the second clause, the goal is unaltered 
p_goal( GlPt, Names, NewGlPt) :-
	Plus = pplus( T, [ sub( atom, [C], Cpt),  
				sub( args(A), Arg, APt)], 5),
	unify( GlPt, Plus), 
	member( [C, A, NewC], Names), !,
	parse( atom, [NewC], NewCPt),
	Plus1 = pplus( T, [ sub( atom, [NewC], NewCPt), 
                                sub( args(A), Arg, APt)], 5),
	unify( NewGlPt, Plus1).
p_goal( GlPt, _, GlPt).
	

