/* dylan-phrases.y -- yacc parseer for infix dylan phrase grammar */

%start	dylan_program

/* lexical objects */

%term	SYMBOL
%term	KEYWORD
%term	BINARY_OPERATOR
%term	NUMBER
%term	STRING_LITERAL
%term	CHARACTER_LITERAL
%term	HASHQUOTED_SYMBOL

/* reserved words */

%term	ABOVE BEGIN BELOW BLOCK BY CASE CLASS CLEANUP CONSTANT
%term	CREATE DEFINE ELSE END EXCEPTION EXPORT FINALLY FOR FROM
%term	GENERIC HANDLER IF IN INSTANCE LET LOCAL METHOD MODULE
%term	OTHERWISE SELECT SLOT SUBCLASS THEN TO UNLESS UNTIL USE
%term	VARIABLE VIRTUAL WHILE

/* punctuation */

%term	COLONCOLON	/* :: */
%term	EQEQ		/* == */
%term	EQARROW		/* => */
%term	HASHBACKQ	/* #` */
%term	HASHPAREN	/* #( */
%term	HASHBRACKET	/* #[ */
%term	HASHT		/* #t */
%term	HASHF		/* #f */
%term	HASHKEY		/* #key */
%term	HASHNEXT	/* #next */
%term	HASHREST	/* #rest */

%%

dylan_program
	: body

body	: constituents semicolon_opt

constituents
	: constituent
	| constituents ';' constituent

constituent
	: defining_form
	| local_declaration
	| expression

defining_form
	: DEFINE CLASS class_definition
	| DEFINE CONSTANT bindings
	| DEFINE GENERIC generic_function_definition
	| DEFINE METHOD named_method
	| DEFINE VARIABLE bindings
	| DEFINE MODULE module_definition

local_declaration
	: LET bindings
	| LET HANDLER SYMBOL '=' expression
	| LET HANDLER '(' expression property_list_opt ')' '=' expression
	| LOCAL local_methods 

bindings
	: variables '=' expression
	| '(' variables ')' '=' expression

variables
	: HASHREST SYMBOL
	| variable
	| VARIABLE ',' variables

variable
	: SYMBOL
	| SYMBOL COLONCOLON operand

local_methods
	: local_method
	| local_methods ',' local_method

local_method
	: method_opt named_method

expression
	: KEYWORD
	| operand binop_series

binop_series
	:
	| binop_series BINARY_OPERATOR operand
	| binop_series '-' operand
	| binop_series '=' operand
	| binop_series EQEQ operand

operand	: '-' operand
	| '~' operand
	| leaf

leaf	: const
	| SYMBOL
	| leaf '[' arguments_opt ']'
	| leaf '(' arguments_opt ')'
	| anonymous_method
	| leaf '.' SYMBOL
	| '(' expression ')'
	| statement

arguments
	: argument
	| arguments ',' argument

argument
	: keyword_opt expression

property_list
	: ',' KEYWORD expression
	| property_list ',' KEYWORD expression

return_type_list
	: HASHREST expression
	| expression ',' expression
	| expression ',' return_type_list

literal	: STRING_LITERAL
	| CHARACTER_LITERAL
	| NUMBER
	| HASHQUOTED_SYMBOL

const	: literal
	| HASHT
	| HASHF
	| HASHPAREN dotted_list ')'
	| HASHPAREN bare_literals_opt ')'
	| HASHBRACKET bare_literals_opt ']'

dotted_list
	: bare_literals '.' bare_literal

bare_literals
	: bare_literal
	| bare_literals ',' bare_literal

bare_literal
	: '(' dotted_list ')'
	| '(' bare_literals_opt ')'
	| '[' bare_literals_opt ']'
	| const
	| SYMBOL
	| KEYWORD

statement
	: BEGIN body_opt end_opt
	| BLOCK '(' symbol_opt ')' body block_epilog_opt END block_opt
	| CASE condition_body END case_opt
	| IF '(' expression ')' body else_part_opt END if_opt
	| FOR '(' for_header ')' body_opt final_part_opt END for_opt
	| SELECT '(' expression by_part_opt ')' condition_body END select_opt
	| UNLESS '(' expression ')' body END unless_opt
	| UNTIL '(' expression ')' body_opt END until_opt
	| WHILE '(' expression ')' body_opt END while_opt

block_epilog
	: exception_clauses_opt cleanup_part exception_clauses_opt
	| exception_clauses

condition_body
	: complete_condition_clauses

for_header
	: UNTIL expression
	| WHILE expression
	| for_clause
	| for_clause ',' for_header

exception_clauses
	: exception_clause
	| exception_clauses exception_clause

exception_clause
	: EXCEPTION SYMBOL body
	| EXCEPTION '(' expression property_list_opt ')' body
	| EXCEPTION '(' SYMBOL COLONCOLON expression property_list_opt ')' body

condition_clauses
	: complete_condition_clauses
	| incomplete_condition_clauses

complete_condition_clauses
	: OTHERWISE eqarrow_opt body
	| condition_clause semicolon_opt
	| condition_clause ';' condition_clauses

incomplete_condition_clauses
	: constituent semicolon_opt
	| constituent ';' condition_clauses

condition_clause
	: expression EQARROW constituent
	| expression ',' condition_clause

for_clause
	: VARIABLE '=' expression THEN expression
	| VARIABLE IN expression
	| VARIABLE FROM expression to_part_opt by_part_opt

by_part	: BY expression

cleanup_part
	: CLEANUP body

else_part
	: ELSE body

final_part
	: FINALLY body

to_part	: TO expression
	| ABOVE expression
	| BELOW expression

class_definition
	: SYMBOL '(' superclasses ')' slot_specs_opt END class_opt symbol_opt

superclasses
	: expression
	| superclasses ',' expression

slot_specs
	: slot_spec semicolon_opt
	| slot_spec ';' slot_specs

slot_spec
	: allocation_opt SLOT symbol_opt slot_type_opt property_list_opt

allocation
	: INSTANCE
	| CLASS
	| SUBCLASS
	| CONSTANT
	| VIRTUAL

slot_type
	: COLONCOLON expression

generic_function_definition
	: SYMBOL '(' gf_parameters_opt ')' gf_suffix

gf_suffix
	: property_list_opt
	| COLONCOLON expression property_list_opt
	| COLONCOLON '(' return_type_list ')' property_list_opt
	| COLONCOLON return_type_list

gf_parameters
	: gf_positional_parameters
	| gf_positional_parameters ',' gf_rest_parameters
	| gf_rest_parameters

gf_positional_parameters
	: SYMBOL
	| gf_positional_parameters ',' SYMBOL

gf_rest_parameters
	: rest_parameter
	| rest_parameter ',' HASHKEY gf_keyword_parameters_opt
	| HASHKEY gf_keyword_parameters_opt

gf_keyword_parameters
	: KEYWORD
	| gf_keyword_parameters ',' KEYWORD

anonymous_method
	: METHOD method_description END method_opt

named_method
	: SYMBOL method_description END method_opt symbol_opt

method_description
	: '(' parameters_opt ')' return_type ';' body
	| '(' parameters_opt ')' return_type semicolon_opt
	| '(' parameters_opt ')' body_opt

return_type
	: COLONCOLON expression
	| COLONCOLON return_type_list
	| COLONCOLON '(' return_type_list ')'

parameters
	: positional_parameters
	| positional_parameters ',' next_parameters
	| next_parameters

positional_parameters
	: positional_parameter
	| positional_parameters ',' positional_parameter

positional_parameter
	: SYMBOL
	| SYMBOL COLONCOLON expression
	| SYMBOL EQEQ expression

next_parameters
	: next_parameter
	| next_parameter ',' rest_parameters
	| rest_parameters

next_parameter
	: HASHNEXT SYMBOL

rest_parameters
	: rest_parameter
	| rest_parameter ',' HASHKEY keyword_parameters_opt
	| HASHKEY keyword_parameters_opt

rest_parameter
	: HASHREST SYMBOL

keyword_parameters
	: keyword_parameter
	| keyword_parameters ',' keyword_parameter

keyword_parameter
	: KEYWORD SYMBOL
	| KEYWORD SYMBOL '(' expression ')'


/*
 * modules -- note that use clause options are not handled here
 */

module_definition
	: module_name module_clauses_opt END module_opt module_name_opt

module_clauses
	: module_clause semicolon_opt
	| module_clause ';' module_clauses

module_clause
	: module_use_clause
	| module_export_clause
	| module_create_clause

module_export_clause
	: EXPORT variable_names

module_create_clause
	: CREATE variable_names

module_use_clause
	: USE module_name property_list_opt

variable_names
	: variable_name
	| variable_name ',' variable_names

variable_name
	: SYMBOL

module_name
	: SYMBOL


/*
 * optional reductions
 */

allocation_opt
	:
	| allocation

arguments_opt
	:
	| arguments

bare_literals_opt
	:
	| bare_literals

block_epilog_opt
	:
	| block_epilog

block_opt
	:
	| BLOCK

body_opt
	:
	| body

by_part_opt
	:
	| by_part

case_opt
	:
	| CASE

class_opt
	:
	| CLASS

else_part_opt
	:
	| else_part

end_opt	:
	| END

eqarrow_opt
	:
	| EQARROW

exception_clauses_opt
	:
	| exception_clauses

final_part_opt
	:
	| final_part

for_opt	:
	| FOR

gf_keyword_parameters_opt
	:
	| gf_keyword_parameters

gf_parameters_opt
	:
	| gf_parameters

if_opt	:
	| IF

keyword_opt
	:
	| KEYWORD

keyword_parameters_opt
	:
	| keyword_parameters

method_opt
	:
	| METHOD

module_clauses_opt
	:
	| module_clauses

module_name_opt
	:
	| module_name

module_opt
	:
	| MODULE

parameters_opt
	:
	| parameters

property_list_opt
	:
	| property_list

select_opt
	:
	| SELECT

semicolon_opt
	:
	| ';'

slot_specs_opt
	:
	| slot_specs

slot_type_opt
	:
	| slot_type

symbol_opt
	:
	| SYMBOL

to_part_opt
	:
	| to_part

unless_opt
	:
	| UNLESS

until_opt
	:
	| UNTIL

while_opt
	:
	| WHILE
