/*
    symbol.d -- Symbols.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

#ifndef MTCL
object string_register;
object gensym_prefix;
object gentemp_prefix;
object token;
#endif MTCL

struct symbol Cnil_body, Ct_body;

/******************************* ------- ******************************/

object siSpname;

int gentemp_counter;
int gensym_counter;

void setup_string_register(char *s)
{
	string_register->st.st_dim =
	(string_register->st.st_fillp = strlen(s))+1;
	string_register->st.st_self = s;
}

object
make_symbol(object st)
{
	object x;

	x = alloc_object(t_symbol);
	x->s.s_dbind = OBJNULL;
	x->s.s_sfdef = NOT_SPECIAL;
	x->s.s_fillp = st->st.st_fillp;
	x->s.s_self = NULL;
	x->s.s_gfdef = OBJNULL;
	x->s.s_plist = Cnil;
	x->s.s_hpack = Cnil;
	x->s.s_stype = (short)stp_ordinary;
	x->s.s_mflag = FALSE;
	if (st->st.st_self < heap_end)
		x->s.s_self = st->st.st_self;
	else {
		x->s.s_self = alloc_contblock(x->s.s_fillp+1);
		memcpy(x->s.s_self, st->st.st_self, st->st.st_fillp+1);
	}
	return(x);
}

/*
	Make_ordinary(s) makes an ordinary symbol from C string s
	and interns it in lisp package as an external symbol.
*/
object
make_ordinary(char *s)
{
	int j;
	object x, l, *ep;

	setup_string_register(s);
	j = pack_hash(string_register);
	ep = &lisp_package->p.p_external[j];
	for (l = *ep;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register))
			return(CAR(l));
	x = make_symbol(string_register);
	x->s.s_hpack = lisp_package;
	*ep = CONS(x, *ep);
	return(x);
}

/*
	Make_special(s, v) makes a special variable from C string s
	with initial value v in lisp package.
*/
object
make_special(char *s, object v)
{
	object x;

	x = make_ordinary(s);
	x->s.s_stype = (short)stp_special;
	x->s.s_dbind = v;
	return(x);
}

/*
	Make_constant(s, v) makes a constant from C string s
	with constant value v in lisp package.
*/
object
make_constant(char *s, object v)
{
	object x;

	x = make_ordinary(s);
	x->s.s_stype = (short)stp_constant;
	x->s.s_dbind = v;
	return(x);
}

/*
	Make_si_ordinary(s) makes an ordinary symbol from C string s
	and interns it in system package as an external symbol.
	It assumes that the (only) package used by system is lisp.
*/
object
make_si_ordinary(char *s)
{
	int j;
	object x, l, *ep;

	setup_string_register(s);
	j = pack_hash(string_register);
	ep = &system_package->p.p_external[j];
	for (l = *ep;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register))
			return(CAR(l));
	for (l = lisp_package->p.p_external[j];
	     type_of(l) == t_cons;
	     l = CDR(l))
		if (string_eq(CAR(l), string_register))
		    error("name conflict --- can't make_si_ordinary.");
	x = make_symbol(string_register);
	x->s.s_hpack = system_package;
	*ep = CONS(x, *ep);
	return(x);
}

/*
	Make_si_special(s, v) makes a special variable from C string s
	with initial value v in system package.
*/
object
make_si_special(char *s, object v)
{
	object x;

	x = make_si_ordinary(s);
	x->s.s_stype = (short)stp_special;
	x->s.s_dbind = v;
	return(x);
}

/*
	Make_si_constant(s, v) makes a constant from C string s
	with constant value v in system package.
*/
object
make_si_constant(char *s, object v)
{
	object x;

	x = make_si_ordinary(s);
	x->s.s_stype = (short)stp_constant;
	x->s.s_dbind = v;
	return(x);
}

/*
	Make_keyword(s) makes a keyword from C string s.
*/
object
make_keyword(char *s)
{
	int j;
	object x, l, *ep;

	setup_string_register(s);
	j = pack_hash(string_register);
	ep = &keyword_package->p.p_external[j];
	for (l = *ep;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register))
			return(CAR(l));
	x = make_symbol(string_register);
	x->s.s_hpack = keyword_package;
	x->s.s_stype = (short)stp_constant;
	x->s.s_dbind = x;
	*ep = CONS(x, *ep);
	return(x);
}

object
symbol_value(object s)
{
	if (s->s.s_dbind == OBJNULL)
		FEunbound_variable(s);
	return(s->s.s_dbind);
}

object
getf(object place, object indicator, object deflt)
{
	object l;

	for (l = place;  !endp(l);  l = CDDR(l)) {
		if (endp(CDR(l)))
			odd_plist(place);
		if (CAR(l) == indicator)
			return(CADR(l));
	}
	return(deflt);
}

object
get(object s, object p, object d)
{
	if (type_of(s) != t_symbol)
		not_a_symbol(s);
	return(getf(s->s.s_plist, p, d));
}

/*
	Putf(p, v, i) puts value v for property i to property list p
	and returns the resulting property list.
*/
object
putf(object p, object v, object i)
{
	object l, l0 = p;

	for (l = p;  !endp(l);  l = CDDR(l)) {
		if (endp(CDR(l)))
			odd_plist(l0);
		if (CAR(l) == i) {
			CADR(l) = v;
			return(p);
		}
	}
	l = CONS(v, p);
	l = CONS(i, l);
	return(l);
}

object
putprop(object s, object v, object p)
{
	if (type_of(s) != t_symbol)
		not_a_symbol(s);
	s->s.s_plist = putf(s->s.s_plist, v, p);
	return(v);
}

/*
	Remf(p, i) removes property i
	from the property list pointed by p,
	which is a pointer to an object.
	The returned value of remf(p, i) is:

		TRUE    if the property existed
		FALSE   otherwise.
*/
bool
remf(object *p, object i)
{
	object l0 = *p;

	for(;  !endp(*p);  p = &CDDR((*p))) {
		if (endp(CDR((*p))))
			odd_plist(l0);
		if (CAR((*p)) == i) {
			*p = CDDR((*p));
			return(TRUE);
		}
	}
	return(FALSE);
}

object
remprop(object s, object p)
{
	if (type_of(s) != t_symbol)
		not_a_symbol(s);
	if (remf(&s->s.s_plist, p))
		return(Ct);
	else
		return(Cnil);
}

bool
keywordp(object s)
{
	return(type_of(s) == t_symbol && s->s.s_hpack == keyword_package);
}

@(defun get (sym indicator &optional deflt)
@
	check_type_symbol(&sym);
	@(return `getf(sym->s.s_plist, indicator, deflt)`)
@)

Lremprop(int narg, object sym, object prop)
{
	check_arg(2);

	check_type_symbol(&sym);
	if (remf(&sym->s.s_plist, prop))
		VALUES(0) = Ct;
	else
		VALUES(0) = Cnil;
	RETURN(1);
}

Lsymbol_plist(int narg, object sym)
{
	check_arg(1);

	check_type_symbol(&sym);
	VALUES(0) = sym->s.s_plist;
	RETURN(1);
}

@(defun getf (place indicator &optional deflt)
@
	@(return `getf(place, indicator, deflt)`)
@)

@(defun get_properties (place indicator_list)
	object l, m;
@
	for (l = place;  !endp(l);  l = CDDR(l)) {
		if (endp(CDR(l)))
			odd_plist(place);
		for (m = indicator_list;  !endp(m);  m = CDR(m))
			if (CAR(l) == CAR(m))
				@(return `CAR(l)`
					 `CADR(l)`
					 l)
	}
	@(return Cnil Cnil Cnil)
@)

object
symbol_name(object x)
{       object y;
	check_type_symbol(&x);
	if ((y = getf(x->s.s_plist, siSpname, Cnil)) != Cnil)
	  return(y);
	y = alloc_simple_string(x->s.s_fillp);
	if (x->s.s_self < heap_end)
		y->st.st_self = x->s.s_self;
	else {
		y->st.st_self = alloc_contblock(x->s.s_fillp+1);
		memcpy(y->st.st_self, x->st.st_self, x->st.st_fillp+1);
	}
	x->s.s_plist = putf(x->s.s_plist, y, siSpname);
	return(y);
}

Lsymbol_name(int narg, object sym)
{
	check_arg(1);
    VALUES(0)=symbol_name(sym);
	RETURN(1);
}

Lmake_symbol(int narg, object str)
{
	check_arg(1);

	check_type_string(&str);
	VALUES(0) = make_symbol(str);
	RETURN(1);
}

@(defun copy_symbol (sym &optional cp &aux x)
@
	check_type_symbol(&sym);
	x = make_symbol(sym);
	if (Null(cp))
		@(return x)
	x->s.s_stype = sym->s.s_stype;
	x->s.s_dbind = sym->s.s_dbind;
	x->s.s_mflag = sym->s.s_mflag;
	x->s.s_gfdef = sym->s.s_gfdef;
	x->s.s_plist = copy_list(sym->s.s_plist);
	@(return x)
@)

@(defun gensym (&optional (x gensym_prefix) &aux sym)
	int i, j;
@
	if (type_of(x) == t_string)
		gensym_prefix = x;
	else {
		check_type_non_negative_integer(&x);
		if (FIXNUMP(x))
			gensym_counter = fix(x);
		else
			gensym_counter = 0;
			/*  incorrect implementation  */
	}
	for (j = gensym_counter, i = 0;  j > 0;  j /= 10)
		i++;
	if (i == 0)
		i++;
	i += gensym_prefix->st.st_fillp;
	setup_string_register("");
	sym = make_symbol(string_register);
	sym->s.s_fillp = i;
	sym->s.s_self = alloc_contblock(i+1);
	sym->s.s_self[i] = '\0';
	for (j = 0;  j < gensym_prefix->st.st_fillp;  j++)
		sym->s.s_self[j] = gensym_prefix->st.st_self[j];
	if ((j = gensym_counter) == 0)
		sym->s.s_self[--i] = '0';
	else
		for (;  j > 0;  j /= 10)
			sym->s.s_self[--i] = j%10 + '0';
	gensym_counter++;
	@(return sym)
@)

@(defun gentemp (&optional (prefix gentemp_prefix)
			   (pack `current_package()`)
		 &aux smbl)
	int i, j;
@
	check_type_string(&prefix);
	check_type_package(&pack);
/*
	gentemp_counter = 0;
*/
ONCE_MORE:
	for (j = gentemp_counter, i = 0;  j > 0;  j /= 10)
		i++;
	if (i == 0)
		i++;
	i += prefix->st.st_fillp;
        string_register->st.st_dim = (string_register->st.st_fillp = i) + 1;
        string_register->st.st_self = alloc_contblock(i+1);
	string_register->st.st_self[i] = '\0';
        for (j = 0;  j < prefix->st.st_fillp;  j++)
                string_register->st.st_self[j] = prefix->st.st_self[j];
	if ((j = gentemp_counter) == 0)
		string_register->st.st_self[--i] = '0';
	else
		for (;  j > 0;  j /= 10)
			string_register->st.st_self[--i] = j%10 + '0';
	gentemp_counter++;
	smbl = intern(string_register->st.st_self, pack);
	if (intern_flag != 0)
		goto ONCE_MORE;
	@(return smbl)
@)

Lsymbol_package(int narg, object sym)
{
	check_arg(1);

	check_type_symbol(&sym);
	VALUES(0) = sym->s.s_hpack;
	RETURN(1);
}

Lkeywordp(int narg, object sym)
{
	check_arg(1);

	if (type_of(sym) == t_symbol && keywordp(sym))
		VALUES(0) = Ct;
	else
		VALUES(0) = Cnil;
	RETURN(1);
}

/*
	(SI:PUT-F plist value indicator)
	returns the new property list with value for property indicator.
	It will be used in SETF for GETF.
*/
siLput_f(int narg, object plist, object value, object indicator)
{
	check_arg(3);

	VALUES(0) = putf(plist, value, indicator);
	RETURN(1);
}

/*
	(SI:REM-F plist indicator) returns two values:

		* the new property list
		  in which property indcator is removed

		* T     if really removed
		  NIL   otherwise.

	It will be used for macro REMF.
*/
siLrem_f(int narg, object plist, object indicator)
{
	check_arg(2);

	if (remf(&plist, indicator))
		VALUES(1) = Ct;
	else
		VALUES(1) = Cnil;
	VALUES(0) = plist;
	RETURN(2);
}

siLset_symbol_plist(int narg, object sym, object plist)
{
	check_arg(2);

	check_type_symbol(&sym);
	sym->s.s_plist = plist;
	VALUES(0) = plist;
	RETURN(1);
}

siLputprop(int narg, object sym, object value, object indicator)
{
	check_arg(3);

	check_type_symbol(&sym);
	sym->s.s_plist = putf(sym->s.s_plist, value, indicator);
	VALUES(0) = value;
	RETURN(1);
}

odd_plist(object place)
{
	FEerror("The length of the property-list ~S is odd.", 1, place);
}

/* Added for defstruct. Beppe */
siLput_properties(int narg, object sym, ...)
{       va_list ind_values; object prop;

	va_start(ind_values, sym);
	while (--narg >= 2) {
	  prop = va_arg(ind_values, object);
	  putprop(sym, va_arg(ind_values, object), prop);
	  narg--;
	}
	VALUES(0) = sym;
	RETURN(1);
}

init_symbol()
{
	Cnil_body.t = (short)t_symbol;
	Cnil_body.s_dbind = Cnil;
	Cnil_body.s_sfdef = NOT_SPECIAL;
	Cnil_body.s_fillp = 3;
	Cnil_body.s_self = "NIL";
	Cnil_body.s_gfdef = OBJNULL;
	Cnil_body.s_plist = Cnil;
	Cnil_body.s_hpack = Cnil;
	Cnil_body.s_stype = (short)stp_constant;
	Cnil_body.s_mflag = FALSE;

	Ct_body.t = (short)t_symbol;
	Ct_body.s_dbind = Ct;
	Ct_body.s_sfdef = NOT_SPECIAL;
	Ct_body.s_fillp = 1;
	Ct_body.s_self = "T";
	Ct_body.s_gfdef = OBJNULL;
	Ct_body.s_plist = Cnil;
	Ct_body.s_hpack = Cnil;
	Ct_body.s_stype = (short)stp_constant;
	Ct_body.s_mflag = FALSE;

	string_register = alloc_simple_string(0);
	gensym_prefix = make_simple_string("G");
	gensym_counter = 0;
	gentemp_prefix = make_simple_string("T");
	gentemp_counter = 0;
	token = alloc_simple_string(LISP_PAGESIZE);
	token->st.st_fillp = 0;
	token->st.st_self = alloc_contblock(LISP_PAGESIZE);
	token->st.st_hasfillp = TRUE;
	token->st.st_adjustable = TRUE;

	enter_mark_origin(&string_register);
	enter_mark_origin(&gensym_prefix);
	enter_mark_origin(&gentemp_prefix);
	enter_mark_origin(&token);
}

init_symbol_function()
{
	make_function("GET", Lget);
	make_function("REMPROP", Lremprop);
	make_function("SYMBOL-PLIST", Lsymbol_plist);
	make_function("GETF", Lgetf);
	make_function("GET-PROPERTIES", Lget_properties);
	make_function("SYMBOL-NAME", Lsymbol_name);
	make_function("MAKE-SYMBOL", Lmake_symbol);
	make_function("COPY-SYMBOL", Lcopy_symbol);
	make_function("GENSYM", Lgensym);
	make_function("GENTEMP", Lgentemp);
	make_function("SYMBOL-PACKAGE", Lsymbol_package);
	make_function("KEYWORDP", Lkeywordp);

	make_si_function("PUT-F", siLput_f);
	make_si_function("REM-F", siLrem_f);
	make_si_function("SET-SYMBOL-PLIST", siLset_symbol_plist);

	make_si_function("PUTPROP", siLputprop);

	siSpname = make_si_ordinary("PNAME");
	enter_mark_origin(&siSpname);

	make_si_function("PUT-PROPERTIES", siLput_properties);
}
