/*
    structure.c -- Structure interface.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

object siSstructure_print_function;
object siSstructure_slot_descriptions;

/******************************* ------- ******************************/

#ifdef CLOS
#define T_STRUCTURE	t_instance
#define STYPE(x)	CLASS_OF(x)
#define SLOTS(x)	(x)->in.in_slots
#define SLENGTH(x)	(x)->in.in_length
#define SLOT(x,i)	(x)->in.in_slots[i]
#define SNAME(x)	CLASS_NAME(CLASS_OF(x))
#define STRUCTUREP(x)	(type_of(x) == t_instance && \
			 structure_subtypep(CLASS_OF(x), Sstructure_object))
#else
#define T_STRUCTURE	t_structure
#define STYPE(x)	x->str.str_name
#define SLOTS(x)	(x)->str.str_self
#define SLENGTH(x)	(x)->str.str_length
#define SLOT(x,i)	(x)->str.str_self[i]
#define SNAME(x)	x->str.str_name
#define STRUCTUREP(x)	(type_of(x) == t_structure)
#endif

#ifdef CLOS
object Sstructure_object;

bool
structure_subtypep(object x, object y)
{ object superiors;
  if (CLASS_NAME(x) == y)
    return(TRUE);
  for (superiors=CLASS_SUPERIORS(x); superiors!=Cnil;
       superiors=CDR(superiors)) {
    if (structure_subtypep(CAR(superiors), y))
      return(TRUE);
  }
  return(FALSE);
}
#else
object siSstructure_include;

bool
structure_subtypep(object x, object y)
{
	do {
		if (type_of(x) != t_symbol)
			return(FALSE);
		if (x == y)
			return(TRUE);
		x = get(x, siSstructure_include, Cnil);
	} while (x != Cnil);
	return(FALSE);
}
#endif CLOS

siLstructure_subtype_p(int narg, object x, object y)
{
	VALUES(0) = (type_of(x) == T_STRUCTURE
		     && structure_subtypep(STYPE(x), y) ? Ct : Cnil);
	RETURN(1);
      }

object
structure_ref(object x, object name, int n)
{

	if (type_of(x) != T_STRUCTURE ||
	    !structure_subtypep(STYPE(x), name))
		FEwrong_type_argument(name, x);
	return(SLOT(x, n));
}

object
structure_set(object x, object name, int n, object v)
{

	if (type_of(x) != T_STRUCTURE ||
	    !structure_subtypep(STYPE(x), name))
		FEwrong_type_argument(name, x);
	SLOT(x, n) = v;
	return(v);
}

#ifndef CLOS
/* This is only used for printing. Should not cons!! */ 
object
structure_to_list(object x)
{
	object *p, r, s;
	int i, n;

	s = getf(SNAME(x)->s.s_plist,
	         siSstructure_slot_descriptions, Cnil);
	p = &CDR(r = CONS(SNAME(x), Cnil));
	for (i=0, n=SLENGTH(x);  !endp(s) && i<n;  s=CDR(s), i++) {
		p = &(CDR(*p = CONS(car(CAR(s)), Cnil)));
		p = &(CDR(*p = CONS(SLOT(x, i), Cnil)));
	}
	return(r);
}
#else
object
structure_to_list(object x)
{ FEerror("Should never be called!",0);
}
#endif CLOS

siLmake_structure(int narg, object type, ...)
{
	va_list args;
	object x;
	int i;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	x = alloc_object(T_STRUCTURE);
	STYPE(x) = type;
	SLOTS(x) = NULL;	/* for GC sake */
	SLENGTH(x) = --narg;
	SLOTS(x) = (object *)alloc_relblock(sizeof(object)*narg,
					    sizeof(object));
	va_start(args, type);
	for (i = 0;  i < narg;  i++)
		SLOT(x, i) = va_arg(args, object);
	VALUES(0) = x;
	RETURN(1);
}

siLcopy_structure(int narg, object x)
{
	int j, size;
	object y;

	check_arg(1);
	if (!STRUCTUREP(x))
		FEwrong_type_argument(Sstructure, x);
	y = alloc_object(T_STRUCTURE);
	STYPE(y) = STYPE(x);
	SLENGTH(y) = j = SLENGTH(x);
	size = sizeof(object)*j;
	SLOTS(y) = NULL;	/* for GC sake */
	SLOTS(y) = (object *)alloc_relblock(size, sizeof(object));
	memcpy(SLOTS(y), SLOTS(x), size);
	VALUES(0) = y;
	RETURN(1);
}

/* Kept only for compatibility. One should use class-of or type-of. */
siLstructure_name(int narg, object s)
{
	check_arg(1);
	if (!STRUCTUREP(s))
		FEwrong_type_argument(Sstructure, s);
	VALUES(0) = SNAME(s);
	RETURN(1);
}

siLstructure_ref(int narg, object x, object type, object index)
{
	check_arg(3);

	if (type_of(x) != T_STRUCTURE ||
	    !structure_subtypep(STYPE(x), type))
		FEwrong_type_argument(type, x);
	VALUES(0) = SLOT(x, fix(index));
	RETURN(1);
}

siLstructure_set(int narg, object x, object type, object index, object val)
{
	check_arg(4);

	if (type_of(x) != T_STRUCTURE ||
	    !structure_subtypep(STYPE(x), type))
		FEwrong_type_argument(type, x);
	SLOT(x, fix(index)) = val;
	VALUES(0) = val;
	RETURN(1);
}

siLstructurep(int narg, object s)
{
	check_arg(1);
	VALUES(0) = (STRUCTUREP(s)) ? Ct : Cnil;
	RETURN(1);
}

siLrplaca_nthcdr(int narg, object x, object idx, object v)
{
/*
	Used in DEFSETF forms generated by DEFSTRUCT.
	(si:rplaca-nthcdr x i v) is equivalent to 
	(progn (rplaca (nthcdr i x) v) v).
*/
	int i;
	object l;

	check_arg(3);
	if (!FIXNUMP(idx) || fix(idx) < 0)
		FEerror("~S is not a non-negative fixnum.", 1, idx);
	if (type_of(x) != t_cons)
		FEerror("~S is not a cons.", 1, x);

	for (i = fix(idx), l = x;  i > 0; --i) {
		l = CDR(l);
		if (endp(l))
			FEerror("The offset ~S is too big.", 1, idx);
	}
	CAR(l) = v;
	VALUES(0) = v;
	RETURN(1);
}

siLlist_nth(int narg, object idx, object x)
{
/*
	Used in structure access functions generated by DEFSTRUCT.
	si:list-nth is similar to nth except that
	(si:list-nth i x) is error if the length of the list x is less than i.
*/
	int i;
	object l;

	check_arg(2);
	if (!FIXNUMP(idx) || fix(idx) < 0)
		FEerror("~S is not a non-negative fixnum.", 1, idx);
	if (type_of(x) != t_cons)
		FEerror("~S is not a cons.", 1, x);

	for (i = fix(idx), l = x;  i > 0; --i) {
		l = CDR(l);
		if (endp(l))
			FEerror("The offset ~S is too big.", 1, idx);
	}

	VALUES(0) = CAR(l);
	RETURN(1);
}

init_structure_function()
{
	siSstructure_print_function
	= make_si_ordinary("STRUCTURE-PRINT-FUNCTION");
	enter_mark_origin(&siSstructure_print_function);
	siSstructure_slot_descriptions
	= make_si_ordinary("STRUCTURE-SLOT-DESCRIPTIONS");
	enter_mark_origin(&siSstructure_slot_descriptions);
#ifndef CLOS
	siSstructure_include = make_si_ordinary("STRUCTURE-INCLUDE");
	enter_mark_origin(&siSstructure_include);
#else
	Sstructure_object = make_ordinary("STRUCTURE-OBJECT");
	enter_mark_origin(&Sstructure_object);
#endif CLOS
	make_si_function("MAKE-STRUCTURE", siLmake_structure);
	make_si_function("COPY-STRUCTURE", siLcopy_structure);
	make_si_function("STRUCTURE-NAME", siLstructure_name);
	make_si_function("STRUCTURE-REF", siLstructure_ref);
	make_si_function("STRUCTURE-SET", siLstructure_set);
	make_si_function("STRUCTUREP", siLstructurep);
	make_si_function("STRUCTURE-SUBTYPE-P",siLstructure_subtype_p);
	make_si_function("RPLACA-NTHCDR", siLrplaca_nthcdr);
	make_si_function("LIST-NTH", siLlist_nth);
}
