/*
    number.c -- Numeric constants.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

/*
	IMPLEMENTATION-DEPENDENT

	This file creates some implementation dependent constants.
*/

#include "config.h"
#include "num_include.h"


int
fixint(object x)
{
	if (!FIXNUMP(x))
		FEerror("~S is not a fixnum.", 1, x);
	return(fix(x));
}

int
fixnnint(object x)
{
	if (!FIXNUMP(x) || fix(x) < 0)
		FEerror("~S is not a non-negative fixnum.", 1, x);
	return(fix(x));
}

object
make_ratio(object num, object den)
{
	object g, r, integer_divide1(), get_gcd();

	if (number_zerop(num))
		return(MAKE_FIXNUM(0));
	if (number_zerop(den))
		FEerror("Zero denominator.", 0);
	if (FIXNUMP(den) && fix(den) == 1)
		return(num);
	if (number_minusp(den)) {
		num = number_negate(num);
		den = number_negate(den);
	}
	g = get_gcd(num, den);
	num = integer_divide1(num, g);
	den = integer_divide1(den, g);
	if(FIXNUMP(den) && fix(den) == 1)
		return(num);
	if(FIXNUMP(den) && fix(den) == -1) {
		num = number_negate(num);
		return(num);
	}
	r = alloc_object(t_ratio);
	r->rat.rat_num = num;
	r->rat.rat_den = den;
	return(r);
}

object
make_shortfloat(float f)
{
	object x;

	if (f == (shortfloat)0.0)
		return(shortfloat_zero);
	x = alloc_object(t_shortfloat);
	sf(x) = f;
	return(x);
}

object
make_longfloat(double f)
{
	object x;

	if (f == (longfloat)0.0)
		return(longfloat_zero);
	x = alloc_object(t_longfloat);
	lf(x) = f;
	return(x);
}

object
make_complex(object r, object i)
{
	object c;

	switch (type_of(r)) {
	case t_fixnum:
	case t_bignum:
	case t_ratio:
		switch (type_of(i)) {
		case t_fixnum:
			if (fix(i) == 0)
				return(r);
			break;
		case t_shortfloat:
			r = make_shortfloat((shortfloat)number_to_double(r));
			break;
		case t_longfloat:
			r = make_longfloat(number_to_double(r));
			break;
		}
		break;
	case t_shortfloat:
		switch (type_of(i)) {
		case t_fixnum:
		case t_bignum:
		case t_ratio:
			i = make_shortfloat((shortfloat)number_to_double(i));
			break;
		case t_longfloat:
			r = make_longfloat((double)(sf(r)));
			break;
		}
		break;
	case t_longfloat:
		switch (type_of(i)) {
		case t_fixnum:
		case t_bignum:
		case t_ratio:
		case t_shortfloat:
			i = make_longfloat(number_to_double(i));
			break;
		}
		break;
	}			
	c = alloc_object(t_complex);
	c->cmp.cmp_real = r;
	c->cmp.cmp_imag = i;
	return(c);
}

double
number_to_double(object x)
{
	switch(type_of(x)) {
	case t_fixnum:
		return((double)(fix(x)));

	case t_bignum:
		return(big_to_double((struct bignum *)x));

	case t_ratio:
		return(number_to_double(x->rat.rat_num) /
		       number_to_double(x->rat.rat_den));

	case t_shortfloat:
		return((double)(sf(x)));

	case t_longfloat:
		return(lf(x));

	default:
		wrong_type_argument(TSor_rational_float, x);
	}
}

init_number()
{
	shortfloat_zero = alloc_object(t_shortfloat);
	sf(shortfloat_zero) = (shortfloat)0.0;
	longfloat_zero = alloc_object(t_longfloat);
	lf(longfloat_zero) = (longfloat)0.0;
	enter_mark_origin(&shortfloat_zero);
	enter_mark_origin(&longfloat_zero);

  	make_constant("MOST-POSITIVE-FIXNUM", MAKE_FIXNUM(MOST_POSITIVE_FIX));
	make_constant("MOST-NEGATIVE-FIXNUM", MAKE_FIXNUM(MOST_NEGATIVE_FIX));

	init_num_pred();
	init_num_comp();
	init_num_arith();
	init_num_co();
	init_num_log();
	init_num_sfun();
	init_num_rand();
}
