/*
    array.c --	Array routines
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

#define	ADIMLIM		16*1024*1024
#define	ATOTLIM		16*1024*1024


enum aelttype
get_aelttype(object x)
{
	if (x == Sstring_char)
		return(aet_ch);
	else if (x == Sbit)
		return(aet_bit);
	else if (x == Sfixnum)
		return(aet_fix);
	else if (x == Ssingle_float || x == Sshort_float)
		return(aet_sf);
	else if (x == Slong_float || x == Sdouble_float)
		return(aet_lf);
/*	else if (x == Ssigned_char)
		return(aet_char);
	else if (x == Sunsigned_char)
		return(aet_uchar);
	else if (x == Ssigned_short)
		return(aet_short);
	else if (x == Sunsigned_short)
		return(aet_ushort);
*/	else
		return(aet_object);
}

enum aelttype
array_elttype(object x)
{
	switch(type_of(x)) {
	case t_array:
	case t_vector:
		return((enum aelttype)x->a.a_elttype);

	case t_string:
		return(aet_ch);

	case t_bitvector:
		return(aet_bit);

	default:
		FEwrong_type_argument(Sarray, x);
	}
}

char *
array_address(object x, int inc)
{
	switch(array_elttype(x)) {
	case aet_object:
	case aet_fix:
	case aet_sf:
		return((char *)(x->a.a_self + inc));
/*
        case aet_char:
        case aet_uchar:
*/
	case aet_ch:
		return(x->st.st_self + inc);
/*
        case aet_short:
        case aet_ushort:
		return ((char *)(x->ust.ust_self + inc);
*/
	case aet_lf:
		return((char *)(x->lfa.lfa_self + inc));

	default:
		FEerror("Bad array type", 0);
	}
}

void array_allocself(object x, bool staticp)
{
	int i, d;
	char *(*f)();

	d = x->a.a_dim;
	if (staticp)
		f = alloc_contblock;
	else
		f = alloc_relblock;
#ifdef THREADS
	start_critical_section(); /* avoid losing elts */
#endif THREADS
	switch (array_elttype(x)) {

	/* assign self field only after it has been filled, for GC sake  */
	case aet_object: {
		object *elts;
		elts = (object *)(*f)(sizeof(object)*d, sizeof(object));
		for (i = 0;  i < d;  i++)
			elts[i] = Cnil;
		x->a.a_self = elts;
		break;
	      }
	case aet_ch: {
		char *elts;
		elts = (*f)(d,sizeof(char));
		for (i = 0;  i < d;  i++)
			elts[i] = ' ';
		if (type_of(x) == t_string) elts[d-1] = '\0';
		x->st.st_self = elts;
		break;
	      }
	case aet_bit: {
		char *elts;
		d = (d+(CHAR_BIT-1))/CHAR_BIT;
		elts = (*f)(d,sizeof(char));
		for (i = 0;  i < d;  i++)
			elts[i] = '\0';
		x->bv.bv_offset = 0;
		x->bv.bv_self = elts;
		break;
	      }
	case aet_fix: {
		fixnum *elts;
		elts = (fixnum *)(*f)(sizeof(fixnum)*d, sizeof(fixnum));
		for (i = 0;  i < d;  i++)
			elts[i] = 0;
		x->fixa.fixa_self = elts;
		break;
	      }
	case aet_sf: {
		shortfloat *elts;
		elts = (shortfloat *)(*f)(sizeof(shortfloat)*d,
					  sizeof(shortfloat));
		for (i = 0;  i < d;  i++)
			elts[i] = 0.0;
		x->sfa.sfa_self = elts;
		break;
	      }
	case aet_lf: {
		longfloat *elts;
		elts = (longfloat *)(*f)(sizeof(longfloat)*d,
					 sizeof(longfloat));
		for (i = 0;  i < d;  i++)
			elts[i] = 0.0;
		x->lfa.lfa_self = elts;
		break;
	      }
	}
#ifdef THREADS
	end_critical_section();
#endif THREADS
}

object
aref(object x, int index)
{
  if (index >= x->a.a_dim)
    FEerror("The index, ~D, is too large.", 1, MAKE_FIXNUM(index));
  switch (array_elttype(x)) {
  case aet_object:
    return(x->a.a_self[index]);

  case aet_ch:
    return(code_char(x->ust.ust_self[index]));

  case aet_bit:
    index += x->bv.bv_offset;
    return(MAKE_FIXNUM(x->bv.bv_self[index/CHAR_BIT]
		       & (0200>>index%CHAR_BIT)));
  case aet_fix:
    return(MAKE_FIXNUM(x->fixa.fixa_self[index]));

  case aet_sf:
    return(make_shortfloat(x->sfa.sfa_self[index]));

  case aet_lf:
    return(make_longfloat(x->lfa.lfa_self[index]));
  }
}

object
aset(object x, int index, object value)
{
  int i;

  if (index >= x->a.a_dim)
    FEerror("The index, ~D, too large.", 1, MAKE_FIXNUM(index));
  switch (array_elttype(x)) {
  case aet_object:
    x->a.a_self[index] = value;
    break;

  case aet_ch:
    if (type_of(value) != t_character)
      FEerror("~S is not a character.", 1, value);
    x->st.st_self[index] = char_code(value);
    break;

  case aet_bit:
    i = fixint(value);
    if (i != 0 && i != 1)
      FEerror("~S is not a bit.", 1, value);
    index += x->bv.bv_offset;
    if (i == 0)
      x->bv.bv_self[index/CHAR_BIT] &= ~(0200>>index%CHAR_BIT);
    else
      x->bv.bv_self[index/CHAR_BIT] |= 0200>>index%CHAR_BIT;
    break;

  case aet_fix:
    x->fixa.fixa_self[index] = fixint(value);
    break;

  case aet_sf:
    x->sfa.sfa_self[index] = object_to_float(value);
    break;

  case aet_lf:
    x->lfa.lfa_self[index] = object_to_double(value);
    break;
  }
  return(value);
}

object
aref1(object v, int index)
{
  int i;
  object l;

  if (index < 0)
    FEerror("Negative index: ~D.", 1, MAKE_FIXNUM(index));
  switch (type_of(v)) {
  case t_vector:
  case t_bitvector:
    return(aref(v, index));

  case t_string:
    if (index >= v->st.st_dim)
      goto E;
    return(code_char(v->ust.ust_self[index]));

  default:
    FEerror("~S is not a vector.", 1, v);
  }

 E:
  FEerror("The index, ~D, is too large.", 1, MAKE_FIXNUM(index));
}

object
aset1(object v, int index, object val)
{
  int i;
  object l;

  if (index < 0)
    FEerror("Negative index: ~D.", 1, MAKE_FIXNUM(index));
  switch (type_of(v)) {
  case t_vector:
  case t_bitvector:
    return(aset(v, index, val));

  case t_string:
    if (index >= v->st.st_dim)
      goto E;
    if (type_of(val) != t_character)
      FEerror("~S is not a character.", 1, val);
    v->st.st_self[index] = char_code(val);
    return(val);

  default:
    FEerror("~S is not a vector.", 1, v);
  }

 E:
  FEerror("The index, ~D, is too large", 1, MAKE_FIXNUM(index));
}

/*
	Displace(from, to, offset) displaces the from-array
	to the to-array (the original array) by the specified offset.
	It changes the a_displaced field of both arrays.
	The field is a cons; the car of the from-array points to
	the to-array and the cdr of the to-array is a list of arrays
	displaced to the to-array, so the from-array is pushed to the
	cdr of the to-array's a_displaced.
*/
displace(object from, object to, object offset)
{
	int j;
	enum aelttype totype, fromtype;

	j = fixnnint(offset);
	totype = array_elttype(to);
	fromtype = array_elttype(from);
	if (totype != fromtype)
		FEerror("Cannot displace the array,~%\
because the element types don't match.", 0);
	if (j + from->a.a_dim > to->a.a_dim)
		FEerror("Cannot displace the array,~%\
because the total size of the to-array is too small.", 0);
	from->a.a_displaced = CONS(to, Cnil);
	if (Null(to->a.a_displaced))
		to->a.a_displaced = CONS(Cnil, Cnil);
	CDR(to->a.a_displaced) =
	CONS(from, CDR(to->a.a_displaced));
	if (fromtype == aet_bit) {
		j += to->bv.bv_offset;
		from->bv.bv_self = to->bv.bv_self + j/CHAR_BIT;
		from->bv.bv_offset = j%CHAR_BIT;
	}
#ifndef BYTE_ADDRESS
	else if (fromtype != aet_ch)
		from->a.a_self = (object *)(array_address(to, j));
#endif
	else
		from->st.st_self = array_address(to, j);
}

/*
	Undisplace(from) destroys the displacement from the from-array.
*/
void undisplace(object from)
{
	object *p;
	object to = CAR(from->a.a_displaced);

	if (Null(to))
		return;
	CAR(from->a.a_displaced) = Cnil;
	for (p = &(CDR(to->a.a_displaced));;  p = &(CDR((*p))))
		if (CAR(*p) == from) {
			*p = CDR((*p));
			return;
		}
}

/*
	Check_displaced(dlist, orig, newdim) checks if the displaced
	arrays can keep the displacement when the original array is
	adjusted.
	Dlist is the list of displaced arrays, orig is the original array
	and newdim is the new dimension of the original array.
*/
check_displaced(object dlist, object orig, int newdim)
{
	object x;

	for (;  dlist != Cnil;  dlist = CDR(dlist)) {
		x = CAR(dlist);
		if (x->a.a_self == NULL)
			continue;
		if (array_elttype(x) != aet_bit) {
			if (array_address(x, x->a.a_dim) >
			    array_address(orig, newdim))
				FEerror("Can't keep displacement.", 0);
		} else {
			if ((x->bv.bv_self - orig->bv.bv_self)*CHAR_BIT +
			    x->bv.bv_dim - newdim +
			    x->bv.bv_offset - orig->bv.bv_offset > 0)
				FEerror("Can't keep displacement.", 0);
		}
		check_displaced(CDR(x->a.a_displaced), orig, newdim);
	}
}

/*
	Adjust_displaced(x, diff) adds the int value diff
	to the a_self field of the array x and all the arrays displaced to x.
	This function is used in siLreplace_array (ADJUST-ARRAY) and
	the garbage collector.
*/
void adjust_displaced(object x, int diff)
{
	if (x->a.a_self != NULL)
		x->a.a_self = (object *)((int)(x->a.a_self) + diff);
	for (x = CDR(x->a.a_displaced);  x != Cnil;  x = CDR(x))
		adjust_displaced(CAR(x), diff);
}

/*
	Internal function for making arrays of more than one dimension:

		(si:make-pure-array element-type adjustable
			            displaced-to displaced-index-offset
				    static
			            dim0 dim1 ... )
*/
siLmake_pure_array(int narg, object etype, object adj, object displ,
		   object disploff, object staticp, ...)
{ int r, s, i, j;
  object x;
  va_list dims;
  va_start(dims, staticp);

  r = narg - 5;
  if (r < 0)
    FEtoo_few_arguments(&narg);
  x = alloc_object(t_array);
  x->a.a_displaced = Cnil;
  x->a.a_self = NULL;		/* for GC sake */
  x->a.a_rank = r;
  x->a.a_elttype = (short)get_aelttype(etype);
  x->a.a_dims = (int *)alloc_relblock(sizeof(int)*r, sizeof(int));
  if (r >= ARANKLIM)
    FEerror("The array rank, ~R, is too large.", 1, MAKE_FIXNUM(r));
  for (i = 0, s = 1;  i < r;  i++) {
    object index = va_arg(dims, object);
    if ((j = fixnnint(index)) > ADIMLIM)
      FEerror("The ~:R array dimension, ~D, is too large.",
	      2, MAKE_FIXNUM(i+1), index);
    s *= (x->a.a_dims[i] = j);
  }
  if (s > ATOTLIM)
    FEerror("The array total size, ~D, is too large.", 1, MAKE_FIXNUM(s));
  x->a.a_dim = s;
  x->a.a_adjustable = adj != Cnil;
  if (Null(displ))
    array_allocself(x, staticp != Cnil);
  else
    displace(x, displ, disploff);
  VALUES(0) = x;
  RETURN(1);
}

/*
	Internal function for making vectors:

		(si:make-vector element-type dimension adjustable fill-pointer
				displaced-to displaced-index-offset
			        static)
*/
siLmake_vector(int narg, object etype, object dim, object adj, object fillp,
	       object displ, object disploff, object staticp)
{ int d, f;
  object x;
  enum aelttype aet;

  check_arg(7);
  aet = get_aelttype(etype);
  if ((d = fixnnint(dim)) > ADIMLIM)
    FEerror("The vector dimension, ~D, is too large.", 1, dim);
  f = d;
  if (aet == aet_ch) {
    x = alloc_object(t_string);
    d++;			/* extra for null terminator */
  }
  else if (aet == aet_bit)
    x = alloc_object(t_bitvector);
  else {
    x = alloc_object(t_vector);
    x->v.v_elttype = (short)aet;
  }
  x->v.v_self = NULL;		/* for GC sake */
  x->v.v_displaced = Cnil;
  x->v.v_dim = d;
  x->v.v_adjustable = adj != Cnil;

  if (Null(fillp))
    x->v.v_hasfillp = FALSE;
  else if (fillp == Ct)
    x->v.v_hasfillp = TRUE;
  else if ((f = fixnnint(fillp)) > d)
    FEerror("The fill-pointer ~S is too large.", 1, fillp);
  else
    x->v.v_hasfillp = TRUE;
  x->v.v_fillp = f;

  if (Null(displ))
    array_allocself(x, staticp != Cnil);
  else
    displace(x, displ, disploff);
  VALUES(0) = x;
  RETURN(1);
}

Laref(int narg, object x, ...)
{ int r, s, i, j;
  object index;
  va_list indx;
  va_start(indx, x);
  r = narg - 1;
  if (r < 0)
    FEtoo_few_arguments(&narg);
  switch (type_of(x)) {
  case t_array:
    if (r != x->a.a_rank)
      FEerror("Wrong number of indices.", 0);
    for (i = j = 0;  i < r;  i++) {
      index = va_arg(indx, object);
      if ((s = fixnnint(index)) >= x->a.a_dims[i])
	FEerror("The ~:R index, ~S, to the array~%\
~S is too large.", 3, MAKE_FIXNUM(i+1), index, x);
      j = j*(x->a.a_dims[i]) + s;
    }
    break;

  case t_vector:
  case t_string:
  case t_bitvector:
    if (r != 1)
      FEerror("Wrong number of indices.", 0);
    index = va_arg(indx, object);
    j = fixnnint(index);
    if (j >= x->v.v_dim)
      FEerror("The first index, ~S, to the array ~S is too large.",
	      2, index, x);
    break;

  default:
    FEwrong_type_argument(Sarray, x);
  }
  VALUES(0) = aref(x, j);
  RETURN(1);
}

/*
	Internal function for setting array elements:

		(si:aset value array dim0 ... dimN)
*/
siLaset(int narg, object v, object x, ...)
{ int r, s, i, j;
  object index;
  va_list dims;
  va_start(dims, x);

  r = narg - 2;
  if (r < 0)
    FEtoo_few_arguments(&narg);
  switch (type_of(x)) {
  case t_array:
    if (r != x->a.a_rank)
      FEerror("Wrong number of indices.", 0);
    for (i = j = 0;  i < r;  i++) {
      index = va_arg(dims, object);
      if ((s = fixnnint(index)) >= x->a.a_dims[i])
	FEerror("The ~:R index, ~S, to the array ~S is too large.",
		3, MAKE_FIXNUM(i+1), index, x);
      j = j*(x->a.a_dims[i]) + s;
    }
    break;

  case t_vector:
  case t_string:
  case t_bitvector:
    if (r != 1)
      FEerror("Wrong number of indices.", 0);
    index = va_arg(dims, object);
    j = fixnnint(index);
    if (j >= x->v.v_dim)
      FEerror("The first index, ~S, to the array ~S is too large.",
	      2, index, x);
    break;

  default:
    FEwrong_type_argument(Sarray, x);
  }
  VALUES(0) = aset(x, j, v);
  RETURN(1);
}

Larray_element_type(int narg, object a)
{
  check_arg(1);

  switch (array_elttype(a)) {
  case aet_object:
    VALUES(0) = Ct;
    break;

  case aet_ch:
    VALUES(0) = Sstring_char;
    break;

  case aet_bit:
    VALUES(0) = Sbit;
    break;

  case aet_fix:
    VALUES(0) = Sfixnum;
    break;

  case aet_sf:
    VALUES(0) = Sshort_float;
    break;

  case aet_lf:
    VALUES(0) = Slong_float;
    break;
  }
  RETURN(1);
}

Larray_rank(int narg, object a)
{
  check_arg(1);
  check_type_array(&a);
  VALUES(0) = (type_of(a) == t_array) ? MAKE_FIXNUM(a->a.a_rank)
				      : MAKE_FIXNUM(1);
  RETURN(1);
}

Larray_dimension(int narg, object a, object index)
{ int i;

  check_arg(2);
  check_type_array(&a);
  i = fixnnint(index);
  if (type_of(a) == t_array) {
    if (i >= a->a.a_rank)
      goto ILLEGAL;
    VALUES(0) = MAKE_FIXNUM(a->a.a_dims[i]);
  } else {
    if (i != 0)
      goto ILLEGAL;
    VALUES(0) = (type_of(a) == t_string) ?
      MAKE_FIXNUM(a->st.st_fillp)
	: MAKE_FIXNUM(a->v.v_dim);
  }
  RETURN(1);

 ILLEGAL:
  FEerror("~S is an illegal axis-number to the array ~S.",
	  2, index, a);
}

Larray_total_size(int narg, object a)
{
  check_arg(1);
  check_type_array(&a);
  VALUES(0) = MAKE_FIXNUM(a->a.a_dim);
  RETURN(1);
}

Ladjustable_array_p(int narg, object a)
{
  check_arg(1);
  check_type_array(&a);
  VALUES(0) = (a->a.a_adjustable) ? Ct : Cnil;
  RETURN(1);
}

/*
	Internal function for checking if an array is displaced.
*/
siLdisplaced_array_p(int narg, object a)
{
  check_arg(1);
  check_type_array(&a);
  VALUES(0) = (CAR(a->a.a_displaced) != Cnil) ? Ct : Cnil;
  RETURN(1);
}

Lsvref(int narg, object x, object index)
{ int i;

  check_arg(2);
  if (type_of(x) != t_vector ||
      x->v.v_adjustable ||
      x->v.v_hasfillp ||
      CAR(x->v.v_displaced) != Cnil ||
      (enum aelttype)x->v.v_elttype != aet_object)
    FEerror("~S is not a simple general vector.", 1, x);
  if ((i = fix(index)) >= x->v.v_dim)
    illegal_index(x, index);
  VALUES(0) = x->v.v_self[i];
  RETURN(1);
}

siLsvset(int narg, object x, object index, object v)
{ int i;

  check_arg(3);
  if (type_of(x) != t_vector ||
      x->v.v_adjustable ||
      x->v.v_hasfillp ||
      CAR(x->v.v_displaced) != Cnil ||
      (enum aelttype)x->v.v_elttype != aet_object)
    FEerror("~S is not a simple general vector.", 1, x);
  if ((i = fixnnint(index)) >= x->v.v_dim)
    illegal_index(x, index);
  VALUES(0) = x->v.v_self[i] = v;
  RETURN(1);
}

Larray_has_fill_pointer_p(int narg, object a)
{
  check_arg(1);
  check_type_array(&a);
  VALUES(0) = (type_of(a) == t_array) ? Cnil
    : (a->v.v_hasfillp) ? Ct : Cnil;
  RETURN(1);
}

Lfill_pointer(int narg, object a)
{
  check_arg(1);
  check_type_vector(&a);
  if (a->v.v_hasfillp) {
    VALUES(0) = MAKE_FIXNUM(a->v.v_fillp);
	RETURN(1);
	}
  else
    FEerror("The vector ~S has no fill pointer.", 1, a);
}

/*
	Internal function for setting fill pointer.
*/
siLfill_pointer_set(int narg, object a, object fp)
{ int i;

  check_arg(2);
  check_type_vector(&a);
  i = fixnnint(fp);
  if (a->v.v_hasfillp)
    if (i > a->v.v_dim)
      FEerror("The fill-pointer ~S is too large", 1, fp);
    else
      a->v.v_fillp = i;
  else
    FEerror("The vector ~S has no fill pointer.", 1, a);
  VALUES(0) = fp;
  RETURN(1);
}

/*
	Internal function for replacing the contents of arrays:

		(si:replace-array old-array new-array).

	Used in ADJUST-ARRAY.
*/
siLreplace_array(int narg, object old, object new)
{ object displaced, dlist;
  int diff;

  check_arg(2);

  if (type_of(old) != type_of(new))
    goto CANNOT;
  if (type_of(old) == t_array && old->a.a_rank != new->a.a_rank)
    goto CANNOT;
  if (!old->a.a_adjustable)
    FEerror("~S is not adjustable.", 1, old);
  diff = (int)(new->a.a_self) - (int)(old->a.a_self);
  dlist = CDR(old->a.a_displaced);
  displaced = CONS(CAR(new->a.a_displaced), dlist);
  check_displaced(dlist, old, new->a.a_dim);
  adjust_displaced(old, diff);
  undisplace(old);
  switch (type_of(old)) {
  case t_array:
  case t_vector:
  case t_bitvector:
    old->a = new->a;
    break;

  case t_string:
    old->st = new->st;
    break;

  default:
    goto CANNOT;
  }
  old->a.a_displaced = displaced;
  VALUES(0) = old;
  RETURN(1);

 CANNOT:
  FEerror("Cannot replace the array ~S by the array ~S.", 2, old, new);
}

init_array_function()
{
	make_constant("ARRAY-RANK-LIMIT", MAKE_FIXNUM(ARANKLIM));
	make_constant("ARRAY-DIMENSION-LIMIT", MAKE_FIXNUM(ADIMLIM));
	make_constant("ARRAY-TOTAL-SIZE-LIMIT", MAKE_FIXNUM(ATOTLIM));

	make_si_function("MAKE-PURE-ARRAY", siLmake_pure_array);
	make_si_function("MAKE-VECTOR", siLmake_vector);
	make_function("AREF", Laref);
	make_si_function("ASET", siLaset);
	make_function("ARRAY-ELEMENT-TYPE", Larray_element_type);
	make_function("ARRAY-RANK", Larray_rank);
	make_function("ARRAY-DIMENSION", Larray_dimension);
	make_function("ARRAY-TOTAL-SIZE", Larray_total_size);
	make_function("ADJUSTABLE-ARRAY-P", Ladjustable_array_p);
	make_si_function("DISPLACED-ARRAY-P", siLdisplaced_array_p);

	make_function("SVREF", Lsvref);
	make_si_function("SVSET", siLsvset);

	make_function("ARRAY-HAS-FILL-POINTER-P",
		      Larray_has_fill_pointer_p);
	make_function("FILL-POINTER", Lfill_pointer);
	make_si_function("FILL-POINTER-SET", siLfill_pointer_set);

	make_si_function("REPLACE-ARRAY", siLreplace_array);
}
