;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: chungl $
;;; $Source: /pic2/picasso/src/toolkit/resource/RCS/cursor.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 1992/04/06 01:07:28 $
;;;

(in-package "PT")

;;;
;;; cursor class
;;;

;; a cursor must be associated with a display
;; the display has a hash table in its 'cursor-table slot that contains
;; all cursors associated with the display.

(defclass cursor (pmc)
  ((instances
    :initform nil
    :type list
    :accessor instances
    :allocation :class)
   (ref-count
    :initform 0
    :type integer
    :reader ref-count) ;; not used at the moment.
   (name 
    :initarg :name
    :initform *default-cursor-name*
    :type string
    :reader name)
   (res
    :initform nil
    :type vector
    :accessor res)
   (foreground
    :initform "black"
    :initarg :foreground
    :type t
    :reader foreground)
   (background
    :initform "white"
    :initarg :background
    :type t
    :reader background)
   (file
    :initform "arrow.cursor"
    :initarg :file
    :accessor file)
   (mask-file
    :initform "arrow_mask.cursor"
    :initarg :mask-file
    :accessor mask-file)
   (image
    :initform nil
    :initarg :image
    :accessor image)
   (mask
    :initform nil
    :initarg :mask
    :accessor mask)
   (x-hot
    :initform nil
    :initarg :x-hot
    :accessor x-hot)
   (y-hot
    :initform nil
    :initarg :y-hot
    :accessor y-hot)
   (source-font
    :initform nil
    :initarg :source-font
    :accessor source-font)
   (mask-font
    :initform nil
    :initarg :mask-font
    :accessor mask-font)
   (source-index
    :initform nil
    :initarg :source-index
    :accessor source-index)
   (mask-index
    :initform nil
    :initarg :mask-index
    :accessor mask-index)
   (display
    :initform nil
    :initarg :display
    :type t
    :reader display)))

(defmethod (setf foreground) (val (self cursor) &aux color)
  (cond ((stringp val) 
	 (setq color (get-color val))
	 (if (not (color-p color))
	     (warn "cursor.setf.foreground: illegal color ~S~%" val)
	     (progn 
	      (when (attached-p self)
		    (do-detach (slot-value self 'foreground))
		    (do-attach color)
		    (change-cursor-color self))
	      (setf (slot-value self 'foreground) color))))
	((color-p val)
	 (when (attached-p self)
	       (do-detach (slot-value self 'foreground))
	       (do-attach color)
	       (change-cursor-color self))
	 (setf (slot-value self 'foreground) color))))

(defmethod (setf background) (val (self cursor) &aux color)
  (cond ((stringp val) 
	 (setq color (get-color val))
	 (if (not (color-p color))
	     (warn "cursor.setf.background: illegal color ~S~%" val)
	     (progn 
	      (when (attached-p self)
		    (do-detach (slot-value self 'background))
		    (do-attach color)
		    (change-cursor-color self))
	      (setf (slot-value self 'background) color))))
	((color-p val)
	 (when (attached-p self)
	       (do-detach (slot-value self 'background))
	       (do-attach color)
	       (change-cursor-color self))
	 (setf (slot-value self 'background) color))))

(defun change-cursor-color (cursor)
  (when (attached-p cursor)
	(xlib:recolor-cursor (res cursor) (res (foreground cursor))
			     (res (background cursor)))))

;;;
;;;	Lookup cursor in table
;;;

(defun get-cursor (&optional name spec &aux display)
  (if (null name)
      nil
      (progn
       (setq display
	     (cond ((window-p spec)
		    (display spec))
		   ((display-p spec)
		    spec)
		   (t (current-display))))
       (gethash name (cursor-table display)))))

;;;
;;; cursor initalization method
;;;

;; there are two ways to create a cursor: through a font, or through
;; bitmaps.
;; if :index or :source-index is specified, then the cursor is created
;; through a font, otherwise, through bitmaps.

(defun make-cursor (&rest args 
		    &key
		    (name *default-cursor-name*)
		    (image nil)
		    (file nil)
		    (font nil)
		    (index nil)
		    (source-font nil)
		    (source-index nil)
		    (bitmap-file nil)
		    (display (current-display))
		    &allow-other-keys)
  (if font
      (setq source-font font))
  (if index
      (setq source-index index))
  (when bitmap-file 
	(setq file bitmap-file)
	(setf (getf args :file) file))
  (cond ((and name (not (or image file source-index)))
	 (let ((c (get-cursor name display)))
	      (unless c 
		      (warn "make-cursor: no cursor named \'~s\' exists." name))
	      c))
	(name (apply #'make-instance 'cursor :allow-other-keys t args))
	(file
	 (setf (getf args :name) file)
	 (apply #'make-instance 'cursor :allow-other-keys t args))
	(image 
	 (unless (image-p image)
		 (error "make-cursor: invalid image \`~s\`." image))
	 (setf (getf args :name) (name image))
	 (apply #'make-instance 'cursor :allow-other-keys t args))
	(t
	 (get-cursor *default-cursor-name* display))))

(defmethod new-instance ((self cursor)
			 &key
			 (font nil)
			 (index nil)
			 (source-font nil) 
			 (source-index nil) 

			 (foreground "black")
			 (background "white")
			 (display nil)
			 &allow-other-keys)

  ;; a cursor must be associated with a valid display
  (unless (display-p display) 
	  (setq display (current-display)))
  (setf (slot-value self 'display) display)

  ;; these are the same thing, just different names
  (if font (setq source-font font))
  (if index (setq source-index index))

  ;; update the slots.
  (setf (source-font self) source-font)
  (setf (source-index self) source-index)

  ;; set foreground and background
  ;; to real color objects if the values passed in were strings.
  (when (stringp foreground)
	(let ((fg (get-color foreground display)))
	     (unless fg
		     (warn "cursor.new-instance: can't find color ~s. 
			   Resetting to black."
			   foreground)
		     (setq fg (get-color "black" display)))
	     (setq foreground fg)))
  (when (stringp background)
	(let ((bg (get-color background display))) 
	     (unless bg 
		     (warn "cursor.new-instance: can't find color ~s. Resetting to white."
			   background)
		     (setq bg (get-color "white" display)))
	     (setq background bg)))

  ;; i'm not sure why we need this below.
  (unless foreground (setq foreground (get-color "black" display)))
  (unless background (setq background (get-color "white" display)))

  ;; make sure they're valid
  (unless (color-p foreground) 
	  (warn "cursor.new-instance: invalid foreground color \`~S\`.  Resetting to black." foreground)
	  (setq foreground (get-color "white")))

  (unless (color-p background) 
	  (warn "cursor.new-instance: invalid background color \`~S\`.  Resetting to white." background)
	  (setq background (get-color "black")))

  ;; stick them back into the slot
  (setf (slot-value self 'foreground) foreground)
  (setf (slot-value self 'background) background)

  ;; register in hashtable
  (setf (gethash (name self) (cursor-table display)) self)

  ;; add to instances list
  (setf (instances self) (cons self (instances self)))

  ;; return self
  self)


(defmethod do-attach ((self cursor))
  (let ((foreground (foreground self))
	(background (background self))
	(si (source-index self))
	(mi (mask-index self))
	(sf (source-font self))
	(mf (mask-font self))
	temp
	
	(file (file self))
	(image (image self))
	(mask (mask self))
	(mask-file (mask-file self))
	(x-hot (x-hot self))
	(y-hot (y-hot self))
	pm 
	(rw (res (root-window)))
	cursor
	imagep
	maskp
	)
       
       (attach foreground)
       (setq foreground (res foreground))
       
       (attach background)
       (setq background (res background))
       
       
       
       ;; 
       ;; there are two ways of specifying a cursor:
       ;; 1. a font, and an index into the font
       ;; 2. bitmap files for the cursor and mask.
       ;;
       
       (cond (si
	      ;; cursor created from font
	      
	      ;; check source-font
	      (setq temp sf)
	      (cond ((null sf)
		     (setq sf (make-font :name "cursor")))
		    ((stringp sf) 
		     (setq sf (get-font sf))))
	      (if (null sf)
		  (error "make-cursor: bad source-font ~S" temp))
	      (if (not (attached-p sf))
		  (attach sf))
	      
	      ;; check mask-font
	      (setq temp mf)
	      (cond ((null mf) (setq mf sf))
		    ((stringp mf) (setq mf (get-font mf))))
	      (if (null mf)
		  (error "make-cursor: bad mask-font ~S" temp))
	      (if (not (attached-p mf))
		  (attach mf))
	      
	      ;; check source-index
	      (if (not (integerp si))
		  (error "make-cursor: bad source-index ~S" si))
	      
	      ;; check mask-index
	      (cond ((null mi)
		     (setq mi (1+ si)))
		    ((not (integerp mi)) 
		     (error "make-cursor: bad mask-index ~S" mi)))
	      
	      ;; finally make the cursor
	      (setq cursor 
		    (xlib:create-glyph-cursor
		     :source-font (res sf)
		     :mask-font (res mf)
		     :source-char si
		     :mask-char mi
		     :foreground foreground
		     :background background)))
	     
	     (t
	      ;; font created from images or bitmaps.

	      (when (image-p image) 
		    (setf imagep (if image t nil))
		    (attach image))
	      (when (image-p mask) 
       		    (setf maskp (if mask t nil))
		    (attach mask))
	      
	      (setq image
		    (if file
			(xlib:read-bitmap-file (find-library-file file))
			(res image)))
	      (setq pm (xlib:create-pixmap :width (xlib:image-width image)
					   :height (xlib:image-height image)
					   :depth 1 :drawable rw))
	      (setq mask
		    (cond (mask-file 
			   (xlib:image-pixmap 
			    pm 
			    (xlib:read-bitmap-file 
			     (find-library-file mask-file))))
			  ((image-p mask) (xlib:image-pixmap
					   pm (res mask)))
			  (t nil)))
	      
	      ;; register hotspots
	      (unless (integerp x-hot)
		      (setq x-hot (xlib:image-x-hot image)))
	      (unless (integerp y-hot)
		      (setq y-hot (xlib:image-y-hot image)))
	      
	      ;; coerce image to pixmap
	      (setq image (xlib:image-pixmap pm image))
	      
	      ;; set id slot
	      (setq cursor
		    (if mask
			(xlib:create-cursor :source image :mask mask
					    :x x-hot :y y-hot 
					    :foreground foreground 
					    :background background)
			(xlib:create-cursor :source image
					    :x x-hot :y y-hot 
					    :foreground foreground 
					    :background background)))
	      
	      ;; Free the pixmaps when necessary
	      (xlib:free-pixmap pm)
	      (unless imagep 
		      (xlib:free-pixmap image))
	      (when (and mask (not maskp))
		    (xlib:free-pixmap mask))))
       
       ;; set resource slot
       (setf (slot-value self 'res) cursor)
       
       )
  ;; return self
  self)

(defmethod do-detach ((self cursor))
  (when (attached-p self)
	(detach (foreground self))
	(detach (background self))
	(when (image-p (image self)) (detach (image self)))
	(when (image-p (mask self)) (detach (mask self)))
	(xlib:free-cursor (res self))
	(setf (res self) nil)
	)
  nil)

;;;
;;; 	Retrieves best size closest to desired width & height
;;;

(defun cursor-best-size (width height &optional (display (current-display)))
  (xlib:query-best-cursor width height (res display)))
