;;|=========================================================================|
;;|                         COPYRIGHT NOTICE                                |
;;|                                                                         |
;;|             Copyright 1990, 1991, 1992, 1993, Mark Tarver               |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun prooftool
       nil
       (prog (run-time assumptions conclusion answer)
             (setq *type-check* t)
             (setq *tactical-inferences* 0)
             (header)
             (initstacks)
             (setq *proofhistory* nil)
             (setq assumptions (enter-assumptions))
             (setq conclusion (enter-conclusion))
             (setq *problem* (list
                               (list assumptions
                                     (return-turnstile)
                                     conclusion)))
             (setq start-time (get-internal-run-time))
             (if *graphics*
                 (update-everything))
             (setq answer (enter-prooftool-loop *problem*))
             (setq stop-time (get-internal-run-time))
             (setq run-time (calibrate (- stop-time start-time)))
             (if *graphics*
                 (update-statistics))
             (if (eval '*stats*)
                 (stats run-time))
             (return answer)))

(defun calibrate
       (fp1)
       (/ fp1 1000000.0))

(defun stats
       (fp1)
       (format t "CPU time ~F secs~%" fp1)
       (format t
               "~D tactical inferences~%"
               *tactical-inferences*)
       (format t
               "~D TIPS~%"
               (round (/ *tactical-inferences* fp1))))

(defun stats+ nil (set '*stats* t))

(defun stats- nil (set '*stats* nil))

(defun enter-prooftool-loop
       (fp1)
       (if (catch 'success
                  (prooftool-loop fp1 1))
           'yes
           'no))

(defun rpt
       nil
       (cond ((boundp '*problem*)
              (prog (run-time answer)
                    (setq *type-check* t)
                    (setq *tactical-inferences* 0)
                    (setq *proofhistory* nil)
                    (header)
                    (initstacks)
                    (setq start-time (get-internal-run-time))
                    (if *graphics*
                        (update-everything))
                    (setq answer (enter-prooftool-loop *problem*))
                    (setq stop-time (get-internal-run-time))
                    (setq run-time (calibrate (- stop-time start-time)))
                    (if *graphics*
                        (update-statistics))
                    (if (eval '*stats*)
                        (stats run-time))
                    (return answer)))
             (t (raise "code 23: no previous problem"))))

(defun header
       nil
       (format t "~%~%SEQUEL Proof Tool ~%~%"))

(defun enter-assumptions
       nil
       (prog2 (format t "ASSUMPTIONS:- ~% ~%")
              (ea1 1
                   (enter-assumption 1)
                   nil)))

(defun ea1
       (fp1 fp2 fp3)
       (cond ((eq 'ok fp2) (external-vars (reverse fp3)))
             ((typed-expr fp2)
              (ea1 (1+ fp1)
                   (enter-assumption (1+ fp1))
                   (cons fp2 fp3)))
             (t
              (prog2 (warn "not a t-expr")
                     (ea1 fp1
                          (enter-assumption fp1)
                          fp3)))))

(defun external-vars
       (fp1)
       (mapcar 'create-internal-variables fp1))

(defun create-internal-variables
       (fp1)
       (cond ((var fp1) (create-internal-variable fp1))
             ((consp fp1) (mapcar 'create-internal-variables
                                  fp1))
             (t fp1)))

(defun create-internal-variable
       (fp1)
       (cond ((ends-in-question-mark fp1) (list '$
                                                (strip-question-mark fp1)))
             (t (list '$ fp1))))

(defun ends-in-question-mark
       (fp1)
       (equal (last (explode fp1))
              (list #\?)))

(defun strip-question-mark
       (fp1)
       (read-from-string (string-right-trim (list #\?)
                                            (format nil "~A" fp1))))

(defun external-vars-out
       (fp1)
       (mapcar 'create-external-variables fp1))

(defun create-external-variables
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '$ (car fp1)))
              (create-external-variable (cadr fp1)))
             ((consp fp1) (mapcar 'create-external-variables
                                  fp1))
             (t fp1)))

(defun create-external-variable
       (fp1)
       (cond ((member fp1 (list 'w 'x 'y 'z)) fp1)
             (t (concat fp1 '?))))

(defun enter-assumption
       (fp1)
       (prog (input)
             (format t "~A> " fp1)
             (setq input (read))
             (if (eq input 'ok)
                 (return 'ok)
                 (return (external-syntax-in input)))))

(defun typed-expr
       (fp1)
       (check-if-typed
         (catch 'delay
                (phi (prep-typecheck fp1)
                     't-expr
                     nil
                     nil))))

(defun check-if-typed
       (fp1)
       (cond ((eq 'delayed fp1) nil)
             (t fp1)))

(defun check-type-2-command
       (fp1)
       (check-if-typed
         (catch 'delay
                (phi (append (mapcar 'prep-typecheck fp1)
                             (list '&&proof))
                     'proof
                     (list (list '&&proof '* 'proof))
                     nil))))

(defun prep-typecheck
       (fp1)
       (remove-quotes (sequel-form1 fp1)))

(defun enter-conclusion
       nil
       (prog2 (format t " ~%CONCLUSION:- ~%~%?- ")
              (ec1 (external-syntax-in (read)))))

(defun ec1
       (fp1)
       (cond ((typed-expr fp1) (prog1 (external-vars fp1)
                                      (terpri)))
             (t (prog2 (warn "not a t-expr")
                       (enter-conclusion)))))

(defun prooftool-loop
       (fp1 fp2)
       (cond ((null fp1) t)
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'back (car fp1)))
              (clrhash *binding-array*)
              (prooftool-loop (go-back (cadr fp1))
                              (- fp2 (cadr fp1) 1)))
             ((consp fp1)
              (prog (command)
                    (print-sequents fp1 fp2 (length fp1))
                    (setq command (external-vars (read-command)))
                    (update-*proof-history* (list fp2 fp1 command))
                    (return
                      (prooftool-loop (execute-proof-command command fp1)
                                      (1+ fp2)))))
             (t (raise "code 13: No Patterns have Fired in prooftool-loop"))))

(defun print-sequents
       (fp1 fp2 fp3)
       (cond ((and (consp fp1)
                   (eval '*allsequents*))
              (mapcar #'(lambda (e)
                                (short-print-sequent e))
                      (reverse (cdr fp1)))
              (print-sequent (car fp1) fp2 fp3))
             ((consp fp1) (print-sequent (car fp1) fp2 fp3))
             (t (raise "code 13: No Patterns have Fired in print-sequents"))))

(defun short-print-sequent
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (null (car fp1))
                   (turnstile (cadr fp1)))
              (rptprint "-" 50)
              (format t "~%?- ")
              (print-formula
                (external-syntax-out (external-vars-out (caddr fp1))))
              (format t "~% ~%"))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (turnstile (cadr fp1)))
              (rptprint "-" 50)
              (format t "~%?- ")
              (print-formula
                (external-syntax-out (external-vars-out (caddr fp1))))
              (format t "~% ~%")
              (print-assumptions (car fp1)))
             (t
              (raise "code 13: No Patterns have Fired in short-print-sequent"))))

(defun update-*proof-history*
       (fp1)
       (push fp1 *proofhistory*))

(defun go-back
       (fp1)
       (cond ((equal 0 fp1) (second (pop *proofhistory*)))
             (t (pop *proofhistory*) (go-back (1- fp1)))))

(defun read-command
       nil
       (prog (command)
             (format t "~A>> " *framework*)
             (if *graphics*
                 (setq command (call-interface))
                 (setq command (lineread)))
             (if (eq (car command) 'lemma)
                 (setq command (list 'lemma
                                     (external-syntax-in (cadr command)))))
             (if (command-well-typed command)
                 (return command)
                 (prog2 (issue-type-warning command)
                        (return (read-command))))))

(defun issue-type-warning
       (fp1)
       (if *graphics*
           (alert)
           (format t
                   ";;; Warning: ~A is not well-typed~%"
                   fp1)))

(defun call-interface
       nil
       (update-statistics)
       (setq *buffer* nil)
       (main-event-loop)
       *buffer*)

(defun command-well-typed
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'abort (car fp1)))
              t)
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'back (car fp1)))
              t)
             ((and (consp fp1)
                   (type-2-tactic (car fp1)))
              (check-type-2-command fp1))
             ((and (consp fp1)
                   (type-1-tactic (car fp1)))
              (check-type-1-command fp1))
             ((and (consp fp1)
                   (type-0-tactic (car fp1)))
              t)
             (t
              (prog2 (format nil
                             ";;; Warning: ~A is not a tactic"
                             fp1)
                     nil))))

(defun is-tactic
       (fp1)
       (or (type-2-tactic fp1)
           (type-1-tactic fp1)
           (type-0-tactic fp1)))

(defun type-2-tactic (fp1)
  (or (member fp1 '(swap refine xtt rotate inst rewrite thin lemma))
      (member fp1 *tactics2*)))

(defun type-1-tactic
       (fp1)
       (member fp1 *tactics1*))

(defun type-0-tactic
       (fp1)
       (member fp1 *tactics0*))

(defun check-type-1-command
       (fp1)
       (check-if-typed
         (catch 'delay
                (phi (append fp1 (list nil))
                     'proof-object
                     nil
                     nil))))

(defun execute-proof-command
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'refine (car fp1)))
              (refine (cadr fp1) (caddr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'thin (car fp1)))
              (thin (cadr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'rewrite (car fp1)))
              (rewrite (cadr fp1) (caddr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'rewrite (car fp1)))
              (rewrite (cadr fp1)
                       (caddr fp1)
                       fp2
                       (caddr (cdr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'xtt (car fp1)))
              (xtt fp2))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'inst (car fp1)))
              (inst (cadr fp1) (caddr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'lemma (car fp1)))
              (lemma (cadr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'swap (car fp1)))
              (swap (cadr fp1) (caddr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'rotate (car fp1)))
              (rotate (cadr fp1) (caddr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'back (car fp1)))
              (list 'back (cadr fp1) fp2))
             ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'abort (car fp1)))
              (throw 'success nil))
             (t (apply-tactic fp1 fp2))))

(defun inst
       (fp1 fp2 fp3)
       (cond ((null fp3) nil)
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '$ (car fp1)))
              (subst fp2
                     (list '$ (cadr fp1))
                     fp3
                     ':test
                     'equal))
             (t fp3)))

(defun rotate (fp1 fp2 fp3)
  (declare (type integer fp1) (type integer fp2) (type list fp3))
  (the list (cond ((null fp3) nil)
                  ((and (>= (length fp3) fp1)
                        (>= (length fp3) fp2)
                        (> fp1 0)
                        (> fp2 0))
                        (+tips) (exchange fp1 fp2 fp3))
                  (t (+tips) fp3))))

(defun repseq
       (fp1 fp2)
       (cond ((null fp2) nil)
             ((consp fp2) (cons fp1 (cdr fp2)))
             (t (raise "code 13: No Patterns have Fired in repseq"))))

(defun swap
       (fp1 fp2 fp3)
       (cond ((null fp3) nil)
             ((equal fp2 fp1) fp3)
             ((and (consp fp3)
                   (consp (car fp3))
                   (= (length (car fp3)) 3)
                   (turnstile (cadar fp3))
                   (> fp1 0)
                   (> fp2 0)
                   (>= (length (caar fp3))
                       fp1)
                   (>= (length (caar fp3)) fp2))
              (+tips)
              (cons
                (list (exchange fp1 fp2 (caar fp3))
                      (cadar fp3)
                      (caddr (car fp3)))
                (cdr fp3)))
             (t fp3)))

(defun exchange
       (fp1 fp2 fp3)
       (cond ((and (equal 1 fp1) (consp fp3))
              (cons (nth (1- fp2) fp3)
                    (put-first (1- fp2)
                               (car fp3)
                               (cdr fp3))))
             ((and (equal 1 fp2) (consp fp3))
              (cons (nth (1- fp1) fp3)
                    (put-first (1- fp1)
                               (car fp3)
                               (cdr fp3))))
             ((consp fp3)
              (cons (car fp3)
                    (exchange (1- fp1)
                              (1- fp2)
                              (cdr fp3))))
             (t (raise "code 13: No Patterns have Fired in exchange"))))

(defun put-first
       (fp1 fp2 fp3)
       (cond ((and (equal 1 fp1) (consp fp3)) (cons fp2 (cdr fp3)))
             ((consp fp3)
              (cons (car fp3)
                    (put-first (1- fp1) fp2 (cdr fp3))))
             (t (raise "code 13: No Patterns have Fired in put-first"))))

(defun rewrite-dispatch
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (null (caddr fp1)))
              nil)
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (null (caddr (cdr fp1))))
              nil)
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (consp (caddr fp1)))
              (+tips)
              (cons (rewrite1 (car fp1)
                              (cadr fp1)
                              (caadr (cdr fp1)))
                    (cdadr (cdr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (consp (caddr (cdr fp1))))
              (+tips)
              (cons
                (rewrite3 (car fp1)
                          (cadr fp1)
                          (caadr (cddr fp1))
                          (caddr fp1))
                (cdadr (cddr fp1))))
             (t (raise "code 13: No Patterns have Fired in rewrite-dispatch"))))

(defun rewrite1
       (fp1 fp2 fp3)
       (cond ((and (equal 0 fp2)
                   (consp fp3)
                   (= (length fp3) 3)
                   (turnstile (cadr fp3)))
              (list (car fp3)
                    (cadr fp3)
                    (funcall fp1 (caddr fp3))))
             ((and (consp fp3)
                   (= (length fp3) 3)
                   (turnstile (cadr fp3)))
              (list (rewrite2 fp1 fp2 (car fp3))
                    (cadr fp3)
                    (caddr fp3)))
             (t (raise "code 13: No Patterns have Fired in rewrite1"))))

(defun rewrite2
       (fp1 fp2 fp3)
       (cond ((null fp3) nil)
             ((and (equal 1 fp2) (consp fp3))
              (cons (funcall fp1 (car fp3))
                    (cdr fp3)))
             ((consp fp3)
              (cons (car fp3)
                    (rewrite2 fp1 (1- fp2) (cdr fp3))))
             (t (raise "code 13: No Patterns have Fired in rewrite2"))))

(defun rewrite3
       (fp1 fp2 fp3 fp4)
       (cond ((and (equal 0 fp2)
                   (consp fp3)
                   (= (length fp3) 3)
                   (turnstile (cadr fp3)))
              (list (car fp3)
                    (cadr fp3)
                    (funcall fp1 (caddr fp3) fp4)))
             ((and (consp fp3)
                   (= (length fp3) 3)
                   (turnstile (cadr fp3)))
              (list (rewrite4 fp1 fp2 (car fp3) fp4)
                    (cadr fp3)
                    (caddr fp3)))
             (t (raise "code 13: No Patterns have Fired in rewrite3"))))

(defun rewrite4
       (fp1 fp2 fp3 fp4)
       (cond ((null fp3) nil)
             ((and (equal 1 fp2) (consp fp3))
              (cons (funcall fp1 (car fp3) fp4)
                    (cdr fp3)))
             ((consp fp3)
              (cons (car fp3)
                    (rewrite4 fp1 (1- fp2) (cdr fp3) fp4)))
             (t (raise "code 13: No Patterns have Fired in rewrite4"))))

(defun +tips
       nil
       (incf *tactical-inferences*))

(defun thin
       (fp1 fp2)
       (cond ((consp fp2) (cons (thin1 (car fp2) fp1)
                                (cdr fp2)))
             (t fp2)))

(defun thin1
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (turnstile (cadr fp1)))
              (+tips)
              (list (remove-nth (car fp1) fp2)
                    (cadr fp1)
                    (caddr fp1)))
             (t (raise "code 13: No Patterns have Fired in thin1"))))

(defun remove-nth
       (fp1 fp2)
       (cond ((null fp1) nil)
             ((and (consp fp1) (equal 1 fp2)) (cdr fp1))
             ((consp fp1)
              (cons (car fp1)
                    (remove-nth (cdr fp1) (1- fp2))))
             (t (raise "code 13: No Patterns have Fired in remove-nth"))))

(defun lemma
       (fp1 fp2)
       (cond ((null fp2) nil)
             ((and (consp fp2)
                   (consp (car fp2))
                   (= (length (car fp2)) 3)
                   (turnstile (cadar fp2)))
              (+tips)
              (cons (list (caar fp2) (cadar fp2) fp1)
                    (cons
                      (list (cons fp1 (caar fp2))
                            (cadar fp2)
                            (caddr (car fp2)))
                      (cdr fp2))))
             (t (+tips) fp2)))

(defun apply-tactic
       (fp1 fp2)
       (apply-tactic1 fp1 fp2))

(defun apply-tactic1
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (= (length fp1) 1))
              (funcall (car fp1) fp2))
             ((consp fp1) (apply (car fp1)
                                 (append (cdr fp1)
                                         (list fp2))))
             (t (raise "code 13: No Patterns have Fired in apply-tactic1"))))

(defun xtt
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 3)
                   (and (consp (caddr (car fp1)))
                        (= (length (caddr (car fp1)))
                           3))
                   (eq '*
                       (cadar (cddar fp1)))
                   (turnstile (cadar fp1)))
              (+tips)
              (if (can-solve-by-phi (caar fp1)
                                    (caadr (cdar fp1))
                                    (caddr (caddr (car fp1))))
                  (deref-sequents (cdr fp1))
                  (cons
                    (list (caar fp1)
                          (cadar fp1)
                          (list (caadr (cdar fp1))
                                '*
                                (caddr (caddr (car fp1)))))
                    (cdr fp1))))
             (t (+tips) fp1)))

(defun can-solve-by-phi
       (fp1 fp2 fp3)
       (prog (result)
             (set '*auto* t)
             (setq result (catch 'delay
                                 (phi fp2
                                      fp3
                                      (rewrite-delta fp1)
                                      nil)))
             (set '*auto* nil)
             (if (eq result t)
                 (return t)
                 (return nil))))

(defun rewrite-delta
       (fp1)
       (cond ((null fp1) nil)
             ((consp fp1)
              (prog (temp)
                    (setq temp (rewrite-auto (car fp1)))
                    (if (equal (car fp1) temp)
                        (return (cons (car fp1)
                                      (rewrite-delta (cdr fp1))))
                        (return (rewrite-delta (append temp
                                                       (cdr fp1)))))))
             (t (raise "code 13: No Patterns have Fired in rewrite-delta"))))

(defun deftactic1
       (fp1)
       (cond ((consp fp1) (define1 fp1) (add-tactic (car fp1)))
             (t (raise "code 13: No Patterns have Fired in deftactic1"))))

(defun add-tactic
       (fp1)
       (add-tactic1 fp1 (signature fp1)))

(defun add-tactic1
       (fp1 fp2)
       (cond ((sound fp2)
              (setq *tactics0* (remove fp1 *tactics0*))
              (setq *tactics1* (remove fp1 *tactics1*))
              (format t
                      "Sound Tactic (type 2): ~A~%"
                      fp1)
              (pushnew fp1 *tactics2*)
              (setq *tactics2* (sort *tactics2* '<-symbol))
              fp1)
             ((syntactically-correct fp2)
              (setq *tactics0* (remove fp1 *tactics0*))
              (setq *tactics2* (remove fp1 *tactics2*))
              (format t
                      "Syntax Preserving Tactic (type 1): ~A~%"
                      fp1)
              (pushnew fp1 *tactics1*)
              (setq *tactics1* (sort *tactics1* '<-symbol))
              fp1)
             (t
              (format t
                      "Unvalidated Tactic (type 0): ~A~%"
                      fp1)
              (setq *tactics1* (remove fp1 *tactics1*))
              (setq *tactics2* (remove fp1 *tactics2*))
              (pushnew fp1 *tactics0*)
              (setq *tactics0* (sort *tactics0* '<-symbol))
              fp1)))

(defun <-symbol
       (fp1 fp2)
       (not
         (null (string-lessp (format nil "~A" fp1)
                             (format nil "~A" fp2)))))

(defun document-tactic
       (fp1 fp2)
       (cond ((and (symbolp fp1)
                   (stringp fp2))
              (put-prop fp1 'doc fp2))
             (t nil)))

(defun document-theory
       (fp1 fp2)
       (cond ((and (symbolp fp1)
                   (stringp fp2))
              (put-prop fp1 'doc fp2))
             (t nil)))

(defun get-document
       (fp1)
       (if (member fp1 *theories*)
           (format (not *graphics*)
                   "Theory: ~A~%Information: ~A"
                   fp1
                   (get-prop fp1 'doc nil))
           (format (not *graphics*)
                   "Tactic: ~A~%Signature: ~A~%Information: ~A"
                   fp1
                   (signature fp1)
                   (get-prop fp1 'doc nil))))

(defun sound
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'proof (car fp1))
                   (eq '-> (cadr fp1))
                   (eq 'proof (caddr fp1)))
              t)
             ((consp fp1) (sound (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in sound"))))

(defun syntactically-correct
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'proof-object (car fp1))
                   (eq '-> (cadr fp1))
                   (eq 'proof-object (caddr fp1)))
              t)
             ((consp fp1) (syntactically-correct (cdr fp1)))
             (t
              (raise "code 13: No Patterns have Fired in syntactically-correct"))))

(defun print-sequent
       (fp1 fp2 fp3)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (null (car fp1))
                   (turnstile (cadr fp1)))
              (rptprint "=" 50)
              (if (> fp3 0)
                  (format t
                          "~%Step ~A [~A]~% ~%?- "
                          fp2
                          fp3)
                  (format t "~%Step ~A ~% ~%?- " fp2))
              (print-formula
                (external-syntax-out (external-vars-out (caddr fp1))))
              (format t "~% ~%"))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (turnstile (cadr fp1)))
              (rptprint "=" 50)
              (if (> fp3 0)
                  (format t
                          "~%Step ~A [~A] ~% ~%?- "
                          fp2
                          fp3)
                  (format t "~%Step ~A ~% ~%?- " fp2))
              (print-formula
                (external-syntax-out (external-vars-out (caddr fp1))))
              (format t "~% ~%")
              (print-assumptions (car fp1)))
             (t (raise "code 13: No Patterns have Fired in print-sequent"))))

(defun rptprint
       (fp1 fp2)
       (cond ((equal 0 fp2) 0)
             (t (write-string fp1) (rptprint fp1 (1- fp2)))))

(defun pprint-sequent
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (null (car fp1))
                   (turnstile (cadr fp1)))
              (print-formula
                (external-syntax-out (external-vars-out (caddr fp1))))
              (format t "~% ~%"))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (turnstile (cadr fp1)))
              (print-formula
                (external-syntax-out (external-vars-out (caddr fp1))))
              (format t "~% ~%")
              (print-assumptions (car fp1)))
             (t (raise "code 13: No Patterns have Fired in pprint-sequent"))))

(defun print-assumptions (fp1) (pa1 fp1 1))

(defun pa1
       (fp1 fp2)
       (cond ((null fp1) (terpri))
             ((consp fp1)
              (format t "~A. " fp2)
              (print-formula
                (external-syntax-out (external-vars-out (car fp1))))
              (terpri)
              (pa1 (cdr fp1) (1+ fp2)))
             (t (raise "code 13: No Patterns have Fired in pa1"))))

(defun print-formula
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq '* (cadr fp1)))
              (write-expr (car fp1))
              (write-string " * ")
              (write (caddr fp1) :pretty t))
             (t (write fp1 :pretty t))))

(defun write-expr
       (fp1)
       (write fp1 ':pretty t))

(defun refine
       (fp1 fp2 fp3)
       (cond ((null fp3) (throw 'success t))
             ((consp fp3)
              (+tips)
              (refine2 (car fp3)
                       (do-refinement fp1
                                      fp2
                                      (car fp3)
                                      (append (cdr fp3)
                                              (list '$)))
                       (cdr fp3)))
             (t (raise "code 13: No Patterns have Fired in refine"))))

(defun do-refinement
       (fp1 fp2 fp3 fp4)
       (cond ((and (consp fp3)
                   (= (length fp3) 3)
                   (and (consp (caddr fp3))
                        (= (length (caddr fp3))
                           3)))
              (prog (l-expr)
                    (setq l-expr (nth (1- fp2)
                                      (get-prop fp1 'inter nil)))
                    (if (null l-expr)
                        (return nil))
                    (return
                      (funcall l-expr
                               (car fp3)
                               (caadr (cdr fp3))
                               (caddr (caddr fp3))
                               fp4))))
             (t (raise "code 13: No Patterns have Fired in do-refinement"))))

(defun refine2
       (fp1 fp2 fp3)
       (cond ((null fp2) (cons fp1 fp3))
             ((and (consp fp2)
                   (= (length fp2) 1)
                   (eq '$ (car fp2)))
              (throw 'success t))
             (t (butlast fp2))))

(defun deref-sequents
       (fp1)
       (prog1 (deref fp1)
              (clrhash *binding-array*)))

(defun divert-proof
       (fp1)
       (dribble fp1)
       (why)
       (dribble)
       (list 'proof 'diverted))

(defun why
       nil
       (prog2 (mapcar 'print-proof-step
                      (reverse *proofhistory*))
              'q.e.d.))

(defun print-proof-step
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (consp (cadr fp1)))
              (print-sequent (caadr fp1)
                             (car fp1)
                             (length (car (cdr fp1))))
              (format t "~A>> " *framework*)
              (lineprint (caddr fp1)))
             (t (raise "code 13: No Patterns have Fired in print-proof-step"))))

(defun lineprint
       (fp1)
       (cond ((null fp1) (terpri))
             ((consp fp1) (format t "~A " (car fp1)) (lineprint (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in lineprint"))))

(defun get-theory-length
       (fp1)
       (length (get-prop fp1 'inter nil)))

(defun infs nil *tactical-inferences*)

(defun unsolved-nodes (fp1) (length fp1))

(defun external-syntax-out (fp1) fp1)

(defun external-syntax-in (fp1) fp1)

(defun newterm
       nil
       (newterm1 'a
                 (eval '*proofhistory*)))

(defun newterm1
       (fp1 fp2)
       (cond ((and (eigen fp1 fp2)
                   (not (member fp1 *constants*)))
              fp1)
             (t (newterm1 (nterm-succ fp1) fp2))))

(defun nterm-succ
       (fp1)
       (cond ((eq 'a fp1) 'b)
             ((eq 'b fp1) 'c)
             ((eq 'c fp1) 'd)
             ((eq 'd fp1) 'e)
             ((eq 'e fp1) 'f)
             ((eq 'f fp1) 'g)
             ((eq 'g fp1) 'h)
             ((eq 'i fp1) 'j)
             ((eq 'j fp1) 'k)
             ((eq 'k fp1) 'l)
             ((eq 'l fp1) 'm)
             ((eq 'm fp1) 'n)
             ((eq 'n fp1) 'o)
             ((eq 'o fp1) 'p)
             ((eq 'p fp1) 'q)
             ((eq 'q fp1) 'r)
             ((eq 'r fp1) 's)
             ((eq 's fp1) t)
             ((equal t fp1) 'u)
             ((eq 'u fp1) 'v)
             (t (gentemp))))

(defun clone (fp1) (clone1 fp1 fp1))

(defun clone1
       (fp1 fp2)
       (cond ((logical-var fp1) (newvar))
             ((and (consp fp1)
                   (logical-var (car fp1)))
              (clone1 (cdr fp1)
                      (subst (newvar)
                             (car fp1)
                             fp2
                             ':test
                             'equal)))
             ((and (consp fp1)
                   (consp (car fp1)))
              (clone1 (append (car fp1) (cdr fp1))
                      fp2))
             ((consp fp1) (clone1 (cdr fp1) fp2))
             (t fp2)))
