;;|=========================================================================|
;;|                           COPYRIGHT NOTICE                              |
;;|                                                                         |
;;|                   Copyright 1992, Mark Tarver                           |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun read-file
       (fp1)
       (if (not (probe-file (my-home-directory fp1)))
           (progn (warn "~A does not exist" fp1) (restart)))
       (with-open-file (in-stream (my-home-directory fp1) 
                        :direction :input)
                       (prog (input output)
                             loop 
                             (setq input (read in-stream nil 'eof))
                             (if (eq input 'eof)
                                 (return (nreverse output))
                                 (push input output))
                             (go loop))))

(defun my-home-directory (filename)
  (declare (special *toplevel-directory*)
           (type string *toplevel-directory*))
   (if (equal (car (coerce filename 'list)) #\~)
       (format nil "~A~A" *toplevel-directory*
           (coerce (cdr (coerce filename 'list)) 'string))
       filename))

(defun consult (fp1)
  (prog (contents)
        (setq contents (evaluate-reserved-first
                         (mapcar 'sequel-form
                                 (mapcar 'list
                                         (read-file fp1)))))
        (overwrite-definitions contents)
        (overwrite-signatures contents)
        (if *type-check*
            (propagate *definitions* 'no)
            (mapcar 'eval contents))
        (print-delayed-functions)
        (return (list fp1 'consulted))))

(defun evaluate-reserved-first
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (reserved-expr (car fp1)))
              (eval (car fp1)) (evaluate-reserved-first (cdr fp1)))
             ((consp fp1)
              (cons (car fp1)
                    (evaluate-reserved-first (cdr fp1))))
             (t
              (raise
                "code 13: No Patterns have Fired in evaluate-reserved-first"))))

(defun reserved-expr
       (fp1)
       (cond ((and (consp fp1)
                   (member (car fp1)
                           (list 'define
                                 'defrew
                                 'deftactic 
                                 'mutual)))
              nil)
             ((consp fp1) (reserved (car fp1)))
             (t nil)))

(defun propagate (fp1 fp2)
  (cond ((and (null fp1) (eq 'no fp2)) t)
        ((and (null fp1) (eq 'yes fp2))
         (if (consp *definitions*)
             (propagate *definitions* 'no)
             t))
        ((consp fp1)
         (if (not (member (typecheck (car fp1) (type-var (car fp1))) 
                    '(nil delayed)))
             (progn
               (setq *definitions* (remove (car fp1)
                                           *definitions*
                                           ':test
                                           'equal))
               (eval (car fp1))
               (propagate (cdr fp1) 'yes))
             (propagate (cdr fp1) fp2)))
        (t (raise "code 13: No Patterns have Fired in propagate"))))

(defun overwrite-definitions
       (fp1)
       (mapcar 'overwrite-definition fp1))

(defun overwrite-definition
       (fp1)
       (prog (temp)
             (if (not *type-check*)
                 (return))
             (setq temp (find-same-def fp1 *definitions*))
             (if temp
                 (nsubst fp1 temp *definitions* ':test 'equal)
                 (push fp1 *definitions*))))

(defun find-same-def
       (fp1 fp2)
       (cond ((null fp2) nil)
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'define (car fp1))
                   (consp fp2)
                   (and (consp (car fp2))
                        (> (length (car fp2)) 1))
                   (eq 'define (caar fp2))
                   (equal (cadar fp2) (cadr fp1)))
              (cons 'define
                    (cons (cadr fp1) (cddar fp2))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'defrew (car fp1))
                   (consp fp2)
                   (and (consp (car fp2))
                        (> (length (car fp2)) 1))
                   (eq 'defrew (caar fp2))
                   (equal (cadar fp2) (cadr fp1)))
              (cons 'defrew
                    (cons (cadr fp1) (cddar fp2))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'deftactic (car fp1))
                   (consp fp2)
                   (and (consp (car fp2))
                        (> (length (car fp2)) 1))
                   (eq 'deftactic (caar fp2))
                   (equal (cadar fp2) (cadr fp1)))
              (cons 'deftactic
                    (cons (cadr fp1) (cddar fp2))))
             ((consp fp2) (find-same-def fp1 (cdr fp2)))
             (t (raise "code 13: No Patterns have Fired in find-same-def"))))

(defun overwrite-signatures (fp1)
   (mapcar 'overwrite-signature fp1))

(defun overwrite-signature
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'define (car fp1)))
              (setf (gethash (cadr fp1) *sfht*)
                    nil))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'deftactic (car fp1)))
              (setf (gethash (cadr fp1) *sfht*)
                    nil))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'defrew (car fp1)))
              (setf (gethash (cadr fp1) *sfht*)
                    nil))
             (t t)))

(defun print-delayed-functions
       nil
       (pdf1 (remove-if 'signature
                        (mapcar 'cadr *definitions*))))

(defun signature
       (fp1)
       (cond ((symbolp fp1) (gethash fp1 *sfht*))
             (t nil)))

(defun pdf1
       (fp1)
       (cond ((null fp1) t)
             (t
              (format t
                      "~%The following functions are still delayed:-~%~%")
              (mapcar 'print-delayed-function fp1)
              (terpri))))

(defun print-delayed-function
       (fp1)
       (format t "~A~%" fp1))

(defun sequel-form-if-needed
       (fp1)
       (cond ((and (consp fp1)
                   (reserved (car fp1)))
              fp1)
             (t (sequel-form1 (first (bracket (list fp1)))))))

(defun divert
       (fp1)
       (setq stream (open (concatenate 'string fp1 ".out")
                          :direction
                          :output
                          :if-exists
                          :overwrite
                          :if-does-not-exist
                          :create))
       (setq *outfile* t)
       (consult fp1)
       (setq *outfile* nil)
       (close stream)
       (list 'code
             'diverted
             'to
             (concatenate 'string fp1 ".out")))

(defun reserved
       (fp1)
       (cond (t (member fp1 *reserved*))
             (t nil)))

(defun sequel-form1
       (fp1)
       (cond ((selfeval fp1) fp1)
             ((atom fp1) (list 'quote fp1))
             ((hasdbar fp1)
              (cons 'append
                    (mapcar 'sequel-form1
                            (remove '<> fp1))))
             ((hasbar fp1) (sequel-form2 fp1))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (atsign (car fp1)))
              (cons (cadr fp1)
                    (mapcar 'sequel-form1 (cddr fp1))))
             ((and (consp fp1)
                   (eval '*type-check*))
              (list 'cons
                    (sequel-form1 (car fp1))
                    (sequel-form1 (cdr fp1))))
             ((consp fp1) (cons 'list
                                (mapcar 'sequel-form1 fp1)))
             (t (raise "code 13: No Patterns have Fired in sequel-form1"))))

(defun sequel-form2 (fp1)
  (cond ((> (length fp1) 2)
         (insert-conses (mapcar 'sequel-form1 (remove '\| fp1))))
        (t (raise "code 13: No Patterns have Fired in sequel-form2"))))

